#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""dnat extension"""

import abc
import six
from neutron import manager
from neutron.api.v2 import base
from neutron.api import extensions
from neutron.api.extensions import ResourceExtension
try:
    from neutron.common.exceptions import NotFound, Conflict, InUse
except ImportError:
    from neutron_lib.exceptions import NotFound, Conflict, InUse
try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase
try:
    from neutron_lib.plugins import directory
except ImportError:
    pass
try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor
from networking_huawei._i18n import _
from networking_huawei.drivers.ac.common import constants
from networking_huawei.drivers.ac.extensions import dnat as extension
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

extensions.append_api_extensions_path(extension.__path__)


class PluginNotFound(NotFound):
    """Plugin Not Found"""
    message = _("Plugin %(plugin)s could not be found.")


class DNATNotFound(NotFound):
    """DNAT Not Found"""
    message = _("DNAT %(id)s could not be found.")


class RouterForPortNotFound(NotFound):
    """Router For Port Not Found"""
    message = _("Router for port %(id)s could not be found.")


class SubnetForPortNotFound(NotFound):
    """Subnet For Port Not Found"""
    message = _("Subnet for port %(id)s could not be found.")


class DNATConflict(Conflict):
    """DNAT Conflict"""
    message = _("The DNAT %(id)s is already exist.")


class FixedIPForDNATConflict(Conflict):
    """Fixed IP For DNAT Conflict"""
    message = _("The fixed IP %(ip)s does not match with port %(id)s.")


class RouterForDNATConflict(Conflict):
    """Router For DNAT Conflict"""
    message = _("The router %(router)s does not match with DNAT %(dnat)s.")


class FloatingIPInUse(InUse):
    """Floating IP In Use"""
    message = _("Unable to create DNAT. The floatingip %(id)s is in use.")


DNAT = 'dnat'
DNATS = '%ss' % DNAT
TENANT_ID_MAX_LEN = 255
PORT_RANGE = [1, 65535]

RESOURCE_ATTRIBUTE_MAP = {
    DNATS: {
        'id': {
            'allow_post': False, 'allow_put': False,
            'validate': {'type:uuid': None},
            'is_visible': True,
            'primary_key': True
        },
        'tenant_id': {
            'allow_post': True, 'allow_put': False,
            'required_by_policy': False,
            'validate': {'type:string': TENANT_ID_MAX_LEN},
            'is_visible': True
        },
        'floating_ip_id': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:uuid': None},
            'is_visible': True
        },
        'router_id': {
            'allow_post': False, 'allow_put': False,
            'validate': {'type:uuid': None},
            'is_visible': True,
            'default': None
        },
        'port_id': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:uuid': None},
            'is_visible': True
        },
        'fixed_ip_address': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:ip_address': None},
            'is_visible': True
        },
        'protocol': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:values': ['tcp', 'udp']},
            'is_visible': True
        },
        'floating_ip_port': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:range': PORT_RANGE},
            'is_visible': True
        },
        'fixed_ip_port': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:range': PORT_RANGE},
            'is_visible': True
        },
        'status': {
            'allow_post': False, 'allow_put': False,
            'validate': {'type:string': TENANT_ID_MAX_LEN},
            'is_visible': True,
            'default': None
        },
    }
}


class Dnat(ExtensionDescriptor):
    """Dnat extension class"""
    @classmethod
    def get_name(cls):
        """get name"""
        return DNAT

    @classmethod
    def get_alias(cls):
        """get alias"""
        return DNAT

    @classmethod
    def get_description(cls):
        """get description"""
        return DNAT

    @classmethod
    def get_namespace(cls):
        """get namespace"""
        return ''

    @classmethod
    def get_updated(cls):
        """get udated"""
        return '2018-02-22T12:00:00-00:00'

    @classmethod
    def get_resources(cls):
        """get resources"""
        ops_version = ncu.get_ops_version()
        if ops_version in [constants.OPS_O, constants.OPS_P, constants.OPS_Q,
                           constants.OPS_R, constants.OPS_T, constants.OPS_W,
                           constants.FSP_21_0]:
            plugin = directory.get_plugin(DNAT)
        else:
            plugins = manager.NeutronManager.get_service_plugins()
            plugin = plugins[DNAT]

        resources = []
        for resource_name in [DNAT]:
            collection_name = '{}s'.format(resource_name)
            params = RESOURCE_ATTRIBUTE_MAP.get(collection_name, dict())
            member_actions = {}
            controller = base.create_resource(
                collection_name, resource_name, plugin, params,
                allow_bulk=True,
                member_actions=member_actions,
                allow_pagination=True,
                allow_sorting=True)
            resource = ResourceExtension(
                collection_name, controller,
                member_actions=member_actions,
                attr_map=params)
            resources.append(resource)
        return resources


@six.add_metaclass(abc.ABCMeta)
class DnatBase(ServicePluginBase):
    """Dnat Base"""
    def get_plugin_name(self):
        """get plugin name"""
        pass

    def get_plugin_type(self):
        """get plugin type"""
        pass

    def get_plugin_description(self):
        """get plugin description"""
        pass

    @abc.abstractmethod
    def create_dnat(self, context, dnat):
        """create dnat"""
        pass

    @abc.abstractmethod
    def delete_dnat(self, context, id):
        """delete dnat"""
        pass

    @abc.abstractmethod
    def get_dnat(self, context, id, fields=None):
        """get dnat"""
        pass

    @abc.abstractmethod
    def get_dnats(self, context, filters=None, fields=None):
        """get dnats"""
        pass
