#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""resource sync extension"""

import abc
import six
from oslo_config import cfg

try:
    from neutron_lib.api import validators
except ImportError:
    validators = None
try:
    from neutron.api.v2 import attributes
except ImportError:
    attributes = None
from neutron import manager
from neutron.api.extensions import ResourceExtension
try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor
from neutron.api.v2 import base
try:
    from neutron.api.v2.attributes import convert_to_boolean
except ImportError:
    from neutron_lib.api.converters import convert_to_boolean
try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase
try:
    from neutron_lib.plugins import directory
except ImportError:
    pass
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu


LOG = ncu.ac_log.getLogger(__name__)

RESOURCE_SYNC = 'resourcesync'
RESOURCE_SYNCS = '%ss' % RESOURCE_SYNC
MAX_LENGTH_FILE_NAME = 255


def _validate_list_of_res_type(data, valid_values=None):
    LOG.debug("validate list_of_res_type :%s", data)
    msg = "'%s' is not a valid list" % data
    if not data:
        return None
    if not isinstance(data, list):
        return msg
    for res_type in data:
        if res_type not in ac_const.SYNC_TYPE_LIST:
            return msg
    return None


if validators and hasattr(validators, 'add_validator') and \
        'type:list_of_res_type' not in validators.validators:
    validators.add_validator('list_of_res_type', _validate_list_of_res_type)

if attributes and hasattr(attributes, 'validators') and \
        'type:list_of_res_type' not in attributes.validators:
    attributes.validators['type:list_of_res_type'] = _validate_list_of_res_type

# Attribute Map
RESOURCE_ATTRIBUTE_MAP = {
    'resourcesyncs': {
        'server': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None
        },
        'state': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None
        },
        'sync_res': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None
        },
        'sync_start_time': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None
        },
        'expected_completion_time': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': 0
        },
        'tenant_id': {
            'allow_post': True, 'allow_put': False,
            'required_by_policy': True,
            'is_visible': True, 'default': None
        },
        'sync_data': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {
                'type:values': [None, 'exclude-delete', 'include-delete']
            }
        },
        'compare_data': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {
                'type:dict_or_nodata': {
                    'overwrite': {
                        'type:boolean': None,
                        'required': False,
                        'convert_to': convert_to_boolean},
                    'filename': {
                        'type:not_empty_string': MAX_LENGTH_FILE_NAME,
                        'required': False}
                }
            },
        },
        'sync_resources': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None
        },
        'include_bindports': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': False,
        },
        'resource_type': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:list_of_res_type': None},
        },
        'sync_segments': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
        },
    }
}


class Resource_sync(ExtensionDescriptor):
    """Perform neutron sync through cli"""

    @classmethod
    def get_name(cls):
        """Name for this extension"""
        return "Resource sync"

    @classmethod
    def get_alias(cls):
        """core_plugin class to load the extension"""
        return 'resourcesync'

    @classmethod
    def get_description(cls):
        """A small description about this extension"""
        return "Perform neutron sync through cli"

    @classmethod
    def get_namespace(cls):
        """The XML namespace for this extension"""
        return "http://docs.openstack.org/ext/neutron/" \
               "resourcesync/api/v2.0"

    @classmethod
    def get_updated(cls):
        """Specify when was this extension last updated"""
        return "2017-02-8T19:48:50-00:00"

    @classmethod
    def get_resources(cls):
        """Returns Ext Resources."""

        ops_version = ncu.get_ops_version()

        if ops_version in ac_const.OPS_VERSION_O_PQRTW_6_21:
            plugin = directory.get_plugin(RESOURCE_SYNC)
        else:
            plugin = \
                manager.NeutronManager.get_service_plugins()[RESOURCE_SYNC]
        resources = []

        for resource_name in [RESOURCE_SYNC]:
            collection_name = '{}s'.format(resource_name)
            params = RESOURCE_ATTRIBUTE_MAP.get(collection_name, dict())
            member_actions = {}
            if ops_version in ac_const.OPS_VERSION_O_PQRTW_6_21:
                controller = base.create_resource(
                    collection_name, resource_name, plugin, params,
                    allow_bulk=True,
                    member_actions=member_actions,
                    allow_pagination=True,
                    allow_sorting=True)
            else:
                controller = base.create_resource(
                    collection_name, resource_name, plugin, params,
                    allow_bulk=True, member_actions=member_actions,
                    allow_pagination=cfg.CONF.allow_pagination,
                    allow_sorting=cfg.CONF.allow_sorting)
            res = ResourceExtension(collection_name,
                                    controller,
                                    member_actions=member_actions,
                                    attr_map=params)
            resources.append(res)
        return resources


@six.add_metaclass(abc.ABCMeta)
class ResourcesyncBase(ServicePluginBase):
    """Resourcesync class"""

    def get_plugin_type(self):
        """get plugin type"""
        return RESOURCE_SYNC

    def get_plugin_description(self):
        """get plugin description"""
        return 'Perform neutron sync through cli'

    def get_plugin_name(self):
        """get plugin name"""
        pass

    @abc.abstractmethod
    def get_resourcesyncs(self, context, filters=None, fields=None):
        """the abstractmethod to get resourcesync"""
        pass

    @abc.abstractmethod
    def create_resourcesync(self, context, resourcesync):
        """the abstractmethod to get resourcesync"""
        pass
