#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ac keystone interface"""

from oslo_config import cfg
from networking_huawei._i18n import _LE, _LI
from networking_huawei.drivers.ac.common import security_util
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

LOG = ncu.ac_log.getLogger(__name__)

# 到处都是类似判断，考虑利用外观模式进行重构：将与ops版本相关的方法调用抽象出来，屏蔽版本之间差异，对外提供统一的调用接口
OPS_VERSIONS = {ac_const.OPS_K, ac_const.OPS_L, ac_const.OPS_M, ac_const.FSP_6_1, ac_const.FSP_6_3_0,
                ac_const.FSP_6_3_1, ac_const.FSP_6_5_1, ac_const.FSP_8_0_0, ac_const.FSP_8_0_3, ac_const.FSP_21_0}


class ACKeyStoneIf(object):
    """ac keystone interface"""

    @staticmethod
    def get_keystone_v2(keystone_info):
        """get keystone v2"""
        ops_version = ncu.get_ops_version()
        try:
            LOG.debug("using keystone v2 client")
            url = ACKeyStoneIf.__join_url(keystone_info['auth_url'], 'v2.0')
            if ops_version in OPS_VERSIONS:
                from keystoneclient.v2_0 import client
                keystone = client.Client(
                    username=keystone_info['user_name'],
                    password=keystone_info['password'],
                    tenant_name=keystone_info['tenant_name'],
                    auth_url=url, timeout=5)
            else:
                from keystoneauth1.identity import v2
                from keystoneauth1 import session
                from keystoneclient.v2_0 import client
                auth = v2.Password(
                    auth_url=url,
                    username=keystone_info['user_name'],
                    password=keystone_info['password'],
                    tenant_name=keystone_info['tenant_name'])
                sess = session.Session(auth=auth)
                keystone = client.Client(session=sess, timeout=5)
            LOG.debug("form keystone v2, got keystone.")
        except Exception as ex:
            LOG.error(_LE("[AC] Get keystone failed for %s."), ex)
            raise
        return keystone

    @staticmethod
    def get_keystone_v3(keystone_info):
        """get keystone v3"""
        ops_version = ncu.get_ops_version()
        try:
            LOG.debug("using keystone v3 client")
            if ops_version in OPS_VERSIONS:
                from keystoneclient.v3 import client
                keystone = client.Client(
                    user_domain_name='Default',
                    username=keystone_info['user_name'],
                    password=keystone_info['password'],
                    project_domain_name='Default',
                    project_name=keystone_info['tenant_name'],
                    auth_url=keystone_info['auth_url'],
                    insecure=True, timeout=5)
            else:
                from keystoneauth1.identity import v3
                from keystoneauth1 import session
                from keystoneclient.v3 import client
                auth = v3.Password(
                    user_domain_name='Default',
                    username=keystone_info['user_name'],
                    password=keystone_info['password'],
                    project_domain_name='Default',
                    project_name=keystone_info['tenant_name'],
                    auth_url=keystone_info['auth_url'])
                sess = session.Session(auth=auth, verify=False)
                keystone = client.Client(session=sess, insecure=False, timeout=5)
            LOG.debug("form keystone v3, got keystone.")
        except Exception as ex:
            LOG.error(_LE("[AC] Get keystone failed for %s."), ex)
            raise
        return keystone

    # Read the tenant name from the keystone
    @staticmethod
    def get_tenant_name_by_id_from_keystone(tenant_id):
        """get tenant name by id from keystone"""
        if not tenant_id:
            return ''
        keystone_info = ACKeyStoneIf.get_keystone_user_info()
        try:
            if keystone_info.get('auth_url', '').endswith('v3'):
                keystone = ACKeyStoneIf.get_keystone_v3(keystone_info)
                tenant = keystone.projects.get(tenant_id)
                tenant_name = tenant.name
            else:
                keystone = ACKeyStoneIf.get_keystone_v2(keystone_info)
                tenant = keystone.tenants.get(tenant_id)
                tenant_name = tenant.name
        except Exception as ex:
            LOG.error(_LE("[AC] Get tenant name fail for: %s,use the default name"), ex)
            tenant_name = tenant_id
        return tenant_name

    # List of tenants in the keystone
    @staticmethod
    def get_tenant_list_from_keystone():
        """get tenant list from keystone"""
        keystone_info = ACKeyStoneIf.get_keystone_user_info()
        try:
            if keystone_info.get('auth_url', '').endswith('v3'):
                keystone = ACKeyStoneIf.get_keystone_v3(keystone_info)
                tenants = keystone.projects.list()
            else:
                keystone = ACKeyStoneIf.get_keystone_v2(keystone_info)
                tenants = keystone.tenants.list()
        except Exception as ex:
            LOG.error(_LE("[AC]Get tenants failed, %s"), ex)
            return []
        return tenants

    @staticmethod
    def __join_url(base_url, path):
        """Join url"""
        url = ''
        if base_url:
            if base_url[-1:] != '/' and path[:1] != '/':
                url = '%s/%s' % (base_url, path)
            elif base_url[-1:] == '/' and path[:1] == '/':
                url = '%s%s' % (base_url[:-1], path)
            else:
                url = '%s%s' % (base_url, path)
        return url

    @staticmethod
    def get_keystone_user_info():
        """get keystone user info"""
        keystone_info = {}
        try:
            keystone_info['auth_url'] = cfg.CONF.huawei_ac_config.auth_url
        except cfg.NoSuchOptError:
            if ncu.get_ops_version() in {ac_const.OPS_R, ac_const.OPS_T, ac_const.OPS_W, ac_const.FSP_21_0}:
                keystone_info['auth_url'] = 'http://controller:5000/v3'
            else:
                keystone_info['auth_url'] = 'http://controller:35357'
        try:
            keystone_info['tenant_name'] = cfg.CONF.huawei_ac_config.keystone_tenant
        except cfg.NoSuchOptError:
            keystone_info['tenant_name'] = 'admin'

        try:
            keystone_info['user_name'] = cfg.CONF.huawei_ac_config.keystone_user
        except cfg.NoSuchOptError:
            keystone_info['user_name'] = 'admin'
        try:
            crypt_password = cfg.CONF.huawei_ac_config.keystone_passwd
            if ncu.after_fsp_6_3_0():
                keystone_info['password'] = crypt_password
            else:
                keystone_info['password'] = security_util.decrypt_data(
                    crypt_password, data_type=ac_const.KEYSTONE_AUTH_SECURE_KEY)
        except cfg.NoSuchOptError:
            LOG.error(_LE('keystone auth data missing in config file '))

        keystone_info = ACKeyStoneIf.get_domain_info(keystone_info)

        return keystone_info

    @staticmethod
    def get_domain_info(keystone_info):
        """Get domain info from config file"""
        try:
            keystone_info['domain_id'] = cfg.CONF.huawei_ac_config.domain_id
            if keystone_info['domain_id'] == 'None':
                keystone_info['domain_id'] = None
        except cfg.NoSuchOptError:
            keystone_info['domain_id'] = None

        try:
            keystone_info['user_domain_id'] = cfg.CONF.huawei_ac_config.user_domain_id
            if keystone_info['user_domain_id'] == 'None':
                keystone_info['user_domain_id'] = None
        except cfg.NoSuchOptError:
            keystone_info['user_domain_id'] = None

        try:
            keystone_info['project_domain_id'] = cfg.CONF.huawei_ac_config.project_domain_id
            if keystone_info['project_domain_id'] == 'None':
                keystone_info['project_domain_id'] = None
        except cfg.NoSuchOptError:
            keystone_info['project_domain_id'] = None

        return keystone_info

    @staticmethod
    def get_tenant_name(resource, context):
        """get tenant name"""
        tenant_id = resource['tenant_id']
        context_tenant_name = context.tenant_name
        context_tenant_id = context.tenant
        if context_tenant_id == tenant_id:
            tenant_name = context_tenant_name + "(" + tenant_id + ")"
            LOG.info(_LI("get_tenant_name from context %s"), tenant_name)
        else:
            tenant_name = ACKeyStoneIf.get_tenant_name_by_id_from_keystone(tenant_id)
            if tenant_name != tenant_id:
                tenant_name = tenant_name + "(" + tenant_id + ")"
            LOG.info(_LI("get_tenant_name from keystone %s"), tenant_name)
        return tenant_name
