#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Huawei subnet model."""

from oslo_config import cfg
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei._i18n import _LE

LOG = logging.getLogger(__name__)


class ACSubnetModel(object):
    """Huawei subnet model."""
    @staticmethod
    def ac_model_format(subnet, tenant_name):
        """Convert specific subnet data to Huawei model.

        :param subnet: subnet data.
        :param tenant_name: tenant name of subnet.
        :return: Huawei subnet model.
        """
        try:
            subnet_info = {'uuid': subnet['id'],
                           'name': subnet['name'],
                           'network-id': subnet['network_id'],
                           'tenant-name': tenant_name,
                           'tenant-id': subnet['tenant_id']}

            # description is not supported in kilo
            if 'description' in subnet:
                subnet_info['description'] = subnet['description']

            ACSubnetModel.get_subnet_time(subnet, subnet_info)
            if subnet.get('ip_version') is not None:
                subnet_info['ip-version'] = \
                    ac_const.NEUTRON_CONSTANTS +\
                    'ip-version-v' + str(subnet['ip_version'])

            if subnet.get('enable_dhcp') is not None:
                subnet_info['enable-dhcp'] = subnet['enable_dhcp']

            if subnet.get('allocation_pools') is not None:
                subnet_info['allocation-pools'] = \
                    subnet['allocation_pools']

            if subnet.get('cidr') is not None:
                subnet_info['cidr'] = subnet['cidr']

            if subnet.get('gateway_ip') is not None:
                subnet_info['gateway-ip'] = subnet['gateway_ip']

            if subnet.get('host_routes') is not None:
                subnet_info['host-routes'] = subnet['host_routes']

            if subnet.get('dns_nameservers') is not None:
                subnet_info['dns-nameservers'] = subnet['dns_nameservers']

            ACSubnetModel.get_subnet_external_access_mode(subnet, subnet_info)
            ACSubnetModel.get_subnet_ipv6_info(subnet, subnet_info)

            if subnet.get('enable_internet') is not None:
                subnet_info['enable-internet'] = subnet['enable_internet']

        except KeyError as ex:
            LOG.error(_LE("[AC]Key Error, doesn't contain all fields %s."), ex)
            raise KeyError

        return subnet_info

    @staticmethod
    def get_subnet_external_access_mode(subnet, subnet_info):
        """Query external access mode of specific subnet.

        :param subnet: subnet data.
        :param subnet_info: Huawei subnet model.
        :return: None.
        """
        if cfg.CONF.huawei_ac_config.external_access_mode in \
                [ac_const.FLEX_SNAT]:
            subnet_info['external-access-mode'] = 0

        for subnet_name in ac_const.SUBNET_RDS_LIST:
            if subnet['name'].startswith(subnet_name):
                subnet_info['external-access-mode'] = 0
                break

    @staticmethod
    def get_subnet_time(subnet, subnet_info):
        """Query create and update time of specific subnet.

        :param subnet: subnet data.
        :param subnet_info: Huawei subnet model.
        :return: None.
        """
        if subnet.get('created_at'):
            subnet_info['created-at'] = subnet['created_at']
        if subnet.get('updated_at'):
            subnet_info['updated-at'] = subnet['updated_at']

    @staticmethod
    def get_subnet_ipv6_info(subnet, subnet_info):
        """Query IPv6 info of specific subnet.

        :param subnet: subnet data.
        :param subnet_info: Huawei subnet model.
        :return: None.
        """
        if subnet.get('ipv6_ra_mode') is not None:
            subnet_info['ipv6-ra-mode'] = subnet['ipv6_ra_mode']

        if subnet.get('ipv6_address_mode') is not None:
            subnet_info['ipv6-address-mode'] = subnet['ipv6_address_mode']
