#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""resource sync"""

import argparse
import os.path
import string
import uuid

import six
from neutronclient.common import extension

from networking_huawei.drivers.ac.common import constants as ac_const

COMPARE_OVERWRITE_ERROR = "Compare-result file already exists."


class ACCommonCtrl(extension.NeutronClientExtension):
    """ACControlCommands class

    """
    resource = 'resourcesync'
    resource_plural = '%ss' % resource
    object_path = '/%s' % resource_plural
    resource_path = '/%s/%%s' % resource_plural
    versions = ['2.0']


class ValidateCredits(argparse.Action):
    """validate credits"""

    @classmethod
    def _is_valid_result_filename(cls, filepath):
        """ Function to validate the compare result filename.

        :param filepath: filename to be validated.
        :return: True/False, msg[reason for failure]
        """
        safechars = string.letters + string.digits + "-_."
        if not os.path.dirname(filepath):
            return False, 'Invalid COMPARE_RESULT_FILE_NAME.'
        filename = os.path.basename(filepath)
        if filename.startswith('-') or filename.startswith('.'):
            return False, 'COMPARE_RESULT_FILE_NAME is not allowed to start ' \
                          'with \'-\' or \'.\' characters.'
        str_len = len(filename)
        if str_len > 255:
            return False, 'Length of COMPARE_RESULT_FILE_NAME is greater ' \
                          'than 255.'
        ext = os.path.splitext(filename)[1][1:]
        if ext != 'csv':
            return False, 'COMPARE_RESULT_FILE_NAME is not in .csv format.'
        if set(filename) - set(safechars):
            return False, 'Invalid characters in COMPARE_RESULT_FILE_NAME.'
        return True, None

    def __call__(self, parser, args, values, option_string=None):
        if option_string == '--compare-data':
            setattr(args, self.dest, values)
        elif option_string == '--sync-data':
            setattr(args, self.dest, values)
        elif option_string == '--overwrite-compare-result':
            setattr(args, self.dest, True)
        elif option_string == '--include-bindports':
            setattr(args, self.dest, True)
        elif option_string == '--resource-type':
            setattr(args, self.dest, values)
        elif option_string == '--sync-segments':
            setattr(args, self.dest, values)
        return parser


class PerformNeutronSync(extension.ClientExtensionCreate, ACCommonCtrl):
    """Perform neutron sync."""

    shell_command = 'acctrl-neutron-sync'
    list_columns = []

    @classmethod
    def add_known_arguments(cls, parser):
        """This method is used to define the arguments that this CLI"""
        # command expects. When a user hits "neutron acctrl-neutron-sync
        # --help", information from these are displayed.
        parser.add_argument(
            '--compare-data', nargs='?',
            const=os.path.realpath(
                '/var/log/neutron/neutron_sync_comp_res.csv'),
            metavar='COMPARE_RESULT_FILE_NAME', action=ValidateCredits,
            help='Perform compare data and store result in '
                 'COMPARE_RESULT_FILE_NAME (.csv file). (default '
                 'filename: %(''const)s, Max length of filename including '
                 'format: 255, Characters allowed [A-Z,a-z,0-9,\'-\',\'_\'])')
        parser.add_argument(
            "--sync-data", nargs='?', choices=['exclude-delete',
                                               'include-delete'],
            const='exclude-delete',
            action=ValidateCredits, help='Perform neutron sync operation. ('
                                         'default: %(const)s)')
        parser.add_argument(
            "--overwrite-compare-result", nargs=0, action=ValidateCredits,
            help='Overwrite compare result file. Option is valid only with '
                 '--compare-data.')
        parser.add_argument(
            "--include-bindports", nargs=0, action=ValidateCredits,
            help='Include bindports or not, default: False.')
        parser.add_argument(
            "--sync-segments", nargs='?',
            const="sync_segments",
            help='Clean invalid segmentations.')
        parser.add_argument(
            "--resource-type", nargs='*', choices=ac_const.SYNC_TYPE_LIST,
            action=ValidateCredits,
            help='Perform compare data or neutron sync operation with the '
                 'type of resource.')
        return parser

    def run(self, parsed_args):
        """start run"""
        if parsed_args.compare_data or parsed_args.sync_data or parsed_args.sync_segments:
            neutron_client = self.get_client()
            neutron_client.format = parsed_args.request_format
            body = {'resourcesync': {}}
            body['resourcesync']['include_bindports'] = \
                bool(parsed_args.include_bindports)
            if parsed_args.compare_data:
                body['resourcesync']['compare_data'] = {
                    'filename': parsed_args.compare_data}
                body['resourcesync']['compare_data']['overwrite'] = bool(
                    parsed_args.overwrite_compare_result)
                body['resourcesync']['sync_resources'] = {
                    'id': str(uuid.uuid4())
                }
            if parsed_args.sync_data:
                if parsed_args.overwrite_compare_result and not \
                        parsed_args.compare_data:
                    print('Error: --overwrite-compare-result is allowed '
                          'only with --compare-data')
                    return
                body['resourcesync']['sync_data'] = parsed_args.sync_data
            if parsed_args.resource_type:
                body['resourcesync']['resource_type'] = \
                    parsed_args.resource_type
            if parsed_args.sync_segments:
                body['resourcesync']['sync_segments'] = \
                    parsed_args.sync_segments
                print("Info: Begin to clean invalid data from networksegments.")
            else:
                print('Info: Performing neutron sync operation. This might take a'
                      ' while. Please check the status using command \'neutron '
                      'acctrl-neutron-sync-status\'.')
            try:
                neutron_client.create_resourcesync(body)
            except Exception as except_msg:
                if self.check_sync_message(except_msg, body, neutron_client):
                    return
        else:
            if parsed_args.overwrite_compare_result:
                print('Error: --overwrite-compare-result is allowed only with'
                      ' --compare-data')
                return
            print('Warning: Need to input the operation to perform '
                  'in neutron-sync.')
            return
        return

    @classmethod
    def check_sync_message(cls, except_msg, body, neutron_client):
        """check sync message"""
        if except_msg.message.startswith(COMPARE_OVERWRITE_ERROR):
            print('Warning: Compare-result file already exists.')
            while True:
                user_input = six.moves.input('Press [Y] to overwrite '
                                             'compare-result file or press [N] '
                                             'to give up comparing#[Y/N]')
                if user_input == "Y" or user_input == "y":
                    break
                elif user_input == "N" or user_input == "n":
                    return True
                else:
                    continue
            body['resourcesync']['compare_data']['overwrite'] = True
            try:
                neutron_client.create_resourcesync(body)
            except Exception as ex:
                print('Error: %s' % str(ex))
        else:
            print('Error: %s' % except_msg.message)
        return False


class StatusNeutronSync(extension.ClientExtensionList, ACCommonCtrl):
    """Get neutron sync status."""

    shell_command = 'acctrl-neutron-sync-status'
    list_columns = ['server', 'state', 'sync_res', 'sync_start_time',
                    'expected_completion_time']

    def take_action(self, parsed_args):
        """take action"""
        return super(StatusNeutronSync, self).take_action(parsed_args)
