# !/usr/bin/env python
# -*- coding:utf-8 -*-
"""
| 功能：扩展实现 neutron 插件的顶级抽象类，剥离与 neutron 多版本依赖的部分
| 修改记录：2022-03-09 10:15 创建
"""
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase

try:
    from neutron.common.exceptions import BadRequest
except ImportError:
    from neutron_lib.exceptions import BadRequest

from networking_huawei._i18n import _
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_log

LOG = ac_log.getLogger(__name__)


class UnsupportedOperation(BadRequest):
    """unsupported operate"""
    message = _('Unsupported Operation: %(operate)s.')


class AbstractServicePlugin(ServicePluginBase):
    """插件顶级抽象类"""

    def __init__(self, extension):
        super(AbstractServicePlugin, self).__init__()
        # 插件对应的扩展，extension:AbstractServiceExt
        self.extension = extension
        # neutron 根据该属性值寻找对应扩展
        self.supported_extension_aliases.append(extension.get_alias())

    def get_plugin_type(self):
        """see ServicePluginBase"""
        return self.extension.get_name()

    def get_plugin_description(self):
        """see ServicePluginBase"""
        return "Description of plugin '%s'" % self.get_plugin_type()

    def show_core(self, request, pk_id, **kwargs):
        """查询单个资源

        :param request: neutron.api.v2.resource.Request,请求
        :param pk_id: str,资源ID
        :param kwargs: dict,其它参数
        :return: model_base.BASEV2
        """
        raise UnsupportedOperation(operate='show')

    def list_core(self, request, **kwargs):
        """查询指定资源

        :param request: neutron.api.v2.resource.Request,请求
        :param kwargs: dict,其它参数
        :return: list
        """
        raise UnsupportedOperation(operate='list')

    def create_core(self, request, body=None, **kwargs):
        """创建资源

        :param request: neutron.api.v2.resource.Request,请求
        :param body: model_base.BASEV2 or dict,请求体
        :param kwargs: dict,其它参数
        :return: model_base.BASEV2
        """
        raise UnsupportedOperation(operate='create')

    def update_core(self, request, pk_id, body=None, **kwargs):
        """更新指定资源

        :param request: neutron.api.v2.resource.Request,请求
        :param pk_id: str,资源ID
        :param body: dict,请求体
        :param kwargs: dict,其它参数
        :return: model_base.BASEV2
        """
        raise UnsupportedOperation(operate='update')

    def delete_core(self, request, pk_id, **kwargs):
        """删除指定资源

        :param request: neutron.api.v2.resource.Request,请求
        :param pk_id: str,资源ID
        :param kwargs: dict,其它参数
        :return: model_base.BASEV2
        """
        raise UnsupportedOperation(operate='delete')
