# !/usr/bin/env python
# -*- coding:utf-8 -*-
"""
| 功能：检查AC 配置信息的插件
| 修改记录：2022-03-08 15:24 创建
"""
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import os
import tempfile

import six

from networking_huawei.drivers.ac.common.ac_config_utils import AC_CONFIG_FILES, read_ac_config
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_log
from networking_huawei.drivers.ac.common.ssh_utils import SshExecutor
from networking_huawei.drivers.ac.db.config_check.ac_basic_config_db import AcBasicConfigDBMixin
from networking_huawei.drivers.ac.db.config_check.ac_config_schema import ACBasicConfig
from networking_huawei.drivers.ac.extensions.config_check.config_check_ext import Config_check_ext
from networking_huawei.drivers.ac.plugins.abstract_neutron_service_plugin import AbstractServicePlugin

LOG = ac_log.getLogger(__name__)


class AcConfigCheckPlugin(AbstractServicePlugin):
    """云平台插件巡检插件"""

    def __init__(self):
        super(AcConfigCheckPlugin, self).__init__(Config_check_ext())
        # 数据库
        self.db = AcBasicConfigDBMixin()

    def check_config(self, context, ssh_client, **kwargs):
        """比较ac远程节点的配置项和当前节点的配置项是否一样

        :param context: neutron_lib.context.Context,上下文
        :param ssh_client: SshExecutor,远程机器SSH客户端
        :param kwargs: dict,其它参数
        :return: list[dict]
        """
        src_config = {}
        #  get ac config of source machine from db
        for elem in self.db.list(context):
            src_config[elem.config_file] = elem

        # 是否显示相同的配置项,NestedMultiDict：bool类型变成了字符串
        should_show_same = (kwargs.get('should_show_same', 'False') == 'True')

        LOG.debug('[AC]ac config in local:%s', src_config)
        # get ac config of target machine from file and compare difference
        result = []
        for elem in _read_remote_config(ssh_client, neutron_user=kwargs.get('neutron_user'),
                                        neutron_password=kwargs.get('neutron_password')):
            base = src_config.get(elem.config_file)
            # if db present,read from file and add to db
            LOG.debug('[AC]compare config %s:%s,base=%s', elem.config_file, elem, base)
            result.extend(_compare_config(base, elem, should_show_same))

        return result

    def list_core(self, request, **kwargs):
        """see AbstractServicePlugin.list_core"""
        body = kwargs.get('body', request.params)
        ssh_client = SshExecutor(body.get('host'),
                                 body.get('username'),
                                 body.get('password'),
                                 port=int(body.get('port', 22)))
        try:
            return {self.extension.get_alias(): self.check_config(request.context, ssh_client, **dict(body))}
        finally:
            ssh_client.close()


class AssertElem:
    """比较对象"""

    def __init__(self, filepath, name, excepted, actual, **kwargs):
        self.filepath = filepath
        self.name = name
        self.excepted = excepted
        self.actual = actual

        # 为了解决python 2版本处理字符串用unicode类型导致的问题.fix不兼容问题:isinstance(self.excepted, unicode)在python 3环境下编译报错
        if isinstance(self.excepted, six.text_type):
            self.excepted = self.excepted.encode('utf8')
        if isinstance(self.actual, six.text_type):
            self.actual = self.actual.encode('utf8')

        # 期望值原始值，因为原始为list可能经过处理
        self._excepted = kwargs.get('_excepted', excepted)
        # 实际值原始值，因为原始为list可能经过处理
        self._actual = kwargs.get('_actual', actual)
        # 是否一样，未比较时为None
        self._is_equal = None

    def __repr__(self):
        return 'AssertElem<%s,name=%s>excepted=%s,actual=%s(%s?%s)' % (
            self.filepath, self.name, self._excepted, self._actual, self.excepted, self.actual)

    def to_dict(self):
        """返回固定格式的比较结果

        :return: dict
        """
        if self._is_equal is None:  # 未进行过比较
            self.is_equal()
        return {'filepath': self.filepath, 'name': self.name, 'expected': self._excepted, 'actual': self._actual,
                'is_same': 'Y' if self._is_equal else 'N'}

    def is_equal(self):
        """比较excepted和actual是否一样

        :return: bool
        """
        LOG.debug('[AC]is_equal() start:%s', self)
        self._is_equal = False
        if not isinstance(self.excepted, type(self.actual)):
            return False
        if isinstance(self.excepted, list):
            if sorted(self.excepted) != sorted(self.actual):
                return False
        elif isinstance(self.excepted, dict):
            for key, value in self.excepted.items():
                if not AssertElem(self.filepath, '%s.%s' % (self.name, key), value, self.actual.get(key)).is_equal():
                    return False

            # 如果 actual 有而 expected 里没有的键值对，返回 False。排除值为None的
            for key, value in self.actual.items():
                if (value is not None) and (self.excepted.get(key) is None):
                    return False
        else:
            if self.excepted != self.actual:
                return False
        self._is_equal = True
        return True


def _read_remote_config(ssh_client, neutron_user=None, neutron_password=None):
    """读取远端机器的配置文件

    :param ssh_client: SshExecutor,远程机器SSH客户端
    :param neutron_user: str,有权限读取配置文件的用户
    :param neutron_password: str,有权限读取配置文件的用户的密码
    :return: list[ACBasicConfig]
    """
    _, tmp_file = tempfile.mkstemp(suffix='.tmp', prefix='neutron_ac_config_')
    try:
        for elem in AC_CONFIG_FILES:
            cat_cmd = 'su - %s -c "cat %s"' % (neutron_user, elem) if neutron_user else 'cat %s' % elem
            flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
            with os.fdopen(os.open(tmp_file, flags, 0o664), 'w') as fw:
                fw.write(ssh_client.run_command(cat_cmd, user_input=neutron_password))
            result = read_ac_config(tmp_file, elem)

            ll_cmd = 'su - %s -c "ls -l %s"' % (neutron_user, elem) if neutron_user else 'ls -l %s' % elem
            # 获取原始文件权限等信息,结果示例:-rwxr-xr-x.  1 root    root     2015 Jan 12 00:03 conf.tmp
            file_meta = ssh_client.run_command(ll_cmd, user_input=neutron_password).split()
            result.right = file_meta[0][:10]
            result.owner = file_meta[2]
            result.user_group = file_meta[3]
            yield result
    finally:
        if os.path.isfile(tmp_file):
            os.remove(tmp_file)


def _should_exclude(name):
    """排除不需要进行比较的配置项

    :param name: 配置项名称
    :return: bool
    """
    return name.find('password') != -1 or name in {'huawei_ac_agent_config.host_ip', 'huawei_ac_config.keystone_passwd'}


def _compare_config_value(filepath, excepted, actual):
    """比较配置文件内容

    :param filepath: str,配置文件
    :param excepted: 配置内容
    :param actual: 配置内容
    :return: list[dict]
    """
    if isinstance(excepted, dict):
        for key, value in excepted.items():
            # 不需要比对的配置项及需要比较处理后的内容
            if _should_exclude(key) or ACBasicConfig.get_list_key(key) in excepted:
                continue
            kwargs = {}
            name = key
            if key.startswith(ACBasicConfig.prefix_list()):
                name = key[len(ACBasicConfig.prefix_list()):]
                kwargs = {'_excepted': excepted.get(name), '_actual': actual.get(name)}
            yield AssertElem(filepath, name, value, actual.get(key), **kwargs)

        # 处理 actual 有而 expected 里没有的配置项
        for key, value in actual.items():
            # 不需要比对的配置项
            if _should_exclude(key) or key.startswith(ACBasicConfig.prefix_list()):
                continue
            if (value is not None) and (excepted.get(key) is None):
                yield AssertElem(filepath, key, None, value)
    else:
        yield AssertElem(filepath, None, excepted, actual)


def _compare_config(excepted, actual, should_show_same=False):
    """比较2个配置是否一样，并返回不一样的配置项

    :param excepted: ACBasicConfig,期望的配置项
    :param actual: ACBasicConfig,实际配置项
    :param should_show_same: bool,是否显示值相同的配置项
    :return: list[dict]
    """
    if not (excepted is not None and actual is not None):
        return [AssertElem(getattr(actual, 'config_file', None), 'config', str(excepted), str(actual)).to_dict()]
    result = []
    filepath = excepted.config_file
    for elem in (AssertElem(filepath, 'owner', excepted.owner, actual.owner),
                 AssertElem(filepath, 'user_group', excepted.user_group, actual.user_group),
                 AssertElem(filepath, 'right', excepted.right, actual.right)):

        if should_show_same:
            result.append(elem.to_dict())
        else:
            if not elem.is_equal():
                result.append(elem.to_dict())

    LOG.debug('excepted=<%s>%s,actual=<%s>%s,' % (
        type(excepted.config_value), excepted.config_value, type(actual.config_value), actual.config_value))
    for elem in _compare_config_value(filepath, excepted.config_value, actual.config_value):
        if should_show_same:
            result.append(elem.to_dict())
        else:
            if not elem.is_equal():
                result.append(elem.to_dict())
    return result
