#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""HuaweiL2brPlugin class """

from oslo_log import log as logging

import networking_huawei.drivers.ac.common.constants as ac_constants
from networking_huawei.drivers.ac.model.l2br_model import ACL2BRModel
from networking_huawei.drivers.ac.extensions.l2br import l2br as l2aas
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei.drivers.ac.db.l2br import l2br_db
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler


LOG = logging.getLogger(__name__)


class HuaweiL2brPlugin(l2aas.L2brBase, l2br_db.L2brDbMixin):
    """Implementation of the Huawei AC L2BR Service Plugin."""
    supported_extension_aliases = ['l2br']

    def __init__(self):
        LOG.info(_LI("[AC] Init Huawei L2br plugin."))
        super(HuaweiL2brPlugin, self).__init__()
        self.l2br_reliability = ACReliabilityAPI(ac_constants.NW_HW_L2BR)

        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for Huawei L2br plugin."))

    def get_plugin_type(self):
        """get plugin type"""
        return ac_constants.NW_HW_L2BR

    def get_plugin_description(self):
        """get plugin description"""
        return 'Huawei L2BR service plugin'

    def create_l2br(self, context, l2br):
        """create l2br"""
        LOG.info(_LI("[AC] Begin to create l2br : %s "), l2br)
        ac_osprofiler.record_chain_start("create l2br start")
        l2br_dicts = l2br
        status = ac_constants.NEUTRON_STATUS_ACTIVE
        l2br_dict = super(HuaweiL2brPlugin, self).\
            create_db_l2br(context, l2br_dicts, status)
        LOG.info(_LI("[AC] Create l2br record in Neutron DB successful: %s "),
                 l2br_dict)
        ac_osprofiler.record_call_chain("l2br id:" + l2br_dict['id'])
        l2br_info = ACL2BRModel.ac_model_format(l2br_dict)
        try:
            self.l2br_reliability.\
                update_plugin_record(context,
                                     l2br_info['uuid'],
                                     l2br_info,
                                     ac_constants.NW_HW_CREATE_L2BR)
        except Exception as ex:
            LOG.error(_LE("[AC] Huawei AC create l2br failed. "
                          "Roll back: Delete l2br in Neutron DB. catch: %s"),
                      ex)
            ac_osprofiler.record_chain_exception_end("create l2br fail")
            super(HuaweiL2brPlugin, self).\
                delete_db_l2br(context, l2br_info['uuid'])
            raise ex
        ac_osprofiler.record_chain_end_with_reason("create l2br success")
        LOG.info(_LI("[AC] Huawei AC create l2br successful."))

        return l2br_dict

    def delete_l2br(self, context, l2br_id):
        """delete l2br"""
        LOG.info(_LI("[AC] Begin to delete l2br : %s"), l2br_id)
        ac_osprofiler.record_chain_start("delete l2br start,ID:" + l2br_id)
        try:
            self.l2br_reliability.\
                update_plugin_record(context,
                                     l2br_id,
                                     {},
                                     ac_constants.NW_HW_DELETE_L2BR)
        except Exception as value:
            LOG.error(_LE("[AC] AC delete l2br failed for %s."), value)
            ac_osprofiler.record_chain_exception_end("delete l2br fail")
            raise value

        LOG.info(_LI("[AC] AC delete l2br successful."))

        try:
            super(HuaweiL2brPlugin, self).delete_db_l2br(context, l2br_id)
        except Exception as value:
            LOG.error(_LE("[AC] Failed to delete l2br "
                          "in Neutron DB for %s."), value)
            raise value
        ac_osprofiler.record_chain_end_with_reason("delete l2br success")
        LOG.info(_LI("[AC] Delete l2br record in Neutron DB successful."))

    def update_l2br(self, context, l2br_id, l2br):
        """update l2br"""
        LOG.info(_LI("[AC] Begin to update l2br : %s"), l2br)
        ac_osprofiler.record_chain_start("update l2br start,ID:" + l2br_id)

        original_l2br = super(HuaweiL2brPlugin, self).\
            get_db_l2br(context, l2br_id)
        LOG.debug("[AC] Get original l2br in Neutron DB, original_l2br= %s",
                  original_l2br)
        update_l2br = super(HuaweiL2brPlugin, self).\
            update_db_l2br(context, l2br_id, l2br)
        LOG.info(_LI("[AC] Updated l2br record in Neutron DB "
                     "successful: %s"), update_l2br)

        l2br_new_info = ACL2BRModel.\
            ac_model_format(update_l2br)
        try:
            self.l2br_reliability.\
                update_plugin_record(context,
                                     l2br_new_info['uuid'],
                                     l2br_new_info,
                                     ac_constants.NW_HW_UPDATE_L2BR)
        except Exception as value:
            LOG.error(_LE("[AC] AC update l2br failed for %s. "
                          "Roll back to the previous l2br info."), value)
            ac_osprofiler.record_chain_exception_end("update l2br fail")
            super(HuaweiL2brPlugin, self).\
                update_db_l2br(context, l2br_id,
                               {'l2br': original_l2br})
            raise value

        LOG.info(_LI("[AC] Huawei AC update l2br successful."))
        ac_osprofiler.record_chain_end_with_reason("update l2br success")
        return l2br_new_info

    def get_l2br(self, context, l2br_id, fields=None):
        """get l2br"""
        LOG.info(_LI("[AC] Begin to get l2br : %s"), l2br_id)
        rest_info = {}
        rest_info = super(HuaweiL2brPlugin, self). \
            get_db_l2br(context, l2br_id, fields=fields)
        LOG.info(_LI("[AC] End to get l2br."))
        return rest_info

    def get_l2brs(self, context, filters=None, fields=None):
        """get l2brs"""
        LOG.info(_LI("[AC] Begin to get l2brs."))

        return super(HuaweiL2brPlugin, self). \
            get_db_l2brs(context, filters=filters, fields=fields)
