#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Implementation of the Neutron L3 Router Service Plugin."""
import time

import netaddr
from neutron.db import models_v2

try:
    from neutron.extensions.external_net import EXTERNAL
except ImportError:
    from neutron_lib.api.definitions.external_net import EXTERNAL
from oslo_config import cfg

from networking_huawei._i18n import _LE
from networking_huawei._i18n import _LI
# import config register for fusionsphere
from networking_huawei.drivers.ac.common import constants as ac_cnst
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.db.dnat.dnat import DNAT
from networking_huawei.drivers.ac.extensions.exroute import l3_exroutes
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.model.port_model import ACPortModel
from networking_huawei.drivers.ac.model.router_model import ACRouterModel

try:
    from neutron_lib import exceptions as n_exc
except ImportError:
    from neutron.common import exceptions as n_exc

try:
    from neutron_lib.db import model_query
except ImportError:
    pass

try:
    from neutron.common.exceptions import NeutronException, BadRequest
except ImportError:
    from neutron_lib.exceptions import NeutronException, BadRequest

try:
    from neutron_lib.api.definitions import l3 as l3_apidef
except ImportError:
    l3_apidef = None

try:
    from neutron.db.l3_db import RouterPort, FloatingIP
except ImportError:
    from neutron.db.models.l3 import RouterPort, FloatingIP

try:
    from neutron.extensions.l3 import RouterInUse
except ImportError:
    from neutron_lib.exceptions.l3 import RouterInUse

try:
    from neutron.services.timestamp.timestamp_db import TimeStamp_db_mixin
except ImportError:
    TimeStamp_db_mixin = None
try:
    from neutron.objects import router as l3_obj
except ImportError:
    l3_obj = None

LOG = ncu.ac_log.getLogger(__name__)
TIME_FORMAT_WHOLE_SECONDS = '%Y-%m-%dT%H:%M:%S'


def _extend_resource_dict_timestamp(resource_res, resource_db):
    if (resource_db and resource_db.created_at and
            resource_db.updated_at):
        resource_res['created_at'] = \
            (resource_db.created_at.strftime(TIME_FORMAT_WHOLE_SECONDS)) + 'Z'
        resource_res['updated_at'] = \
            (resource_db.updated_at.strftime(TIME_FORMAT_WHOLE_SECONDS)) + 'Z'


def _disable_qos_extension_by_plugins(aliases):
    if all(p not in cfg.CONF.service_plugins for p in ac_cnst.QOS_CLASSES):
        if 'qos-fip' in aliases:
            aliases.remove('qos-fip')
        if 'qos-gateway-ip' in aliases:
            aliases.remove('qos-gateway-ip')


class ExRoutesException(NeutronException):
    """Exroutes failed exception."""
    message = 'AC %(opr)s exroutes %(ex)s failed.'


class ExRoutesTypeException(NeutronException):
    """Exroutes is not same type."""
    message = 'Delete failed, exroute:%(ex1)s is not same type with %(ex2)s.'


class HuaweiACL3Util(ncu.base_db.CommonDbMixin):
    """Huawei AC L3 Util"""

    @classmethod
    def modify_router_info(cls, router):
        """Modify router information."""
        if router['router'].get('external_gateway_info', {}) and \
                not cfg.CONF.huawei_ac_config.enable_snat:
            router['router']['external_gateway_info']['enable_snat'] = False
            LOG.info(_LI("[AC]SNAT is disabled due to the configuration"))

        if router['router'].pop('distributed', None):
            LOG.info(_LI("[AC]AC not allowed to config DVR manually."))

        if router['router'].pop('ha', None):
            LOG.info(_LI("[AC]AC not allowed to config HA manually."))

    @classmethod
    def check_public_network_ipv6(cls, context, public_network_id):
        """Check public network IPv6."""
        LOG.info(_LI("Check public network %s whether has an IPV6 subnet."),
                 public_network_id)

        if not public_network_id:
            return None
        subnets = context.session.query(models_v2.Subnet). \
            filter(models_v2.Subnet.network_id == public_network_id).all()
        LOG.info(_LI("Get subnets of network %s is: %s")
                 % (public_network_id, subnets))
        for subnet in subnets:
            LOG.info(_LI("Check subnet %s : "), subnet)
            subnet_id = subnet.get('id')
            ip_version = subnet.get('ip_version')
            if ip_version == 6:
                LOG.info(_LI("The network %s has an ipv6 subnet :"
                             "%s") % (public_network_id, subnet_id))
                return subnet_id
        LOG.info(_LI("The network %s does not has ipv6 subnet :"),
                 public_network_id)
        return None

    @classmethod
    def _validate_firewall_routers_not_in_use(cls, context, router_id):
        try:
            try:
                # import firewall router insertion db for openstack
                from neutron_fwaas.db.firewall import \
                    firewall_router_insertion_db
            except ImportError:
                # import firewall router insertion db for fusionsphere
                from neutron.db.firewall import firewall_router_insertion_db
        except ImportError:
            pass
        else:
            fw_db = firewall_router_insertion_db. \
                FirewallRouterInsertionDbMixin()
            fw_db.validate_firewall_routers_not_in_use(context, [router_id])

    @classmethod
    def _check_router_used_by_vpc_connection(cls, context, router_id):
        from networking_huawei.drivers.ac.db.vpc_connection import \
            vpc_connection_db
        vpc_db = vpc_connection_db.VpcConnectionDbMixin()
        vpc_conns = vpc_db.get_db_vpc_connections(context)
        for vpc_conn in vpc_conns:
            if router_id in [vpc_conn['local_router'],
                             vpc_conn['peer_router']]:
                erro_msg = "is in use by vpc connection %s" % vpc_conn['id']
                ac_osprofiler.record_chain_exception_end(erro_msg)
                raise RouterInUse(
                    router_id=router_id,
                    reason=erro_msg)

    @classmethod
    def _check_router_used_by_bgp_route(cls, context, router_id):
        from networking_huawei.drivers.ac.db.bgp_route import \
            bgp_route_db
        bgp_db = bgp_route_db.BgpRouteDbMixin()
        bgp_routes = bgp_db.get_db_bgp_routes(context)
        for bgp_route in bgp_routes:
            if bgp_route['router_id'] == router_id:
                erro_msg = "is in use by bgp route %s" % bgp_route['id']
                ac_osprofiler.record_chain_exception_end(erro_msg)
                raise RouterInUse(
                    router_id=router_id,
                    reason=erro_msg)

    def _check_router_not_in_use(self, context, router_id):
        if ac_cnst.NW_HW_VPC_CONNECTION in \
                ac_cnst.NW_HW_NEUTRON_SYNC_SUPPORT_RES:
            self._check_router_used_by_vpc_connection(
                context, router_id)
        if ac_cnst.NW_HW_BGP_ROUTE in \
                ac_cnst.NW_HW_NEUTRON_SYNC_SUPPORT_RES:
            self._check_router_used_by_bgp_route(
                context, router_id)

    @classmethod
    def _validate_sub_floatingip(cls, context, fip_old, fip):
        if ncu.IS_FSP and ACCommonUtil.is_sub_floatingip(context, fip_old) \
                and set(fip.keys()) & {'port_id', 'fixed_ip_address'}:
            msg = 'sub floating IP could not be updated'
            LOG.error(_LE('[AC] Failed to update floating IP: %s'), msg)
            ac_osprofiler.record_chain_exception_end(msg)
            raise BadRequest(resource='floatingip', msg=msg)

    @classmethod
    def _validate_dnat(cls, context, fip_id):
        try:
            dnats = context.session.query(DNAT).filter(
                DNAT.floating_ip_id == fip_id).all()
        except Exception as ex:
            LOG.info(_LI('[AC] Failed to query DNATs: %s, ignore it'), ex)
            return

        if dnats:
            msg = 'floating IP with DNATs could not be updated'
            LOG.error(_LE('[AC] Failed to update floating IP: %s'), msg)
            raise BadRequest(resource='floatingip', msg=msg)

    @classmethod
    def check_dnat_from_db(cls, context, fip_id):
        """Check DNAT from DB."""
        try:
            dnats = context.session.query(DNAT).filter(
                DNAT.floating_ip_id == fip_id).all()
        except Exception as ex:
            LOG.info(_LI('[AC] Failed to query DNATs: %s, ignore it'), ex)
            dnats = []

        if len(dnats) > 1:
            msg = 'floating IP with more than 1 DNAT could not be deleted'
            LOG.error(_LE('[AC] Failed to delete floating IP: %s'), msg)
            raise BadRequest(resource='floatingip', msg=msg)
        elif len(dnats) == 1:
            if 'dnat' not in ncu.get_service_plugin():
                msg = 'huawei ac dnat plugin is not configured'
                LOG.error(_LE('[AC] Failed to delete floating IP: %s'), msg)
                raise BadRequest(resource='floatingip', msg=msg)
            ncu.get_service_plugin()['dnat'].delete_dnat(context, dnats[0].id)

    @classmethod
    def _get_destination(cls, exroute):
        try:
            return str(netaddr.IPNetwork(exroute['destination']).cidr)
        except netaddr.AddrFormatError:
            raise l3_exroutes.ParamInvalid(
                param='destination', exroute=exroute)

    def _get_exroute_format(self, exroute):
        exroute_format = {'type': exroute.get('type')}
        if 'nexthop' in exroute:
            exroute_format['nexthop'] = \
                str(netaddr.IPAddress(exroute['nexthop']))
        if 'destination' in exroute:
            exroute_format['destination'] = self._get_destination(exroute)
        if 'ports' in exroute:
            exroute_format['ports'] = exroute['ports']
        return exroute_format

    @classmethod
    def _update_exroute_ip_version(cls, exroute):
        if not exroute.get('ip_version') and exroute.get('destination'):
            exroute['ip_version'] = netaddr.IPNetwork(
                exroute['destination']).version

    def _hill_stone_not_config(self):
        return not self._hill_stone_config()

    @classmethod
    def _hill_stone_config(cls):
        return not ac_cnst.HILL_STONE_PLUGIN.isdisjoint(
            set(cfg.CONF.service_plugins))

    def _extend_exroutes_dict(self, exroutes_dict):
        try:
            router_id = exroutes_dict.get('id')
            response = self.rest_service.send_vtep_ip_request(router_id)
        except Exception as ex:
            LOG.error(_LE('[AC] Huawei AC get local Vtep IP failed: %s'), ex)
            response = None

        vtep_ip = None
        if response:
            output_info = response.get(ac_cnst.EXROUTES_OUTPUT, {})
            vtep_ip = output_info.get('vtepip')

        if vtep_ip:
            LOG.info(_LI('[AC] Huawei AC get local Vtep IP: %s'), vtep_ip)
            exroutes_dict['local_ip'] = vtep_ip

        return exroutes_dict

    def _get_session(self, context):
        if hasattr(context, 'session'):
            return context.session
        return self.db_if.get_session('write')

    def is_father_floatingip(self, context, floatingip):
        """Is father floating IP or not."""
        if floatingip.get('fixed_ip_address') or \
                (not floatingip.get('floating_ip_address')):
            return False

        filters = {'floating_ip_address': [floatingip['floating_ip_address']],
                   'tenant_id': [floatingip['tenant_id']],
                   'floating_network_id': [floatingip['floating_network_id']]}

        if ncu.get_ops_version() in [ac_cnst.FSP_21_0]:
            floatingip_count = model_query.get_collection_count(
                context.elevated(), FloatingIP, filters=filters)
        else:
            floatingip_count = self._get_collection_count(
                context.elevated(), FloatingIP, filters=filters)

        if int(floatingip_count) <= 1:
            return False

        LOG.info(_LI('[AC] %s is father floating IP.'), floatingip)
        return True

    def _fixed_subnet_not_ready(self, current_gw_info, original_gw_info):
        current_fixed_ips = current_gw_info.get('external_fixed_ips', []) \
            if isinstance(current_gw_info, dict) else []
        original_fixed_ips = original_gw_info.get('external_fixed_ips', []) \
            if isinstance(original_gw_info, dict) else []

        if not current_fixed_ips or original_fixed_ips == current_fixed_ips:
            return False
        for fixed_ips_info in current_fixed_ips:
            subnet_id = fixed_ips_info.get('subnet_id')
            if not subnet_id:
                continue
            session = self.db_if.get_session()
            with session.begin(subtransactions=True):
                subnet_info = session.query(models_v2.Subnet). \
                    filter(models_v2.Subnet.id == subnet_id).all()
                subnet_flag = self.db_if.get_config_record(key=subnet_id)
                if not subnet_info or subnet_flag:
                    LOG.info(_LI('The subnet has not created in db '
                                 'or not create in AC: %s'), subnet_id)
                    return True
        return False

    def get_default_public_service_networks(self, context):
        """Get default public service networks."""
        if cfg.CONF.huawei_ac_config.default_public_service_network:
            filters = {
                'name': [
                    cfg.CONF.huawei_ac_config.default_public_service_network],
                EXTERNAL: [True]
            }
            fields = ['id']
            fip_networks = self._core_plugin.get_networks(
                context, filters=filters, fields=fields)
            if not fip_networks or len(fip_networks) > 1:
                return None
            return fip_networks
        return None

    def get_public_network(self, context, external_network_name):
        """Get public network."""
        admin_context = ncu.neutron_context.get_admin_context()
        for public_service_name in \
                cfg.CONF.huawei_ac_config.public_service_networks:
            LOG.info(_LI("[AC]All public networks : %s"), public_service_name)
            filters = {
                'name': [public_service_name],
                'router:external': [True]
            }
            fields = ['id']
            if external_network_name.endswith(public_service_name):
                public_networks = self._core_plugin.get_networks(
                    admin_context, filters=filters, fields=fields)
                LOG.info(_LI('[AC] Public service network found: %s'),
                         public_networks)
                if not public_networks or len(public_networks) > 1:
                    LOG.error(_LE('[AC] No public service network found for '
                                  'name %s'), public_service_name)
                else:
                    public_network_id = public_networks[0]['id']
                    public_network_ipv6 = self. \
                        check_public_network_ipv6(context, public_network_id)
                    if not public_network_ipv6:
                        LOG.debug(_LI("[AC] public service network %s does not"
                                      " exist for router to creat interface "
                                      "port."), public_network_id)
                    else:
                        return public_network_id, public_network_ipv6
        default_public_networks = self. \
            get_default_public_service_networks(admin_context)
        if default_public_networks:
            LOG.info(_LI('[AC] Get default public service '
                         'networks: %s'), default_public_networks)
            public_network_id = default_public_networks[0]['id']
            public_network_ipv6 = self. \
                check_public_network_ipv6(context, public_network_id)
            if not public_network_ipv6:
                LOG.debug(_LI("[AC] public service network does not exist "
                              "for router to creat interface port."))
            else:
                return public_network_id, public_network_ipv6
        return None

    def _validate_tenant_id(self, context, port, router_id):
        if not port.get('tenant_id'):
            LOG.error(_LE('[AC] Tenant id of interface port is None.'))
            return None

        router = self.get_router(context, router_id)
        if not router.get('tenant_id'):
            LOG.error(_LE('[AC] Tenant id of router is None.'))
            return None

        network = self._core_plugin.get_network(context, port['network_id'])
        if network.get('tenant_id') == router.get('tenant_id'):
            LOG.info(_LI('[AC] Tenant id of interface port and router '
                         'are the same. No need to create public FIP.'))
            return None

        public_networks = self._get_public_network_id(context, router_id)
        if not public_networks or len(public_networks) > 1:
            LOG.error(_LI("[AC] public service network does not exist "
                          "for port %s in creating fip."), port['id'])
            return None
        return public_networks

    def delete_public_network_ipv6_port(self, context, router_id):
        """Delete public network IPv6 port."""
        LOG.info(_LI("Delete public network ipv6 port from router : "
                     "%s,"), router_id)
        query = context.session.query(models_v2.Port).join(RouterPort)
        ports = query.filter(
            RouterPort.router_id == router_id,
            RouterPort.port_type == 'network:router_interface',
        ).all()
        subnet_id_list = []
        for each_port in ports:
            LOG.info(_LI("The router interface port is: %s"), each_port)
            port = self._core_plugin.get_port(context, each_port['id'])
            LOG.info(_LI("The interface port detail info is: %s"), port)
            subnet_id = self. \
                _process_to_remove_interface(context, port, router_id)
            subnet_id_list.append(subnet_id)
        return subnet_id_list

    def _ensure_router_not_in_use(self, context, router_id):
        """ ensure router not in use """
        router = self._get_router(context, router_id)
        device_owner = self._get_device_owner(context, router)
        if ncu.get_ops_version() in [ac_cnst.OPS_Q, ac_cnst.OPS_R, ac_cnst.OPS_W,
                                     ac_cnst.OPS_T, ac_cnst.FSP_6_5,
                                     ac_cnst.FSP_21_0]:
            router_attached_ports = router.attached_ports
        else:
            router_attached_ports = router.attached_ports.all()
        for port in router_attached_ports:
            if port.port_type != device_owner:
                continue
            LOG.info(_LI("Get router interface port : %s"), port.port_id)
            port_info = self._core_plugin.get_port(context, port.port_id)
            if not port_info:
                raise RouterInUse(router_id=router_id)
            profile = port_info.get('binding:profile', [{}])
            if not profile:
                raise RouterInUse(router_id=router_id)
            service_type = profile.get('service_type')
            enable_internet = profile.get('enable_internet')
            if service_type == 'Internet6' and enable_internet:
                LOG.info(_LI("Get router ipv6 interface port : "
                             "%s"), port.port_id)
                continue
            raise RouterInUse(router_id=router_id)
        return router

    def _rollback_gateway_port(self, context, router, org_gw_port, ext_ips):
        port_data = {
            'port': {
                'id': org_gw_port.id, 'tenant_id': '',
                'network_id': org_gw_port.network_id,
                'mac_address': org_gw_port.mac_address,
                'fixed_ips': ext_ips, 'device_id': router['id'],
                'device_owner': 'network:router_gateway',
                'admin_state_up': True, 'name': ''}}
        LOG.info(_LI('[AC] Begin to create gateway port: %s'), port_data)
        ac_osprofiler.record_call_chain("roll back to create gateway port")
        gw_port = self._core_plugin.create_port(context.elevated(), port_data)

        with context.session.begin(subtransactions=True):
            router.gw_port = self._core_plugin._get_port(
                context.elevated(), gw_port['id'])
            router_port = RouterPort(
                router_id=router.id, port_id=gw_port['id'],
                port_type='network:router_gateway')
            context.session.add(router)
            context.session.add(router_port)

    def process_public_network_ipv6_port(self, context, original_gw_info,
                                         current_gw_info, current_router):
        """Process public network IPv6 port."""
        if not original_gw_info and current_gw_info:
            LOG.info(_LI("Check whether to create public network ipv6 port,"
                         "the gateway_info is : %s"), current_gw_info)
            gateway_network_id = current_gw_info.get('network_id')
            tenant_id = current_router.get('tenant_id')
            subnet_id = self.check_gateway_and_public_network(
                context, gateway_network_id)
            if subnet_id:
                self.add_router_interface_public_network(
                    context, tenant_id, current_router['id'], subnet_id)

    def _handle_rollback_router_failure(self, context, current_router,
                                        original_gw_info, router_id):
        current_gw_port_id = current_router.get('gw_port_id')
        if not original_gw_info and current_gw_port_id:
            # If delete gateway_port failed When Rollback router,
            # We add gateway information return to router.
            LOG.info(_LI("[AC] Add gateway info %s for router %s"),
                     current_gw_port_id, router_id)
            current_gw_port_id = current_router.get('gw_port_id')
            with context.session.begin(subtransactions=True):
                router_object = self._get_router(context, router_id)
                router_object.gw_port_id = current_gw_port_id
                context.session.flush()

    def _get_snat_cidrs(self, context, floatingip):
        LOG.info(_LI('[AC] Begin to get flexible SNAT cidrs.'))
        if floatingip.get('port_id'):
            port_info = self._core_plugin.get_port(context, floatingip['port_id'])
            LOG.info(_LI('[AC]Begin to get flexible SNAT cidrs from port:%s'), port_info)
            snat_cidrs = []

            for fixed_ip in port_info.get('fixed_ips'):
                subnet_info = self._core_plugin.get_subnet(
                    context, fixed_ip['subnet_id'])
                LOG.info(_LI('[AC] Begin to get flexible SNAT cidrs from '
                             'subnet: %s'), subnet_info)
                snat_cidrs.append(subnet_info['cidr'])

            if not snat_cidrs:
                msg = 'flexible SNAT cidrs should not be None'
                LOG.error(_LE('[AC] Failed to get SNAT cidrs: %s.'), msg)
                raise BadRequest(resource='flexible SNAT', msg=msg)

            LOG.info(_LI('[AC] Flexible SNAT cidrs: %s'), snat_cidrs)
            return snat_cidrs
        return None

    def _validate_routes_nexthop(self, cidrs, ips, routes, nexthop):
        pass

    def _make_router_dict(self, router, fields=None, process_extensions=True):
        router_dict = super(HuaweiACL3Util, self)._make_router_dict(
            router, fields, process_extensions)
        if TimeStamp_db_mixin:
            if ncu.get_ops_version() in ac_cnst.OPS_VERSION_PQRTW_6_21:
                _extend_resource_dict_timestamp(router_dict, router)
            else:
                TimeStamp_db_mixin().extend_resource_dict_timestamp(
                    None, router_dict, router)
        return router_dict

    def _make_floatingip_dict(self, floatingip, fields=None,
                              process_extensions=True):
        versions = [ac_cnst.OPS_M, ac_cnst.OPS_N, ac_cnst.OPS_O,
                    ac_cnst.OPS_EZ_M, ac_cnst.FSP_6_5_1, ac_cnst.FSP_8_0_0,
                    ac_cnst.FSP_8_0_3] + ac_cnst.OPS_VERSION_PQRTW_6_21
        if ncu.get_ops_version() in versions:
            floatingip_dict = super(HuaweiACL3Util, self)._make_floatingip_dict(
                floatingip, fields, process_extensions)
        else:
            floatingip_dict = super(HuaweiACL3Util, self)._make_floatingip_dict(
                floatingip, fields)
        if TimeStamp_db_mixin:
            if ncu.get_ops_version() in ac_cnst.OPS_VERSION_PQRTW_6_21:
                _extend_resource_dict_timestamp(floatingip_dict, floatingip)
            else:
                TimeStamp_db_mixin().extend_resource_dict_timestamp(
                    None, floatingip_dict, floatingip)
        return floatingip_dict

    def check_gateway_and_public_network(self, context, gateway_network_id):
        """Check gateway and public network."""
        LOG.info(_LI("[AC]Begin to check gateway and public network :"
                     " %s "), context)
        external_network = self._core_plugin.get_network(
            context, gateway_network_id)
        external_network_name = external_network.get('name', '')
        if not external_network:
            LOG.error(_LE('[AC] No external network found for id %s'),
                      gateway_network_id)
            return None
        public_networks = self.get_public_network(context, external_network_name)
        if public_networks:
            public_network_id = public_networks[0]
            subnet_id = public_networks[1]
            LOG.info(_LI("[AC] Public service network : %s has subnet : %s"
                         "for router to creat interface port.")
                     % (public_network_id, subnet_id))
            return subnet_id
        return None

    def _create_gateway_port_ez(self, context, router_dict, gateway_info):
        """Create gateway port for openstack EZ_Mitaka.

        :param context: context of create router
        :param router_dict: router dict of db
        :param gateway_info: gateway port info
        :return: None
        """
        if ncu.get_ops_version() in [ac_cnst.OPS_EZ_M] and gateway_info:
            try:
                gw_port_id = router_dict.get('gw_port_id')
                if gw_port_id:
                    LOG.info(_LI('[AC] begin to delete floating ip port for '
                                 '%s'), router_dict['id'])
                    self.l3_reliability.update_plugin_record(
                        context, gw_port_id, {},
                        ac_cnst.NW_HW_DELETE_PORT)
                    gw_port = self._core_plugin.get_port(context, gw_port_id)
                    gw_port_info = ACPortModel.ac_model_format(gw_port, None)
                    LOG.info(_LI('[AC] begin to create gateway port for '
                                 '%s'), router_dict['id'])
                    self.l3_reliability.update_plugin_record(
                        context, gw_port_id, gw_port_info, ac_cnst.NW_HW_CREATE_PORT)
            except Exception as ex:
                LOG.error(_LE('[AC] create router failed: %s'), ex)
                super(HuaweiACL3Util, self).delete_router(context, router_dict['id'])
                raise

    def _notify_router_create(self, context, router, router_dict, gateway_info):
        """notify_router_create"""
        tenant_name = ACKeyStoneIf.get_tenant_name(router_dict, context)
        router_info = ACRouterModel.ac_model_format(router_dict, tenant_name)
        try:
            self._send_notify(
                ac_cnst.NW_HW_ROUTERS, ac_cnst.BEFORE_AC_CREATE, router_dict)
            self.l3_reliability.update_plugin_record(
                context, router_dict['id'], router_info, ac_cnst.NW_HW_CREATE_ROUTER)
            self._send_notify(
                ac_cnst.NW_HW_ROUTERS, ac_cnst.AFTER_AC_CREATE, router_dict)
            if ncu.IS_FSP and gateway_info:
                LOG.info(_LI("Check whether to create public network ipv6 port,"
                             "the gateway_info is : %s"), gateway_info)
                gateway_network_id = gateway_info.get('network_id', None)
                tenant_id = router['router'].get('tenant_id', None)
                subnet_id = self.check_gateway_and_public_network(
                    context, gateway_network_id)
                if subnet_id:
                    self.add_router_interface_public_network(
                        context, tenant_id, router_dict['id'], subnet_id)
                else:
                    LOG.info(_LI("No need to create public network ipv6 port."))
        except Exception as ex:
            LOG.error(_LE("[AC]Huawei AC create router failed.Roll back:"
                          "Delete router in Neutron DB. catch: %s"), ex)
            ac_osprofiler.record_chain_exception("crate router fail,delete it")
            super(HuaweiACL3Util, self).delete_router(context, router_dict['id'])
            raise BadRequest(resource='router', msg=ex)

    def _create_public_network_ipv6_port(self, context, fixed_ips, router_id):
        if fixed_ips and ACCommonUtil.is_port_contain_ipv6_address(fixed_ips):
            router = self.get_router(context, router_id)
            gateway_info = router.get('external_gateway_info', None)
            if ncu.IS_FSP and gateway_info:
                LOG.info(_LI("Check whether to create public network ipv6 port,"
                             "the gateway_info is : %s"), gateway_info)
                gateway_network_id = gateway_info.get('network_id', None)
                tenant_id = router.get('tenant_id', None)
                subnet_id = self.check_gateway_and_public_network(
                    context, gateway_network_id)
                if subnet_id:
                    self.add_router_interface_public_network(
                        context, tenant_id, router_id, subnet_id)
                else:
                    LOG.info(_LI("No need to create public network ipv6 port."))

    def _update_floatingip_rollback(self, context, fip_id, fip_old):
        """Update floating IP rollback."""
        super(HuaweiACL3Util, self).update_floatingip(
            context, fip_id, {'floatingip': fip_old})
        with context.session.begin(subtransactions=True):
            floatingip_db = self._get_floatingip(context, fip_id)
            floatingip_db.last_known_router_id = fip_old['last_known_router_id']
            context.session.flush()

    def _get_sub_floatingip_db(self, context, initial_status, fip_data):
        LOG.info(_LI('[AC] Begin to create sub floating IP.'))
        fip, fip_id, dns_integration = fip_data
        father_floatingip = ACCommonUtil.get_father_floatingip(
            context, fip)
        if ncu.get_ops_version() in [ac_cnst.FSP_6_5]:
            floatingip_db = l3_obj.FloatingIP(
                context, id=fip_id, project_id=fip['tenant_id'],
                status=initial_status,
                floating_network_id=fip['floating_network_id'],
                floating_ip_address=fip['floating_ip_address'],
                floating_port_id=father_floatingip.get('floating_port_id'))
        else:
            floatingip_db = FloatingIP(
                id=fip_id, tenant_id=fip['tenant_id'], status=initial_status,
                floating_network_id=fip['floating_network_id'],
                floating_ip_address=fip['floating_ip_address'],
                floating_port_id=father_floatingip.get('floating_port_id'))
        external_port = None
        floatingip_dict, dns_data = self._get_floatingip_dict(
            context, floatingip_db, external_port, (fip_id, fip, dns_integration))
        return dns_data, external_port, floatingip_db, floatingip_dict

    def _send_disassociate_fip(self, context, floating_ip_objs, port_id,
                               router_ids):
        old_fips = {fip.id: fip.to_dict() for fip in floating_ip_objs}
        values = {'fixed_port_id': None,
                  'fixed_ip_address': None,
                  'router_id': None}
        l3_obj.FloatingIP.update_objects(
            context, values, fixed_port_id=port_id)
        for fip in floating_ip_objs:
            try:
                self._send_notify(ac_cnst.NW_HW_FIP, ac_cnst.BEFORE_AC_DELETE,
                                  fip)
                if self._hill_stone_not_config():
                    self.l3_reliability.update_plugin_record(
                        context, fip['id'], {}, ac_cnst.NW_HW_DELETE_FIP)
                self._send_notify(ac_cnst.NW_HW_FIP, ac_cnst.AFTER_AC_DELETE,
                                  fip)
            except Exception as ex:
                LOG.error(_LE("[AC]AC delete floating IP failed for %s."), ex)
                raise ex
            LOG.info(_LI("[AC]AC disassociate floating IPs successful."))
            self.update_floatingip_status(
                context, fip['id'], ncu.FLOATINGIP_STATUS_DOWN)

            LOG.info(_LI("[AC]Update floating IP records in Neutron DB."))

            ncu.registry.notify(ncu.resources.FLOATING_IP,
                                ncu.events.PRECOMMIT_UPDATE,
                                self,
                                context=context,
                                floatingip={l3_apidef.FLOATINGIP: values},
                                floatingip_db=fip,
                                old_floatingip=old_fips[fip.id],
                                router_ids=router_ids)

    def _send_notify_of_delete_event(self, res_type, event, payload):
        LOG.info("[AC]Begin to send notify of %s delete ", res_type)
        ctx = ncu.neutron_context.get_admin_context()
        begin_time = time.time()
        while True:
            time.sleep(2)
            if time.time() - begin_time > \
                    cfg.CONF.huawei_ac_config.request_timeout * \
                    (cfg.CONF.huawei_ac_config.timeout_retry + 1):
                LOG.error("[AC] wait time out to send notify.")
                break
            if res_type == ac_cnst.NW_HW_ROUTERS:
                if self._send_router_del_event(ctx, payload, event):
                    return
            elif res_type == ac_cnst.NW_HW_ROUTER_IF:
                if self._send_router_if_del_event(ctx, payload, event):
                    return
            elif res_type == ac_cnst.NW_HW_FIP:
                if self._send_eip_del_event(ctx, payload, event):
                    return
        LOG.error("[AC]The neutron DB is still exist, do not send notify")

    def _send_router_del_event(self, ctx, payload, event):
        try:
            from neutron.extensions import l3
        except ImportError:
            l3 = None
        try:
            self.get_router(ctx, payload['id'])
        except l3.RouterNotFound:
            ncu.registry.notify(ac_cnst.NW_HW_ROUTERS, event, self, **payload)
            LOG.info("[AC] notify router delete %s", payload['id'])
            return True
        return False

    def _send_router_if_del_event(self, ctx, payload, event):
        try:
            port = self._core_plugin.get_port(ctx, payload['port_id'])
            port_subnets = [fip['subnet_id'] for fip in port['fixed_ips']]
            if payload['subnet_id'] not in port_subnets:
                LOG.info("[AC] send router interface notify when "
                         "subnet not in port's subnets")
                ncu.registry.notify(
                    ac_cnst.NW_HW_ROUTER_IF, event, self, **payload)
                return True
        except n_exc.PortNotFound:
            LOG.info("[AC] send router interface notify when port deleted")
            ncu.registry.notify(ac_cnst.NW_HW_ROUTER_IF, event, self, **payload)
            return True
        return False

    def _send_eip_del_event(self, ctx, payload, event):
        try:
            from neutron.extensions import l3
        except ImportError:
            l3 = None
        try:
            fip = self.get_floatingip(ctx, payload['id'])
            if not fip['fixed_ip_address']:
                LOG.info("[AC] send floatingip notify "
                         "when floatingip has no fixip")
                ncu.registry.notify(
                    ac_cnst.NW_HW_FIP, event, self, **payload)
                LOG.info("fip is %s", fip)
                return True
        except l3.FloatingIPNotFound:
            LOG.info("[AC]send floatingip notify when floatingip deleted")
            ncu.registry.notify(ac_cnst.NW_HW_FIP, event, self, **payload)
            return True
        return False


def get_router_port(router_intf_ports, internal_subnet_id):
    """ get router port """
    router_port_qry = router_intf_ports.join(models_v2.IPAllocation)
    router_port_qry = router_port_qry.filter(
        models_v2.IPAllocation.subnet_id == internal_subnet_id)
    return router_port_qry


def handle_router_exception(context, msg, router):
    """处理异常"""
    LOG.error(msg)
    ac_osprofiler.record_chain_exception_end(msg)
    with context.session.begin(subtransactions=True):
        router.status = 'ERROR'
        context.session.add(router)
    raise BadRequest(resource='router', msg=msg)


def is_router_has_gw_port(context, router_id):
    """判断路由是否有端口"""
    gw_ports = context.session.query(models_v2.Port).filter(
        models_v2.Port.device_id == router_id,
        models_v2.Port.device_owner == 'network:router_gateway').all()
    if gw_ports:
        return True
    return False


def generate_floatingip_port_desc(fip, fip_id):
    """生成浮动IP的端口信息"""
    port = {'tenant_id': '',  # tenant intentionally not set
            'network_id': fip['floating_network_id'],
            'admin_state_up': True, 'device_id': fip_id,
            'device_owner': ncu.DEVICE_OWNER_FLOATINGIP,
            'status': ncu.PORT_STATUS_NOTAPPLICABLE, 'name': ''}
    fix_ip = {}
    if fip.get('subnet_id'):
        fix_ip['subnet_id'] = fip['subnet_id']
    if fip.get('floating_ip_address'):
        fix_ip['ip_address'] = fip['floating_ip_address']
    if fix_ip:
        port['fixed_ips'] = [fix_ip]

    return port


def get_router_for_fip_fsp(context, port, subnet_id):
    """ncu.IS_FSP:获取浮动IP的路由"""
    router_intf_qry = context.session.query(RouterPort)
    router_intf_qry = router_intf_qry.join(models_v2.Port)
    router_intf_ports = router_intf_qry.filter(
        models_v2.Port.network_id == port['network_id'],
        models_v2.Port.mac_address == port['mac_address'],
        RouterPort.port_type == ncu.DEVICE_OWNER_ROUTER_INTF)

    routerport_qry = router_intf_ports.join(models_v2.IPAllocation)
    routerport_qry = routerport_qry.filter(
        models_v2.IPAllocation.subnet_id == subnet_id)

    router_id = None
    for router_port in routerport_qry:
        router_id = router_port.router.id
        return router_id

    if not router_id:
        router_intf_ports = router_intf_qry.filter(
            models_v2.Port.network_id == port['network_id'],
            RouterPort.port_type == ncu.DEVICE_OWNER_ROUTER_INTF)

        for router_port in get_router_port(router_intf_ports, subnet_id):
            router_id = router_port.router.id
            return router_id
    return router_id
