#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""portforwarding plugin"""

try:
    from neutron.db import api as db_api
except ImportError:
    from neutron_lib.db import api as db_api
from neutron.services.portforwarding.pf_plugin import PortForwardingPlugin
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.model.dnat_model import ACDNATModel
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
LOG = logging.getLogger(__name__)


class HWPortForwardingPlugin(PortForwardingPlugin):
    """HW PortForwarding Plugin"""
    supported_extension_aliases = ['floating-ip-port-forwarding',
                                   'expose-port-forwarding-in-fip']

    __native_pagination_support = True
    __native_sorting_support = True
    __filter_validation_support = True

    required_service_plugins = ['huawei_ac_router']

    def __init__(self):
        super(HWPortForwardingPlugin, self).__init__()
        self.reliability_api = ACReliabilityAPI(ac_constants.NW_HW_PF)
        self.ops_version = ncu.get_ops_version()
        if hasattr(db_api, "context_manager"):
            self.context_manager = db_api.context_manager
        elif hasattr(db_api, "get_context_manager"):
            self.context_manager = db_api.get_context_manager()
        else:
            self.context_manager = None

    def create_floatingip_port_forwarding(self, context, floatingip_id,
                                          port_forwarding):
        """create floatingip port forwarding"""
        LOG.info(_LI('[AC] Begin to create portforwarding: %s'),
                 port_forwarding)
        pf_obj = super(HWPortForwardingPlugin, self).\
            create_floatingip_port_forwarding(context, floatingip_id,
                                              port_forwarding)
        dnat_obj, dnat_info = self.get_dnat_info(context, floatingip_id,
                                                 port_forwarding, pf_obj)
        try:
            self.reliability_api.update_plugin_record(
                context, dnat_obj['id'], dnat_info,
                ac_constants.NW_HW_CREATE_DNAT)
        except Exception as ex:
            LOG.error(_LE('[AC] Huawei AC create portforwarding '
                          'failed: %s'), ex)
            super(HWPortForwardingPlugin, self). \
                delete_floatingip_port_forwarding(context, dnat_obj['id'],
                                                  floatingip_id)
            raise ex
        LOG.info(_LI('[AC] Huawei AC create portforwarding successfully'))
        return pf_obj

    def delete_floatingip_port_forwarding(self, context, id, floatingip_id):
        """delete floatingip port forwarding"""
        LOG.info(_LI('[AC] Begin to delete portforwarding: %s'), id)
        try:
            self.reliability_api.update_plugin_record(
                context, id, {}, ac_constants.NW_HW_DELETE_DNAT)
        except Exception as ex:
            LOG.error(_LE('[AC] Huawei AC delete portforwarding '
                          'failed: %s'), ex)
            raise ex
        super(HWPortForwardingPlugin, self).\
            delete_floatingip_port_forwarding(context, id, floatingip_id)
        LOG.info(_LI('[AC] Huawei AC delete portforwarding successfully'))

    def update_floatingip_port_forwarding(self, context, id, floatingip_id,
                                          port_forwarding):
        """update floatingip port forwarding"""
        LOG.info(_LI('[AC] Begin to update portforwarding: %s'),
                 port_forwarding)
        org_pf = super(HWPortForwardingPlugin, self). \
            get_floatingip_port_forwarding(context, id, floatingip_id)
        pf_obj = super(HWPortForwardingPlugin, self). \
            update_floatingip_port_forwarding(context, id, floatingip_id,
                                              port_forwarding)
        dnat_obj, dnat_info = self.get_dnat_info(context, floatingip_id,
                                                 port_forwarding, pf_obj)
        try:
            self.reliability_api.update_plugin_record(
                context, dnat_obj['id'], dnat_info,
                ac_constants.NW_HW_UPDATE_DNAT)
        except Exception as ex:
            LOG.error(_LE('[AC] Huawei AC update portforwarding '
                          'failed: %s'), ex)
            super(HWPortForwardingPlugin, self). \
                update_floatingip_port_forwarding(context, id,
                                                  floatingip_id,
                                                  {'port_forwarding': org_pf})
            raise ex
        LOG.info(_LI('[AC] Huawei AC update portforwarding successfully'))
        return pf_obj

    def get_dnat_info(self, context, floatingip_id, port_forwarding, pf_obj):
        """get dnat info"""
        if ncu.get_ops_version() in [ac_constants.FSP_8_0_0,
                                     ac_constants.FSP_8_0_3]:
            with db_api.autonested_transaction(context.session):
                fip_obj = self._get_floatingip(context, floatingip_id)
                router_id = self._find_a_router_for_fip_port_forwarding(
                    context, port_forwarding['port_forwarding'], fip_obj)
        else:
            with self.context_manager.writer.using(context):
                fip_obj = self._get_fip_obj(context, floatingip_id)
                router_id = self._find_a_router_for_fip_port_forwarding(
                    context, port_forwarding['port_forwarding'], fip_obj)
        dnat_obj = {
            'router_id': router_id,
            'port_id': pf_obj['internal_port_id'],
            'fixed_ip_address': pf_obj['internal_ip_address'],
            'id': pf_obj['id'],
            'protocol': pf_obj['protocol'],
            'floating_ip_port': pf_obj['external_port'],
            'fixed_ip_port': pf_obj['internal_port']
        }
        return dnat_obj, ACDNATModel.ac_model_format(dnat_obj, fip_obj)
