#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""cloud alarms sync"""

import threading
from datetime import datetime
from oslo_config import cfg
from oslo_db import api as oslo_db_api
from oslo_serialization import jsonutils

from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.sync import http_heart_beat
from networking_huawei.drivers.ac.model.cloud_alarm_model import CloudAlarmModel
from networking_huawei.drivers.ac.sync.verify_cert import CERT_FILE_DICT
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.db import schema as dbschema

LOG = ncu.ac_log.getLogger(__name__)


class CloudAlarmSync(object):
    """Cloud Alarm Sync"""

    def __init__(self):
        self._db_if = dbif.ACdbInterface()
        self.rest_client = ACReSTClient()
        self.http_heart_beat = http_heart_beat.HttpHeatBeat2AC()
        timer_cloud_alarm = threading.Timer(0, self._handle_cloud_alarms)
        timer_cloud_alarm.start()
        LOG.info("Cloud alarm sync schedule thread initialization done.")

    @classmethod
    def is_alarm_recover(cls, cloud_alarm):
        """Check the alarm is recover or not

        :param cloud_alarm: data of alarm
        :return: Bool
        """
        cert_type = cloud_alarm.get('description', {}).get('cert_type')
        with open(CERT_FILE_DICT.get(cert_type), "rb") as cert_file:
            cert = cert_file.read()
            expired_time = ncu.get_cert_expired_time(cert)
            current_time = datetime.utcnow()
            days_available = (expired_time - current_time).days
            alarm_inner_id = cloud_alarm.get('inner_id')
            if alarm_inner_id == ac_const.CLOUD_CERT_EXPIRED_ALARM_ID and \
                    days_available > 0:
                return True
            if alarm_inner_id == ac_const.CLOUD_CERT_ALMOST_EXPIRED_ALARM_ID \
                    and days_available > \
                    cfg.CONF.huawei_ac_config.cert_expiration_warning_time:
                return True
        return False

    def _handle_retry_success(self, cloud_alarm, session):
        LOG.debug("[AC] Send cloud alarm: %s success", cloud_alarm)
        if cloud_alarm.get('type') == ac_const.CLOUD_CERT_ALARM_TYPE:
            cloud_alarm['type'] = ac_const.CLOUD_CERT_RECOVER_ALARM_TYPE
            self._db_if.create_cloud_alarm(cloud_alarm, session)
        self.delete_cloud_alarm(cloud_alarm.get('seq_num'))

    def _handle_cloud_alarms(self):
        try:
            if not self.http_heart_beat.is_ac_alive():
                LOG.info("[AC] AC is not reached, Not sync cloud alarms to AC.")
                return
            session = self._db_if.get_session('write')
            cloud_alarms = self.get_cloud_alarms(session)
            for cloud_alarm in cloud_alarms:
                if cloud_alarm.get('type') == ac_const.CLOUD_CERT_RECOVER_ALARM_TYPE \
                        and not self.is_alarm_recover(cloud_alarm):
                    continue
                LOG.info("[AC] Deal with cloud alarm: %s", cloud_alarm)
                self.add_count_cloud_alarm(cloud_alarm.get('seq_num'), session)
                cloud_alarm_info = CloudAlarmModel.ac_model_format(cloud_alarm)
                if self.rest_client.send_alarm_to_controler(
                        jsonutils.dumps(cloud_alarm_info)):
                    self._handle_retry_success(cloud_alarm, session)
        except Exception as ex:
            LOG.error("[AC] Handle cloud alarms failed with exception: %s", ex)
        finally:
            timer = threading.Timer(ac_const.CLOUD_ALARM_SYNC_THREAD_WAIT,
                                    self._handle_cloud_alarms)
            timer.start()

    def get_cloud_alarms(self, session=None):
        """Get cloud alarms record.

        :param session: Current session info where the operation is
                        performed. Used to access db.
        :return: Cloud alarms
        """

        if not session:
            session = self._db_if.get_session('read')
        return session.query(dbschema.ACAlarms).all()

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES,
                               retry_interval=0,
                               inc_retry_interval=0,
                               max_retry_interval=0)
    def add_count_cloud_alarm(self, seq_num, session=None):
        """Add retry_count of cloud alarm.

        :param seq_alarm: seq_alarm of cloud alarm record
        :param session: Current session info where the operation is
                        performed. Used to access db.
        :return: None
        """

        if not session:
            session = self._db_if.get_session('write')
        with session.begin(subtransactions=True):
            cloud_alarm = session.query(dbschema.ACAlarms).\
                filter_by(seq_num=seq_num).first()
            if cloud_alarm:
                if cloud_alarm.retry_count >= 2:
                    session.delete(cloud_alarm)
                    LOG.debug("[AC] Max retry times reached, delete from cloud "
                              "alarm record for seq_num: %s.", seq_num)
                    return
                cloud_alarm.retry_count = cloud_alarm.retry_count + 1

    @oslo_db_api.wrap_db_retry(max_retries=ac_const.DB_MAX_RETRIES,
                               retry_interval=0,
                               inc_retry_interval=0,
                               max_retry_interval=0)
    def delete_cloud_alarm(self, seq_num, session=None):
        """Delete cloud alarm record.

        :param seq_num: seq_alarm of cloud alarm record
        :param session: Current session info where the operation is
                        performed. Used to access db.
        :return: None
        """

        if not session:
            session = self._db_if.get_session('write')
        with session.begin(subtransactions=True):
            cloud_alarm = session.query(dbschema.ACAlarms).\
                filter_by(seq_num=seq_num).first()
            if cloud_alarm:
                session.delete(cloud_alarm)
                LOG.debug("[AC] Deleted cloud alarm for seq_num: %s.", seq_num)
