#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Http Heart beat"""

import threading
import time
import sched
import eventlet
from oslo_serialization import jsonutils
from oslo_config import cfg
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.common import random_util
from networking_huawei.drivers.ac.common import constants
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.common import exceptions
from networking_huawei.drivers.ac.client.service import ACRestUtils
from networking_huawei.drivers.ac.common.fusion_sphere_alarm import \
    ACPluginAlarm
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

LOG = logging.getLogger(__name__)


class HttpHeatBeat2AC(object):
    """Http Heat Beat to AC"""

    def __init__(self):
        self.rest_client = ACReSTClient()
        self.ops_version = ncu.get_ops_version()
        self.host_ip = ACCommonUtil.get_local_host_ip()
        self.body = {'cloud_name': cfg.CONF.huawei_ac_config.cloud_name,
                     'agent_ip': self.host_ip}
        self.body = ACRestUtils.fix_json(ACRestUtils.fix_json(
            jsonutils.dumps(self.body)))
        self.dbif = dbif.ACdbInterface()
        self.neutron_name = ACCommonUtil.get_neutron_server_name()
        self.http_heart_beat_thread = None

    def init_http_heart_beat(self):
        """init http heart beat"""
        # start get random thread
        random_util.init_get_random()
        self.dbif.delete_config_record(self.neutron_name)
        self.http_heart_beat_thread = threading.Thread(
            target=self.send_http_heart_beat, name='http_heart_beat')
        LOG.info("[AC]Http heart beat Thread is initialized.")
        self.http_heart_beat_thread.start()
        schedule = sched.scheduler(time.time, time.sleep)
        schedule.enter(0, 0, self._monitor_http_heart_sync_thread, ())
        schedule.run()

    def _monitor_http_heart_sync_thread(self):
        try:
            if self.http_heart_beat_thread.is_alive():
                return
            self.http_heart_beat_thread = threading.Thread(
                target=self.send_http_heart_beat, name='http_heart_beat')
            LOG.info("[AC]Http heart beat Thread is initialized.")
            self.http_heart_beat_thread.start()
        except Exception as ex:
            LOG.error("[AC] Monitor http heart beat thread exception %s", ex)
        finally:
            timer = threading.Timer(constants.HTTP_HEART_TIME,
                                    self._monitor_http_heart_sync_thread)
            timer.start()

    def send_http_heart_beat(self):
        """send http heart beat"""
        down_count = 0
        active_count = 0
        first_flag = 1
        ac_connect_state_origin = constants.HTTP_CONNECTION_ACTIVE
        heart_beat_timeout = 6 * constants.HEARTBEAT_TIMEOUT
        while True:
            with eventlet.Timeout(heart_beat_timeout, False):
                heart_beat_result = self._http_heart_beat_check(
                    down_count, active_count, first_flag, ac_connect_state_origin)
                down_count = heart_beat_result.get("down_count", "")
                active_count = heart_beat_result.get("active_count", "")
                first_flag = heart_beat_result.get("first_flag", "")
                ac_connect_state_origin = heart_beat_result.get("ac_connect_state_origin", "")

    def _http_heart_beat_check(self, down_count, active_count, first_flag,
                               ac_connect_state_origin):
        heartbeart_start = time.time()
        ac_connect_state, reason = self.rest_client. \
            send_http_heart_to_ac(self.body)

        # 北向心跳异常,补充一次重试
        if ac_connect_state == constants.HTTP_CONNECTION_DOWN:
            ac_connect_state, reason = self.rest_client. \
                send_http_heart_to_ac(self.body)
        if ncu.IS_FSP:
            if first_flag:
                down_count = int(
                    ac_connect_state == constants.HTTP_CONNECTION_DOWN)
                active_count = int(
                    ac_connect_state == constants.HTTP_CONNECTION_ACTIVE)
                first_flag = 0
                ac_connect_state_origin = ac_connect_state
                self.change_ac_connection_state(ac_connect_state)
                return {"down_count": down_count, "active_count": active_count,
                        "first_flag": first_flag, "ac_connect_state_origin": ac_connect_state_origin}
            else:
                active_count, down_count = self.http_detect(
                    ac_connect_state, ac_connect_state_origin, active_count,
                    down_count)
                active_count, down_count = self._http_heart_beat_alarm(
                    active_count, down_count, reason)
        ac_connect_state_origin = ac_connect_state
        self.change_ac_connection_state(ac_connect_state)
        heartbeat_time_diff = int(time.time() - heartbeart_start)
        if 0 <= heartbeat_time_diff < constants.HTTP_HEART_TIME:
            time.sleep(constants.HTTP_HEART_TIME - heartbeat_time_diff)
        return {"down_count": down_count, "active_count": active_count,
                "first_flag": first_flag, "ac_connect_state_origin": ac_connect_state_origin}

    @classmethod
    def _proc_connect_down(cls, status_in_db, active_count, down_count):
        """proc connect down"""
        active_flg = (status_in_db == constants.HTTP_CONNECTION_ACTIVE)
        down_flg = (status_in_db == constants.HTTP_CONNECTION_DOWN)
        if active_flg and active_count != 0:
            active_count = 0
        if (active_flg and down_count == 0) or (down_flg and down_count != 0):
            down_count = down_count + 1
        return active_count, down_count

    @classmethod
    def _proc_connect_active(cls, status_in_db, active_count, down_count):
        """proc connect active"""
        active_flg = (status_in_db == constants.HTTP_CONNECTION_ACTIVE)
        down_flg = (status_in_db == constants.HTTP_CONNECTION_DOWN)
        if down_flg and down_count != 0:
            down_count = 0
        if (down_flg and active_count == 0) or (
                active_flg and active_count != 0):
            active_count = active_count + 1
        return active_count, down_count

    def http_detect(self, ac_connect_state, status_in_db, active_count,
                    down_count):
        """http detect"""
        if ac_connect_state == constants.HTTP_CONNECTION_DOWN:
            return self._proc_connect_down(
                status_in_db, active_count, down_count)
        if ac_connect_state == constants.HTTP_CONNECTION_ACTIVE:
            return self._proc_connect_active(
                status_in_db, active_count, down_count)
        return active_count, down_count

    @classmethod
    def _http_heart_beat_alarm(cls, active_count, down_count, reason):
        if down_count == constants.HTTP_FAIL_COUNT_MAX:
            alarm_info = ACPluginAlarm.get_http_connection_fail_alarm(reason)
            LOG.info("[AC]HTTP connection failed. Send alarm message.")
            ACPluginAlarm.send_alarm(alarm_info)
            down_count = 0

        if active_count == constants.HTTP_RECOVER_COUNT_MAX:
            alarm_info = ACPluginAlarm.get_http_connection_recovery_alarm()
            LOG.info("[AC]HTTP connection recovered. Send alarm message.")
            ACPluginAlarm.send_alarm(alarm_info)
            active_count = 0
        return active_count, down_count

    def is_ac_alive(self):
        """is ac alive"""
        ac_connection_status = self.get_ac_connection_status_in_db()
        return ac_connection_status

    def get_ac_connection_status_in_db(self):
        """get ac connection status in db"""
        try:
            neutron_server_record = self.dbif.get_server_record(
                neutron_name=self.neutron_name)
            if neutron_server_record:
                if neutron_server_record.ac_connection_state in \
                        [constants.HTTP_CONNECTION_DOWN,
                         constants.HTTP_CONNECTION_ACTIVE]:
                    return neutron_server_record.ac_connection_state
                self.init_ac_http_connection_status()
                return constants.HTTP_CONNECTION_ACTIVE

            LOG.error("[AC] neutron server record not create: %s",
                      self.neutron_name)
            return constants.HTTP_CONNECTION_DOWN
        except Exception as ex:
            LOG.error("[AC] get http connection status failed: %s", ex)
            return constants.HTTP_CONNECTION_DOWN

    def change_ac_connection_state(self, status_in_detect):
        """change ac connection state"""
        try:
            status_in_db = self.get_ac_connection_status_in_db()
            if status_in_db != status_in_detect:
                LOG.info("[AC]change status from %s to %s",
                         self.get_ac_status_for_log(status_in_db),
                         self.get_ac_status_for_log(status_in_detect))
                self.dbif.update_ac_connection_state_record(self.neutron_name,
                                                            status_in_detect)
        except Exception as ex:
            LOG.error("[AC] update http connection status failed: %s", ex)

    def init_ac_http_connection_status(self):
        """init ac http connection status"""
        try:
            LOG.info("[AC]begin to initial ac http connection status.")
            self.dbif.update_ac_connection_state_record(
                self.neutron_name,
                constants.HTTP_CONNECTION_ACTIVE)
        except Exception as ex:
            LOG.error("[AC] initial http connection status failed: %s", ex)

    def block_send_request_to_ac(self):
        """block send request to ac"""
        if not self.is_ac_alive():
            LOG.error("[AC] AC is not reached")
            raise exceptions.ACConnectionException()

    @classmethod
    def get_ac_status_for_log(cls, status):
        """get ac status for log"""
        if status == constants.HTTP_CONNECTION_ACTIVE:
            return "ACTIVE"
        return "DOWN"
