#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""neutron sync router util"""

import ast
import copy

from oslo_config import cfg
from oslo_serialization import jsonutils
from six.moves import zip

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.client.service import ACRestUtils
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.sync import util

LOG = ncu.ac_log.getLogger(__name__)


class NeutronSyncRouterUtil(object):
    """router sync router util"""

    def _process_with_device_id(self, admin_ctx, device_id, rec_list, session):
        """process with device id"""
        create_router_ids = \
            [r['uuid'] for r in self._create_router_list]
        create_router_tenants = \
            [r['tenant-name'] for r in self._create_router_list]
        create_id_tenant = \
            dict(zip(create_router_ids, create_router_tenants))
        LOG.debug('[AC] create id tenant map %s', create_id_tenant)
        if device_id in create_id_tenant:
            self._process_create_router(admin_ctx, create_id_tenant, device_id,
                                        session)
        else:
            self._process_update_router(admin_ctx, session,
                                        (device_id, rec_list))

    def _process_create_router(self, admin_ctx, create_id_tenant, device_id,
                               session):
        """process create router"""
        LOG.debug('[AC] begin to sync create router %s first',
                  device_id)
        result = self._sync_create_router(admin_ctx, create_id_tenant,
                                          device_id,
                                          session)
        create_router_list_copy = \
            copy.deepcopy(self._create_router_list)
        for create_router in create_router_list_copy:
            if create_router['uuid'] == device_id:
                self._create_data_list.remove(create_router)
                if result == ac_const.NW_HW_TIMEOUT:
                    res_status = ac_const.STAT_TIME_OUT
                else:
                    res_status = result
                data_info = \
                    {'id': create_router['uuid'],
                     'name': create_router['name'],
                     'tenant_name': create_router['tenant-name'],
                     'created_at': create_router.get('created_at',
                                                     ac_const.EMPTY_STR),
                     'updated_at': create_router.get('updated_at',
                                                     ac_const.EMPTY_STR),
                     'result': res_status}
                sync_create_result = \
                    self._neutron_sync_result[
                        ac_const.OPER_CREATE]
                if ac_const.NW_HW_ROUTERS in \
                        sync_create_result:
                    sync_create_result[
                        ac_const.NW_HW_ROUTERS]. \
                        append(data_info)
                else:
                    sync_create_result[
                        ac_const.NW_HW_ROUTERS] = [data_info]

    def _process_update_router(self, admin_ctx, session, sync_router_data):
        """process update router"""
        device_id, rec_list = sync_router_data
        port_info = self._data_list[rec_list[0]]
        LOG.debug("[AC] gateway port: %s", port_info)
        router_res_list = self._db_if.get_router_neutron_record_by_gw_port(
            session, port_info["uuid"])
        LOG.debug("[AC] query router list by routers: %s", router_res_list)
        for router_res in router_res_list:
            if cfg.CONF.huawei_ac_config.neutron_sync_enable_white_list and \
                    router_res.id in self._white_list['routers']:
                continue
            update_router_ids = \
                [r['uuid'] for r in self._update_router_list]
            if router_res.id in update_router_ids:
                continue
            tenant_name = self._get_tenant_name(
                router_res['tenant_id'])
            LOG.debug('Tenant Name: %s', tenant_name)
            router_info = {'res': ac_const.NW_HW_ROUTERS,
                           'uuid': router_res.id,
                           'tenant-name': tenant_name,
                           'status': ac_const.STAT_WAIT}
            LOG.debug('Adding entry to plugin for UPDATE: %s',
                      router_info)
            self._update_router_list.append(router_info)
        update_router_ids = \
            [r['uuid'] for r in self._update_router_list]
        update_router_tenants = \
            [r['tenant-name'] for r in self._update_router_list]
        update_id_tenant = \
            dict(zip(update_router_ids, update_router_tenants))
        LOG.debug('[AC] update id tenant map %s',
                  update_id_tenant)
        LOG.debug('[AC] begin to sync update router %s first',
                  device_id)
        if device_id in update_id_tenant:
            self._sync_update_router(admin_ctx, device_id, update_id_tenant,
                                     session)

    def _sync_update_router(self, admin_ctx, device_id, update_id_tenant,
                            session):
        """sync update router"""
        (method, res_url) = \
            util.ACUtil.get_method_and_resource_url(
                ac_const.NW_HW_UPDATE_ROUTER)
        entry_info = \
            self._get_formatted_data_for_neutron_sync(
                admin_ctx, device_id,
                (update_id_tenant[device_id],
                 method, ac_const.NW_HW_ROUTERS))
        entry_info[ac_const.CLOUD_NAME_PARAM] = \
            cfg.CONF.huawei_ac_config.cloud_name
        rest_info = ACRestUtils.wrap_entry_info(
            method, ac_const.NW_HW_ROUTERS, [entry_info])
        rest_data = ACRestUtils.fix_json(
            jsonutils.dumps(rest_info))
        LOG.debug('[AC] update rest data %s', rest_data)
        url = util.ACUtil.form_resource_url(
            ac_const.NW_HW_URL, method, res_url, device_id)
        result, response = self._send_data_to_ac(
            rest_data, url, None,
            (ac_const.NW_HW_ROUTERS, method, device_id))
        LOG.debug('[AC] sync update router %s finished, '
                  'result %s, response %s',
                  device_id, result, response)
        if result == ac_const.NW_HW_SUCCESS:
            LOG.debug('create plugin record for the router: %s [2]', device_id)
            self._db_if.create_plugin_record(session, {},
                                             (device_id, ac_const.OPER_UPDATE,
                                              ac_const.NW_HW_ROUTERS),
                                             ac_const.NEUTRON_SYNC)

    def _sync_create_router(self, admin_ctx, create_id_tenant, device_id,
                            session):
        """sync create router"""
        (method, res_url) = \
            util.ACUtil.get_method_and_resource_url(
                ac_const.NW_HW_CREATE_ROUTER)
        entry_info = self._get_formatted_data_for_neutron_sync(
            admin_ctx, device_id,
            (create_id_tenant[device_id], method, ac_const.NW_HW_ROUTERS))
        entry_info[ac_const.CLOUD_NAME_PARAM] = \
            cfg.CONF.huawei_ac_config.cloud_name
        rest_info = ACRestUtils.wrap_entry_info(
            method, ac_const.NW_HW_ROUTERS, [entry_info])
        rest_data = ACRestUtils.fix_json(
            jsonutils.dumps(rest_info))
        LOG.debug('[AC] create rest data %s', rest_data)
        url = util.ACUtil.form_resource_url(
            ac_const.NW_HW_URL, method, res_url, device_id)
        result, response = self._send_data_to_ac(
            rest_data, url, None,
            (ac_const.NW_HW_ROUTERS, method, device_id))
        LOG.debug('[AC] sync create router %s finished, '
                  'result %s, response %s',
                  device_id, result, response)
        if result == ac_const.NW_HW_SUCCESS:
            LOG.debug('create plugin record for the router: %s [1]', device_id)
            self._db_if.create_plugin_record(session, {},
                                             (device_id, ac_const.OPER_CREATE,
                                              ac_const.NW_HW_ROUTERS),
                                             ac_const.NEUTRON_SYNC)
        return result

    def _process_router_related_port(self, router):
        """process router relatd port"""
        LOG.debug("[AC]Process router related ports: %s", router)
        try:
            from neutron.db.db_base_plugin_v2 import NeutronDbPluginV2
            ml2_db = NeutronDbPluginV2()
            for port in router['ports']:
                port_attr = self._db_if.get_port_attribute(res_id=port['uuid'])
                if port_attr:
                    LOG.info(_LI("[AC]Delete ports related with router: %s"),
                             router['uuid'])
                    ml2_db.delete_port(port['uuid'])
                data_info = \
                    {"id": port['uuid'],
                     "related_router": router['uuid'],
                     "name": port['name'],
                     "tenant_name": port['tenant-name'],
                     "created_at": port.get('created_at', ac_const.EMPTY_STR),
                     "updated_at": port.get('updated_at', ac_const.EMPTY_STR),
                     "result": ac_const.NW_HW_SUCCESS}
                if (ac_const.NW_HW_PORTS in
                        self._neutron_sync_result[ac_const.OPER_DELETE]):
                    self._neutron_sync_result[ac_const.OPER_DELETE][
                        ac_const.NW_HW_PORTS].append(data_info)
                else:
                    self._neutron_sync_result[ac_const.OPER_DELETE][
                        ac_const.NW_HW_PORTS] = [data_info]
            LOG.debug("[AC]Neutron sync result: %s", self._neutron_sync_result)
        except Exception as ex:
            LOG.debug("[AC]Process related router failed: %s", str(ex))

    def _compare_and_generate_diff_requests_for_exroute(self, neutron_data_list, ac_data_list, res_type):
        """ compare and generate diff request for exroute """
        total_record_count = 0
        if (self._sync_op in [ac_const.SYNC_OP_SYNC_DATA,
                              ac_const.SYNC_OP_SYNC_AND_COMPARE_DATA,
                              ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE] and self._need_delete) \
                or self._sync_op == ac_const.SYNC_OP_COMPARE_DATA:
            if self._sync_op == ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE:
                ac_data_list = self.filter_exroute_for_single_recover(ac_data_list, self.single_delete_lists)
            delete_list = self._left_diff_exroutes(ac_data_list, neutron_data_list, delete=True)
            for entry_info in delete_list:
                delete_data = {
                    'uuid': entry_info['router-id'],
                    'nexthop': entry_info['nexthop'],
                    'destination': entry_info['destination'],
                    'type': entry_info['type'],
                    'tenant-name': "",
                    'res': res_type,
                    'status': ac_const.STAT_WAIT
                }
                self._data_list.append(delete_data)
                total_record_count += 1

        if self._sync_op == ac_const.SYNC_OP_SYNC_SINGLE_INSTANCE:
            neutron_data_list = self._intersect_exroutes(neutron_data_list, self.single_create_lists)
        create_list = self._left_diff_exroutes(ac_data_list, neutron_data_list)
        for entry_info in create_list:
            create_data = {
                'uuid': entry_info['router-id'],
                'nexthop': entry_info['nexthop'],
                'destination': entry_info['destination'],
                'tenant-name': entry_info['tenant-name'],
                'type': entry_info.get('type'),
                'res': res_type,
                'status': ac_const.STAT_WAIT
            }
            self._create_data_list.append(create_data)
        return total_record_count

    def filter_exroute_for_single_recover(self, ac_data_list, req_list):
        ac_data_list_after_filter = []
        if not ac_data_list or not req_list:
            return ac_data_list_after_filter
        for req_item in req_list:
            ac_data = self.find_ac_resource(ast.literal_eval(req_item), ac_data_list)
            if ac_data:
                ac_data_list_after_filter.append(ac_data)
        return ac_data_list_after_filter

    def find_ac_resource(self, req_item, data_list):
        for ac_item in data_list:
            if (ac_item['router-id'] == req_item['router_id']
                    and ac_item['nexthop'] == req_item['nexthop']
                    and ac_item['destination'] == req_item['destination']):
                return ac_item
        return None

    def _form_bulk_rest_body_for_exroute(
            self, admin_ctx, rec_list, operation):
        """form bulk rest body for exroute"""
        rest_data = None
        length = len(self._data_list)
        for index in rec_list:
            if index >= length:
                LOG.error(_LE('index error in func _form_bulk_rest_body, '
                              'index %d, length %d.'), index, length)
                continue
            record = self._data_list[index]
            if operation == ac_const.OPER_CREATE:
                entry_info = self.neutron_formatter.set_exroute(
                    admin_ctx, record['tenant-name'],
                    router_id=record['uuid'],
                    attribute=record)
            else:
                entry_info = {
                    'router-id': self._data_list[index]['uuid'],
                    'nexthop': self._data_list[index]['nexthop'],
                    'type': self._data_list[index]['type'],
                    'destination': self._data_list[index]['destination']}
            rest_data = entry_info

        if not rest_data:
            return None, None
        if (operation == ac_const.OPER_CREATE or
                operation == ac_const.OPER_DELETE):
            rest_info = {ac_const.EXROUTES_INPUT: rest_data}
        else:
            LOG.error(_LE("Exroute don't support the operation of %s"),
                      operation)
        LOG.debug('Rest info formed: %s', rest_info)
        rest_data_copy = copy.deepcopy(rest_data)
        return ACRestUtils.fix_json(jsonutils.dumps(rest_info)), rest_data_copy

    def _neutron_sync_update_firewall_policy(self, attr_list, ac_data, entry_info, res_type):
        """neutron sync update firewall policy"""
        neutron_updated_at = attr_list[entry_info['uuid']] \
            if entry_info['uuid'] in attr_list else None
        if not ac_data:
            return
        if ac_data['uuid'] == entry_info['uuid'] \
                and ((lambda x, y: (x > y) - (x < y))
                         (ac_data['firewall_rules'],
                          entry_info['firewall_rules'])
                     or neutron_updated_at != ac_data['updated-at']):
            update_info = {
                'res': res_type,
                'uuid': entry_info['uuid'],
                'tenant-name': entry_info['tenant-name'],
                'status': ac_const.STAT_WAIT,
                'name': entry_info.get('name', ac_const.EMPTY_STR),
                'created_at': entry_info.get('created_at', ac_const.EMPTY_STR),
                'updated_at': entry_info.get('updated_at', ac_const.EMPTY_STR)
            }
            self._update_data_list.append(update_info)

    @classmethod
    def _intersect_exroutes(cls, s1_data_list, s2_data_list):
        """获取s1和s2的交集

        :param s1_data_list: s1_data_list
        :param s2_data_list: s2_data_list
        :return: list of exroutes
        """
        s1_data_dict = dict()
        for exroute in s1_data_list:
            key, value = cls._format_exroute(exroute)
            s1_data_dict.update({key: value})
        s2_data_dict = {cls._format_exroute(exroute)[0]: exroute for exroute in s2_data_list}
        intersection = set(s1_data_dict.keys()).intersection(set(s2_data_dict.keys()))
        return [s1_data_dict.get(key) for key in intersection]

    @classmethod
    def _left_diff_exroutes(cls, s1_data_list, s2_data_list, delete=False):
        """获取s1和s2的差集，delete为true表示差集以s1为主体，否则以s2为主体

        :param s1_data_list: s1_data_list
        :param s2_data_list: s2_data_list
        :param delete: delete
        :return: list of exroutes
        """
        s1_data_dict = dict()
        for exroute in s1_data_list:
            key, value = cls._format_exroute(exroute)
            s1_data_dict.update({key: value})
        s2_data_dict = {cls._format_exroute(exroute)[0]: exroute for exroute in s2_data_list}
        if delete:
            diff_set = set(s1_data_dict.keys()) - set(s2_data_dict.keys())
            return [s1_data_dict.get(key) for key in diff_set]
        else:
            diff_set = set(s2_data_dict.keys()) - set(s1_data_dict.keys())
            return [s2_data_dict.get(key) for key in diff_set]

    @staticmethod
    def _format_exroute(exroute):
        if isinstance(exroute, str):
            exroute = ast.literal_eval(exroute)
        key = '%s_%s_%s' % (exroute.get('router-id') or exroute.get('router_id'),
                            exroute['nexthop'], exroute['destination'])
        return key, exroute
