#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""verify cert thread class"""

from datetime import datetime
from threading import Timer
import six
from oslo_config import cfg
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging

from networking_huawei._i18n import _LE

from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import fusion_sphere_alarm as fsa
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.common import constants as ac_constants

LOG = logging.getLogger(__name__)

PLUGIN_TYPE = "iMaster NCE-Fabric"

# one day
VERIFY_CIRCLE_TIME = 60 * 60 * 24

CERT_FILE_DICT = {
    'https_ca_cert': ncu.HTTPS_CA_CERT,
    'websocket_ca_cert': ncu.WEBSOCKET_CA_CERT,
    'websocket_client_cert': ncu.WEBSOCKET_CLIENT_CERT
}


class ACCertSync(object):
    """ verify cert thread class """

    def __init__(self):
        self.rest_client = ACReSTClient()
        timer_verify_cert = Timer(0, self._verify_cert_expire_thread)
        timer_verify_cert.start()
        LOG.debug("[AC] verify cert expired time thread initialization done.")

    def _verify_cert_expire_thread(self):
        try:
            current_time = datetime.utcnow()
            cert_expiration_warning_time = \
                cfg.CONF.huawei_ac_config.cert_expiration_warning_time
            if ncu.SUPPORT_VERIFY_CERT:
                CERT_FILE_DICT.update({'http_client_cert': ncu.FS_CERT_FILE})
            self.process_cert(current_time, cert_expiration_warning_time)
        except Exception as ex:
            LOG.error(_LE("[AC] verify cert expired time error:  %s"),
                      ex)
        finally:
            timer = Timer(VERIFY_CIRCLE_TIME, self._verify_cert_expire_thread)
            timer.start()

    @classmethod
    def send_recovery(cls, reason, cert_type):
        """send cert alarm recovery.

        :param reason: alarm reason
        :param cert_type: type of certification
        :return: None
        """
        if not ncu.SUPPORT_VERIFY_CERT:
            return
        reason['detail'] = "%s alarm recovered." % cert_type
        alarm_info = fsa.ACPluginAlarm.get_cert_recovery_alarm(
            fsa.CERT_ABOUT_TO_EXPIRED, reason)
        fsa.ACPluginAlarm.send_alarm(alarm_info)
        alarm_info = fsa.ACPluginAlarm.get_cert_recovery_alarm(
            fsa.CERT_EXPIRED, reason)
        fsa.ACPluginAlarm.send_alarm(alarm_info)

    def process_cert(self, current_time, cert_expiration_warning_time):
        """ process cert """
        for cert_type, cert_file in six.iteritems(CERT_FILE_DICT):
            with open(cert_file, "rb") as cert:
                cer = cert.read()
            expired_time = ncu.get_cert_expired_time(cer)
            reason = {
                'plugin_type': PLUGIN_TYPE,
                'cert_type': cert_type
            }
            ACCertSync.send_recovery(reason, cert_type)
            days_to_go = (expired_time - current_time).days
            if days_to_go <= 0:
                # send alarm: already expired
                self._send_cert_expired_alarm_to_controller(cert_file,
                                                            cert_type)
                if not ncu.SUPPORT_VERIFY_CERT:
                    continue
                reason['detail'] = "%s has expired." % cert_type
                alarm_info = fsa.ACPluginAlarm.get_cert_expired_alarm(
                    fsa.CERT_EXPIRED, reason)
                fsa.ACPluginAlarm.send_alarm(alarm_info)
            elif days_to_go <= cert_expiration_warning_time:
                # send alarm: about to expired
                self._send_cert_will_expired_alarm_to_controller(cert_file,
                                                                 cert_type,
                                                                 days_to_go)
                if not ncu.SUPPORT_VERIFY_CERT:
                    continue
                reason['detail'] = "%s is about %s days to expired." % \
                                   (cert_type, days_to_go)
                alarm_info = fsa.ACPluginAlarm.get_cert_expired_alarm(
                    fsa.CERT_ABOUT_TO_EXPIRED, reason)
                fsa.ACPluginAlarm.send_alarm(alarm_info)

    def _send_cert_expired_alarm_to_controller(self, cert_file, cert_type):
        LOG.info("[AC]cert:%s has expired. Send alarm message.",
                 cert_file)
        alarm_description = {
            'cert_type': cert_type,
            'info': "Certification already expired."
        }
        alarm_category = {
            'alarm_inner_id':
                ac_constants.CLOUD_CERT_EXPIRED_ALARM_ID,
            'alarm_level':
                ac_constants.CLOUD_CERT_EXPIRED_ALARM_LEVEL,
            'alarm_type': ac_constants.CLOUD_CERT_ALARM_TYPE
        }
        self.rest_client.send_cloud_alarm(
            alarm_description, None, alarm_category)

    def _send_cert_will_expired_alarm_to_controller(self, cert_file, cert_type,
                                                    days_to_go):
        LOG.info("[AC]cert:%s is about to expire. Send alarm "
                 "message.", cert_file)
        alarm_description = {
            'cert_type': cert_type,
            'remain_days': days_to_go,
            'info': "Certification near to expire."
        }
        alarm_category = {
            'alarm_inner_id':
                ac_constants.CLOUD_CERT_ALMOST_EXPIRED_ALARM_ID,
            'alarm_level':
                ac_constants.CLOUD_CERT_ALMOST_EXPIRED_ALARM_LEVEL,
            'alarm_type': ac_constants.CLOUD_CERT_ALARM_TYPE
        }
        self.rest_client.send_cloud_alarm(
            alarm_description, None, alarm_category)
