#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""health detect"""

from __future__ import print_function

import os
import re
import subprocess
import six
import requests
import netaddr
from oslo_config import cfg
from oslo_serialization import jsonutils

try:
    from FSSecurity import crypt
except ImportError:
    import crypt

from networking_huawei.drivers.ac.common.util import ConfigParas, \
    DisplayProcess, check_father_process
from networking_huawei.drivers.ac.client.https_adapter import HWHTTPSAdapter
from networking_huawei.drivers.ac.common import config
from networking_huawei.drivers.ac.common import security_util
from networking_huawei.drivers.ac.common import constants as ac_constants

HUAWEI_CONFIG_FILE = os.path.realpath('/etc/neutron/huawei_driver_config.ini')
FSP_LOG = os.path.realpath('/var/log/fusionsphere/component/neutron-server/'
                           'neutron-server_info.log.200')
OPS_LOG = os.path.realpath('/var/log/neutron/server.log')

cfg.CONF.register_opts(config.HUAWEI_AC_DRIVER_OPTS, "huawei_ac_config")

BEFORE_FSP_6_3_1 = ['FusionSphere6.1', 'FusionSphere6.3.0']


class ManageBase(object):
    """Manage base class"""

    def __init__(self):
        self.huawei_config = ConfigParas(HUAWEI_CONFIG_FILE)

    @staticmethod
    def get_cmd(cmd):
        """get cmd"""
        pipe = os.popen(cmd)
        result = pipe.read().rstrip('\n')
        pipe.close()
        return result

    @staticmethod
    def choose_scene():
        """choose scene"""
        scene_info = 'Please choose scenario:\n' \
                     '[1]Openstack network overlay scenario\n' \
                     '[2]Openstack hybrid overlay scenario\n' \
                     '[3]FusionSphere network overlay scenario\n' \
                     '[4]FusionSphere hybrid overlay scenario\n' \
                     'Please choose from [1/2/3/4]: '
        while True:
            scene_num = six.moves.input(scene_info)
            if scene_num in ['1', '2', '3', '4']:
                break
        while True:
            bare_mod = six.moves.input('\nNeed to check bare metal bond mode?[y/n]: ')
            if bare_mod in ['y', 'n']:
                return scene_num, bare_mod

    def check_required_configuration(self, group_name, config_name):
        """check required filed general"""
        detail = self.huawei_config.get_config_detail(group_name, config_name)
        if detail:
            return config_name, 'configured', detail
        return config_name, 'failed', 'not configured.'

    def check_optional_configuration(self, group_name, config_name):
        """check optional filed general"""
        detail = self.huawei_config.get_config_detail(group_name, config_name)
        if detail:
            return config_name, 'configured', detail
        return config_name, 'not configured', 'not configured.'

    @staticmethod
    def if_ip(ip_address):
        """if ip or not"""
        if netaddr.valid_ipv4(ip_address) or netaddr.valid_ipv6(ip_address):
            return True
        return False

    def check_ip(self, ip_address):
        """check ip"""
        result = [False, False]
        cmd_ping4 = ['ping', '-c', '3', ip_address]
        if self.if_ip(ip_address):
            result[0] = True
        if not subprocess.call(cmd_ping4):
            result[1] = True
        return result

    def check_rpc_server_ip(self):
        """cehck rpc server ip"""
        rpc_server_ips = re.sub(r'\s', '', self.huawei_config.get_config_detail(
            'huawei_ac_agent_config', 'rpc_server_ip'))
        if not rpc_server_ips:
            return 'rpc_server_ip', 'failed', 'not configured.'
        for ac_ip in rpc_server_ips.split(','):
            if not self.check_ip(ac_ip)[0]:
                return 'rpc_server_ip', 'failed', \
                       rpc_server_ips + ' is not correct format.'
            if self.check_ip(ac_ip)[1]:
                return 'rpc_server_ip', 'ok', rpc_server_ips
        return 'rpc_server_ip', 'failed', rpc_server_ips + ' can not ping.'

    def check_host_ip(self):
        """check host ip"""
        host_ip = self.huawei_config.get_config_detail(
            'huawei_ac_agent_config', 'host_ip')
        if not host_ip:
            return 'host_ip', 'failed', 'not configured.'
        if netaddr.valid_ipv4(host_ip) or netaddr.valid_ipv6(host_ip):
            return 'host_ip', 'ok', host_ip
        return 'host_ip', 'failed', 'not in correct format.'

    def check_vhost_user(self, scene_num, true_num):
        """check vhost user"""
        vhost_user = self.huawei_config.get_config_detail(
            'huawei_ac_config', 'vhost_user', 'false')
        if (scene_num in true_num) == (vhost_user.lower() == 'true'):
            return 'vhost_user', 'ok', vhost_user
        return 'vhost_user', 'failed', vhost_user

    @staticmethod
    def check_req_result(req):
        """check req result"""
        if req.status_code == 401:
            return 'wrong password. can not get token with the password.'
        if req.status_code not in [200, 201, 202, 203, 204]:
            return 'something wrong with Agile Controller.'
        if not req.content:
            return 'no content in AC token response.'
        result_data = jsonutils.loads(req.content)
        if result_data['data']['token_id']:
            return 'success'
        return 'no token info in response.'

    def get_token_id(self, host, username, password):
        """get token id"""
        headers = {"Content-type": "application/json",
                   "Accept": "application/json",
                   "Accept-Language": "en-US"}
        auth_url = "%s%s%s%s%s" % (ac_constants.HTTPS_HEADER,
                                   host, ":",
                                   ac_constants.rest_server_port,
                                   ac_constants.AUTH_BASE_URL)
        auth = {"userName": username,
                "password": password}
        auth_data = jsonutils.dumps(auth)
        adapter = HWHTTPSAdapter(
            cert_file=None, key_file=None, password=None,
            pool_connections=100, pool_maxsize=100)
        requests.session().mount("https://", adapter)
        try:
            requests.packages.urllib3.disable_warnings()
            req = requests.session().request('POST',
                                             url=auth_url,
                                             headers=headers,
                                             data=auth_data,
                                             verify=False,
                                             timeout=10)
            return self.check_req_result(req)
        except Exception:
            return 'sending checking request failed.'

    def check_ac_passwd(self):
        """check ac passwd"""
        username = self.huawei_config.get_config_detail(
            'huawei_ac_config', 'ac_auth_username')
        password = self.huawei_config.get_config_detail(
            'huawei_ac_config', 'ac_auth_password')
        ops_version = self.huawei_config.get_config_detail(
            'huawei_ac_config', 'OPS_version')
        if not ops_version:
            return 'ac_auth_password', 'failed', 'OPS_version not configured'
        rpc_server_ips = re.sub(r'\s', '', self.huawei_config.get_config_detail(
            'huawei_ac_agent_config', 'rpc_server_ip')).split(',')
        if not password:
            return 'ac_auth_password', 'failed', 'not configured.'
        if re.match('^FusionSphere.+', ops_version) and \
                ops_version not in BEFORE_FSP_6_3_1:
            password_real = crypt.decrypt(password)
        else:
            password_real = six.text_type(security_util.decrypt_data(password),
                                          errors='ignore')
        for host in rpc_server_ips:
            token_result = self.get_token_id(host, username, password_real)
            if token_result == 'success':
                return 'ac_auth_password', 'ok', password
        return 'ac_auth_password', 'failed', token_result

    @staticmethod
    def _join_url(base_url, path):
        if base_url[-1:] != '/' and path[:1] != '/':
            url = '%s/%s' % (base_url, path)
        elif base_url[-1:] == '/' and path[:1] == '/':
            url = '%s%s' % (base_url[:-1], path)
        else:
            url = '%s%s' % (base_url, path)
        return url

    @classmethod
    def get_auth_url_ops(cls):
        """get auth url ops"""
        neutron_conf = ConfigParas('/etc/neutron/neutron.conf')
        auth_url = neutron_conf.get_config_detail('keystone_authtoken',
                                                  'auth_url')
        return auth_url

    def decrpt_keystone_passwd(self):
        """decrypt keystone password"""
        keystone_passwd = self.huawei_config.get_config_detail(
            'huawei_ac_config', 'keystone_passwd')
        ops_version = self.huawei_config.get_config_detail(
            'huawei_ac_config', 'OPS_version')
        if re.match('^FusionSphere.+', ops_version) and \
                ops_version not in BEFORE_FSP_6_3_1:
            password_real = crypt.decrypt(keystone_passwd)
        else:
            password_real = six.text_type(
                security_util.decrypt_data(keystone_passwd,
                                           data_type=2))
        return password_real

    def check_keystone_passwd(self):
        """check keystone password"""
        keystone_passwd = self.huawei_config.get_config_detail(
            'huawei_ac_config', 'keystone_passwd')
        if not keystone_passwd:
            return 'keystone_passwd', 'failed', 'not configured.'
        ops_version = self.huawei_config.get_config_detail(
            'huawei_ac_config', 'OPS_version')
        if not ops_version:
            return 'keystone_passwd', 'failed', 'OPS_version not configured.'
        password_real = self.decrpt_keystone_passwd()
        if re.match('^FusionSphere.+', ops_version):
            bash_cmd = os.path.realpath("/bin/bash")
            cmd = ['source', '/usr/bin/set_env']
            process_obj = subprocess.Popen(args=cmd, stdin=subprocess.PIPE,
                                           universal_newlines=True,
                                           executable=bash_cmd)
            process_obj.stdin.write("1\n")
            process_obj.stdin.write(password_real + "\n")
            std_output = process_obj.communicate()[0]
            process_result = process_obj.wait()
            if not process_result:
                cmd_net = ['neutron', 'net-list']
                source_result = subprocess.call(cmd_net)
            else:
                print(std_output)
                source_result = None
        else:
            keystone_tenant = self.huawei_config.get_config_detail(
                'huawei_ac_config', 'keystone_tenant', 'admin')
            keystone_user = self.huawei_config.get_config_detail(
                'huawei_ac_config', 'keystone_user', 'admin')
            auth_url = self.get_auth_url_ops()
            os.environ['OS_PROJECT_DOMAIN_NAME'] = 'Default'
            os.environ['OS_USER_DOMAIN_NAME'] = 'Default'
            os.environ['OS_PROJECT_NAME'] = 'admin'
            os.environ['OS_PROJECT_NAME'] = keystone_tenant
            os.environ['OS_USERNAME'] = keystone_user
            os.environ['OS_PASSWORD'] = password_real
            os.environ['OS_AUTH_URL'] = auth_url
            cmd_net = ['neutron', 'net-list']
            source_result = subprocess.call(cmd_net)
        if source_result == 0:
            return 'keystone_password', 'ok', keystone_passwd
        return 'keystone_password', 'failed', 'wrong keystone password'


    def check_neutron_server(self):
        """check neutron server"""
        server_inf = self.get_cmd("ps -ef | grep 'neutron-server' | "
                                  "grep '^neutron*'").split('\n')
        server_inf.extend(self.get_cmd("ps -ef | grep 'neutron-server' | "
                                       "grep '^opensta*'").split('\n'))
        server_inf = [_f for _f in server_inf if _f]
        if not server_inf:
            return 'neutron-server', 'failed', 'neutron-server is not running.'
        len_ppid1 = check_father_process(server_inf)
        if len(len_ppid1) == 1:
            return 'neutron-server', 'ok', ''
        return 'neutron-server', 'failed', \
               'there are more than 1 father process of neutron-server'

    @staticmethod
    def get_log():
        """get log"""
        if os.path.exists(FSP_LOG):
            return FSP_LOG
        elif os.path.exists(OPS_LOG):
            return OPS_LOG
        return None

    @staticmethod
    def check_thread_time(gap, std):
        """check thread time"""
        if len(gap.split('.')[0]) > 4:
            return False
        if float(gap) > std:
            return False
        return True

    def check_normal_sync(self):
        """check normal sync"""
        neutron_log = self.get_log()
        sync_time = self.get_cmd("grep 'normal sync heart beat time stamp' %s "
                                 "| tail -1" % neutron_log).split('stamp: ')[-1]
        monitor_time = self.get_cmd("grep 'normal sync monitor time stamp' %s |"
                                    " tail -1" %
                                    neutron_log).split('stamp:')[-1]
        now_time = self.get_cmd("date +'%s'")
        sync_gap = self.get_cmd("echo '%s - %s'| bc -l" % (now_time, sync_time))
        monitor_gap = self.get_cmd("echo '%s - %s'| bc -l" %
                                   (now_time, monitor_time))
        if not self.check_thread_time(sync_gap, 150):
            return 'normal_sync thread', 'failed', 'normal_sync thread is dead.'
        if not self.check_thread_time(monitor_gap, 450):
            return 'normal_sync thread', 'failed', 'monitor thread is dead.'
        return 'normal_sync thread', 'ok', ''


def main():
    """Main function of health detect.

    :return: None
    """
    manage_base = ManageBase()
    results = []
    scene_num, bare_mode = manage_base.choose_scene()
    results.append(manage_base.check_rpc_server_ip())
    if scene_num in ['1', '2']:
        results.append(manage_base.check_host_ip())
    results.append(manage_base.check_required_configuration(
        'huawei_ac_config', 'ac_auth_username'))
    results.append(manage_base.check_ac_passwd())
    results.append(manage_base.check_required_configuration(
        'huawei_ac_config', 'cloud_name'))
    results.append(manage_base.check_required_configuration(
        'huawei_ac_config', 'OPS_version'))
    results.append(manage_base.check_required_configuration(
        'huawei_ac_config', 'keystone_tenant'))
    results.append(manage_base.check_required_configuration(
        'huawei_ac_config', 'keystone_user'))
    results.append(manage_base.check_keystone_passwd())
    results.append(manage_base.check_required_configuration(
        'huawei_ac_config', 'physical_network'))
    DisplayProcess.display_result('required configurations', results)
    print('\n')

    results = []
    results.append(manage_base.check_vhost_user(scene_num, ['2', '4']))
    if scene_num in ['3', '4']:
        results.append(manage_base.check_optional_configuration(
            'huawei_ac_config', 'logical_hostnames'))
        results.append(manage_base.check_optional_configuration(
            'huawei_ac_config', 'vcenter_host_mappings'))
        results.append(manage_base.check_optional_configuration(
            'huawei_ac_config', 'vcenter_cluster_mappings'))
        results.append(manage_base.check_optional_configuration(
            'huawei_ac_config', 'vcenter_network_mappings'))
        results.append(manage_base.check_optional_configuration(
            'huawei_ac_config', 'primary_interface'))
        results.append(manage_base.check_optional_configuration(
            'huawei_ac_config', 'vpc_peering'))
    if bare_mode == 'y':
        results.append(manage_base.check_optional_configuration(
            'huawei_ac_config', 'bare_metal_bond_mode'))
    DisplayProcess.display_result('optional configurations', results)
    print('\n')

    results = [manage_base.check_neutron_server(),
               manage_base.check_normal_sync()]
    DisplayProcess.display_result('process & thread', results)


if __name__ == '__main__':
    main()
