#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.

"""Exceptions used by ML2."""

from webob.exc import HTTPError

# try part for M\O ops version,and exception part for L/K ops version.
NEUTRON_LIB_EXIST_FLG = False
try:
    from neutron.common.exceptions import NeutronException
except ImportError:
    from neutron_lib.exceptions import NeutronException

    NEUTRON_LIB_EXIST_FLG = True
from networking_huawei._i18n import _


class MechanismDriverError(NeutronException):
    """Mechanism driver call failed."""
    message = _("Request failed, method: %(method)s, "
                "url: %(url)s, error: %(error)s")


class OamException(Exception):
    """unknown exception"""
    message = "An unknown exception occurred."

    def __init__(self, message=None, **kwargs):
        """ init OamException """
        if message is None:
            message = self.message
        try:
            message = message % kwargs
            super(OamException, self).__init__(message)
        except KeyError:
            super(OamException, self).__init__(self.message)


class TimeoutException(OamException):
    """timeout exception"""

    def __init__(self, message="Timeout", **kwargs):
        super(TimeoutException, self).__init__(message, **kwargs)


class ContinueTimeoutException(OamException):
    """continue timeout"""
    message = "ContinueTimeout"


class JsonLengthException(OamException):
    """ Exceed max length """

    def __init__(self, message="Exceed max length", **kwargs):
        """init JsonLengthException"""
        super(JsonLengthException, self).__init__(message, **kwargs)


class SyncConnectionException(OamException):
    """ Sync connection exception """

    def __init__(self, message="Sync connection exception", **kwargs):
        """ init SyncConnectionException """
        super(SyncConnectionException, self).__init__(message, **kwargs)


class NoTerminalFromIpException(OamException):
    """ no terminal exception """
    message = "There is no terminal from your ip"


class NoConnectionFromIpException(OamException):
    """ no connection exception """
    message = "There is no connection from your ip"


class NoRpcClientFromIpException(OamException):
    """ no rpcclient exception """
    message = "There is no RpcClient from your ip"


class CrtFileNotFoundException(OamException):
    """ crt file not found """
    message = "Crt file not found"


class NeutronSyncError(OamException):
    """Neutron sync operation failed."""
    message = _("Neutron sync operation failed(Reason: %(reason)s).")


class NeutronSyncOrCompareError(NeutronException):
    """Neutron sync operation failed."""
    message = _("Neutron sync operation failed.(Operation: %(operation)s, "
                "Reason: %(reason)s).")


class CompareOverwriteError(NeutronException):
    """ compare overwrite """
    message = _("Compare-result file already exists.")


class AuthenticationException(Exception):
    """AC account authorize fail."""
    message = _("AC authorize failed")


class ACConnectionException(Exception):
    """ AC Northbound heart beat Failed."""
    message = _("Connect to AC failed")


class InternalServerException(Exception):
    """ AC status is not normal."""
    message = _("Internal Server Error")


class OtherException(Exception):
    """ Get AC token Failed for other reasons"""
    message = _("Unknown Error")


class ResourceReferenceException(NeutronException):
    """ Get AC token Failed for other reasons"""
    message = _("Resource be Referenced ")


class WrongSANException(NeutronException):
    """ subjectAltName in AC server cert is not permitted """
    message = "subjectAltName in AC server cert is not permitted"


class CertExpiredException(NeutronException):
    """ cert has been expired  """
    message = "cert has been expired"


class CertRevokedException(NeutronException):
    """ cert is revoked  """
    message = "%(cert)s cert is revoked"


class CertNotExistException(NeutronException):
    """ There is no such cert  """
    message = "There is no such cert"


class CertGetFailedException(NeutronException):
    """Get cert failed"""
    message = "Get cert failed"


class QueryWhiteListError(OamException):
    """Query white list file Failed for other reasons"""
    message = _("Query white list failed(Reason: %(reason)s).")


class BlockedByNeutronSyncException(NeutronException):
    """ operation blocked by neutron sync """
    message = "Operation blocked by neutron sync"


class CanNotUpdateFixedIpException(NeutronException):
    """Update fixed ips to None Exception"""
    message = "Can not update fixed_ips to None"


class HostStatusError(NeutronException):
    """Controller host link status error """
    message = "Host status of controller error: %(reason)s"


class DryRunError(NeutronException):
    """dry run occur error"""
    message = "Dry run occur error:%(reason)s"


class ExecuteShellError(NeutronException):
    """
    run shell occur error,因为neutron版本问题不得不重写初始化方法
    """

    def __init__(self, **kwargs):
        self.message = _("Execute command occur error:%(error)s")
        if not NEUTRON_LIB_EXIST_FLG:
            self.msg = self.message % kwargs
        else:
            super(ExecuteShellError, self).__init__(**kwargs)


class ACHttpException(HTTPError):
    """处理ac返回HTTP响应的异常情况"""

    def __init__(self, ac_response, *args, **kwargs):
        super(ACHttpException, self).__init__(*args, **kwargs)
        self.msg = 'ac response:%s %s' % (ac_response, ac_response.content)
        self.code = ac_response.status_code

    def __str__(self):
        """解决neutron.wsgi.JSONDictSerializer.default()序列化成''的问题"""
        return self.msg


class GetAcDataError(NeutronException):
    """ Get data from ac failed for unknown error"""
    message = _("Get ac data failed(Reason: %(reason)s).")