#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ac agent api"""
import ast
import os
import time
import traceback
import uuid
from collections import OrderedDict

from neutron.db import models_v2
from oslo_config import cfg
from oslo_serialization import jsonutils

try:
    from neutron import context
except ImportError:
    try:
        from neutron_lib import context
    except ImportError:
        context = None

from networking_huawei._i18n import _LI, _LE
from networking_huawei.common import constants as constants
from networking_huawei.common import exceptions as exceptions
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.db.ac_proc_status.ac_proc_status_db import ACProStatusDbMixin
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.schema import ACFailedResources
from networking_huawei.drivers.ac.extended_support import AcExtendedSupport, NeutronSyncConflict
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.model.port_model import ACPortModel
from networking_huawei.drivers.ac.neutron_client.health_check import HealthCheck
from networking_huawei.drivers.ac.sync.message_reliability_api import ACReliabilityAPI
from networking_huawei.drivers.ac.sync.check_and_recover import CheckAndRecover

LOG = ncu.ac_log.getLogger(__name__)

NAME_MAX_LENGTH = 128
STRING_MAX_LENGTH = 255

SYNC_RESOURCE_DICT = {
    ac_constants.NW_HW_MIGRATE_PORT: "_process_migrate_port_sync",
    ac_constants.NW_HW_DHCP_PORT: "_process_dhcp_port_sync",
    ac_constants.NW_HW_PORTS: "_process_port_sync",
    ac_constants.NW_HW_ROUTERS: "_process_router_sync",
    ac_constants.NW_HW_FIP: "_process_fip_sync",
    ac_constants.NW_HW_FIREWALL: "_process_firewall_sync",
    ac_constants.NW_HW_FIREWALL_RULE: "_process_firewall_rule_sync",
    ac_constants.NW_HW_LOADBALANCER: "_process_loadbalancer_sync",
    ac_constants.NW_HW_POOL: "_process_pool_sync",
    ac_constants.NW_HW_LISTENER: "_process_listener_sync",
    ac_constants.NW_HW_MEMBER: "_process_member_sync",
    ac_constants.NW_HW_HEALTH_MONITOR: "_process_health_monitor_sync",
    ac_constants.NW_HW_VPN_SERVICE: "_process_vpn_service_sync",
    ac_constants.NW_HW_IPSEC_SITE_CONNECTION:
        "_process_ipsec_site_connection_sync",
    ac_constants.NW_HW_L2BR: "_process_l2br_sync",
    ac_constants.NW_HW_VPC_CONNECTION:
        "_process_vpc_connection",
    ac_constants.NW_HW_UP_PORT: "_process_up_port",
    ac_constants.NW_HW_DNAT: "_process_dnat_sync",
    ac_constants.NW_HW_LAST_DHCP_PORT: "_process_last_dhcp_port"
}

REQUEST_TO_OPERATION = {
    constants.REST_POST: ac_constants.OPER_CREATE,
    constants.REST_DELETE: ac_constants.OPER_DELETE,
    constants.REST_UPDATE: ac_constants.OPER_UPDATE,
    constants.REST_GET: ac_constants.OPER_GET
}


class ACAgentAPI(object):
    """ac agent api"""

    def __init__(self):
        LOG.info(_LI("[AC]AC agent api init"))
        self.reliability_api = ACReliabilityAPI()
        self.db_if = ACdbInterface()
        self.ops_version = ncu.get_ops_version()
        self.ac_proc_db_func = ACProStatusDbMixin()
        self.vhost_user = cfg.CONF.huawei_ac_config.vhost_user
        self.health_check = HealthCheck()
        self.client = ACReSTClient()
        self.ac_ext_support = AcExtendedSupport()
        self.check_and_recover = CheckAndRecover()

    def sync_msg(self, res_id, name, attribute, description=None):
        """ Sync information from AC to OpenStack
        :param res_id: resource id
        :param name: resource name(port or dhcp port)
        :param attribute: feature dict
        :param description: description information
        :return: result dict
        """

        if not self._check_data_type(name, attribute):
            return {"result": False,
                    "error_msg": "Input parameters error"}

        LOG.info(_LI("[AC]Start to sync msg from AC, res_id: %(res_id)s, "
                     "name: %(name)s, attribute: %(attribute)s, "
                     "decription: %(description)s"),
                 {"res_id": res_id, "name": name,
                  "attribute": attribute,
                  "description": description})
        return self._process_resource_sync(name, res_id,
                                           attribute, description)

    def _process_resource_sync(self, name, res_id, attribute, description):
        """Process resource sync"""
        if name in SYNC_RESOURCE_DICT:
            func = getattr(self, SYNC_RESOURCE_DICT[name])
            return func(res_id, attribute, description)
        LOG.error(_LE("[AC]Don't support to sync the resource: %s"), name)
        return {"result": False,
                "error_msg": "The resource don't support to sync"}

    def _get_port_status(self, res_id):
        """Get port status"""
        session = self.db_if.get_session()
        rec = session.query(models_v2.Port).filter_by(id=res_id).first()
        return rec

    def _process_port_sync(self, res_id, attribute, description):
        """Process port sync"""
        try:
            port_status = attribute['status']
            operation = attribute['operation']
            if operation == constants.REST_POST:
                opr = ac_constants.OPER_CREATE
            elif operation == constants.REST_UPDATE:
                opr = ac_constants.OPER_UPDATE
            else:
                _msg = "[AC]Sync port Status not support the operation: %s" % operation
                raise Exception(_msg)
            session = self.db_if.get_session('read')
            session.bind.pool._max_overflow = 70
            if port_status == ac_constants.NEUTRON_STATUS_ERROR:
                LOG.debug("[AC]Port status is error: %s", description)
                failed_resource = ACFailedResources(id=res_id, res_type=ac_constants.NW_HW_PORTS,
                                                    operation='%s_%s' % (opr, ac_constants.NW_HW_PORTS))
                self.db_if.create_or_update_failed_resource(failed_resource, False)
            else:
                self.db_if.delete_failed_resource(res_id)

            if "vm_report" in attribute and attribute.get("vm_report"):
                session = self.db_if.get_session('write')
                self.db_if.update_neutron_db_state(
                    session, res_id, ac_constants.NW_HW_PORTS,
                    port_status)
            else:
                self.reliability_api.update_resource_status(
                    res_id, opr, ac_constants.NW_HW_PORTS, port_status)
                self._update_ac_proc_status_table(res_id, port_status)
        except Exception as ex:
            LOG.error(_LE("[AC]Sync port status exception: %s"), str(ex))
            return {"result": False, "error_msg": str(ex)}

        return {"result": True, "error_msg": ""}

    def _update_ac_proc_status_table(self, res_id, port_status):
        # write port status from AC to ac_proc_status_table
        if ncu.IS_FSP:
            if ac_constants.NW_HW_AC_STATUS_PORT in [cfg.CONF.ml2.extension_drivers]:
                device_info = {'id': res_id, 'type': ac_constants.NW_HW_PORTS, 'ac_proc_status': port_status}
                self.ac_proc_db_func.update_db_ac_proc_status(device_info)

    def _process_migrate_port_sync(self, res_id, attribute, description=None):
        """ When migratting vm with ce1800v, vm will report the
            event of up_port to AC , AC will report the event
            of migrate_port to AC plugin, then AC plugin will
            send the request of update_port to AC.
        """
        LOG.info(_LI("[AC]Process migrate port: %s, attribute: %s, "
                     "description: %s"), res_id, attribute, description)
        try:
            ctx = context.get_admin_context()
            query_port = ncu.get_core_plugin().get_port(ctx, res_id)
            if query_port['binding:host_id'] == attribute['hostid']:
                return {"result": True, "error_msg": ""}

            tenant_name = ACKeyStoneIf.get_tenant_name(
                query_port, ctx)

            query_port['binding:profile'].update(
                {'migrating-host': attribute['hostid']})
            port_info = ACPortModel.ac_model_format(
                query_port, tenant_name)
            self.reliability_api.update_plugin_record(
                ctx, res_id, port_info, ac_constants.NW_HW_UPDATE_PORT)
        except Exception as ex:
            LOG.error(_LE("[AC]Process migrate port failed: %s"), ex)

        return {"result": True, "error_msg": ""}

    def _process_dhcp_port_sync(self, res_id, attribute, description=None):
        """Process dhcp port sync"""
        LOG.debug("[AC]Process dhcp port, operation: %s",
                  attribute["operation"])
        ctx = context.get_admin_context()
        if attribute["operation"] == constants.REST_POST:
            result = self._notify_create_dhcp_port(ctx, attribute)
        elif attribute["operation"] == constants.REST_DELETE:
            result = self._notify_delete_dhcp_port(ctx, res_id)
        elif attribute["operation"] == constants.REST_UPDATE:
            result = self._notify_update_dhcp_port(ctx, res_id, attribute)
        else:
            result = {"result": False,
                      "error_msg": "Can't process the operation"}
        return result

    def wait_for_subnet_complete(self, attribute):
        """Wait for subnet complete create or update"""
        retry_time = 0
        max_retry_times = \
            cfg.CONF.huawei_ac_config.ac_response_time // \
            ac_constants.REPORT_STATUS_INTERVAL_TIME
        # if the subnet is during processing, wait
        session = self.db_if.get_session()
        while retry_time < max_retry_times - 1:
            loop_flag = False
            for subnet_attr in attribute["subnet_id"]:
                res_subnet = self.db_if.get_neutron_record(
                    session,
                    subnet_attr["subnet_id"],
                    ac_constants.NW_HW_SUBNETS)
                if res_subnet and not res_subnet.enable_dhcp:
                    loop_flag = True
            if loop_flag:
                retry_time += 1
                time.sleep(ac_constants.REPORT_STATUS_INTERVAL_TIME)
            else:
                break

    def _notify_create_dhcp_port(self, ctx, attribute):
        """Notify create dhcp port"""
        try:

            mac = ACCommonUtil.get_mac(self.ops_version)

            self.wait_for_subnet_complete(attribute)
            self.deal_dhcp_port(attribute, ctx, mac)
        except Exception as ex:
            LOG.error(_LE("[AC]Process creating dhcp port error, %s, "
                          "traceback: %s"), str(ex),
                      traceback.format_exc())
            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    @classmethod
    def deal_dhcp_port(cls, attribute, context_arg, mac):
        """call by :_notify_create_dhcp_port , create_dhcp_port_by_subnet"""
        device_uuid = uuid.uuid1()
        device_id = "dhcp%s-%s" % (device_uuid, device_uuid)
        if 'port_security' in cfg.CONF.ml2.extension_drivers:
            security_groups = []
        else:
            security_groups = object()
        dhcp_port = {
            "port":
                {
                    "dns_name": "",
                    "binding: host_id": "",
                    "allowed_address_pairs": [],
                    "extra_dhcp_opts": None,
                    "binding: vnic_type": "normal",
                    "device_owner": "network:dhcp",
                    "qos_policy_id": None,
                    "mac_address": mac,
                    "binding: profile": object(),
                    "security_groups": security_groups,
                    "name": "",
                    "admin_state_up": True,
                    "network_id": attribute["network_id"],
                    "tenant_id": attribute["tenant_id"],
                    "fixed_ips": attribute["subnet_id"],
                    "device_id": device_id
                }
        }
        if ncu.IS_FSP:
            dhcp_port['port']['security_groups'] = []
        return ncu.get_core_plugin().create_port(context_arg, dhcp_port)

    def _process_last_dhcp_port(self, attribute):
        """sleep for 10s in case the subnet is deleting,
           it may cause port create fail.
        """
        time.sleep(10)
        session = self.db_if.get_session()
        subnet_flag = False
        for subnet_attr in attribute["subnet_id"]:
            res_subnet = self.db_if.get_neutron_record(
                session,
                subnet_attr["subnet_id"],
                ac_constants.NW_HW_SUBNETS)
            if not res_subnet:
                subnet_flag = True
        if not subnet_flag:
            LOG.debug("[AC] create dhcp port when the last dhcp port is "
                      "deleted.")
            ctx = context.get_admin_context()
            return self._notify_create_dhcp_port(ctx, attribute)
        LOG.info("[AC] subnet not exist, no need to create dhcp port.")
        return {"result": True,
                "error_msg": ""}

    @classmethod
    def _notify_delete_dhcp_port(cls, ctx, res_id):
        """Notify delete dhcp port"""
        try:
            ncu.get_core_plugin().delete_port(ctx, res_id, l3_port_check=False)
        except Exception as ex:
            LOG.error(_LE("[AC]Process deleting dhcp port error, %s, "
                          "traceback: %s"), str(ex),
                      traceback.format_exc())
            return {"result": False,
                    "error_msg": str(ex)}
        return {"result": True,
                "error_msg": ""}

    @classmethod
    def _notify_update_dhcp_port(cls, ctx, res_id, attribute):
        """Notify update dhcp port"""
        try:
            dhcp_port = {
                "id": res_id,
                "port": {
                    "network_id": attribute["network_id"],
                    "fixed_ips": attribute['fixed_ips'],
                    "binding: host_id": attribute['binding: host_id']
                }
            }
            ncu.get_core_plugin().update_port(ctx, res_id, dhcp_port)
        except Exception as ex:
            LOG.error(_LE("[AC]Process updating dhcp port error, %s, "
                          "traceback: %s"), str(ex),
                      traceback.format_exc())
            return {"result": False,
                    "error_msg": str(ex)}
        return {"result": True,
                "error_msg": ""}

    def _process_router_sync(self, res_id, attribute, description):
        """Process router sync"""
        try:
            router_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_ROUTERS, router_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync router status exception: %s"), str(ex))
            alarm_description = {
                'operation': REQUEST_TO_OPERATION[operation],
                'res_type': ac_constants.NW_HW_ROUTERS,
                'res_id': res_id,
                'reason': "Update router status in db failed."
            }
            self.client.send_cloud_alarm(alarm_description)
            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_fip_sync(self, res_id, attribute, description):
        """Process fip sync"""
        try:
            fip_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_FIP, fip_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync fip status exception: %s"), str(ex))
            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_dnat_sync(self, res_id, attribute, description):
        """Process dnat sync"""
        try:
            status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_DNAT, status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync DNAT status exception: %s"), str(ex))
            return {"result": False, "error_msg": str(ex)}

        return {"result": True, "error_msg": ""}

    def _process_firewall_sync(self, res_id, attribute, description):
        """Process firewall sync"""
        try:
            firewall_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_FIREWALL, firewall_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync firewall status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_firewall_rule_sync(self, res_id, attribute, description):
        """Process firewall rule sync"""
        try:
            firewall_rule_status = attribute['status']
            operation = attribute['operation']

            if ncu.IS_FSP:
                if ac_constants.NW_HW_AC_STATUS_FIREWALL_POLICY in \
                        cfg.CONF.ml2.extension_drivers:
                    device_dict = {
                        'id': res_id,
                        'type': 'firewall_rule',
                        'ac_proc_status': firewall_rule_status
                    }
                    ac_status_dict = self.ac_proc_db_func. \
                        update_db_ac_proc_status(device_dict)
                    LOG.info(_LI("[AC] sync msg. Update ac process status for "
                                 "firewall-rule successful: %s. operation: %s "),
                             ac_status_dict, operation)
        except Exception as ex:
            LOG.error(_LE("[AC]Sync firewall rule status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}
        return {"result": True,
                "error_msg": ""}

    def _process_loadbalancer_sync(self, res_id, attribute, description):
        """Process loadbalancer sync"""
        try:
            loadbalancer_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_LOADBALANCER, loadbalancer_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync loadbalancer status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_pool_sync(self, res_id, attribute, description):
        """Process pool sync"""
        try:
            pool_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_POOL, pool_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync pool status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_listener_sync(self, res_id, attribute, description):
        """Process listener sync"""
        try:
            listener_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_LISTENER, listener_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync listener status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_member_sync(self, res_id, attribute, description):
        """Process member sync"""
        try:
            member_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_MEMBER, member_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync member status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_health_monitor_sync(self, res_id, attribute, description):
        """Process health monitor sync"""
        try:
            health_monitor_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_HEALTH_MONITOR, health_monitor_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync health monitor status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_vpn_service_sync(self, res_id, attribute, description):
        """Process vpn service sync"""
        try:
            vpn_service_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_VPN_SERVICE, vpn_service_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync vpn service status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_ipsec_site_connection_sync(self, res_id,
                                            attribute, description):
        """Process ipsec site connection sync"""
        try:
            ipsec_connection_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_IPSEC_SITE_CONNECTION,
                ipsec_connection_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync ipsec site connection status "
                          "exception: %s"), str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_vpc_connection(self, res_id, attribute, description):
        """Process vpc connection sync"""
        try:
            vpc_connection_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_VPC_CONNECTION,
                vpc_connection_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync vpc connection status "
                          "exception: %s"), str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_l2br_sync(self, res_id, attribute, description):
        """Process l2br sync"""
        try:
            l2br_status = attribute['status']
            operation = attribute['operation']

            self.reliability_api.update_resource_status(
                res_id, REQUEST_TO_OPERATION[operation],
                ac_constants.NW_HW_L2BR, l2br_status)

        except Exception as ex:
            LOG.error(_LE("[AC]Sync l2br status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _process_up_port(self, res_id, attribute, description):
        """Create vm with ce1800v, when vm started, ce1800v will report
           the event of up_port, then AC will report the event to AC plugin,
           AC plugin check that if the status of the port need to be update
        """
        try:
            if not cfg.CONF.huawei_ac_config.vhost_user:
                return {"result": True, "error_msg": ""}
            up_port_status = attribute['status']
            self.reliability_api.notify_nova_vif_plugged_and_update_status(
                res_id, up_port_status,
                attribute.get('description') == 'up_port_4_hot_migrate')

        except Exception as ex:
            LOG.error(_LE("[AC]Sync ce1800v port status exception: %s"),
                      str(ex))

            return {"result": False,
                    "error_msg": str(ex)}

        return {"result": True,
                "error_msg": ""}

    def _check_data_type(self, name, attribute):
        """Check data type"""
        try:

            # check name length
            self._check_str_length(name, NAME_MAX_LENGTH)

            # check attribute
            for key in attribute:
                self._check_str_length(str(attribute[key]),
                                       STRING_MAX_LENGTH)

        except Exception as ex:
            LOG.error(_LE("[AC]Input error, %s"), str(ex))
            return False

        return True

    @classmethod
    def _check_str_length(cls, message, length):
        """Check str length"""

        if len(message) > length:
            raise Exception("[AC]String length error")

    def get_sync_body(self):
        """Get sync body prepare to sync"""
        session = self.db_if.get_session()
        (status, _) = self.db_if.check_is_neutron_sync_in_progress(session)
        if status:
            raise NeutronSyncConflict()

        body = {
            'resourcesync': {
                'sync_start_time': None,
                'compare_data': None,
                'sync_res': None,
                'tenant_id': '',
                'expected_completion_time': 0,
                'sync_resources': '',
                'server': None,
                'state': None,
                'sync_data': None,
                'include_bindports': False,
                'resource_type': None
            }
        }
        return body, session

    def inconsistency_check(self, req):
        """Inconsistency check"""
        LOG.info(_LI("Create inconsistency check :%s"), req)
        try:
            body, session = self.get_sync_body()
            filename = ac_constants.NEUTRON_SYNC_COMPARE_RESULT_FILE
            body['resourcesync']['compare_data'] = {'overwrite': False, 'filename': filename}
            body.get('resourcesync')['sync_resources'] = str(req)

            LOG.info(_LI("Prepare to start inconsistency check: %s"), body)
            self.ac_ext_support.create_resourcesync(session, body)

            json_ret = {"result": "exec successfully", "error_msg": ""}
            return json_ret

        except Exception as ex:
            json_ret = {"result": "",
                        "error_msg": str(ex)}
            LOG.error(_LE("[AC]Inconsistency check raise exception: %s"),
                      json_ret)
            return json_ret

    def inconsistency_recover(self, req):
        """Inconsistency recover"""
        LOG.info(_LI("Create inconsistency recover :%s"), req)
        try:
            method_info = req.get('method')
            if method_info != 'inconsistency_recover':
                req['method'] = 'inconsistency_recover'
            body, session = self.get_sync_body()
            body['resourcesync']['sync_data'] = 'exclude-delete'
            body.get('resourcesync')['sync_resources'] = str(req)

            LOG.info(_LI("Prepare to start inconsistency recover: %s"), body)
            self.ac_ext_support.create_resourcesync(session, body)

            json_ret = {"result": "exec successfully", "error_msg": ""}
            return json_ret

        except Exception as ex:
            json_ret = {"result": "", "error_msg": str(ex)}
            LOG.error(_LE("[AC]Perform inconsistency recover raise exception: "
                          "%s"), json_ret)
            return json_ret

    def single_feature_recover(self, req):
        LOG.info(_LI("Create single feature recover :%s"), req)
        return self.inconsistency_recover(req)

    def multi_instance_recover(self, req):
        """Single instance recover"""
        LOG.info(_LI("Create single instance recover :%s"), req)
        try:
            body, session = self.get_sync_body()
            body.get('resourcesync')['sync_resources'] = str(req)

            LOG.info(_LI("Prepare to start single instance sync: %s"), body)
            self.ac_ext_support.create_resourcesync(session, body)

            json_ret = {"result": "exec successfully", "error_msg": ""}
            return json_ret

        except Exception as ex:
            json_ret = {"result": "", "error_msg": str(ex)}
            LOG.error(_LE("[AC]Perform single instance recover raise "
                          "exception: %s"), json_ret)
            return json_ret

    def single_feature_check(self, req):
        """单特性"""
        LOG.info(_LI("Create single instance check :%s"), req)
        try:
            check_type = []
            body, session = self.get_sync_body()
            params = req.get('params')
            if params and isinstance(params, list):
                check_type.append(params[0].get('attribute'))
                body['resourcesync']['resource_type'] = check_type
            else:
                LOG.error(_LI("Invalid params get from ac :%s"), params)
                return {"result": "", "error_msg": "Invalid params get from ac"}

            filename = ac_constants.NEUTRON_SYNC_COMPARE_RESULT_FILE
            body['resourcesync']['compare_data'] = {'overwrite': False, 'filename': filename}
            body.get('resourcesync')['sync_resources'] = str(req)

            LOG.info(_LI("Prepare to start single instance check: %s"), body)
            self.ac_ext_support.create_resourcesync(session, body)

            json_ret = {"result": "exec successfully", "error_msg": ""}
            return json_ret

        except Exception as ex:
            json_ret = {"result": "", "error_msg": "neutron plugin error"}
            LOG.error(_LE("[AC]Perform single instance check raise exception: %s,result: %s"),
                      str(ex.message), json_ret)
            return json_ret

    def single_instance_check(self, req):
        """单实例"""
        LOG.info(_LI("Create single data check :%s"), req)
        try:
            params = req.get('params')
            if params and isinstance(params, list):
                res_info = params[0]
            else:
                LOG.error(_LI("Invalid params get from ac :%s"), params)
                return {"result": "", "error_msg": "Invalid params get from ac"}

            res_data = self.check_and_recover.check_inconsistency(res_info)

            json_ret = {"result": res_data, "error_msg": ""}
            return json_ret

        except Exception as ex:
            json_ret = {"result": "", "error_msg": "neutron plugin error"}
            LOG.error(_LE("[AC]Perform single data check raise exception: %s,result: %s"), str(ex.message), json_ret)
            return json_ret

    def single_instance_recover(self, req):
        """Single data recover"""
        LOG.info(_LI("Create single data recover :%s"), req)
        try:
            params = req.get('params')
            if params and isinstance(params, list):
                res_info = params[0]
            else:
                LOG.error(_LI("Invalid params get from ac :%s"), params)
                return {"result": "", "error_msg": "Invalid params get from ac"}

            res_data = self.check_and_recover.recover_inconsistency(res_info)

            json_ret = {"result": res_data, "error_msg": ""}
            return json_ret
        except Exception as ex:
            json_ret = {"result": "", "error_msg": "neutron plugin error"}
            LOG.error(_LE("[AC]Perform single data recover raise exception: %s,result: %s"),
                      str(ex.message), json_ret)
            return json_ret

    @classmethod
    def query_whitelist(cls, req):
        """Query whitelist"""
        LOG.info(_LI("Query white list :%s"), req)
        whitelist_file = ac_constants.NEUTRON_SYNC_WHITELIST_FILENAME
        try:
            f_size = os.path.getsize(whitelist_file)
            if f_size > ac_constants.LIMIT_FILE_SIZE:
                raise exceptions.QueryWhiteListError(
                    reason=('Whitelist file-size is greater than %s'
                            % ac_constants.LIMIT_FILE_SIZE))

            with open(whitelist_file, 'r') as r_file:
                white_list = r_file.read()
                LOG.info(_LI("Get white list data :%s"), white_list)
                white_list = ast.literal_eval(white_list)

            json_ret = {"result": white_list, "error_msg": ""}
            return json_ret
        except Exception as ex:
            json_ret = {"result": "", "error_msg": str(ex)}
            LOG.error(_LE("[AC]Query white list raise exception: %s"), json_ret)
            return json_ret

    def set_whitelist(self, request):
        """Set whitelist function"""
        LOG.info(_LI("Start to set white list :%s"), request)
        params = request.get("params")
        if not params or not isinstance(params, list):
            LOG.error(_LE("Cannot get params or params is null, return"))
            return {"result": "",
                    "error_msg": "The params get from ac is not correct"}
        whitelist_info = params[0]
        whitelist_path = ac_constants.NEUTRON_SYNC_WHITELIST_FILENAME

        try:
            self.save_data_into_whitelist(whitelist_path, whitelist_info)
            LOG.info(_LI("Set white list success, return"))
            json_ret = {"result": "set successfully", "error_msg": ""}
            return json_ret

        except Exception as ex:
            json_ret = {"result": "", "error_msg": str(ex)}
            LOG.error(_LE("Set white list raise exception :%s"), json_ret)
            return json_ret

    @classmethod
    def save_data_into_whitelist(cls, whitelist_path, whitelist_info):
        """Save data into whitelist file"""
        request_keys = whitelist_info.keys()
        annotate = ""
        with open(whitelist_path, 'r') as r_list_file:
            ori_dic = ast.literal_eval(r_list_file.read())
            lines = r_list_file.readlines()
            for line in lines:
                if line.startswith("#"):
                    annotate += line

        for key in ori_dic.keys():
            if key not in ac_constants.WHITELIST_SEQUENCE:
                ac_constants.WHITELIST_SEQUENCE.append(key)
        OrderedDict(sorted(whitelist_info.items(), key=lambda x: x[0]))
        whitelist_dic = OrderedDict()
        for key in ac_constants.WHITELIST_SEQUENCE:
            if key in request_keys:
                whitelist_dic[key] = whitelist_info[key]

        json_data = \
            str(jsonutils.dumps(whitelist_dic, indent=4)).replace('"', "'")
        if len(json_data) > ac_constants.LIMIT_FILE_SIZE:
            raise exceptions.QueryWhiteListError(
                reason=('Whitelist file-size is greater than %s'
                        % ac_constants.LIMIT_FILE_SIZE))
        cls.write_whitelist_file(json_data, whitelist_path, annotate)

    @classmethod
    def write_whitelist_file(cls, json_data, whitelist_path, annotate):
        """
        write whitelist file
        """
        LOG.info(_LI("White list data is :%s"), json_data)
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        with os.fdopen(os.open(whitelist_path, flags, 0o600), 'w') as f_out:
            f_out.write(annotate)
            f_out.write(json_data)

    def health_detect(self, req):
        """Health detect"""
        LOG.info(_LI("Create health detect :%s"), req)
        try:
            scene = req.get('params')[0]['scene']
            result = self.health_check.check_ac_plugin(scene)

            json_ret = {"result": result, "error_msg": ""}
            LOG.info(_LI("Health detect info is : %s"), json_ret)
            return json_ret

        except Exception as ex:
            json_ret = {"result": "", "error_msg": str(ex)}
            LOG.error(_LE("Health detect raise exception :%s"), json_ret)
            return json_ret
