#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""REST client to Agile Controller."""

import base64
import os
import ssl
import time
import traceback
from contextlib import closing
from socket import create_connection

import eventlet
import netaddr
import six
from _ssl import CERT_NONE, CERT_REQUIRED
from oslo_config import cfg
from oslo_serialization import jsonutils
from oslo_utils import timeutils

from networking_huawei._i18n import _LI, _LE
from networking_huawei.common import constants
from networking_huawei.common.exceptions import AuthenticationException, \
    CertRevokedException, CertGetFailedException
from networking_huawei.common.exceptions import CertExpiredException
from networking_huawei.common.exceptions import CertNotExistException
from networking_huawei.common.exceptions import HostStatusError
from networking_huawei.common.exceptions import InternalServerException
from networking_huawei.common.exceptions import OtherException
from networking_huawei.common.exceptions import WrongSANException
from networking_huawei.drivers.ac.client.https_adapter import HWHTTPSAdapter
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common import security_util
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.common.fusion_sphere_alarm import \
    ACPluginAlarm
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.db.decrypt_factor.decrypt_factor_db import \
    DecryptFactorDbMixin
from networking_huawei.drivers.ac.db.lock import lockedobjects_db as lock_db
from networking_huawei.drivers.ac.encode_convert import convert_to_bytes
from networking_huawei.drivers.ac.model.cloud_alarm_model import CloudAlarmModel

try:
    from FSSecurity import crypt as fs_crypt
except ImportError:
    pass

requests = eventlet.import_patched('requests.__init__')

LOG = logging.getLogger(__name__)

HEARTBEAT_RETRY_TIMES = 1


class ACReSTClient(object):
    """REST client to Agile Controller."""

    def __init__(self):
        self.port = ac_constants.rest_server_port
        self.db_inter = dbif.ACdbInterface()
        self.host_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip. \
            replace(' ', '').split(',')
        self.host_list = [x.lower() for x in self.host_list]
        self.req_token_id = {self.host_list[0]: None}
        if len(self.host_list) == 2:
            self.req_token_id[self.host_list[1]] = None
        self.callback_restart_rpc = None
        self.neutron_name = ACCommonUtil.get_neutron_server_name()
        self.dfactor_db = DecryptFactorDbMixin()
        self.token_variable = None

    def send(self, method, url, res_id, body, max_times=None, timeout=None,
             headers=None, log_response=True):
        """Send the json data to the controller
        :param method: rest request method
        :param url: rest request
        :param res_id: resource id
        :param body: rest request data
        :param max_times: max retry count when timeout
        :param timeout: timeout of the rest call
        :param headers: header of the rest message
        :param log_response: Whether to log the response or not
        :return:
        """
        LOG.debug("[AC] Begin to send %s request to %s for %s", method, url, res_id)
        rest_data = body
        original_host = url.split(':' + str(self.port))[0].split('://')[1]

        original_host = self.update_host_ip(original_host)

        if original_host not in self.req_token_id:
            ac_osprofiler.record_chain_exception("send to IP:" + original_host
                                                 + " failed")
            raise requests.Timeout

        # if the first request, get the token id
        if not ('controller/dc/v2/neutronapi/cloud/northheartbeat' in url and self.token_variable):
            self.get_token_first_time(original_host, max_times)
            if self.token_variable != self.req_token_id.get(original_host):
                self.token_variable = self.req_token_id.get(original_host)

        token_retry = cfg.CONF.huawei_ac_config.token_retry
        ret = None

        if not headers:
            headers = {"Content-type": "application/json",
                       "Accept": "application/json",
                       "Accept-Language": "en-US"}

        while token_retry >= 0:
            if 'controller/dc/v2/neutronapi/cloud/northheartbeat' in url:
                headers["X-ACCESS-TOKEN"] = self.token_variable
            else:
                headers["X-ACCESS-TOKEN"] = self.req_token_id.get(original_host)
            ret = self._process_request(
                (headers, method, url, rest_data),
                max_times=max_times, timeout=timeout, print_message=True)
            LOG.debug("[AC] AC request response, status_code: %(status)s, "
                      "headers: %(headers)s", {'status': ret.status_code,
                                               'headers': ret.headers})
            if len(ret.content) < ac_constants.NW_HW_MAX_DEBUG_CONTENT_SIZE \
                    and log_response:
                LOG.debug("[AC] AC request response, content: %(content)s",
                          {'content': ret.content})

            # authorize fail, get token_id from ac
            if ret.status_code == requests.codes.unauthorized:
                try:
                    if token_retry > 0:
                        self.req_token_id[original_host] = \
                            self._get_token_id(original_host, self.port)
                        self.token_variable = self.req_token_id.get(original_host)
                except Exception as ex:
                    if token_retry == 0:
                        ac_osprofiler.record_chain_exception(
                            "AC authorize failed")
                        LOG.error(_LE("[AC] AC authorize failed: %s"), ex)
                        raise
                token_retry -= 1
            else:
                break

        return ret

    @classmethod
    def _format_data_log(cls, data):
        """format data log"""
        data_log = jsonutils.loads(data)
        if data_log.get('ipsec-site-connections'):
            data_log = data_log['ipsec-site-connections']
            if data_log.get('ipsecsiteconnection'):
                data_log = validate.validate_log_record(
                    data_log['ipsecsiteconnection'][0],
                    ac_constants.NW_HW_IPSEC_SITE_CONNECTION)
                data_log = {
                    'ipsec-site-connections': {
                        'ipsecsiteconnection': [data_log]
                    }
                }
        elif data_log.get('ipsecsiteconnection'):
            data_log = validate.validate_log_record(
                data_log['ipsecsiteconnection'][0],
                ac_constants.NW_HW_IPSEC_SITE_CONNECTION)
            data_log = {'ipsecsiteconnection': [data_log]}
        elif data_log.get('bgp-route'):
            data_log = validate.validate_log_record(
                data_log['bgp-route'][0],
                ac_constants.NW_HW_BGP_ROUTE)
            data_log = {'bgp-route': [data_log]}

        return jsonutils.dumps(data_log)

    def get_token_first_time(self, original_host, max_times):
        """Get token from Agile Controller for the first time."""
        get_token_info = self.db_inter.get_config_record(key=self.neutron_name)
        if not get_token_info:
            LOG.debug("[AC] Begin to get token for %s", self.neutron_name)
            self.req_token_id[original_host] = self._get_token_id(
                original_host, self.port, max_times=max_times)
        else:
            try:
                admin_ctx = ncu.neutron_context.get_admin_context()
                decrypt_dict = self.dfactor_db.get_db_decrypt_factor(
                    admin_ctx, self.neutron_name)
                decrypt_token_info = six.text_type(security_util.decrypt_data(
                    get_token_info.comment,
                    pw_key=base64.b64decode(convert_to_bytes(
                        decrypt_dict['pw_key'])),
                    pw_iv=base64.b64decode(
                        convert_to_bytes(decrypt_dict['pw_iv'])),
                    key_iv=base64.b64decode(
                        convert_to_bytes(decrypt_dict['key_iv']))
                ), errors='ignore')
                self.req_token_id[original_host] = decrypt_token_info.rsplit(
                    ':', 1)[0] if get_token_info else ''
            except Exception as ex:
                LOG.error("[AC] Decrypt token_info failed: %s", ex)
                self.req_token_id[original_host] = self._get_token_id(
                    original_host, self.port, max_times=max_times)

    def update_host_ip(self, original_host):
        """Get active host ip of Agile Controller from DB and update."""
        if len(self.host_list) != 1 and \
                original_host == ac_constants.DEFAULT_AC_IP:
            session = self.db_inter.get_session()
            active_host = self.db_inter.get_active_ac_ip(session)
            if active_host is not None and \
                    active_host != ac_constants.DEFAULT_AC_IP:
                original_host = active_host.lower()
        return original_host

    @classmethod
    def _process_request_msg_log(cls, method, data, active_url):
        """process request msg log"""
        if method in [constants.REST_DELETE, constants.REST_GET]:
            msg = _LI("[AC] Start send the request information, "
                      "method: %s, url: %s") % (method, active_url)
        else:
            if data:
                data_log = cls._format_data_log(data)
                msg = _LI("[AC] Start send the request "
                          "information, method: %s, url: %s, "
                          "data: %s") % (method, active_url,
                                         data_log)
            else:
                msg = _LI("[AC] Start send the request "
                          "information, method: %s, "
                          "url: %s") % (method, active_url)
        LOG.info(msg)

    @classmethod
    def _get_timeout_retry(cls, max_times):
        """get timeout retry"""
        if max_times is not None:
            return max_times
        return cfg.CONF.huawei_ac_config.timeout_retry

    def _get_host_index(self, url):
        """get host index"""
        original_host = url.split(':' + str(self.port))[0].split('://')[1]
        detect_request_index = url.find(ac_constants.NW_HW_CLUSTERS_URL)
        if len(self.host_list) == 1 or \
                (url.find(ac_constants.AUTH_BASE_URL) > 0 and
                 original_host in self.host_list):
            detect_request_index = 0
        return original_host, detect_request_index

    def get_active_url(self, url, detect_request_index, original_host):
        """Get active host ip of Agile Controller from DB and form url."""
        active_host = original_host
        if detect_request_index < 0:
            session = self.db_inter.get_session()
            active_host = self.db_inter.get_active_ac_ip(session)
            if active_host is None:
                raise AuthenticationException()
            if active_host == ac_constants.DEFAULT_AC_IP:
                LOG.error(_LE("[AC] Can not know which AC is active."))
                raise requests.Timeout
            if isinstance(active_host, str):
                active_host = active_host.lower()

        self.verify_ac_cert(active_host, url)

        active_url = url
        if original_host != active_host:
            active_url = url.replace(original_host, active_host)
        temp_host = active_url.split(':' + str(self.port))[0].split('://')[1]
        if netaddr.valid_ipv6(temp_host):
            active_url = active_url.replace(temp_host, '[' + temp_host + ']')
        return active_url

    @classmethod
    def get_res_content(cls, method, req):
        """Get content of the apply from Agile Controller."""
        if method in [constants.REST_DELETE, constants.REST_GET]:
            res_content = None
        else:
            res_content = req.content
        return res_content

    def _process_request(self, req_param, max_times=None,
                         timeout=None, print_message=False):
        headers, method, url, data = req_param
        timeout_retry = self._get_timeout_retry(max_times)
        # 2017.8.24 1) decrease 5s, for softcom project, use neutron api
        # timeout is 120s
        timeout = timeout or (cfg.CONF.huawei_ac_config.request_timeout -
                              ac_constants.REQUEST_TIMEOUT_DECREASE)
        LOG.debug('Sending request to controller with timeout: %d, '
                  'retry:%d', timeout, timeout_retry)
        original_host, detect_request_index = self._get_host_index(url)
        active_url, req = '', ''
        while timeout_retry >= 0:
            start_time = time.time()
            try:
                active_url = self. \
                    get_active_url(url, detect_request_index, original_host)
                if print_message:
                    self._process_request_msg_log(method, data, active_url)

                ac_osprofiler.record_call_chain("send request to controller")
                req_start = time.time()
                req = self._request_client(
                    {'headers': headers, 'method': method,
                     'url': active_url, 'data': data, 'timeout': timeout})
                if print_message:
                    LOG.info(_LI("[AC] Send the request response, status_code: %s, res_content: %s, req_cost: %ss"),
                             req.status_code, self.get_res_content(method, req), time.time() - req_start)
                # if return http connetion problem,
                # process it as timeout(status 502,503,504)
                self._handle_connection_pro(req)
                break
            except (requests.Timeout, requests.ConnectionError):
                LOG.error(_LE("[AC] Request sent to controller timed out, "
                              "request information: method: %s, url: %s "
                              "traceback: %s"), method,
                          active_url, traceback.format_exc())
                self._handle_req_timeout_alarm(active_url, method)
                self._process_request_retry_wait(timeout, start_time)
                if timeout_retry == 0:
                    self._process_first_timeout_retry(detect_request_index,
                                                      active_url)
                timeout_retry -= 1
                if timeout_retry >= 0:
                    ac_osprofiler.record_call_chain("time out,try it again")
            except Exception:
                LOG.error(_LE("[AC] Request send to controller failed, "
                              "request information: method: %s, url: %s "
                              "traceback: %s"), method,
                          active_url, traceback.format_exc())
                raise
        return req

    def _handle_req_timeout_alarm(self, active_url, method):
        # token, 心跳、告警、集群查询接口超时，不进行告警
        if ac_constants.AUTH_BASE_URL in active_url or ac_constants.NW_HW_CLOUD_HEART_BEAT in active_url:
            return
        if ac_constants.NW_HW_CLOUD_ALARM_URL in active_url and ac_constants.NW_HW_CLUSTERS_URL in active_url:
            return
        self.send_cloud_alarm(
                {'method': method, 'url': active_url,
                 'reason': 'Request sent to controller timed out'})

    @classmethod
    def _process_request_retry_wait(cls, req_timeout, start_time):
        end_time = time.time()
        sleep_time = req_timeout - (end_time - start_time)
        if sleep_time > 0:
            time.sleep(sleep_time)

    def _process_first_timeout_retry(self, detect_request_index, active_url):
        if detect_request_index < 0:
            LOG.error(_LE("%s response timeout, detect "
                          "whether controller had switchover."), active_url)
            session = self.db_inter.get_session()
            self.db_inter.update_active_ac_record(
                session, ac_constants.DEFAULT_AC_IP)
            self.detect_active_ac()
        raise requests.Timeout

    @classmethod
    def _handle_connection_pro(cls, req):
        """ handle connection process"""
        if ((req.status_code == requests.codes.bad_gateway) or
                (req.status_code == requests.codes.unavailable) or
                (req.status_code == requests.codes.gateway_timeout)):
            ac_osprofiler.record_call_chain("AC response status is" +
                                            str(req.status_code))
            raise requests.Timeout

    @classmethod
    def _get_certificate(cls):
        """get plugin cert path"""
        cert_paths = {}
        dir_path = os.path.dirname(os.path.realpath(__file__))
        ca_file = os.path.join(dir_path, 'ssl_cacert.pem')
        if not os.path.exists(ca_file):
            LOG.error(_LE('[AC] cert file does not exist'))
            raise CertNotExistException
        cert_paths['ca_file'] = ca_file
        if ncu.SUPPORT_VERIFY_CERT:
            fs_cert_file = ncu.FS_CERT_FILE
            fs_key_file = ncu.FS_KEY_FILE
            if not os.path.exists(fs_cert_file) \
                    or not os.path.exists(fs_key_file):
                LOG.error(_LE('[AC] cert file does not exist'))
                raise CertNotExistException
            else:
                cert_paths['cert_file'] = fs_cert_file
                cert_paths['key_file'] = fs_key_file
        return cert_paths

    @classmethod
    def get_ac_server_certificate(cls, addr, ssl_version=ssl.PROTOCOL_SSLv23,
                                  ca_certs=None):
        """get ac server certificate"""
        cert_reqs = CERT_NONE if ca_certs is None else CERT_REQUIRED
        context = ssl._create_stdlib_context(ssl_version,
                                             cert_reqs=cert_reqs,
                                             cafile=ca_certs)
        context.set_ciphers(ac_constants.SUPPORT_CIPHERS)
        local_address = ACCommonUtil.get_local_host_ip()
        src_addr = (local_address, 0)
        with closing(create_connection(addr, source_address=src_addr)) as sock:
            with closing(context.wrap_socket(sock)) as sslsock:
                dercert = sslsock.getpeercert(True)
        return ssl.DER_cert_to_PEM_cert(dercert)

    def verify_ac_cert(self, host, url):
        """verify ac cert"""
        # check peer cert in request for auth
        if ac_constants.AUTH_BASE_URL not in url:
            return
        if not os.path.exists(ncu.HTTPS_CRL) and not ncu.SUPPORT_VERIFY_CERT:
            return
        pem = ""
        with eventlet.Timeout(ac_constants.GET_SERVER_CERTIFICATE_TIME, False):
            pem = self.get_ac_server_certificate(
                (host.strip('[').strip(']'), ac_constants.rest_server_port))

        if not pem:
            LOG.error(_LE("get AC https cert failed"))
            raise CertGetFailedException()
        if ncu.is_revoked(ncu.HTTPS_CRL, pem):
            LOG.error(_LE("AC https cert is revoked"))
            raise CertRevokedException(cert="AC https cert")

        if ncu.is_cert_expired(pem):
            LOG.error(_LE("AC https cert is expired"))
            raise CertExpiredException()
        else:
            server_cert = ssl.PEM_cert_to_DER_cert(pem)
            # verify san
            if not ncu.verify_san(server_cert):
                raise WrongSANException()

    def _request_client(self, req_dict):
        """request client"""
        if ncu.SUPPORT_VERIFY_CERT:
            certs = ACReSTClient._get_certificate()
            ncu.verify_plugin_cert(certs, connection='https')

        if req_dict.get('method') in [constants.REST_DELETE,
                                      constants.REST_GET]:
            req_dict.pop('data')

        # not use proxy in fsp env
        if os.environ.get("no_proxy") and ncu.IS_FSP:
            for controller_ip in self.host_list:
                if controller_ip not in os.environ.get("no_proxy").split(','):
                    os.environ["no_proxy"] = os.environ.get("no_proxy") + "," + controller_ip
        return self._get_request(req_dict)

    @classmethod
    def _get_request(cls, req_dict):
        """get request"""
        cert_paths = cls._get_certificate()
        with requests.session() as session:
            adapter = ACReSTClient._get_adapter(cert_paths)
            session.mount("https://", adapter)
            LOG.info("[AC]start to request. method: %s, url: %s, timeout: %s", req_dict.get('method'),
                     req_dict.get('url'), req_dict.get('timeout'))
            req = session.request(req_dict.get('method'),
                                  url=req_dict.get('url'),
                                  headers=req_dict.get('headers'),
                                  data=req_dict.get('data'),
                                  verify=cert_paths.get('ca_file'),
                                  timeout=req_dict.get('timeout'))
        return req

    @staticmethod
    def _get_adapter(cert_paths):
        """ get adapter """
        cert_file = cert_paths.get('cert_file')
        key_file = cert_paths.get('key_file')
        key_file_passwd = None
        if ncu.SUPPORT_VERIFY_CERT:
            key_file_passwd = fs_crypt.decrypt(ncu.FS_PRIVATE_KEY_PASSWORD)
        adapter = HWHTTPSAdapter(cert_file=cert_file, key_file=key_file,
                                 password=key_file_passwd,
                                 pool_connections=100, pool_maxsize=100)
        return adapter

    def _get_token_id(self, host, port, max_times=None, is_websocket=False):
        """Get token id from controller.

        :param host: ip address of northbound in controller.
        :param port: port of northbound in controller.
        :param max_times: max times to request controller
        :param is_websocket: whether or not used by ws
        :return: token id of controller.
        """
        LOG.debug("[AC] get token ID from %s:%s", host, port)

        ret = self._get_ret(host, max_times)

        if ((ret.status_code >= requests.codes.ok) and
                (ret.status_code < requests.codes.multiple_choices)):
            if ret.content:
                result_data = jsonutils.loads(ret.content)
                token_id = result_data['data']['token_id']
                if token_id and is_websocket:
                    LOG.debug("[AC] AC authorize successfully from websocket")
                    return token_id
                if token_id:
                    LOG.debug("[AC] AC authorize successfully")
                    token_time = time.time()
                    comment_info = token_id + ':' + str(token_time)
                    admin_ctx = ncu.neutron_context.get_admin_context()
                    self.dfactor_db.delete_decrypt_factor(
                        admin_ctx, self.neutron_name)
                    encrypt_comment_info = self.dfactor_db. \
                        create_db_decrypt_factor(admin_ctx,
                                                 self.neutron_name,
                                                 comment_info)

                    self.create_update_token_info(
                        value=0, token_id=self.neutron_name,
                        comment_info=encrypt_comment_info)
                    return token_id
        elif ret.status_code == requests.codes.unauthorized:
            LOG.error(_LE("[AC] AC authorize fail, "
                          "please check the ac account!"))
            raise AuthenticationException()
        elif ret.status_code == requests.codes.internal_server_error:
            LOG.error(_LE("Internal server error. "
                          "Please check AC account status!"))
            raise InternalServerException()
        else:
            raise OtherException()

        return None

    def _get_ret(self, host, max_times):
        headers = {"Content-type": "application/json",
                   "Accept": "application/json",
                   "Accept-Language": "en-US"}

        auth_url = "%s%s%s%s%s" % (ac_constants.HTTPS_HEADER,
                                   host, ":",
                                   ac_constants.rest_server_port,
                                   ac_constants.AUTH_BASE_URL)

        if ncu.after_fsp_6_3_0():
            password = cfg.CONF.huawei_ac_config.ac_auth_password
        else:
            password = six.text_type(security_util.decrypt_data(
                cfg.CONF.huawei_ac_config.ac_auth_password), errors='ignore')

        auth = {"userName": cfg.CONF.huawei_ac_config.ac_auth_username,
                "password": password}

        auth_data = jsonutils.dumps(auth)

        ret = self._process_request(
            (headers, constants.REST_POST, auth_url, auth_data),
            max_times=max_times,
            timeout=ac_constants.SECONDS_EVRY_MINUTE // 3)
        return ret

    @lock_db.wrap_db_lock(lock_db.RESOURCE_TOKEN_CONFIG)
    def create_update_token_info(self, value, token_id, comment_info):
        """Create or update token info to DB."""
        LOG.debug("[AC] Create or update token info in db: %s", token_id)
        self.db_inter.update_config_record(key=token_id, value=value,
                                           comment=comment_info)

    def _check_ac_cluster_switchover(self, active_ac, timestamp):
        """check ac cluster switchover"""
        session = self.db_inter.get_session()
        original_active_ac = self.db_inter.get_active_ac_ip(session)
        if isinstance(original_active_ac, str):
            original_active_ac = original_active_ac.lower()
        if original_active_ac != active_ac:
            LOG.info(_LI("[AC] AC cluster has been switchover. "
                         "Time stamp: %s"), timestamp)
            self.db_inter.update_active_ac_record(session, active_ac)

    def detect_active_ac(self):
        """Detect active host ip of Agile Controller."""
        try:
            timestamp = timeutils.utcnow(True)

            active_ac_list = []
            for i in [0, 1]:
                active_ac_list.append(
                    self._get_active_ip(self.host_list[i], timestamp))

            if active_ac_list[0] is not None:
                if active_ac_list[1] is not None and \
                        active_ac_list[0] != active_ac_list[1]:
                    LOG.error(_LE("[AC] Two AC response different status. "
                                  "Time stamp: %s"), timestamp)
                    return
                active_ac = active_ac_list[0]
            else:
                if active_ac_list[1] is None:
                    LOG.error(_LE("[AC] Can not get AC status. "
                                  "Time stamp: %s"), timestamp)
                    return
                active_ac = active_ac_list[1]

            if active_ac not in self.host_list and \
                    active_ac != ac_constants.DEFAULT_AC_IP:
                LOG.error(_LE("[AC] Response active IP (%s) "
                              "is not configured."),
                          active_ac)
                return
            LOG.debug(_LI("[AC] Active AC is %s. Time stamp: %s"
                          % (active_ac, timestamp)))

            # check if ac cluster has been switchover, if switchover, update
            # active ac record
            self._check_ac_cluster_switchover(active_ac, timestamp)

            if active_ac != ac_constants.DEFAULT_AC_IP and \
                    self.callback_restart_rpc is not None:
                self.callback_restart_rpc(active_ac)
        except Exception as ex:
            LOG.error(_LE("[AC] Some exception happen: %s. "
                          "Wait for next periodically detect."),
                      ex)
            return

    def _get_active_ip(self, host, timestamp):
        """get active ip"""
        LOG.debug("[AC] Start to detect AC (%s). Time stamp: %s"
                  % (host, timestamp))
        url = '%s%s%s%s%s' % (ac_constants.HTTPS_HEADER, host, ":",
                              str(self.port), ac_constants.NW_HW_CLUSTERS_URL)
        try:
            ret = self.send(constants.REST_GET, url,
                            ac_constants.DEFAULT_AC_ID, {}, max_times=0)
            if ret.status_code < requests.codes.ok \
                    or ret.status_code >= requests.codes.multiple_choices:
                return None
            if not ret.content:
                return None
            result_data = jsonutils.loads(ret.content)
            ac_list = result_data['cluster']
            active_num = 0
            active_ac = ac_constants.DEFAULT_AC_IP

            active_ac, active_num = \
                ACReSTClient.get_active_ac_and_active_num(ac_list,
                                                          host, active_ac,
                                                          active_num)

            if active_num > 1:
                LOG.error(_LE("[AC] Two AC are all active. "
                              "Time stamp: %s"), timestamp)
                return None
            if active_num == 0:
                LOG.error(_LE("[AC] Two AC are all standby. "
                              "Time stamp: %s"), timestamp)
            return active_ac

        except AuthenticationException as ex:
            LOG.error(_LE("[AC] Failed to detect AC, reason: %s"), str(ex))
            session = self.db_inter.get_session('write')
            LOG.debug("[AC]AC authorize fail, delete the active ac record")
            self.db_inter.delete_active_ac_record(session, host)
        except Exception as ex:
            LOG.error(_LE("[AC] Failed to detect AC (%s). Time stamp: %s. %s"),
                      host, timestamp, ex)
        return None

    @staticmethod
    def get_active_ac_and_active_num(ac_list, host, active_ac,
                                     active_num):
        """ get active ac and active num"""
        for ac_status in ac_list:
            if ac_status['role'] == ac_constants.AC_ACTIVE_ROLE:
                if netaddr.valid_ipv6(host.strip('[').strip(']')):
                    active_ac = ac_status.get('ipv6').lower()
                elif netaddr.valid_ipv4(host):
                    active_ac = ac_status.get('ip')
                active_num += 1
        return active_ac, active_num

    def send_http_heart_to_ac(self, body):
        """Send HTTP heart beat request to Agile Controller."""
        if len(self.host_list) > 1:
            host = ac_constants.DEFAULT_AC_IP
        else:
            host = self.host_list[0]
        try:
            url = '%s%s%s%s%s%s' % (ac_constants.HTTPS_HEADER, host, ":",
                                    str(self.port),
                                    ac_constants.NW_HW_RESTFUL_V3,
                                    ac_constants.NW_HW_CLOUD_HEART_BEAT)
            ret = None
            ret = self.send(method=constants.REST_UPDATE, body=body,
                            res_id="", url=url,
                            max_times=HEARTBEAT_RETRY_TIMES,
                            timeout=ac_constants.HEARTBEAT_TIMEOUT)
            if ret and ((ret.status_code >= requests.codes.ok) and
                        (ret.status_code < requests.codes.multiple_choices)):
                return ac_constants.HTTP_CONNECTION_ACTIVE, ''

            LOG.error(_LE("[AC]AC connection status is down"))
            return ac_constants.HTTP_CONNECTION_DOWN, 'Internal error.'
        except Exception as ex:
            LOG.error("[AC]send http heart beat to ac failed: %s, "
                      "traceback: %s", ex, traceback.format_exc())
            if "Invalid return character or leading space in header" in str(ex):
                self.token_variable = None
                self.db_inter.delete_config_record(self.neutron_name)
            return ac_constants.HTTP_CONNECTION_DOWN, ex

    def get_stream_name_with_cert(self, headers, method, url, body):
        """Get stream name with certification."""
        ret = self._process_request((headers, method, url, body))
        if ((ret.status_code >= requests.codes.ok) and
                (ret.status_code < requests.codes.multiple_choices)):
            if ret.content:
                result_data = jsonutils.loads(ret.content)
                stream_name = result_data['huawei-ac-stream-websocket:output'][
                    'stream-name']
                if stream_name:
                    LOG.debug("[AC] Get stream name successfully")
                    return stream_name
            return None
        else:
            raise Exception("[AC] Fail to get stream name from AC.")

    def get_filter_info_with_cert(self, headers, method, url):
        """Get filter information with certification."""
        ret = self._process_request((headers, method, url, None))
        if ((ret.status_code >= requests.codes.ok) and
                (ret.status_code < requests.codes.multiple_choices)):
            return ret.content
        else:
            raise Exception("[AC] Fail to get filter information from AC.")

    def send_cloud_alarm(self, alarm_description, session=None,
                         alarm_category=None):
        """Send cloud alarm to controller.

        :param alarm_description: alarm description
        :param session: db session
        :param alarm_category: category of the alarm
        :return: None
        """
        neutron_record = self.db_inter.get_server_record(
            neutron_name=self.neutron_name)
        if not neutron_record or neutron_record.ac_connection_state != \
                ac_constants.HTTP_CONNECTION_ACTIVE:
            LOG.info("Controller is not reached, Not send alarm: %s to "
                     "controller.", alarm_description)
            return
        if not alarm_category:
            alarm_category = {
                'alarm_inner_id': ac_constants.CLOUD_BUSINESS_ALARM_ID,
                'alarm_level': ac_constants.CLOUD_BUSINESS_ALARM_LEVEL,
                'alarm_type': ac_constants.CLOUD_BUSINESS_ALARM_TYPE
            }
        alarm_type = alarm_category.get('alarm_type')
        alarm_data = {
            "inner_id": alarm_category.get('alarm_inner_id'),
            "level": alarm_category.get('alarm_level'),
            "type": alarm_type,
            "description": alarm_description,
            "occur_time": ACCommonUtil.get_standard_current_time(),
            "host_id": ACCommonUtil.get_local_host_ip()
        }
        cloud_alarm_info = CloudAlarmModel.ac_model_format(alarm_data)
        if not self.send_alarm_to_controler(jsonutils.dumps(cloud_alarm_info)):
            self.db_inter.create_cloud_alarm(alarm_data, session)
        elif alarm_type == ac_constants.CLOUD_CERT_ALARM_TYPE:
            alarm_data.update(
                {"type": ac_constants.CLOUD_CERT_RECOVER_ALARM_TYPE})
            self.db_inter.create_cloud_alarm(alarm_data, session)

    def send_alarm_to_controler(self, body):
        """Send alarm request to controller.

        :param body: data of the alarm
        :return: Bool
        """
        if len(self.host_list) > 1:
            host = ac_constants.DEFAULT_AC_IP
        else:
            host = self.host_list[0]
        try:
            url = '%s%s:%s%s' % (ac_constants.HTTPS_HEADER, host, self.port,
                                 ac_constants.NW_HW_CLOUD_ALARM_URL)
            ret = self.send(constants.REST_POST, url, "", body)
            if requests.codes.ok <= ret.status_code \
                    < requests.codes.multiple_choices:
                return True
            LOG.error("[AC]send alarm request to controller failed with "
                      "response code: %s, ", ret.status_code)
            return False
        except Exception as ex:
            LOG.error("[AC]send alarm request to controller failed: %s, "
                      "traceback: %s", ex, traceback.format_exc())
            return False

    def query_host_status(self, migrate_host):
        """query host status of controller.

        :param migrate_host: migrate_host
        :return: None
        """
        if len(self.host_list) > 1:
            host = ac_constants.DEFAULT_AC_IP
        else:
            host = self.host_list[0]
        try:
            url = '%s%s:%s%s' % (ac_constants.HTTPS_HEADER, host, self.port,
                                 ac_constants.NW_HW_QUERY_HOST_URL)
            query_host_body = {"hosts": [{"host-id": migrate_host}]}
            query_host_body = jsonutils.dumps(query_host_body)
            ret = self.send(constants.REST_POST, url, "", query_host_body)
            if requests.codes.ok <= ret.status_code \
                    < requests.codes.multiple_choices:
                return
            raise HostStatusError(
                reason="Controller response code:%s." % ret.status_code)
        except Exception as ex:
            LOG.error("[AC]Request to controller query host status failed with:"
                      " %s, traceback: %s", ex, traceback.format_exc())
            raise HostStatusError(reason=str(ex))
