#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Request Service to Agile Controller."""

import copy
import traceback
from enum import Enum

import eventlet
import six
from oslo_config import cfg
from oslo_serialization import jsonutils

from networking_huawei._i18n import _LI, _LE
from networking_huawei.common import exceptions as ml2_exc
from networking_huawei.common.constants import REST_POST, REST_DELETE, REST_UPDATE, REST_GET
from networking_huawei.common.exceptions import TimeoutException, DryRunError
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as ac_osprofiler
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_log as logging
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.schema import ACPluginSchema
from networking_huawei.drivers.ac.extensions.external_whitelist.external_whitelist import ExternalWhitelistNotFound
from networking_huawei.drivers.ac.extensions.flowmirror.flow_mirror import FlowMirrorExceptionFromController
from networking_huawei.drivers.ac.plugins.dry_run import dry_run_util
from networking_huawei.drivers.ac.sync.util import ACUtil

try:
    from oslo_service import loopingcall
except ImportError:
    from neutron.openstack.common import loopingcall
try:
    from neutron import context
except ImportError:
    from neutron_lib import context

requests = eventlet.import_patched('requests.__init__')
LOG = logging.getLogger(__name__)

REQUEST_TO_OPERATION = {REST_POST: ac_constants.OPER_CREATE,
                        REST_DELETE: ac_constants.OPER_DELETE,
                        REST_UPDATE: ac_constants.OPER_UPDATE,
                        REST_GET: ac_constants.OPER_GET}


class DryRunStateEnum(Enum):
    DEFAULT = 1  # 默认
    NO_EXIST = 2  # 数据库不存在
    REAL_DATA = 3  # 真实数据
    DRY_RUN_DATA = 4  # dry-run产生的数据


class RequestServiceParas(object):
    """Request service parameters."""

    def __init__(self, session, seq_num, res_type, method, url, req_id, rest_data,
                 dry_run_state=DryRunStateEnum.DEFAULT):
        self.session = session
        self.seq_num = seq_num
        self.res_type = res_type
        self.method = method.upper()
        self.url = url
        self.req_id = req_id
        self.rest_data = rest_data

        LOG.info('dryRun#orignal:res_type=%s,%s %s,dry_run_state=%s,req_id=%s', self.res_type, self.method,
                 self.url, dry_run_state, self.req_id)
        # 支持 dry-run 的资源
        if res_type not in {ac_constants.NW_HW_PORTS, ac_constants.NW_HW_QOS_POLICY, ac_constants.NW_HW_TAP_SERVICE,
                            ac_constants.NW_HW_TAP_FLOW, ac_constants.NW_HW_EXROUTE}:
            return

        if self.method == 'DELETE' and dry_run_state == DryRunStateEnum.DRY_RUN_DATA:
            if res_type in {ac_constants.NW_HW_TAP_SERVICE, ac_constants.NW_HW_TAP_FLOW}:
                self.url = self.url.replace('/controller/', '/controller/dryrun/', 1)

        # 是否是 dry run 生成的资源
        dry_run_flg = dry_run_util.is_dry_run_data(rest_data, res_type)
        if dry_run_flg:
            # 需要调用 dry-run 相关接口
            self.url = self.url.replace('/restconf/', '/restconf/dryrun/', 1)
            if res_type in {ac_constants.NW_HW_TAP_SERVICE, ac_constants.NW_HW_TAP_FLOW}:
                self.url = self.url.replace('/controller/', '/controller/dryrun/', 1)
            if self.method == 'POST':
                if dry_run_state in {DryRunStateEnum.REAL_DATA, DryRunStateEnum.DRY_RUN_DATA}:
                    raise DryRunError(reason="%s:POST %s,dry_run_state=%s" % (
                        self.res_type, self.url, dry_run_state))
            elif self.method == 'DELETE':
                # 删除非dry run创建的正式数据，上层约束，插件防止错误增加校验
                if dry_run_state in {DryRunStateEnum.REAL_DATA}:
                    raise DryRunError(reason="%s:POST %s,dry_run_state=%s" % (
                        self.res_type, self.url, dry_run_state))
        else:
            if self.method == 'POST':
                if dry_run_state in {DryRunStateEnum.REAL_DATA, DryRunStateEnum.DRY_RUN_DATA}:
                    raise DryRunError(reason="%s:POST %s,dry_run_state=%s" % (
                        self.res_type, self.url, dry_run_state))
            elif self.method == 'PUT':
                # 处理dry-run转正的特殊场景:不带dry-run标识插件数据库里数据带dry-run标志
                if dry_run_state == DryRunStateEnum.DRY_RUN_DATA:
                    self.method = 'POST'
                    if self.res_type not in {ac_constants.NW_HW_EXROUTE,
                                             ac_constants.NW_HW_TAP_SERVICE,
                                             ac_constants.NW_HW_TAP_FLOW}:
                        self.url = self.url[:self.url.rfind('/')]
                        self.url = self.url[:self.url.rfind('/')]
                elif dry_run_state == DryRunStateEnum.NO_EXIST:
                    raise DryRunError(reason="%s:POST %s,dry_run_state=%s" % (
                        self.res_type, self.url, dry_run_state))
            elif self.method == 'DELETE':
                if dry_run_state == DryRunStateEnum.NO_EXIST:
                    raise DryRunError(reason="%s:POST %s,dry_run_state=%s" % (
                        self.res_type, self.url, dry_run_state))
        LOG.info('dryRun#last:res_type=%s,%s %s,dry_run_state=%s,req_id=%s', self.res_type, self.method,
                 self.url, dry_run_state, self.req_id)

    def __str__(self):
        rest_log = copy.deepcopy(self.rest_data)
        rest_log = validate.validate_log_record(rest_log, self.res_type)
        return ("seq_num: %s, res_type: %s, id: %s, method: %s, url: %s, rest_data: %s " %
                (self.seq_num, self.res_type, self.req_id, self.method, self.url, rest_log))


class Singleton(type):
    """Singleton base class."""

    def __call__(cls, *args, **kwargs):
        if not hasattr(cls, '_instance'):
            cls._instance = super(Singleton, cls).__call__(*args, **kwargs)
        return cls._instance


class ACReSTService(six.with_metaclass(Singleton, object)):
    """Request Service to Agile Controller."""

    def __init__(self):
        validate.validate_rpc_server_ip()
        self.host_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip.replace(' ', '').lower().split(',')
        self.client = ACReSTClient()
        self.util = ACUtil()
        self.plugin_db = ACdbInterface()
        self.ctx = context.get_admin_context()

        if len(self.host_list) > 1:
            self.url = '%s%s:%s' % (
                ac_constants.HTTPS_HEADER, ac_constants.DEFAULT_AC_IP, str(ac_constants.rest_server_port))
            self.plugin_db.create_active_ac_record(self.ctx.session)
            ac_detect_timer = loopingcall.FixedIntervalLoopingCall(self.client.detect_active_ac)
            LOG.info(_LI("[AC] Start cluster detect timer."))

            if ncu.get_ops_version() not in [ac_constants.OPS_K]:
                # OPS L\M version
                ac_detect_timer.start(interval=cfg.CONF.huawei_ac_config.ac_detect_period, initial_delay=0,
                                      stop_on_exception=False)
            else:
                # OPS K version
                ac_detect_timer.start(interval=cfg.CONF.huawei_ac_config.ac_detect_period, initial_delay=0)
        else:
            self.url = '%s%s:%s' % (ac_constants.HTTPS_HEADER, self.host_list[0], str(ac_constants.rest_server_port))

    def request_send_service(self, request_para, max_times=None):
        """Rest request service
        :param request_para: RequestServiceParas class
        :param max_times: max time retry when timeout
        :return: request response
        """
        LOG.info(_LI("[AC] Request service is called, %s"), request_para)
        try:
            self._request_send_service_core(request_para, max_times)
        except (requests.Timeout, TimeoutException):
            LOG.error(_LE("[AC]AC request timeout error,method:%s,url:%s,body:%s"), request_para.method,
                      request_para.url, request_para.rest_data)
            ac_osprofiler.record_chain_exception("request AC timeout")
            # if user set the error_retry_count as 0, so when send the request timeout, just throw the exception in
            # plugin processing
            if cfg.CONF.huawei_ac_config.error_retry_count == 0:
                self._handle_timeout_with_no_error_retry(request_para)
                raise
            if request_para.method == REST_DELETE:
                LOG.error(_LE("[AC]AC request method is delete,just raise the exception for user"))
                # if the method is Delete, when Delete operation error
                # just raise the exception,tell user the operation failed
                self.plugin_db.delete_plugin_record(request_para.session, request_para.seq_num)
                raise
            else:
                self._handle_error_timeout_from_ac(request_para.session, request_para.seq_num)
        except ml2_exc.MechanismDriverError as ex:
            LOG.error(_LE("[AC] AC process failed, traceback: %s"), traceback.format_exc())
            ac_osprofiler.record_chain_exception(str(ex))

            self._process_exception_or_ac_return_error(request_para.session, request_para.seq_num)
            raise
        except ml2_exc.ResourceReferenceException:
            self._process_resource_reference_error(request_para)
            raise
        except Exception as ex:
            LOG.error(_LE("[AC] AC request exception, traceback: %s"), traceback.format_exc())
            ac_osprofiler.record_chain_exception(str(ex))
            self._process_exception_or_ac_return_error(request_para.session, request_para.seq_num)
            raise ml2_exc.MechanismDriverError(method=request_para.method, url=request_para.url, error=str(ex))

    def _request_send_service_core(self, request_para, max_times):
        """request_send_service call it"""
        # process request resource status
        self._process_resource_status(request_para)
        rest_data = ACRestUtils.generate_rest_data(request_para)
        ret = self.client.send(request_para.method, '%s%s' % (self.url, request_para.url), request_para.req_id,
                               rest_data, max_times)
        res_code = int(ret.status_code)
        try:
            res_json = jsonutils.loads(ret.content) if ret.content else None
        except ValueError as e:
            LOG.error(_LI("[AC]handle response form ac occur error: %s"), ret.content)
            res_json = ret.content
        if requests.codes.ok <= res_code < requests.codes.multiple_choices:
            LOG.info(_LI("[AC]Send request successfully and AC process ok, response: %s"), res_json)
            ac_osprofiler.record_call_chain("AC return %s,success" % res_code)
            if request_para.method != REST_GET:
                self._handle_success_from_ac(request_para)
        else:
            self._handle_failure_from_ac({'status': res_code, 'errorCode': None, 'reason': None, 'response': None},
                                         res_json, request_para)

    def _handle_timeout_with_no_error_retry(self, request_para):
        LOG.error(_LE("[AC]The parameter error_retry_count is set as 0,just throw exception when sending the request "
                      "timeout"))
        if ACUtil.check_resource_have_state(request_para.res_type) and request_para.method != REST_DELETE:
            LOG.debug("[AC] Request timeout, set the status to error")
            if request_para.seq_num and self.plugin_db.get_plugin_record(request_para.session, request_para.seq_num):
                self.plugin_db.update_neutron_db_state(request_para.session, request_para.req_id, request_para.res_type,
                                                       ac_constants.NEUTRON_STATUS_ERROR)
        self.plugin_db.delete_plugin_record(request_para.session, request_para.seq_num)

    def _handle_failure_from_ac(self, result, res_json, request_para):
        if not res_json:
            ac_osprofiler.record_chain_exception("AC return and no err msg")
            raise ml2_exc.MechanismDriverError(method=request_para.method, url=request_para.url,
                                               error="Request fail, but return content is None")

        if request_para.res_type == ac_constants.NW_HW_EXTERNAL_WHITELIST:
            ac_osprofiler.record_chain_exception("AC return " + str(res_json))
            LOG.error(_LE("[AC]AC process request failed.method:%s, url:%s, result:%s"), request_para.method,
                      request_para.url, res_json)
            error_info = res_json['errors']['error'][0]
            error = 'Ac process external whitelist process failed, %s, %s.' % (
                error_info['error-info']['error-code'], error_info['error-message'])
            raise ml2_exc.MechanismDriverError(method=request_para.method, url=request_para.url, error=error)

        result['response'] = res_json
        error_code = ACRestUtils.get_error_code(res_json)
        result['errorCode'] = error_code
        reason = ACRestUtils.get_error_message(res_json)
        result['reason'] = reason
        ac_osprofiler.record_chain_exception("AC return %s:%s" % (error_code, reason))
        LOG.error(_LE("[AC] AC process request failed. result: %s"), result)

        if (error_code in ac_constants.DATA_CONSISTENCY_ERROR_CODE) and request_para.method != REST_DELETE:
            LOG.error(_LE("[AC] AC return data inconsistent."))
            self._handle_inconsistent_state_in_ac(res_json)
        elif error_code in ac_constants.DATA_CONSISTENCY_ERROR_CODE and request_para.method == REST_DELETE \
                and request_para.res_type == ac_constants.NW_HW_NETWORKS:
            LOG.error("[AC] Delete network,AC return be Referenced")
            raise ml2_exc.ResourceReferenceException()
        elif error_code == ac_constants.AC_OFF_LINE:
            self._handle_ac_offline(request_para)
        else:
            raise ml2_exc.MechanismDriverError(method=request_para.method, url=request_para.url, error=reason)

    def _process_resource_status(self, request_para):
        """
            process request resource status
            if it is vm port or dhcp port, don't set pending status AC process it as long feedback
            for fsp6.5 and Q version, don't set fip pending status
        """

        if not self._res_not_set_pending_status(request_para) and ACUtil.check_resource_have_state(
                request_para.res_type) and request_para.method != REST_DELETE:
            LOG.debug("[AC] Before Request, set the status as pending in neutron db")
            if self._no_need_to_set_pending_status(request_para):
                return
            pending_status = self._get_pending_status(request_para)
            is_ops_ez_m = ncu.get_ops_version() == ac_constants.OPS_EZ_M
            if not is_ops_ez_m or (is_ops_ez_m and request_para.res_type != ac_constants.NW_HW_PORTS):
                if self.plugin_db.get_plugin_record_list(
                        self.plugin_db.get_session('write'),
                        ACPluginSchema(res_uuid=request_para.req_id, neutron_id=-1,
                                       user_oper=ac_constants.OPER_DELETE)):
                    return
                if request_para.seq_num and self.plugin_db.get_plugin_record(request_para.session,
                                                                             request_para.seq_num):
                    self.plugin_db.update_neutron_db_state(request_para.session, request_para.req_id,
                                                           request_para.res_type, pending_status)

    @classmethod
    def _no_need_to_set_pending_status(cls, request_para):
        if request_para.res_type == ac_constants.NW_HW_PORTS and request_para.method == REST_UPDATE:
            if not request_para.rest_data.get('device-owner') and not request_para.rest_data.get('device-id'):
                LOG.debug("[AC]Need not set pending status when update port without device_owner/device_id.")
                return True
        if request_para.res_type == ac_constants.NW_HW_PORTS and request_para.rest_data.get('vnic-type') == 'direct':
            LOG.debug("[AC]Do not set pending status for trunk port.")
            return True
        return False

    @classmethod
    def _get_pending_status(cls, request_para):
        pending_status = ACRestUtils.get_operation_pending_status(request_para.method)
        if ncu.get_ops_version() == ac_constants.OPS_EZ_M and request_para.res_type in {ac_constants.NW_HW_NETWORKS,
                                                                                        ac_constants.NW_HW_ROUTERS}:
            pending_status = ac_constants.NEUTRON_STATUS_BUILD
        return pending_status

    def _res_not_set_pending_status(self, request_para):
        """if it is vm port or dhcp port, don't set pending status"""

        is_vm_port = False
        is_dhcp_port = False
        if request_para.res_type == ac_constants.NW_HW_PORTS:
            port = self.plugin_db.get_neutron_record(request_para.session, request_para.req_id, request_para.res_type)
            is_vm_port = ACUtil.is_compute_port(port)
            is_dhcp_port = ACUtil.is_dhcp_port(port)

        if is_vm_port:
            LOG.debug("[AC] VM port need not set pending status.")
            return True
        elif is_dhcp_port:
            LOG.debug("[AC] DHCP port need not set pending status.")
            return True
        elif request_para.res_type == ac_constants.NW_HW_FIP and ncu.get_ops_version() in {
            ac_constants.FSP_6_5, ac_constants.OPS_Q, ac_constants.OPS_R, ac_constants.OPS_T, ac_constants.OPS_W,
            ac_constants.FSP_21_0}:
            LOG.debug("[AC] floatingIP in ovo db cant set pending status.")
            return True
        return False

    def get_ac_port_info(self, port_id):
        """Get Agile Controller port information."""
        LOG.info(_LI('[AC] Begin to get ac port info: %s'), port_id)
        get_port_url = self.url + ac_constants.GET_PORT_SUFFIX + port_id
        try:
            ret = self.client.send(REST_GET, get_port_url, '', {})
            res_code = int(ret.status_code)
            res_json = jsonutils.loads(ret.content)
            if requests.codes.ok <= res_code < requests.codes.multiple_choices:
                LOG.info(_LI('[AC]Huawei AC process get ac port info request successfully: %s'), res_json)
            else:
                error_info = res_json['ietf-restconf:errors']['error']
                error_code = error_info[0]['error-info']['error-code']
                error_message = error_info[0]['error-message']
                error = 'get ac port info request process failed, %s, %s.' % (error_code, error_message)
                raise ml2_exc.MechanismDriverError(method=REST_GET, url=get_port_url, error=error)
            port_info = res_json.get('huawei-ac-neutron:port', [])[0]
            if not port_info:
                LOG.error(_LE('[AC] Failed to get port info from response.'))
                error = 'huawei-ac-neutron:port is empty.'
                raise ml2_exc.MechanismDriverError(method=REST_GET, url=get_port_url, error=error)
            LOG.info(_LI('[AC] Get port info from AC: %s'), port_info)
            return port_info
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to get ac port info: %s'), ex)
            raise

    def get_ac_external_whitelist_info(self, externalwhitelist_id=None, router_id=None):
        """Get Agile Controller external white list information."""
        if externalwhitelist_id:
            ac_external_whitelist_url = '%s%s/logicnetwork/external_whitelists/%s' % (
                self.url, ac_constants.NW_HW_RESTFUL_V3, externalwhitelist_id)
        elif router_id:
            ac_external_whitelist_url = '%s%s?router_id=%s' % (self.url, ac_constants.EXTERNAL_WHITELIST_URL, router_id)
        else:
            ac_external_whitelist_url = self.url + ac_constants.EXTERNAL_WHITELIST_URL
        try:
            ret = self.client.send(REST_GET, ac_external_whitelist_url, '', {})
            res_code = int(ret.status_code)
            res_json = jsonutils.loads(ret.content) if ret.content else {}
            if requests.codes.ok <= res_code < requests.codes.multiple_choices:
                LOG.info(_LI('[AC] Get AC external whitelist: %s'), res_json)
                if externalwhitelist_id:
                    external_whitelist_info = res_json.get('external_whitelist')
                else:
                    external_whitelist_info = res_json.get('external_whitelists')
                return external_whitelist_info
            elif res_code == requests.codes.not_found:
                raise ExternalWhitelistNotFound(external_whitelist_id=externalwhitelist_id)
            else:
                error_message = res_json.get('errors', {}).get('error', [{}])[0].get('error-message')
                LOG.error(_LE('[AC]Get AC external whitelist failed %s:%s.'), error_message, res_json)
                raise ml2_exc.MechanismDriverError(method=REST_GET, url=ac_external_whitelist_url, error=error_message)
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to get ac external whitelist: %s'), ex)
            raise ex

    def send_bind_port_request(self, bind_port_info):
        """Send bind port request to Agile Controller."""
        LOG.info(_LI('[AC] Begin to send bind port request: %s'),
                 bind_port_info)
        try:
            ac_osprofiler.record_call_chain("send vlan bind to AC")
            bind_port_url = self.url + ac_constants.BIND_PORT_SUFFIX
            ret = self.client.send(REST_POST, bind_port_url, '', jsonutils.dumps(bind_port_info))
            res_code = int(ret.status_code)
            res_json = jsonutils.loads(ret.content)
            if requests.codes.ok <= res_code < requests.codes.multiple_choices:
                LOG.info(_LI('[AC] Huawei AC process bind port request successfully: %s'), res_json)
                ac_osprofiler.record_call_chain("send vlan to AC success")
                return res_json
            else:
                error_info = res_json['ietf-restconf:errors']['error']
                error_code = error_info[0]['error-info']['error-code']
                error_message = error_info[0]['error-message']
                error = 'bind port request process failed, %s, %s.' % (error_code, error_message)
                raise ml2_exc.MechanismDriverError(method=REST_POST, url=bind_port_url, error=error)
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to process bind port request in huawei ac driver: %s'), ex)
            ac_osprofiler.record_chain_exception("send to AC fail:" + str(ex))
            raise

    def send_vtep_ip_request(self, router_id):
        """Send vtep ip request to Agile Controller."""
        LOG.info(_LI('[AC] Begin to send Vtep IP request: %s'), router_id)
        input_info = {ac_constants.EXROUTES_INPUT: {'router-id': router_id}}
        try:
            vtep_ip_url = self.url + ac_constants.VTEP_IP_SUFFIX
            ret = self.client.send(REST_POST, vtep_ip_url, '', jsonutils.dumps(input_info))
            res_code = int(ret.status_code)
            res_json = jsonutils.loads(ret.content)
            if requests.codes.ok <= res_code < requests.codes.multiple_choices:
                LOG.info(_LI('[AC] Huawei AC process Vtep IP request successfully: %s'), res_json)
                return res_json
            else:
                error_info = res_json['ietf-restconf:errors']['error']
                error_code = error_info[0]['error-info']['error-code']
                error_message = error_info[0]['error-message']
                error = 'Vtep IP request process failed, %s, %s.' % (error_code, error_message)
                raise ml2_exc.MechanismDriverError(method=REST_POST, url=vtep_ip_url, error=error)
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to process Vtep IP request: %s'), ex)
            raise

    def _process_exception_or_ac_return_error(self, session, seq_num):
        LOG.debug("[AC] Delete the res data in plugin db and validation db")
        self.util.delete_one_record_in_plugin_db(session, seq_num, on_error=True)

    def _process_resource_reference_error(self, request_para):
        LOG.debug("[AC] Process resource reference error")
        self.plugin_db.delete_plugin_record(request_para.session, request_para.seq_num)
        if request_para.method in [REST_UPDATE] and request_para.res_type in ac_constants.RESOURCE_HAVE_STATE:
            self.plugin_db.update_neutron_db_state(
                request_para.session, request_para.req_id, request_para.res_type, ac_constants.NEUTRON_STATUS_ERROR)

    def _handle_inconsistent_state_in_ac(self, res_content):
        error_paras = ACRestUtils.get_error_paras(res_content)
        if error_paras == "flow mirror error":
            errmsg = res_content['errmsg']
            LOG.error(_LE("[AC]flow mirror error: %s"), errmsg)
            raise FlowMirrorExceptionFromController()
        elif error_paras:
            LOG.error(_LE("[AC] Data inconsistent, error_para %s"), error_paras)
            raise ml2_exc.ResourceReferenceException()

        #   if inconsistent error, do not send parent resource again
        #   user can handle inconsistent error with neutron sync

    def _check_dependency_status(self, session, method, url, dep_list):
        for res in dep_list:
            if not ACUtil.check_resource_have_state(res['type']):
                continue
            res_neutron = self.plugin_db.get_neutron_record(session, res['id'], res['type'])
            if res_neutron:
                res_status = ACUtil.get_status_by_type(res_neutron, res['type'])

                if res_status == ac_constants.NEUTRON_STATUS_ERROR:
                    LOG.error(_LE("[AC] The dependency %(type)s: %(id)s status is error"),
                              {'type': res['type'], 'id': res['id']})
                    raise ml2_exc.MechanismDriverError(method=method, url=url,
                                                       error="Data inconsistent error, the dependency status is error")

    def _handle_success_from_ac(self, request_para):
        session, seq_num, req_id, method, data = request_para.session, request_para.seq_num, request_para.req_id, \
            request_para.method, request_para.rest_data
        res = self.plugin_db.get_plugin_record(session, seq_num=seq_num)
        if not res:
            return

        res_log = validate.validate_log_record(dict(copy.deepcopy(res.get('data'))), res['res_type'])
        if self.is_need_short_feedback(res, data, method):
            LOG.debug("[AC] Set %s status as completed in plugin db, %s", res['res_type'], res_log)
            self.plugin_db.update_plugin_record(
                session, ACPluginSchema(seq_num=res['seq_num'], state=ac_constants.COMPLETE))
        else:
            self._process_no_short_feedback_res(request_para, res, res_log)

    def _process_no_short_feedback_res(self, request_para, res, res_log):
        session, seq_num, req_id, method, data = request_para.session, request_para.seq_num, \
            request_para.req_id, request_para.method, request_para.rest_data

        LOG.debug("[AC] Delete the resource in plugin db, %s", res_log)
        self.plugin_db.delete_plugin_record(session, seq_num=seq_num)

        LOG.debug("[AC] Delete the dependency in validation db,%s", res_log)
        self.plugin_db.delete_dependency(dep_seq_num=seq_num)

        self.plugin_db.delete_failed_resource(res['res_uuid'])

        if method == REST_DELETE:
            self.plugin_db.delete_standard_res_attributes(res['res_type'], req_id, session)
            return

        if ACUtil.check_resource_have_state(res['res_type']):
            state = ACRestUtils.get_neutron_status(data, res['res_type'])
            self.plugin_db.update_neutron_db_state(session, req_id, res['res_type'], state)

    def _handle_ac_offline(self, request_para):
        session, seq_num, req_id, method, url = request_para.session, request_para.seq_num, request_para.req_id, \
                                                request_para.method, request_para.url
        res = self.plugin_db.get_plugin_record(session, seq_num=seq_num)
        if method == REST_DELETE:
            LOG.error("[AC] AC is offline")
            LOG.debug("[AC] Delete the res data in plugin db and validation db")
            self.plugin_db.delete_plugin_record(session, seq_num=seq_num)
            raise ml2_exc.MechanismDriverError(method=method, url=url, error="AC is offline")
        else:
            LOG.error("[AC] AC is offline, create and update consider as success")
            self.plugin_db.delete_plugin_record(session, seq_num=seq_num)
            self.plugin_db.update_neutron_db_state(session, req_id, res['res_type'], ac_constants.NEUTRON_STATUS_ACTIVE)

    def _handle_error_timeout_from_ac(self, session, seq_num):
        res = self.plugin_db.get_plugin_record(session, seq_num=seq_num)
        if res:
            self.plugin_db.update_plugin_record(session,
                                                ACPluginSchema(seq_num=seq_num, state=ac_constants.ERROR_RETRY))

    @classmethod
    def is_need_short_feedback(cls, res, data, method):
        """Check the resource need a short feedback or not."""
        if (res['res_type'] in ac_constants.NW_HW_REPORT_STATUS_RESOURCES) and method in {REST_POST, REST_UPDATE}:
            if res['res_type'] == ac_constants.NW_HW_PORTS:
                return data.get('device-owner') not in {ncu.DEVICE_OWNER_ROUTER_GW, ncu.DEVICE_OWNER_FLOATINGIP}
            return True
        return False


class ACRestUtils(object):
    """Request utils to Agile Controller."""

    @staticmethod
    def get_tenant_name(body):
        """Get tenant name from body."""
        return body['tenant-name']

    @staticmethod
    def get_error_code(ret_content):
        """Get error code from content."""
        if 'errcode' in ret_content:
            return ret_content['errcode'].lower()
        error = ret_content['ietf-restconf:errors']['error'][0]
        return error['error-info']['error-code'].lower()

    @staticmethod
    def get_error_message(ret_content):
        """Get error message from content."""
        if 'errmsg' in ret_content:
            return ret_content['errmsg']
        error = ret_content['ietf-restconf:errors']['error'][0]
        return error['error-message']

    @staticmethod
    def get_error_paras(ret_content):
        """Get error parameters from content."""
        if 'ietf-restconf:errors' in ret_content:
            error = ret_content['ietf-restconf:errors']['error'][0]
            return error['error-info']['error-paras']
        elif 'errmsg' in ret_content:
            return "flow mirror error"
        return ''

    @staticmethod
    def fix_json(json_string):
        """Replace null by None for body."""
        return json_string.replace(r': null', r': "None"')

    @staticmethod
    def get_operation_pending_status(operation):
        """Get pending status by operation type."""
        if operation == REST_POST:
            return ac_constants.NEUTRON_STATUS_PENDING_CREATE
        elif operation == REST_UPDATE:
            return ac_constants.NEUTRON_STATUS_PENDING_UPDATE
        elif operation == REST_DELETE:
            return ac_constants.NEUTRON_STATUS_PENDING_DELETE
        return ''

    @staticmethod
    def wrap_entry_info(method, res_type, entry_info):
        """Wrap entry info according to resource type."""
        if res_type in [ac_constants.NW_HW_TAP_FLOW,
                        ac_constants.NW_HW_TAP_SERVICE]:
            if isinstance(entry_info, list) and entry_info:
                entry_info = entry_info[0]
        if res_type == ac_constants.NW_HW_TAP_FLOW:
            entry_info["quintuples"] = []
            destination_v4ip_prefix = entry_info.get('destination_v4ip_prefix')
            destination_v6ip_prefix = entry_info.get('destination_v6ip_prefix')
            if destination_v4ip_prefix:
                destination_v4ip = destination_v4ip_prefix.split('/')
                quintuples_v4 = {
                    'quintupleDIP': destination_v4ip[0],
                    'quintupleDMask': destination_v4ip[1],
                    'ipType': 'ipv4'
                }
                entry_info["quintuples"].append(quintuples_v4)
            if destination_v6ip_prefix:
                destination_v6ip = destination_v6ip_prefix.split('/')
                quintuples_v6 = {
                    'quintupleDIP': destination_v6ip[0],
                    'quintupleDMask': destination_v6ip[1],
                    'ipType': 'ipv6'
                }
                entry_info["quintuples"].append(quintuples_v6)
        if method in [REST_POST, REST_UPDATE]:
            list_name = ac_constants.NEUTRON_RESOURCES_YANG.get(res_type, {}).get('list_name')
            wrap_entry_info = {list_name: entry_info}
        else:
            wrap_entry_info = {}
        return wrap_entry_info

    @staticmethod
    def generate_rest_data(request_para):
        """Generate request data."""
        method, res_type, data = request_para.method, request_para.res_type, request_para.rest_data
        if res_type == ac_constants.NW_HW_EXROUTE:
            # remove description for restful exroute
            if res_type == ac_constants.NW_HW_EXROUTE:
                data.pop("description", None)
            rest_info = {ac_constants.EXROUTES_INPUT: data}
            return ACRestUtils.fix_json(jsonutils.dumps(rest_info))
        if res_type == ac_constants.NW_HW_QOS_POLICY:
            # remove description for restful qos policy
            data.pop("description", None)
        if res_type in {ac_constants.NW_HW_REMOVE_RULE, ac_constants.NW_HW_INSERT_RULE}:
            return ACRestUtils.fix_json(jsonutils.dumps(data))
        if not ((res_type == ac_constants.NW_HW_HEALTH_MONITOR) or (res_type == ac_constants.NW_HW_MEMBER) or
                (res_type == ac_constants.NW_HW_SEC_GRP_RULE)):
            data[ac_constants.CLOUD_NAME_PARAM] = cfg.CONF.huawei_ac_config.cloud_name
        if res_type in {ac_constants.NW_HW_TAP_SERVICE, ac_constants.NW_HW_TAP_FLOW,
                        ac_constants.NW_HW_EXTERNAL_WHITELIST}:
            entry = data
        else:
            entry = [data]
        rest_info = ACRestUtils.wrap_entry_info(method, res_type, entry)
        return ACRestUtils.fix_json(jsonutils.dumps(rest_info))

    @staticmethod
    def get_neutron_status(data, res_type):
        """Get neutron status."""
        if ('admin-state-up' in data) and not data['admin-state-up'] and res_type == ac_constants.NW_HW_PORTS:
            return ac_constants.NEUTRON_STATUS_DOWN
        return ac_constants.NEUTRON_STATUS_ACTIVE
