# !/usr/bin/env python
# -*- coding:utf-8 -*-
"""
| 功能：SSH 客户端
| 版本：2022-03-28 19:33 创建
"""
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import traceback

import paramiko

from networking_huawei.common.exceptions import ExecuteShellError
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_log

LOG = ac_log.getLogger(__name__)


class SshExecutor:
    """SSH 客户端"""

    def __init__(self, host, username, password, port=22):
        self._client = paramiko.SSHClient()
        self._client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        try:
            self._client.connect(hostname=host, port=port, username=username, password=password)
        except Exception:
            LOG.error('[AC]SSH error:%s', traceback.format_exc())
            raise ExecuteShellError(error='SSH login failed.')

    def run_command(self, sh_cmd, user_input=None):
        """远程执行 Shell 命令

        :param sh_cmd: str,Shell 命令
        :param user_input: str,用户输入
        :return: str
        """
        LOG.debug("[AC]Execute command start:%s", sh_cmd)
        stdin, stdout, stderr = self._client.exec_command(sh_cmd, timeout=30)

        # 处理用户输入
        if user_input:
            try:
                stdin.write('%s\n' % user_input)
            except OSError as e:
                LOG.info("[AC]Execute command error :%s", e.args)

        error = stderr.read()
        if error:
            if error.strip() == b'Password:':  # 提示输入密码，忽略该错误
                LOG.info("[AC]Execute command need input:%s", error)
            else:  # 处理可能的错误
                error = error.strip().replace('Password:', '')
                LOG.error("[AC]Execute command may have error:%s", error)
                if error.find(b'Authentication failure') != -1:  # Authentication failure
                    raise ExecuteShellError(error='Incorrect username or password.')
                elif error.find(b'Permission denied') != -1:  # 无文件权限
                    raise ExecuteShellError(error=error)
                else:
                    raise ExecuteShellError(error=error)

        out = stdout.read()  # read方法读取输出结果
        result = out.decode('utf8').strip()
        LOG.debug("[AC]Execute command end:%s", result)
        return result

    def close(self):
        """关闭客户端"""
        self._client.close()
