#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Function: Common utility functions"""

import os
import json
import base64
import socket
import time
import datetime

import six
from six.moves import range
from six.moves import zip
from oslo_utils import timeutils
import netaddr

from oslo_config import cfg
from neutron.db import l3_db, models_v2

try:
    from neutron.db.l3_db import Router, FloatingIP
except ImportError:
    from neutron.db.models.l3 import Router, FloatingIP
from neutron.db import db_base_plugin_v2 as neutron_db_ml2

try:
    from neutron.objects.qos import policy as policy_object
except ImportError:
    pass
from novaclient import client as nova_client
from oslo_serialization import jsonutils

try:
    from keystoneauth1.identity import v3
    from keystoneauth1 import session as auth_session
except ImportError:
    pass

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
import networking_huawei.drivers.ac.common.constants as ac_const
from networking_huawei.drivers.ac.encode_convert import \
    convert_to_bytes, convert_to_str

LOG = ncu.ac_log.getLogger(__name__)

AES_ENCRYPT_LENGTH = 16
MAX_RETRY_TIMES = 3

INFO_FILE = os.path.realpath(
    "/etc/huawei/fusionsphere/osConfig.network/cfg/neutron_ports_info")


class ACCommonUtil(object):
    """AC common util"""
    # Neutron server name taken from hostname
    neutron_name = ''

    @staticmethod
    def get_fs_host_ip_from_file():
        """get fs host ip from file"""
        if not os.access(INFO_FILE, os.R_OK):
            return None

        with open(INFO_FILE, 'r') as info_file:
            return ACCommonUtil._get_fsp_host_ip(info_file)

    @classmethod
    def _get_fsp_host_ip(cls, info_file):
        for line in info_file.readlines():
            try:
                info = json.loads(line)
                if 'ip' not in info.get('external_om', {}):
                    continue
                host_ip = info['external_om']['ip'].split('/')[0]
                if netaddr.valid_ipv4(host_ip) or netaddr.valid_ipv6(host_ip):
                    return host_ip
            except Exception as ex:
                LOG.error('[AC] get fs host ip from file failed: %s', ex)
        return None

    @classmethod
    def get_local_host_ip(cls):
        """ get host ip"""
        if ncu.after_fsp_6_3_0():
            host_ip = ACCommonUtil.get_fs_host_ip_from_file() or \
                      ACCommonUtil.get_fs_host_ip_by_socket()
        else:
            host_ip = cfg.CONF.huawei_ac_agent_config.host_ip or \
                      ACCommonUtil.get_fs_host_ip_by_socket()
        return host_ip

    @staticmethod
    def get_fs_host_ip_by_socket():
        """get fs host ip by socket"""
        try:
            hostname = socket.gethostname()
            host_ip = socket.gethostbyname(hostname)
            if netaddr.valid_ipv4(host_ip) or netaddr.valid_ipv6(
                    host_ip):
                return host_ip
        except Exception as ex:
            LOG.error('[AC] get fs host ip by socket failed: %s', ex)
        return None

    @staticmethod
    def get_neutron_server_name():
        """get neutron server name"""
        if not ACCommonUtil.neutron_name:
            ACCommonUtil.neutron_name = socket.gethostname()
            if len(ACCommonUtil.neutron_name) >= \
                    ac_const.NEUTRON_SERVER_MAX_LEN:
                raise AssertionError("Invalid hostname!")

        return ACCommonUtil.neutron_name

    @staticmethod
    def get_standard_current_time():
        """get standard current time"""
        time_local = time.time()
        return datetime.datetime.fromtimestamp(time_local)

    @staticmethod
    def get_standard_current_time_utc():
        """get standard current time utc"""
        return timeutils.utcnow()

    @staticmethod
    def is_router_interface(context, port_id):
        """is router interface"""
        neutron_db = neutron_db_ml2.NeutronDbPluginV2()
        port_info = neutron_db.get_port(context, port_id) or {}
        return port_info.get('device_owner') == ncu.DEVICE_OWNER_ROUTER_INTF

    @staticmethod
    def get_father_floatingip(context, floatingip):
        """get father floating ip"""
        LOG.debug(_LI('[AC] Begin to get father floating IP: %s'), floatingip)
        query = context.session.query(FloatingIP)
        query = query.filter_by(
            floating_network_id=floatingip.get('floating_network_id'),
            fixed_ip_address=None,
            floating_ip_address=floatingip.get('floating_ip_address'),
            tenant_id=floatingip.get('tenant_id')).first()
        LOG.info(_LI('[AC] Father floating IP got from DB: %s'), query)
        return query

    @staticmethod
    def is_sub_floatingip(context, floatingip):
        """is sub floating ip"""
        if (not floatingip.get('floating_ip_address')) or \
                (not ACCommonUtil.get_father_floatingip(context, floatingip)):
            LOG.info(_LI('[AC] %s is not sub floating IP.'), floatingip)
            return False

        # check whether it is sub floating IP when create
        port_id = floatingip.get('port_id')
        if port_id and ACCommonUtil.is_router_interface(context, port_id):
            LOG.info(_LI('[AC] %s is sub floating IP.'), floatingip)
            return True

        # check whether it is sub floating IP when update or delete
        fixed_port_id = floatingip.get('fixed_port_id')
        if fixed_port_id and ACCommonUtil.is_router_interface(context,
                                                              fixed_port_id):
            LOG.info(_LI('[AC] %s is sub floating IP.'), floatingip)
            return True

        return False

    @staticmethod
    def qos_plugin_configured():
        """qos plugin configured"""
        if ac_const.NW_HW_QOS_POLICY in \
                ac_const.NW_HW_NEUTRON_SYNC_SUPPORT_RES:
            return True
        if ac_const.NW_HW_CMCC_QOS_POLICY in \
                ac_const.NW_HW_NEUTRON_SYNC_SUPPORT_RES:
            return True
        LOG.debug('[AC]Qos plugin is not configured')
        return False

    @staticmethod
    def get_bind_url():
        """get bind url"""
        host_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip. \
            replace(' ', '').split(',')
        if len(host_list) == 1:
            host = host_list[0]
        else:
            host = ac_const.DEFAULT_AC_IP
        url = '%s%s%s%s' % (ac_const.HTTPS_HEADER,
                            host,
                            ":",
                            str(ac_const.rest_server_port))

        bind_port_url = url + ac_const.BIND_PORT_SUFFIX
        return bind_port_url

    @staticmethod
    def get_vm_name_auth_using_keystoneclient(device_id):
        """get vm name auth using keystone client"""
        LOG.info(_LI('[AC] Begin to get vm port name by device id: %s'),
                 device_id)
        try:
            from networking_huawei.drivers.ac.external.ext_if import \
                ACKeyStoneIf
            tenants = ACKeyStoneIf.get_tenant_list_from_keystone()
            keystone_info = ACKeyStoneIf.get_keystone_user_info()
            tenants_dict = dict(zip([tenant.name for tenant in tenants],
                                    [tenant.id for tenant in tenants]))

            tenant_id = tenants_dict.get(keystone_info.get('tenant_name'))
            if keystone_info.get('auth_url').endswith('v3'):
                LOG.debug("using nova v3 client.")
                if ncu.IS_FSP and \
                        ncu.get_ops_version() not in [ac_const.FSP_6_5,
                                                      ac_const.FSP_21_0]:
                    keystone_info['auth_url'] = \
                        keystone_info.get('auth_url')[:-2] + 'v2.0'
                nova = ACCommonUtil._get_nova_client(keystone_info, tenant_id)
            else:
                LOG.debug("using nova v2 client.")
                keystone_info['auth_url'] += ac_const.KEYSTONE_VERSION_SUFFIX
                kwargs = {
                    "tenant_id": tenant_id,
                    "auth_url": keystone_info.get('auth_url')
                }
                # region name is exist
                try:
                    region_name = cfg.CONF.nova.region_name
                    kwargs['timeout'] = ac_const.NOVA_REQ_TIMEOUT
                    kwargs['region_name'] = region_name
                except Exception as ex:
                    LOG.error('region name error:%s', ex)

                nova = nova_client.Client("2",
                                          keystone_info.get('user_name'),
                                          keystone_info.get('password'),
                                          **kwargs)
            vm_port_name = nova.servers.get(device_id).name
        except Exception as ex:
            LOG.error(_LE('[AC] VM port name got failed: %s'), ex)
            return ""
        LOG.info(_LI("get_vm_port_name vm_port_name is %s"), vm_port_name)
        return vm_port_name

    @classmethod
    def _get_nova_client(cls, keystone_info, tenant_id):
        return nova_client.Client(
            "2",
            keystone_info['user_name'],
            keystone_info['password'],
            tenant_id=tenant_id,
            auth_url=keystone_info.get('auth_url'),
            project_name=keystone_info.get('tenant_name'),
            user_domain_name='Default',
            user_domain_id=keystone_info.get('user_domain_id'),
            project_domain_name='Default',
            project_domain_id=keystone_info.get('project_domain_id'),
            timeout=ac_const.NOVA_REQ_TIMEOUT)

    @staticmethod
    def get_vm_name_auth_using_keystoneauth1(device_id):
        """get vm name auth using keystoneauth1"""
        LOG.info(_LI('[AC] Begin to get vm port name by device id: %s'),
                 device_id)
        try:
            from networking_huawei.drivers.ac.external.ext_if import \
                ACKeyStoneIf
            keystone_info = ACKeyStoneIf.get_keystone_user_info()
            auth = v3.Password(
                auth_url=keystone_info.get('auth_url'),
                username=keystone_info.get('user_name'),
                password=keystone_info.get('password'),
                project_name=keystone_info.get('tenant_name'),
                user_domain_id=keystone_info.get('user_domain_id'),
                project_domain_id=keystone_info.get('project_domain_id'))
            session = auth_session.Session(auth=auth, verify=False)
            nova_server = nova_client.Client(
                "2", session=session,
                region_name=cfg.CONF.nova.region_name,
                timeout=ac_const.NOVA_REQ_TIMEOUT)
            vm_port_name = nova_server.servers.get(device_id).name
            return vm_port_name

        except Exception as ex:
            LOG.error("get vm name failed,%s", ex)
            return ""

    @staticmethod
    def get_vm_port_name(device_id):
        """get vm port name"""
        try:
            if cfg.CONF.huawei_ac_config.nova_client_auth_method == \
                    ac_const.AUTH_METHOD_KEYSTONECLIENT:
                vm_name = ACCommonUtil. \
                    get_vm_name_auth_using_keystoneclient(device_id)
            elif cfg.CONF.huawei_ac_config.nova_client_auth_method == \
                    ac_const.AUTH_METHOD_KEYSTONEAUTH1:
                vm_name = ACCommonUtil. \
                    get_vm_name_auth_using_keystoneauth1(device_id)
            else:
                LOG.error("nova_client_auth_method is %s,not support",
                          cfg.CONF.huawei_ac_config.nova_client_auth_method)
                vm_name = ""
            return ACCommonUtil.get_base64_str(vm_name)
        except Exception as ex:
            LOG.error("get vm name failed,%s", ex)
            return ACCommonUtil.get_base64_str("")

    @staticmethod
    def get_base64_str(str_to_base64):
        """get base64 str"""
        tmp = jsonutils.dumps(str_to_base64)
        tmp = tmp[1:-1]
        tmp = base64.b64encode(convert_to_bytes(tmp))
        return convert_to_str(tmp)

    @staticmethod
    def get_exroute_interface_id(context, nexthop, destination):
        """get exroute interface id"""
        subnet_query = context.session.query(models_v2.Subnet)
        subnets = subnet_query.filter(
            models_v2.Subnet.cidr == destination
        ).all()
        for subnet in subnets:
            interface_query = context.session.query(models_v2.Port)
            interface_query = interface_query.join(models_v2.IPAllocation)
            router_interface = interface_query.filter(
                models_v2.IPAllocation.subnet_id == subnet.id,
                models_v2.Port.device_owner == l3_db.DEVICE_OWNER_ROUTER_INTF
            ).first()
            if not router_interface:
                LOG.info(_LI('[AC] subnet %s has no interface'), subnet.id)
                continue
            nexthop_query = context.session.query(models_v2.Port)
            nexthop_query = nexthop_query.join(models_v2.IPAllocation)
            nexthop_interface = nexthop_query.filter(
                models_v2.Port.device_id == router_interface.device_id,
                models_v2.Port.device_owner == l3_db.DEVICE_OWNER_ROUTER_INTF,
                models_v2.IPAllocation.ip_address == nexthop
            ).first()
            if nexthop_interface:
                LOG.info(_LI('[AC] interface %s found for nexthop %s'),
                         nexthop_interface.id, nexthop)
                return nexthop_interface.id
        return None

    @staticmethod
    def get_secure_random(size=AES_ENCRYPT_LENGTH, retry=MAX_RETRY_TIMES):
        """get secure random"""
        for _ in range(retry):
            try:
                with open("/dev/random", "rb") as ran_file:
                    return ran_file.read(size)
            except Exception as ex:
                LOG.info(_LI("Failed to get secure random, exc=%s", ex))
        raise Exception("Can not get secure random")

    @staticmethod
    def is_port_contain_ipv4_address(fix_ips):
        """is port contian ipv4 address"""
        if not fix_ips:
            return False
        for fix_ip in fix_ips:
            if netaddr.valid_ipv4(fix_ip['ip_address']):
                return True
        return False

    @staticmethod
    def is_port_contain_ipv6_address(fix_ips):
        """is port contain ipv6 address"""
        if not fix_ips:
            return False
        for fix_ip in fix_ips:
            if netaddr.valid_ipv6(fix_ip['ip_address']):
                return True
        return False

    @staticmethod
    def get_port_first_ipv4_subnet(fix_ips):
        """get port first ipv4 subnet"""
        if not fix_ips:
            return None
        for fix_ip in fix_ips:
            if netaddr.valid_ipv4(fix_ip['ip_address']):
                return fix_ip['subnet_id']
        return None

    @classmethod
    def get_mac(cls, ops_version):
        """get mac util"""
        if ops_version in [ac_const.OPS_K]:
            mac = neutron_db_ml2.NeutronDbPluginV2._generate_mac()
        elif ops_version in [ac_const.FSP_21_0]:
            from neutron.db import db_base_plugin_common
            mac = db_base_plugin_common.DbBasePluginCommon._generate_macs()[0]
        else:
            from neutron.db import db_base_plugin_common
            mac = db_base_plugin_common.DbBasePluginCommon._generate_mac()
        return mac


class DataFilterUtil(object):
    """data filter util"""

    def __init__(self):
        self.list_matching = cfg.CONF.huawei_ac_config.network_list_matching
        self.prefix_matching = (self.list_matching == 'prefix')
        self.suffix_matching = (self.list_matching == 'suffix')
        self.network_black_list = cfg.CONF.huawei_ac_config.network_black_list
        for i in range(len(self.network_black_list)):
            if "*" not in self.network_black_list[i]:
                continue
            if self.suffix_matching:
                self.network_black_list[i] = \
                    self.network_black_list[i].split("*")[-1]
            elif self.prefix_matching:
                self.network_black_list[i] = \
                    self.network_black_list[i].split("*")[0]
        self.network_white_list = cfg.CONF.huawei_ac_config.network_white_list
        for i in range(len(self.network_white_list)):
            if "*" not in self.network_white_list[i]:
                continue
            if self.suffix_matching:
                self.network_white_list[i] = \
                    self.network_white_list[i].split("*")[-1]
            elif self.prefix_matching:
                self.network_white_list[i] = \
                    self.network_white_list[i].split("*")[0]
        self.db_base_plugin = neutron_db_ml2.NeutronDbPluginV2()

    def _check_name_matching(self, object_name, name):
        if self.suffix_matching and object_name.endswith(name):
            return True
        if self.prefix_matching and object_name.startswith(name):
            return True
        return False

    def _name_match_with_network_white_list(self, res_name):
        for name in self.network_white_list:
            if self._check_name_matching(res_name, name):
                return True
        return False

    def _check_in_network_white_list_by_network_type(self, res, res_type):
        is_network = (res_type == ac_const.NW_HW_NETWORKS)
        if is_network or 'provider:network_type' in res:
            return self._name_match_with_network_white_list(res.get('name'))
        return False

    def _check_in_network_white_list_by_network_id(self, context, res):
        if res.get('network_id'):
            query_network = self.db_base_plugin.get_network(
                context, res['network_id'])
            network_name = query_network.get('name')
            if self._name_match_with_network_white_list(network_name):
                return True
        return False

    def _check_in_network_white_list_by_device_id(self, context, res, in_list):
        if res.get('device_id'):
            router_list = context.session.query(Router).filter(
                Router.id == res['device_id']
            ).all()
            _device_in_list = False
            for router in router_list:
                if self._name_match_with_network_white_list(router.name):
                    _device_in_list = True
            if not _device_in_list and in_list and router_list:
                return True
        return False

    def _check_in_network_white_list_by_router_id(self, context, res):
        if res.get('router') and res['router'].get('id'):
            router_list = context.session.query(Router).filter(
                Router.id == res['router']['id']
            ).all()
            for router in router_list:
                if self._name_match_with_network_white_list(router.name):
                    return True
        return False

    @staticmethod
    def _get_qos_policy_name(context, policy_id):
        filters = {'id': policy_id}
        policy_list = policy_object.QosPolicy.get_objects(context, **filters)
        resource_data = policy_list[0]
        if not resource_data:
            try:
                from neutron.common import exceptions as n_exc
            except ImportError:
                from neutron_lib.exceptions import qos as n_exc
            raise n_exc.QosPolicyNotFound(policy_id=policy_id)
        return resource_data['name']

    def _in_white_list(self, context, res, res_type, res_id):
        if not self.network_white_list:
            return True

        if res_type == ac_const.NW_HW_QOS_POLICY:
            policy_name = res.name if hasattr(res, 'name') else \
                self._get_qos_policy_name(context, res_id)
            return self._name_match_with_network_white_list(policy_name)

        if self._check_in_network_white_list_by_network_type(res, res_type):
            return True

        _in_list = False
        if self._check_in_network_white_list_by_network_id(context, res):
            _in_list = True

        if self._check_in_network_white_list_by_device_id(context, res,
                                                          _in_list):
            _in_list = False

        if res.get('router') and res['router'].get('name'):
            router_name = res['router']['name']
            if self._name_match_with_network_white_list(router_name):
                return True
            return False

        if self._check_in_network_white_list_by_router_id(context, res):
            _in_list = True

        return _in_list

    def _name_match_with_network_black_list(self, res_name):
        for name in self.network_black_list:
            if self._check_name_matching(res_name, name):
                return True
        return False

    def _check_in_network_black_list_by_network_type(self, res, res_type):
        is_network = (res_type == ac_const.NW_HW_NETWORKS)
        if is_network or 'provider:network_type' in res:
            return self._name_match_with_network_black_list(res.get('name'))
        return False

    def _check_in_network_black_list_by_network_id(self, context, res):
        if not res.get('network_id'):
            return False
        query_network = self.db_base_plugin.get_network(
            context, res['network_id'])
        network_name = query_network.get('name')
        if self._name_match_with_network_black_list(network_name):
            return True
        return False

    def _check_in_network_black_list_by_device_id(self, context, res, in_list):
        if not res.get('device_id'):
            return False
        router_list = context.session.query(Router).filter(
            Router.id == res['device_id']
        ).all()
        _device_in_list = False
        for router in router_list:
            if self._name_match_with_network_black_list(router.name):
                _device_in_list = True
        if _device_in_list and not in_list:
            return True
        return False

    def _check_in_network_black_list_by_router_id(self, context, res):
        if not res.get('router', {}).get('id'):
            return False
        routers = context.session.query(Router).filter(
            Router.id == res['router']['id']
        ).all()
        for router in routers:
            if self._name_match_with_network_black_list(router.name):
                return True
        return False

    def _in_black_list(self, context, res, res_type, res_id):
        if res_type == ac_const.NW_HW_QOS_POLICY:
            policy_name = res.name if hasattr(res, 'name') else \
                self._get_qos_policy_name(context, res_id)
            return self._name_match_with_network_black_list(policy_name)

        if self._check_in_network_black_list_by_network_type(res, res_type):
            return True

        _in_list = False
        if self._check_in_network_black_list_by_network_id(context, res):
            _in_list = True

        if self._check_in_network_black_list_by_device_id(context, res,
                                                          _in_list):
            _in_list = True

        if res.get('router') and res['router'].get('name'):
            router_name = res['router']['name']
            if self._name_match_with_network_black_list(router_name):
                return True
            return False

        if self._check_in_network_black_list_by_router_id(context, res):
            return True

        return _in_list

    @staticmethod
    def port_contain_anti_affinity(res):
        """port contain anti affinity"""
        anti_affinity_port = res.get('binding:profile', {}).get(
            'anti_affinity_port')
        if anti_affinity_port:
            LOG.info(_LI('The port is anti affinity port, '
                         'No need send request'))
            return True
        return False

    def not_in_white_or_in_black_list(self, context, res, res_type, res_id):
        """The resource not in white list or in black list.

        :param context: neutron api request context
        :param res: dictionary describing the resource
        :param res_type: type of the resource
        :param res_id: UUID of the resource
        """
        if not self._in_white_list(context, res, res_type, res_id):
            msg = '%s %s is not in white list: %s, no need to process' % (
                res_type, res_id if res_id else res['id'],
                self.network_white_list)
            LOG.info(msg)
            ac_osprofiler.record_chain_end_with_reason(msg)
            return True

        if self._in_black_list(context, res, res_type, res_id):
            msg = '%s %s is in black list: %s, no need to process' % (
                res_type, res_id if res_id else res['id'],
                self.network_black_list)
            LOG.info(msg)
            ac_osprofiler.record_chain_end_with_reason(msg)
            return True

        return False

    @staticmethod
    def _name_para_process(res_type, current_res, original_res):
        if res_type == ac_const.NW_HW_ROUTERS:
            res_id = current_res['router'].get('id')
            current_name = current_res['router'].get('name')
            original_name = original_res['router'].get('name')
        elif res_type == ac_const.NW_HW_NETWORKS:
            res_id = current_res.get('id')
            current_name = current_res.get('name')
            original_name = original_res.get('name')
        else:
            return None, None, None
        return res_id, current_name, original_name

    def name_can_be_updated(self, context, current_res, original_res, res_type):
        """Resource name can be updated or not.

        :param context: neutron api request context
        :param current_res: dictionary describing the current resource
        :param original_res: dictionary describing the original resource
        :param res_type: type of the resource
        """
        res_id, current_name, original_name = self._name_para_process(
            res_type, current_res, original_res)

        if self._in_black_list(context, original_res, res_type, res_id) and \
                not self._in_black_list(context, current_res, res_type, res_id):
            LOG.error('%s %s original name %s is in black list: %s',
                      res_type, res_id, original_name, self.network_black_list)
            LOG.error('%s %s current name %s is not in black list: %s',
                      res_type, res_id, current_name, self.network_black_list)
            msg = '%s %s name %s is in black list: %s, can not be updated' % (
                res_type, res_id, original_name, self.network_black_list)
            LOG.error(msg)
            ac_osprofiler.record_chain_exception_end(msg)
            return False, msg

        if not self._in_black_list(context, original_res, res_type, res_id) \
                and self._in_black_list(context, current_res, res_type, res_id):
            LOG.error('%s %s original name %s is not in black list: %s',
                      res_type, res_id, original_name, self.network_black_list)
            LOG.error('%s %s current name %s is in black list: %s',
                      res_type, res_id, current_name, self.network_black_list)
            msg = '%s %s name %s is in black list: %s, can not be updated' % (
                res_type, res_id, current_name, self.network_black_list)
            LOG.error(msg)
            ac_osprofiler.record_chain_exception_end(msg)
            return False, msg

        if not self._in_white_list(context, original_res, res_type, res_id) \
                and self._in_white_list(context, current_res, res_type, res_id):
            LOG.error('%s %s original name %s is not in white list: %s',
                      res_type, res_id, original_name, self.network_white_list)
            LOG.error('%s %s current name %s is in white list: %s',
                      res_type, res_id, current_name, self.network_white_list)
            msg = '%s %s name %s is in white list: %s, can not be updated' % (
                res_type, res_id, current_name, self.network_white_list)
            LOG.error(msg)
            ac_osprofiler.record_chain_exception_end(msg)
            return False, msg

        if self._in_white_list(context, original_res, res_type, res_id) and \
                not self._in_white_list(context, current_res, res_type, res_id):
            LOG.error('%s %s original name %s is in white list: %s',
                      res_type, res_id, original_name, self.network_white_list)
            LOG.error('%s %s current name %s is not in white list: %s',
                      res_type, res_id, current_name, self.network_white_list)
            msg = '%s %s name %s is in white list: %s, can not be updated' % (
                res_type, res_id, original_name, self.network_white_list)
            LOG.error(msg)
            ac_osprofiler.record_chain_exception_end(msg)
            return False, msg

        return True, None


class ConfigParas(object):
    """Config params manage,read the params from config file"""

    def __init__(self, config_file):
        """Init function"""
        self.config = six.moves.configparser.ConfigParser()
        self.config.read(config_file)

    def get_config_detail(self, group_name, config_name, default=''):
        """Get config param values from config file"""
        try:
            return self.config.get(group_name, config_name)
        except Exception:
            return default


class DisplayProcess(object):
    """Supply the method for display process"""

    @staticmethod
    def display_red(func, msg_str, solution):
        """Display error message"""
        str_len = len(solution)
        line_count = str_len // 40
        print("|%-30s|\033[1;31m%-25s\033[0m|\033[1;31m%-40s\033[0m|" %
              (func, msg_str, solution[0:40]))
        for count in six.moves.range(line_count):
            print("|%-30s|\033[1;31m%-25s\033[0m|\033[1;31m%-40s\033[0m|" %
                  ('', '', solution[(count + 1) * 40: (count + 2) * 40]))

    @staticmethod
    def display_yellow(func, msg_str, solution):
        """Display warning message"""
        str_len = len(solution)
        line_count = str_len // 40
        print("|%-30s|\033[1;33m%-25s\033[0m|\033[1;33m%-40s\033[0m|" %
              (func, msg_str, solution[0:40]))
        for count in six.moves.range(line_count):
            print("|%-30s|\033[1;33m%-25s\033[0m|\033[1;33m%-40s\033[0m|" %
                  ('', '', solution[(count + 1) * 40: (count + 2) * 40]))

    @staticmethod
    def display_normal(func, msg_str, detail):
        """Display normal message"""
        str_len = len(detail)
        line_count = str_len // 40
        print("|%-30s|%-25s|%-40s|" % (func, msg_str, detail[0:40]))
        for count in six.moves.range(line_count):
            print("|%-30s|%-25s|%-40s|" %
                  ('', '', detail[(count + 1) * 40: (count + 2) * 40]))

    @staticmethod
    def display_title(title):
        """Display title"""
        print('+' + '-' * 97 + '+')
        print('|' + ' ' * 37 + title + ' ' * (60 - len(title)) + '|')
        print('+' + '-' * 30 + '+' + '-' * 25 + '+' + '-' * 40 + '+')

    @staticmethod
    def display_dividing_line():
        """Display dividing line"""
        print('+' + '-' * 30 + '+' + '-' * 25 + '+' + '-' * 40 + '+')

    @staticmethod
    def display_result(title, results):
        """Display result"""
        DisplayProcess.display_title(title)
        for result in results:
            if result[1] == 'failed':
                DisplayProcess.display_red(result[0], result[1], result[2])
            elif result[1] == 'not configured':
                DisplayProcess.display_yellow(result[0], result[1], result[2])
            else:
                DisplayProcess.display_normal(result[0], result[1], result[2])
            DisplayProcess.display_dividing_line()


def check_father_process(server_inf):
    """Check father process"""
    neutron_server_ppid = []
    for row in server_inf:
        row_list = row.split()
        if row_list and len(row_list) > 1:
            ppid = row_list[2]
            neutron_server_ppid.append(ppid)
    len_ppid1 = []
    for ppid in neutron_server_ppid:
        if ppid == '1':
            len_ppid1.append(ppid)
    return len_ppid1
