#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""validate"""

import copy
import six
import netaddr
from oslo_config import cfg

try:
    from neutron.common.exceptions import NotFound, InvalidInput, \
        NeutronException
except ImportError:
    from neutron_lib.exceptions import NotFound, InvalidInput, NeutronException

from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging

LOG = logging.getLogger(__name__)


class ParamNotConfigured(NotFound):
    """ param not configured """
    message = "{} %(para)s not configured.".format(ac_constants.HUAWEI_CONFIG_FILE)


class ParamValueInvalid(NotFound):
    """ param value invalid """
    message = "{} %(para)s=%(value)s invalid.".format(ac_constants.HUAWEI_CONFIG_FILE)


class InvalidBooleanInput(NeutronException):
    """InvalidBooleanInput"""
    message = "Invalid input for operation: %(error_message)s."


class VpcConectionLocalCidrsInvalid(NeutronException):
    """VpcConectionLocalCidrsInvalid"""
    message = "Invalid input for operation: VPC CONNECTION " \
              "local_cidrs %(local_cidr)s is invalid."


class VpcConectionPeerCidrsInvalid(NeutronException):
    """VpcConectionPeerCidrsInvalid"""
    message = "Invalid input for operation: VPC CONNECTION " \
              "peer_cidrs %(peer_cidr)s is invalid."


class VpcConectionFirewallEnableInvalid(NeutronException):
    """VpcConectionFirewallEnableInvalid"""
    message = "Illegal parameter type, parameter value:" \
              "local_firewall_enable and peer_firewall_enable" \
              " can not be none while mode =1."


def validate_rpc_server_ip():
    """validate rpc server ip.

    :return: None
    """
    if not hasattr(cfg.CONF.huawei_ac_agent_config, 'rpc_server_ip'):
        raise ParamNotConfigured(para='rpc_server_ip')
    if not cfg.CONF.huawei_ac_agent_config.rpc_server_ip:
        raise ParamNotConfigured(para='rpc_server_ip')
    server_ips = []
    rpc_server_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip.strip().\
        replace(' ', '').split(',')
    for rpc_server in rpc_server_list:
        if validate_ip_address(rpc_server):
            raise ParamValueInvalid(
                para='rpc_server_ip',
                value=cfg.CONF.huawei_ac_agent_config.rpc_server_ip)
        server_ips.append(str(netaddr.IPAddress(rpc_server)))
    cfg.CONF.huawei_ac_agent_config.rpc_server_ip = ','.join(server_ips)


def validate_ip_address(data, valid_values=None):
    """validate input data has no whitesapce and match ipv4 or ipv6

    :param data: input data
    :param data: valid values
    :return: error msssage if validate false
    """
    try:
        trans_data = data.strip('[').strip(']')
        netaddr.IPAddress(trans_data)
        if data.count('.') != 3 and ':' not in data:
            raise ValueError()
    except Exception:
        msg = "'%s' is invalid IPv4 or IPv6 address" % data
        LOG.debug(msg)
        return msg
    return None


def validate_source_ips(data, valid_values=None):
    """validate_source_ips

    :param data: source ips
    :param valid_values: valid values
    :return: None
    """
    if data is None:
        return None
    if not isinstance(data, list):
        msg = "'%s' is not a valid list" % data
        return msg
    else:
        for ip_address in data:
            msg = validate_ip_address(ip_address, valid_values)
            if msg:
                return msg
    return None


def validate_list_of_strings(data, max_len=None):
    """validate input data as list of string and string length is
       no more than max_len

    :param data: input data
    :param max_len: string length
    """
    if not isinstance(data, list):
        msg = "'%s' is not a list" % data
        raise InvalidInput(error_message=msg)

    for item in data:
        if not isinstance(item, six.string_types):
            msg = "'%s' is not a valid string" % item
            raise InvalidInput(error_message=msg)

        if max_len is not None and len(item) > max_len:
            msg = ("'%(data)s' exceeds maximum length of %(max_len)s" %
                   {'data': item, 'max_len': max_len})
            raise InvalidInput(error_message=msg)


def validate_boolean_or_none(data, valid_values=None):
    """validate input as boolean and is not none

    :param data: input data
    :param valid_values: valid values
    :return: none
    """
    if data is None or data == '':
        return
    elif not isinstance(data, bool):
        msg = "'%s' is not a valid boolean value" % data
        raise InvalidBooleanInput(error_message=msg)


def validate_cidrs(local_cidrs, peer_cidrs):
    """validate cidrs match with ip address style

    :param local_cidrs: local cidrs
    :param peer_cidrs: peerr cidrs
    """
    for local_cidr in local_cidrs:
        try:
            netaddr.IPNetwork(local_cidr)
        except netaddr.AddrFormatError:
            raise VpcConectionLocalCidrsInvalid(local_cidr=local_cidr)
        if netaddr.valid_ipv4(local_cidr) or netaddr.valid_ipv6(local_cidr):
            raise VpcConectionLocalCidrsInvalid(local_cidr=local_cidr)
    for peer_cidr in peer_cidrs:
        try:
            netaddr.IPNetwork(peer_cidr)
        except netaddr.AddrFormatError:
            raise VpcConectionPeerCidrsInvalid(peer_cidr=peer_cidr)
        if netaddr.valid_ipv4(peer_cidr) or netaddr.valid_ipv6(peer_cidr):
            raise VpcConectionPeerCidrsInvalid(peer_cidr=peer_cidr)


def validate_firewall_mode(mode, fw_enabled, local_firewall_enable,
                           peer_firewall_enable):
    """validate firewall mode

    :param mode: mode value
    :param fw_enabled: fw_enabled
    :param local_firewall_enable: local firewall enable switch
    :param peer_firewall_enable: peer firewall enable switch
    """
    if mode == 1 and fw_enabled is None:
        if local_firewall_enable is None or peer_firewall_enable is None:
            raise VpcConectionFirewallEnableInvalid


def _get_dict_key_value(data, key, default):
    tmp = data
    for key_tmp, value in six.iteritems(tmp):
        if key_tmp == key:
            return value
        else:
            if not isinstance(value, dict):
                continue
            ret = _get_dict_key_value(value, key, default)
            if ret is not default:
                return ret
    return default


def _validate_bgp_route(data):
    if _get_dict_key_value(data, "password_text", None):
        data['session_attribute']['auth'].pop('password_text')
    if _get_dict_key_value(data, "password-text", None):
        data['session-attribute']['auth'].pop('password-text')
    return data


def validate_display_record(bgp_route):
    """hide password when return data to user interface

    :param bgp_route: input data
    :return: data
    """
    data = copy.deepcopy(bgp_route)
    if _get_dict_key_value(data, "password_text", None):
        data['session_attribute']['auth']['password_text'] = "*****"
    if _get_dict_key_value(data, "password-text", None):
        data['session_attribute']['auth']['password-text'] = "*****"
    return data


def _validate_ipsec_site_connection(data):
    if _get_dict_key_value(data, "psk", None):
        data.pop('psk')
    return data


def validate_log_record(data, res_type):
    """validate_log_record

    :param data: validate data
    :param res_type: resource type
    :return: valid data or None
    """
    if res_type == ac_constants.NW_HW_BGP_ROUTE:
        return _validate_bgp_route(data)
    elif res_type == ac_constants.NW_HW_IPSEC_SITE_CONNECTION:
        return _validate_ipsec_site_connection(data)
    return None
