#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""AC Process Status DbMixin"""


from oslo_log import log as logging
try:
    from neutron import context as neutron_context
except ImportError:
    from neutron_lib import context as neutron_context
from networking_huawei.drivers.ac.db.ac_proc_status import schema
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei._i18n import _LI, _LE

LOG = logging.getLogger(__name__)


class ACProStatusDbMixin(ncu.base_db.CommonDbMixin):
    """AC process status DB
    """
    def _get_ac_proc_status_resource(self, context, model, v_id):
        try:
            return self._get_by_id(context, model, v_id)
        except Exception as ex:
            LOG.error(_LE('[AC] Get AC process status resource failed : %s'),
                      ex)

    def _make_ac_proc_status_dict(self, ac_proc_status_db, fields=None):
        if ac_proc_status_db:
            res = {'id': ac_proc_status_db['id'],
                   'type': ac_proc_status_db['type'],
                   'ac_proc_status': ac_proc_status_db['ac_proc_status']}
            return self._fields(res, fields)
        return None

    def create_db_ac_proc_status(self, device_info):
        """the function to create ac process status"""
        LOG.info(_LI('[AC] Create ac proc status in Neutron '
                     'DB for device %s.'), device_info['id'])
        context = neutron_context.get_admin_context()
        query_device_status = context.session.query(
            schema.ACProcStatusSchema).filter_by(id=device_info['id']).all()
        if query_device_status:
            self.update_db_ac_proc_status(device_info)
        with context.session.begin(subtransactions=True):
            ac_status_db = schema.ACProcStatusSchema(
                id=device_info['id'],
                type=device_info.get('type', ''),
                ac_proc_status=device_info.get('ac_proc_status', 'PENDING'))
            context.session.add(ac_status_db)
            context.session.flush()
            return self._make_ac_proc_status_dict(ac_status_db)

    def update_db_ac_proc_status(self, device_info):
        """the function to update ac process status"""
        LOG.info(_LI('[AC] Update ac process status in Neutron DB '
                     'by id: %s.'), device_info['id'])
        context = neutron_context.get_admin_context()
        query_device_status = context.session.query(
            schema.ACProcStatusSchema).filter_by(id=device_info['id']).all()
        if not query_device_status:
            LOG.error(_LE('[AC] Update failed, the device is not existed.'
                          ' id: %s'), device_info['id'])
            LOG.info(_LI('[AC] Turn to create a record for device: %s'),
                     device_info['id'])
            self.create_db_ac_proc_status(device_info)
        else:
            device_info['type'] = device_info.get('type', '')
            try:
                with context.session.begin(subtransactions=True):
                    ac_status_db = self._get_ac_proc_status_resource(
                        context, schema.ACProcStatusSchema, device_info['id'])
                    ac_status_db.update(device_info)
                    return self._make_ac_proc_status_dict(ac_status_db)
            except Exception as ex:
                LOG.error(_LE("[AC]update ac process status in Neutron DB "
                              "failed : %s"), str(ex))
        return None

    def delete_db_ac_proc_status(self, device_id):
        """the function to delete ac process status"""
        LOG.info(_LI('[AC] Delete ac process status in Neutron '
                     'DB by id: %s.'), device_id)
        context = neutron_context.get_admin_context()
        with context.session.begin(subtransactions=True):
            ac_status_db = self._get_ac_proc_status_resource(
                context, schema.ACProcStatusSchema, device_id)
            context.session.delete(ac_status_db)

    def get_db_ac_proc_status_list(self, filters=None, fields=None):
        """the function to get ac process status list"""
        LOG.debug(_LI("[AC] Get ac process status from Neutron DB."))
        context = neutron_context.get_admin_context()
        return self._get_collection(context, schema.ACProcStatusSchema,
                                    self._make_ac_proc_status_dict,
                                    filters=filters, fields=fields)

    def get_db_ac_proc_status(self, device_id):
        """the function to get ac process status"""
        LOG.debug(_LI('[AC] Get ac process status from Neutron '
                      'DB by id: %s.'), device_id)
        context = neutron_context.get_admin_context()
        ac_status_db = self._get_ac_proc_status_resource(
            context, schema.ACProcStatusSchema, device_id)
        return self._make_ac_proc_status_dict(ac_status_db)
