#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Bgpneighbor db"""

import copy
import netaddr
from sqlalchemy.orm import exc
from oslo_log import log as logging
from oslo_utils import uuidutils
from neutron.db import models_v2

try:
    from neutron.common.exceptions import NotFound, InvalidInput
except ImportError:
    from neutron_lib.exceptions import NotFound, InvalidInput
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.db.bgp_route import schema
from networking_huawei.drivers.ac.db.bgp_route.bgp_route_db import \
   BgpRouteDbMixin
from networking_huawei.drivers.ac.db.decrypt_factor.decrypt_factor_db import \
    DecryptFactorDbMixin
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.db.dbif import ACdbInterface

LOG = logging.getLogger(__name__)


class BgpRouteNotFound(NotFound):
    """BgpRouteNotFound class"""
    message = "Bgpneighbor %(bgp_route_id)s could not be found."


class BgpNeighborDbMixin(ncu.base_db.CommonDbMixin):
    """Bgp route DB
    """

    def __init__(self):
        self.dfactor_db = DecryptFactorDbMixin()
        self.ac_db_interface = ACdbInterface()

    def get_router_interface_id(self, exsubnet_id):
        """the function to get port ID from subnet which added to a router"""
        session = self.ac_db_interface.get_session()
        interface_query = session.query(models_v2.Port)
        interface_query = interface_query.join(models_v2.IPAllocation)
        router_interface = interface_query.filter(
            models_v2.IPAllocation.subnet_id == exsubnet_id,
            models_v2.Port.device_owner == ncu.DEVICE_OWNER_ROUTER_INTF
        ).first()
        if not router_interface:
            LOG.error(_LE('[AC] No interface found for subnet %s'), exsubnet_id)
            return ""
        return router_interface.id

    def _get_bgp_route_resource(self, context, model, v_id):
        try:
            return self._get_by_id(context, model, v_id)
        except exc.NoResultFound:
            raise BgpRouteNotFound(bgp_route_id=v_id)

    def _make_bgpneighbor_dict(self, bgp_route_db, fields=None):

        res = {'bgpneighbor_id': bgp_route_db['id'],
               'tenant_id': bgp_route_db['tenant_id'],
               'name': bgp_route_db['name'],
               'description': bgp_route_db['description'],
               'router_id': bgp_route_db['router_id'],
               'router_interface_id': bgp_route_db['router_interface_id'],
               'peer_ip_address': bgp_route_db['peer_ip_address'],
               'peer_as_number': bgp_route_db['peer_as_number']}
        if 'ip_version' in bgp_route_db:
            res['ip_version'] = bgp_route_db['ip_version']
        if 'source_ips' in bgp_route_db:
            res['source_ips'] = bgp_route_db['source_ips']
        if 'exsubnet_id' in bgp_route_db:
            res['exsubnet_id'] = bgp_route_db['exsubnet_id']
        if 'local_as_number' in bgp_route_db:
            res['local_as_number'] = bgp_route_db['local_as_number']
        if 'peer_type' in bgp_route_db:
            res['peer_type'] = bgp_route_db['peer_type']
        if 'session_attribute' in bgp_route_db and \
                bgp_route_db['session_attribute'] and \
                'suppress' in bgp_route_db['session_attribute']:
            res['suppress'] = bgp_route_db['session_attribute']['suppress']

        return self._fields(res, fields)

    def _make_bgpneighbor_dict_to_ac(self, bgp_route_db, fields=None):
        res = BgpRouteDbMixin.create_bgp_route_dict(bgp_route_db)
        if 'exsubnet_id' in bgp_route_db:
            res['exsubnet_id'] = bgp_route_db['exsubnet_id']
        if 'local_as_number' in bgp_route_db:
            res['local_as_number'] = bgp_route_db['local_as_number']
        return self._fields(res, fields)

    def create_db_bgpneighbor(self, context, bgpneighbor):
        """the function to create bgpneighbor"""
        bgp_route_log = copy.deepcopy(bgpneighbor)['bgpneighbor']
        LOG.info(_LI('[AC] Create bgpneighbor in Neutron '
                     'DB for %s.'), bgp_route_log)
        bgp_route_info = self._init_bgp_router_info(bgpneighbor)
        with context.session.begin(subtransactions=True):
            bgp_route_db = schema.ACBgpRouteSchema(
                id=bgp_route_info['id'],
                tenant_id=bgp_route_info['tenant_id'],
                name=bgp_route_info['name'],
                description=bgp_route_info['description'],
                router_id=bgp_route_info['router_id'],
                exsubnet_id=bgp_route_info['exsubnet_id'],
                local_as_number=bgp_route_info['local_as_number'],
                bfd=bgp_route_info['bfd'],
                router_interface_id=bgp_route_info['router_interface_id'],
                peer_ip_address=bgp_route_info['peer_ip_address'],
                peer_as_number=bgp_route_info['peer_as_number'],
                ip_version=bgp_route_info['ip_version'],
                session_attribute=bgp_route_info['session_attribute'],
                source_ips=bgp_route_info['source_ips'],
                peer_type=bgp_route_info['peer_type'])
            context.session.add(bgp_route_db)
            context.session.flush()
        return self._make_bgpneighbor_dict_to_ac(bgp_route_db)

    def _init_bgp_router_info(self, bgpneighbor):
        """create_db_bgpneighbor call:init bgp routerinfo"""
        result = bgpneighbor['bgpneighbor']
        result['id'] = result.get('id', uuidutils.generate_uuid())
        exsubnet_id = result.get('exsubnet_id')
        if not exsubnet_id:
            result['router_interface_id'] = ""
            result['peer_type'] = 1
        else:
            if result.get('source_ips'):
                LOG.error(_LE('[AC] exsubnet_id and source_ips cannot be '
                              'configured at the same time.'))
                raise InvalidInput(error_message='exsubnet_id and source_ips '
                                                 'cannot be configured at the '
                                                 'same time.')
            result['router_interface_id'] = \
                self.get_router_interface_id(exsubnet_id)
            result['peer_type'] = 0
        peer_ip_address = result.get('peer_ip_address', None)
        if result.get('ip_version') == ac_constants.VERSION_IPV4 and \
                netaddr.valid_ipv6(peer_ip_address):
            LOG.error(_LE('[AC] Peer ip address : %s, is not ipv4.'),
                      peer_ip_address)
            raise InvalidInput(error_message='[AC] Peer ip address is not '
                                             'match with ip_version')
        if result.get('ip_version') == ac_constants.VERSION_IPV6 and \
                netaddr.valid_ipv4(peer_ip_address):
            LOG.error(_LE('[AC] Peer ip address : %s, is not ipv6.'),
                      peer_ip_address)
            raise InvalidInput(error_message='[AC] Peer ip address is not '
                                             'match with ip_version')
        result['bfd'] = None
        result.update(
            {'session_attribute': {'suppress': result.get('suppress')}})
        result['source_ips'] = result.get('source_ips', [])
        return result

    def update_db_bgpneighbor(self, context, bgp_route_id, bgp_route):
        """the function to update bgpneighbor"""
        LOG.info(_LI('[AC] Update bgpneighbor in Neutron DB '
                     'by id: %s.'), bgp_route_id)
        bgp_route_info = bgp_route['bgpneighbor']
        with context.session.begin(subtransactions=True):
            bgp_route_db = context.session. \
                query(schema.ACBgpRouteSchema). \
                filter_by(id=bgp_route_id).first()

            suppress = bgp_route_info.get('suppress')
            if suppress is not None:
                if bgp_route_info.get('session_attribute'):
                    bgp_route_info['session_attribute'].update(
                        {'suppress': suppress})
                else:
                    bgp_route_info.update(
                        {'session_attribute': {'suppress': suppress}})
            if bgp_route_db:
                bgp_route_db.update(bgp_route_info)
        return self._make_bgpneighbor_dict_to_ac(bgp_route_db)

    def delete_db_bgpneighbor(self, context, bgp_route_id):
        """the function to bgpneighbor"""
        LOG.info(_LI('[AC] Delete bgpneighbor in Neutron '
                     'DB by id: %s.'), bgp_route_id)
        with context.session.begin(subtransactions=True):
            bgp_route_db = self._get_bgp_route_resource(
                context, schema.ACBgpRouteSchema, bgp_route_id)
            context.session.delete(bgp_route_db)

    def get_db_bgpneighbors(self, context, filters=None, fields=None,
                            sorts=None, limit=None, marker_obj=None,
                            page_reverse=False):
        """the function to get bgpneighbors"""
        LOG.info(_LI("[AC] Get bgpneighbors from Neutron DB."))
        bgp_routes = self._get_collection(context, schema.ACBgpRouteSchema,
                                          self._make_bgpneighbor_dict,
                                          filters=filters, fields=fields,
                                          sorts=sorts, limit=limit,
                                          marker_obj=marker_obj,
                                          page_reverse=page_reverse)
        return bgp_routes

    def get_db_bgpneighbor(self, context, bgp_route_id, fields=None):
        """the function to get bgpneighbor"""
        LOG.info(_LI('[AC] Get bgpneighbor from Neutron '
                     'DB by id: %s.'), bgp_route_id)
        bgp_route_db = context.session.query(schema.ACBgpRouteSchema).filter_by(
            id=bgp_route_id).first()
        if bgp_route_db:
            return self._make_bgpneighbor_dict(bgp_route_db, fields=fields)
        else:
            raise BgpRouteNotFound(bgp_route_id=bgp_route_id)
