#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""decrypt factor db"""

try:
    from neutron.common.exceptions import NotFound
except ImportError:
    from neutron_lib.exceptions import NotFound

from sqlalchemy.orm import exc
from oslo_log import log as logging
from networking_huawei.drivers.ac.common.security_util import encrypt_data
from networking_huawei.drivers.ac.db.schema import ACDecryptFactorSchema
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei._i18n import _LI

LOG = logging.getLogger(__name__)


class DecryptFactorNotFound(NotFound):
    """Decrypt factor not found class"""
    message = "Decrypt factor for %(resource_id)s could not be found."


class DecryptFactorDbMixin(ncu.base_db.CommonDbMixin):
    """Decrypt Factor DB
    """
    def _get_decrypt_factor_resource(self, context, model, v_id):
        try:
            return self._get_by_id(context, model, v_id)
        except exc.NoResultFound:
            raise DecryptFactorNotFound(resource_id=v_id)

    def _make_decrypt_factor_dict(self, decrypt_factor_db, fields=None):

        res = {'id': decrypt_factor_db['id'],
               'pw_key': decrypt_factor_db['pw_key'],
               'pw_iv': decrypt_factor_db['pw_iv'],
               'key_iv': decrypt_factor_db['key_iv']}
        return self._fields(res, fields)

    @classmethod
    def create_db_decrypt_factor(cls, context, uuid, text):
        """the function to create decrypt factor"""
        LOG.info(_LI('[AC] Create decrypt factor in Neutron '
                     'DB for %s.'), uuid)
        password_key, password_iv, key_vi, data_encrypt = encrypt_data(text)
        with context.session.begin(subtransactions=True):
            decrypt_factor_db = ACDecryptFactorSchema(
                id=uuid,
                pw_key=password_key,
                pw_iv=password_iv,
                key_iv=key_vi)
            context.session.add(decrypt_factor_db)
            context.session.flush()
        return data_encrypt

    @classmethod
    def update_db_decrypt_factor_for_random(cls, session, random_dict):
        """the function to update decrypt factor"""
        LOG.info(_LI('[AC] Update decrypt factor random in Neutron DB'))
        with session.begin(subtransactions=True):
            factor_random_db = session.query(ACDecryptFactorSchema).filter_by(
                id="random").first()
            if factor_random_db:
                factor_random_db.pw_key = random_dict['work_key_random']
                factor_random_db.pw_iv = random_dict['work_key_iv_random']
                factor_random_db.key_iv = random_dict['pw_iv_random']
                session.merge(factor_random_db)
                session.flush()
            else:
                d_factor_db = ACDecryptFactorSchema(
                    id="random",
                    pw_key=random_dict['work_key_random'],
                    pw_iv=random_dict['work_key_iv_random'],
                    key_iv=random_dict['pw_iv_random'])
                session.add(d_factor_db)
                session.flush()

    @classmethod
    def update_db_decrypt_factor(cls, context, uuid, text):
        """the function to update decrypt factor"""
        LOG.info(_LI('[AC] Update decrypt factor in Neutron DB for %s.'), uuid)
        password_key, password_iv, key_vi, data_encrypt = encrypt_data(text)
        with context.session.begin(subtransactions=True):
            decrypt_factor_db = ACDecryptFactorSchema(
                id=uuid,
                pw_key=password_key,
                pw_iv=password_iv,
                key_iv=key_vi)
            context.session.merge(decrypt_factor_db)
            context.session.flush()
        return data_encrypt

    @classmethod
    def delete_decrypt_factor(cls, context, uuid):
        """the function to delete decrypt factor"""
        LOG.info(_LI('[AC]Delete decrypt factor in Neutron DB by id: %s.'),
                 uuid)
        with context.session.begin(subtransactions=True):
            decrypt_factor_db = context.session.query(ACDecryptFactorSchema).\
                filter_by(id=uuid).first()
            if decrypt_factor_db:
                context.session.delete(decrypt_factor_db)

    def get_db_decrypt_factor(self, context, uuid):
        """the function to get decrypt factor"""
        LOG.info(_LI('[AC] Get decrypt factor from Neutron '
                     'DB by id: %s.'), uuid)
        decrypt_factor_db = self._get_decrypt_factor_resource(
            context, ACDecryptFactorSchema, uuid)
        if decrypt_factor_db:
            return self._make_decrypt_factor_dict(decrypt_factor_db)
        return None
