#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""L2brDbMixin class"""

from sqlalchemy.orm import exc
from oslo_log import log as logging
from networking_huawei.drivers.ac.extensions.l2br import l2br
from networking_huawei.drivers.ac.extensions.l2br.l2br \
    import L2brDeviceInfoParamInvalid
from networking_huawei.drivers.ac.extensions.l2br.l2br \
    import L2brServiceInfoParamInvalid
from networking_huawei.drivers.ac.db.l2br import schema
from networking_huawei._i18n import _LI
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

LOG = logging.getLogger(__name__)


class L2brDbMixin(ncu.base_db.CommonDbMixin):
    """L2BR DB"""

    def _get_l2br_resource(self, context, model, v_id):
        try:
            return self._get_by_id(context, model, v_id)
        except exc.NoResultFound:
            raise l2br.L2brNotFound(l2br_id=v_id)

    @classmethod
    def _validate_l2br_device_info_parameters(cls, l2br_info):
        LOG.info(_LI('[AC] Begin to validate device_info= %s.'),
                 l2br_info["device_info"])
        if l2br_info["device_info"].get('device_id') \
                and l2br_info["device_info"]['device_id'].strip() == '':
            del l2br_info["device_info"]['device_id']
        if l2br_info["device_info"].get('device_mac') \
                and l2br_info["device_info"]['device_mac'].strip() == '':
            del l2br_info["device_info"]['device_mac']
        if l2br_info["device_info"].get('device_ip') \
                and l2br_info["device_info"]['device_ip'].strip() == '':
            del l2br_info["device_info"]['device_ip']
        if l2br_info["device_info"].get('device_id') \
                and l2br_info["device_info"].get('device_mac'):
            raise L2brDeviceInfoParamInvalid()
        if l2br_info["device_info"].get('device_id') \
                and l2br_info["device_info"].get('device_ip'):
            raise L2brDeviceInfoParamInvalid()
        if l2br_info["device_info"].get('device_mac') \
                and l2br_info["device_info"].get('device_ip'):
            raise L2brDeviceInfoParamInvalid()
        if (not l2br_info["device_info"].get('device_id')) and \
                (not l2br_info["device_info"].get('device_mac')) \
                and (not l2br_info["device_info"].get('device_ip')):
            raise L2brDeviceInfoParamInvalid()

    @classmethod
    def _validate_l2br_service_info_parameters(cls, l2br_info):
        LOG.info(_LI('[AC] Begin to validate service_info= %s.'),
                 l2br_info["service_info"])
        if l2br_info["service_info"].get('network_id') \
                and l2br_info["service_info"]['network_id'].strip() == '':
            del l2br_info["service_info"]['network_id']
        if l2br_info["service_info"].get('vni') \
                and l2br_info["service_info"]['vni'] == 0:
            del l2br_info["service_info"]['vni']
        if l2br_info["service_info"].get('vni') \
                and l2br_info["service_info"].get('network_id'):
            raise L2brServiceInfoParamInvalid()
        if (not l2br_info["service_info"].get('vni')) \
                and (not l2br_info["service_info"].get('network_id')):
            raise L2brServiceInfoParamInvalid()

    def _make_l2br_dict(self, l2br_db, fields=None):

        res = {'id': l2br_db['id'],
               'tenant_id': l2br_db['tenant_id'],
               'name': l2br_db['name'],
               'device_info': {
                   'device_id': l2br_db['device_id'],
                   'device_mac': l2br_db['device_mac'],
                   'device_ip': l2br_db['device_ip'],
                   'interface_id': l2br_db['interface_id']},
               'service_info': {
                   'vlan_id': l2br_db['vlan_id'],
                   'vni': l2br_db['vni'],
                   'network_id': l2br_db['network_id']},
               'status': l2br_db['status']}

        return self._fields(res, fields)

    def create_db_l2br(self, context, l2br_dict, status=None):
        """the function to create l2br"""
        LOG.info(_LI('[AC] Create l2br in Neutron DB for %s.'), l2br_dict)
        l2br_info = l2br_dict['l2br']
        try:
            tenant_id = self._get_tenant_id_for_create(context, l2br_info)
        except Exception:
            if context.is_admin and 'tenant_id' in l2br_info:
                tenant_id = l2br_info['tenant_id']
            else:
                tenant_id = context.tenant_id
        device_info, service_info = self._deal_device_and_service(l2br_info)
        with context.session.begin(subtransactions=True):
            l2br_db = schema.ACL2brSchema(
                tenant_id=tenant_id,
                name=l2br_info['name'],
                device_id=device_info['device_id'],
                device_mac=device_info['device_mac'],
                device_ip=device_info['device_ip'],
                interface_id=device_info['interface_id'],
                vlan_id=service_info['vlan_id'],
                vni=service_info['vni'],
                network_id=service_info['network_id'],
                status=status)
            context.session.add(l2br_db)
            context.session.flush()

        return self._make_l2br_dict(l2br_db)

    def _deal_device_and_service(self, l2br_info):
        """deal device_info and service_info in l2br_info"""
        self._validate_l2br_device_info_parameters(l2br_info)
        self._validate_l2br_service_info_parameters(l2br_info)
        device_info = l2br_info['device_info']
        service_info = l2br_info['service_info']
        if not l2br_info.get('name'):
            l2br_info['name'] = ''
        if not device_info.get('device_id'):
            device_info['device_id'] = ''
        if not device_info.get('device_mac'):
            device_info['device_mac'] = ''
        if not device_info.get('device_ip'):
            device_info['device_ip'] = ''
        if not service_info.get('vni'):
            service_info['vni'] = 0
        if not service_info.get('network_id'):
            service_info['network_id'] = ''
        return device_info, service_info

    def update_db_l2br(self, context, l2br_id, l2br_dict):
        """the function to update l2br"""
        LOG.info(_LI('[AC] Update l2br in Neutron DB by id, id= %s.'), l2br_id)
        l2br_to_update = l2br_dict['l2br']
        device_info, service_info = self._deal_device_and_service(
            l2br_to_update)
        with context.session.begin(subtransactions=True):
            l2br_db = self._get_l2br_resource(context,
                                              schema.ACL2brSchema,
                                              l2br_id)
            l2br_info = {
                'name': l2br_to_update['name'],
                'device_id': device_info['device_id'],
                'device_mac': device_info['device_mac'],
                'device_ip': device_info['device_ip'],
                'interface_id': device_info['interface_id'],
                'vlan_id': service_info['vlan_id'],
                'vni': service_info['vni'],
                'network_id': service_info['network_id']
            }
            l2br_db.update(l2br_info)
        return self._make_l2br_dict(l2br_db)

    def update_db_l2br_status(self, context, l2br_id, status):
        """the function to update l2br status"""
        LOG.info(_LI('[AC] Update l2br %s status in Neutron DB'), l2br_id)
        with context.session.begin(subtransactions=True):
            l2br_db = self._get_l2br_resource(
                context, schema.ACL2brSchema, l2br_id)
            l2br_db.status = status
            l2br_db.update(l2br_db)
        return self._make_l2br_dict(l2br_db)

    def delete_db_l2br(self, context, l2br_id):
        """the function to delete l2br"""
        LOG.info(_LI('[AC] Delete l2br in Neutron DB by id, id= %s.'), l2br_id)
        with context.session.begin(subtransactions=True):
            l2br_db = self._get_l2br_resource(
                context, schema.ACL2brSchema, l2br_id)
            context.session.delete(l2br_db)

    def delete_db_l2brs_by_network(self, context, network_id):
        """the function to delete l2br by network"""
        LOG.info(_LI('[AC] Delete l2br in Neutron '
                     'DB by network id: %s.'), network_id)
        query_dbs = context.session.query(schema.ACL2brSchema).filter_by(
            network_id=network_id).all()
        for elem in query_dbs:
            self.delete_db_l2br(context, elem.id)

    def get_db_l2brs(self, context, filters=None, fields=None):
        """the function to get l2brs"""
        LOG.info(_LI("[AC] Get l2brs in Neutron DB."))
        return self._get_collection(context, schema.ACL2brSchema,
                                    self._make_l2br_dict,
                                    filters=filters, fields=fields)

    def get_db_l2br(self, context, l2br_id, fields=None):
        """the function to get l2br"""
        LOG.info(_LI('[AC] Get l2br in Neutron DB by id, id= %s.'), l2br_id)
        l2_db = self._get_l2br_resource(context, schema.ACL2brSchema,
                                        l2br_id)
        return self._make_l2br_dict(l2_db, fields=fields)
