#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Bgpneighbor extension"""

import abc
import six
from oslo_config import cfg
from neutron.api.v2 import resource as wsgi_resource

try:
    from neutron_lib.plugins import directory
except Exception:
    directory = None
from neutron import manager

try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor

try:
    from neutron_lib.api import validators
except ImportError:
    validators = None
try:
    from neutron.api.v2 import attributes
except ImportError:
    attributes = None

from neutron.api.v2 import base

try:
    from neutron.api.v2.base import FAULT_MAP as neutronfaultmap
except ImportError:
    from neutron_lib.api.faults import FAULT_MAP as neutronfaultmap

from neutron.api import extensions

try:
    from neutron.common.exceptions import NeutronException
except ImportError:
    from neutron_lib.exceptions import NeutronException
try:
    from neutron.common.exceptions import NotFound
except ImportError:
    from neutron_lib.exceptions import NotFound
try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase

import networking_huawei.drivers.ac.common.constants as ac_constants
from networking_huawei.drivers.ac.common import validate as validator
from networking_huawei.drivers.ac.extensions import bgpneighbor \
    as bgpneighbor_extension
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
LOG = logging.getLogger(__name__)


class BgpRouteNotFound(NotFound):
    """Bgpneighbor not found class"""
    message = "Bgpneighbor %(bgp_route_id)s could not be found."


extensions.append_api_extensions_path(bgpneighbor_extension.__path__)

BGPNEIGHBOR = 'bgpneighbor'

OPS_VERSION = ncu.get_ops_version()


def create_resource_bgpneighbor(collection, resource, plugin, params,
                                allow_bulk=False, member_actions=None,
                                parent=None, allow_pagination=False,
                                allow_sorting=False):
    """create resource bgpneighbor"""
    controller = base.Controller(plugin, collection, resource, params,
                                 allow_bulk, member_actions=member_actions,
                                 parent=parent,
                                 allow_pagination=allow_pagination,
                                 allow_sorting=allow_sorting)

    return wsgi_resource.Resource(controller, neutronfaultmap, None, None,
                                  dict(create=201, delete=200, update=201))


if validators and hasattr(validators, 'add_validator') and \
        'type:ip_address_list_or_nodata' not in validators.validators:
    validators.add_validator('ip_address_list_or_nodata',
                             validator.validate_source_ips)

if attributes and hasattr(attributes, 'validators') and \
        'type:ip_address_list_or_nodata' not in attributes.validators:
    attributes.validators['type:ip_address_list_or_nodata'] = \
        validator.validate_source_ips


class InvalidIntegerInput(NeutronException):
    """Invalid interger input"""
    message = "Invalid input for operation: %(error_message)s."


def convert_to_int(data):
    """convert to int"""
    try:
        return int(data)
    except (ValueError, TypeError):
        msg = "'%s' is not an integer" % data
        raise InvalidIntegerInput(error_message=msg)


# Attribute Map
RESOURCE_ATTRIBUTE_MAP = {
    "bgpneighbors": {
        'bgpneighbor_id': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:uuid_or_none': None},
            'is_visible': True, 'primary_key': True,
            'default': None},
        'name': {
            'allow_post': True, 'allow_put': True, 'is_visible': True,
            'validate': {'type:string_or_none': 255}},
        'description': {
            'allow_post': True, 'allow_put': True, 'is_visible': True,
            'default': '', 'validate': {'type:string_or_none': 255}},
        'tenant_id': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:string': 255},
            'is_visible': True},
        'router_id': {
            'allow_post': True, 'allow_put': False, 'is_visible': True,
            'validate': {'type:uuid': None}},
        'exsubnet_id': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:string_or_none': 255},
            'is_visible': True,
            'default': ''},
        'local_as_number': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True,
            'validate': {'type:range': [1, 4294967295]}},
        'ip_version': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True,
            'convert_to': convert_to_int,
            'validate': {'type:values': [-1, 4, 6]}},
        'peer_ip_address': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True, 'validate': {'type:ip_address': None}},
        'peer_as_number': {
            'allow_post': True, 'allow_put': False,
            'is_visible': True,
            'validate': {'type:range': [1, 4294967295]}},
        'suppress': {
            'allow_post': True, 'allow_put': True, 'is_visible': True,
            'default': True, 'validate': {'type:boolean': None}},
        'source_ips': {
            'allow_post': True, 'allow_put': True, 'is_visible': True,
            'default': None,
            'validate': {'type:ip_address_list_or_nodata': None}},
    }
}


class Bgpneighbor(ExtensionDescriptor):
    """Consistency report generation"""

    @classmethod
    def get_name(cls):
        """Name for this extension"""
        return "Bgpneighbor"

    @classmethod
    def get_alias(cls):
        """core_plugin class to load the extension"""
        return 'bgpneighbor'

    @classmethod
    def get_description(cls):
        """A small description about this extension"""
        return "Support connecting site with datacenter by bgp"

    @classmethod
    def get_namespace(cls):
        """The XML namespace for this extension"""
        return ""

    @classmethod
    def get_updated(cls):
        """Specify when was this extension last updated"""
        return "2017-04-18T21:10:50-00:00"

    @classmethod
    def get_resources(cls):
        """Returns Ext Resources."""
        if OPS_VERSION in [ac_constants.OPS_O, ac_constants.OPS_P,
                           ac_constants.OPS_Q, ac_constants.OPS_R, ac_constants.OPS_W,
                           ac_constants.FSP_6_5, ac_constants.FSP_21_0]:
            allow_pagination = True
            allow_sorting = True
            plugin = directory.get_plugin(BGPNEIGHBOR)
        else:
            plugin = \
                manager.NeutronManager.get_service_plugins()[BGPNEIGHBOR]
            allow_pagination = cfg.CONF.allow_pagination
            allow_sorting = cfg.CONF.allow_sorting
        resources = []

        for resource_name in ["bgpneighbor"]:
            collection_name = BGPNEIGHBOR + "s"
            params = RESOURCE_ATTRIBUTE_MAP.get("bgpneighbors", dict())
            member_actions = {}
            controller = create_resource_bgpneighbor(
                collection_name, resource_name, plugin, params,
                allow_bulk=True,
                member_actions=member_actions,
                allow_pagination=allow_pagination,
                allow_sorting=allow_sorting)
            res = extensions.ResourceExtension(collection_name,
                                               controller,
                                               member_actions=member_actions,
                                               attr_map=params)
            resources.append(res)

        return resources

    @classmethod
    def get_plugin_interface(cls):
        """get plugin interface"""
        return BgpneighborBase

    def update_attributes_map(self, attributes, extension_attrs_map=None):
        """update attributes map"""
        super(Bgpneighbor, self).update_attributes_map(
            attributes, extension_attrs_map=RESOURCE_ATTRIBUTE_MAP)

    def get_extended_resources(self, version):
        """get extended resources"""
        if version != "2.0":
            return {}
        return RESOURCE_ATTRIBUTE_MAP


@six.add_metaclass(abc.ABCMeta)
class BgpneighborBase(ServicePluginBase):
    """Bgpneighbor class"""

    def get_plugin_type(self):
        """get plugin type"""
        pass

    def get_plugin_description(self):
        """get plugin description"""
        pass

    def get_plugin_name(self):
        """get plugin name"""
        pass

    @abc.abstractmethod
    def create_bgpneighbor(self, context, bgpneighbor):
        """the abstractmethod to create bgpneighbor"""
        pass

    @abc.abstractmethod
    def update_bgpneighbor(self, context, bgpneighbor_id, bgpneighbor):
        """the abstractmethod to update bgpneighbor"""
        pass

    @abc.abstractmethod
    def delete_bgpneighbor(self, context, bgpneighbor_id):
        """the abstractmethod to delete bgpneighbor"""
        pass

    @abc.abstractmethod
    def get_bgpneighbor(self, context, bgpneighbor_id, fields=None):
        """the abstractmethod to get bgpneighbor"""
        pass

    @abc.abstractmethod
    def get_bgpneighbors(self, context, filters=None, fields=None):
        """the abstractmethod to get bgpneighbors"""
        pass
