#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""exroute extension"""

import abc
from neutron import wsgi
from neutron import policy
from neutron.api import extensions
from neutron.api.v2 import base
from neutron.api.v2 import resource as v2_resource
from neutron.extensions import l3

from networking_huawei._i18n import _
from networking_huawei.drivers.ac.common import constants
from networking_huawei.drivers.ac.common import validate as validator
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

try:
    from neutron_lib.api import faults
except ImportError:
    pass
try:
    from neutron.common.exceptions import NeutronException
except ImportError:
    from neutron_lib.exceptions import NeutronException
try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor

try:
    from neutron_lib.api.definitions.l3 import RESOURCE_ATTRIBUTE_MAP
except ImportError:
    from neutron.extensions.l3 import RESOURCE_ATTRIBUTE_MAP

EXROUTES = 'exroutes'
EXROUTES_DESTINATION = 'destination'
EXROUTES_NEXTHOP = 'nexthop'
EXROUTES_TYPE = 'type'
EXROUTES_SEGMENT_TYPE = 'segment_type'
EXROUTES_SEGMENT_ID = 'segment_id'
EXROUTES_PRIORITY = 'priority'
EXROUTES_PROTOCOL = 'protocol'
EXROUTES_PORT = 'port'
EXROUTES_EXTRA_OPT = 'extra_opt'
QUERY_PARAMS_NUM = 2
TYPE_MAX = 9
TYPE_MIN = 0
TYPE_SDN_VPC_PEERING = 400
QUERY_SPLITTER = '&'
QUERY_EQUAL = '='


class ParamNotSpecified(NeutronException):
    """Param Not Specified"""
    message = _("%(param)s is not specified in %(exroute)s.")


class ParamInvalid(NeutronException):
    """Param Invalid"""
    message = _("%(param)s in %(exroute)s is invalid.")


class ParamOutOfRange(NeutronException):
    """Param Out Of Range"""
    message = _("%(param)s in %(exroute)s is not in %(range)s.")


class InvalidParam(NeutronException):
    """参数非法"""
    message = _("The param(%(name)s) value(%(value)s) is invalid.Error detail:%(error_trace)s")


class MissParam(NeutronException):
    """参数缺失"""
    message = _("The param(%(name)s) is miss.")


class ExRoutePortNotFound(NeutronException):
    """BFD路由对应的Port没有或者超过1个"""
    message = _("Can't find the best match port of %(ex_route)s.")


class ExRouteMultiBfdLeader(NeutronException):
    """BFD路由对应的子Port有超过1个被标记成首个子port bfd_leader"""
    message = _("The %(ex_route)s find more then one port which mark as bfd_leader.")


def _get_query_params_values(query_str):
    if not query_str:
        return None, None
    type_value = None
    nexthop_value = None
    std_query = QUERY_SPLITTER.join(
        [s for s in query_str.split(QUERY_SPLITTER) if s]).lower()
    if std_query.count(QUERY_EQUAL) > QUERY_PARAMS_NUM:
        raise Exception
    key_value_pairs = std_query.split(QUERY_SPLITTER)
    for key_value in key_value_pairs:
        name, value = key_value.split(QUERY_EQUAL)
        if name == EXROUTES_TYPE:
            type_value = int(value)
            if type_value < TYPE_MIN or (type_value > TYPE_MAX and type_value != TYPE_SDN_VPC_PEERING):
                raise Exception
        elif name == EXROUTES_NEXTHOP:
            err_msg = validator.validate_ip_address(value)
            if err_msg:
                raise Exception
            nexthop_value = value
        else:
            raise Exception
    return type_value, nexthop_value


class ExroutesController(wsgi.Controller):
    """Exroutes Controller"""

    def get_plugin(self):
        """get plugin"""
        plugin = ncu.get_service_plugin()['L3_ROUTER_NAT']
        if not plugin:
            raise Exception
        return plugin

    def index(self, request, **kwargs):
        """index"""
        policy.enforce(request.context, "get_%s" % EXROUTES, {})
        type_value, nexthop_value = _get_query_params_values(
            request.environ.get('QUERY_STRING'))

        return self.get_plugin().list_router_exroutes(
            request.context, kwargs.get('router_id'),
            type_value=type_value, nexthop_value=nexthop_value)


class L3_exroutes(ExtensionDescriptor):
    """L3 exroutes extension"""

    @classmethod
    def get_name(cls):
        """get name"""
        return "Neutron L3 Router Exroutes"

    @classmethod
    def get_alias(cls):
        """get alias"""
        return "exroutes_ext_alias"

    @classmethod
    def get_description(cls):
        """get description"""
        return "L3 Router exroutes extension"

    @classmethod
    def get_namespace(cls):
        """get namespace"""
        return ''

    @classmethod
    def get_updated(cls):
        """get updated"""
        return "2016-09-19T10:00:00-00:00"

    @classmethod
    def update_floatingips_map(cls, resource):
        """update floatingips map"""
        from neutron.services.l3_router import l3_router_plugin
        supported_extension_aliases = \
            l3_router_plugin.L3RouterPlugin._supported_extension_aliases
        if 'qos-fip' in supported_extension_aliases:
            from neutron.extensions import qos_fip
            resource.attr_map.update(
                dict(qos_fip.EXTENDED_ATTRIBUTES_2_0.items())['floatingips'])

    def get_resources(self):
        """get resources"""
        ops_version = ncu.get_ops_version()
        resources = l3.L3.get_resources()
        for resource in resources:
            if resource.collection == 'routers':
                resource.member_actions['add_exroutes'] = 'PUT'
                resource.member_actions['remove_exroutes'] = 'PUT'
                if ops_version in [constants.OPS_P, constants.OPS_O,
                                   constants.OPS_N]:
                    from neutron.extensions import \
                        l3_ext_gw_mode, l3_ext_ha_mode, dvr, extraroute, \
                        timestamp, standardattrdescription, \
                        router_availability_zone
                    resource.attr_map.update(
                        l3_ext_gw_mode.EXTENDED_ATTRIBUTES_2_0.get('routers'))
                    resource.attr_map.update(
                        l3_ext_ha_mode.EXTENDED_ATTRIBUTES_2_0.get('routers'))
                    resource.attr_map.update(
                        dvr.EXTENDED_ATTRIBUTES_2_0.get('routers'))
                    resource.attr_map.update(
                        extraroute.EXTENDED_ATTRIBUTES_2_0.get('routers'))
                    resource.attr_map.update(
                        timestamp.TIMESTAMP_BODY)
                    resource.attr_map.update(
                        standardattrdescription.DESCRIPTION_BODY)
                    resource.attr_map.update(
                        router_availability_zone.
                            EXTENDED_ATTRIBUTES_2_0.get('routers'))
                if ops_version in [constants.OPS_P]:
                    from neutron.extensions import tagging
                    resource.attr_map.update(
                        tagging.TAG_ATTRIBUTE_MAP)
                if ops_version in [constants.OPS_O]:
                    from neutron.extensions import tag
                    resource.attr_map.update(
                        tag.TAG_ATTRIBUTE_MAP)
            if ops_version in [constants.OPS_P] and \
                    resource.collection == 'floatingips':
                self.update_floatingips_map(resource)

        exts = self.get_exts_by_version(ops_version)
        exts.extend(resources)
        return exts

    def get_exts_by_version(self, ops_version):
        """根据版本获取扩展

        :param ops_version: str,版本
        :return: list
        """
        if ops_version in [constants.OPS_Q, constants.OPS_R, constants.OPS_T, constants.OPS_W,
                           constants.FSP_6_5, constants.FSP_21_0]:
            controller = v2_resource.Resource(ExroutesController(),
                                              faults.FAULT_MAP)
        else:
            controller = v2_resource.Resource(ExroutesController(),
                                              base.FAULT_MAP)
        return [extensions.ResourceExtension(
            EXROUTES, controller,
            dict(member_name="router", collection_name="routers"))]

    def update_attributes_map(self, attributes):
        """update attributes map"""
        super(L3_exroutes, self).update_attributes_map(
            attributes, extension_attrs_map=RESOURCE_ATTRIBUTE_MAP)

    def get_extended_resources(self, version):
        """get extended resources"""
        return {}

    def get_required_extensions(self):
        """get required extensions"""
        return ["router"]


class RouterExroutesBase:
    """Router Exroutes Base"""

    @abc.abstractmethod
    def add_exroutes(self, context, router_id, exroutes_info):
        """add exroutes"""
        pass

    @abc.abstractmethod
    def remove_exroutes(self, context, router_id, exroutes_info):
        """remove exroutes"""
        pass

    @abc.abstractmethod
    def list_router_exroutes(self, context, router_id):
        """list router exroutes"""
        pass
