#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""External whitelist extension."""

import abc

import six

from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.extensions.sync_result.sync_result import \
    Sync_result

try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor
from neutron.api import extensions
from neutron.api.v2 import base
try:
    from neutron_lib.api import validators
except ImportError:
    validators = None
try:
    from neutron.api.v2 import attributes
except ImportError:
    attributes = None

try:
    from neutron.common.exceptions import NotFound
except ImportError:
    from neutron_lib.exceptions import NotFound

try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase

from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.extensions import external_whitelist \
    as external_whitelist_extension

LOG = logging.getLogger(__name__)


class ExternalWhitelistNotFound(NotFound):
    """External Whitelist not found class"""
    message = "External Whitelist %(external_whitelist_id)s could not be found."


extensions.append_api_extensions_path(external_whitelist_extension.__path__)

EXTERNAL_WHITELIST = 'external_whitelist'

if validators and hasattr(validators, 'add_validator') and \
        'type:list_of_strings' not in validators.validators:
    validators.add_validator('list_of_strings',
                             validate.validate_list_of_strings)
if attributes and hasattr(attributes, 'validators') and \
        'type:list_of_strings' not in attributes.validators:
    attributes.validators['type:list_of_strings'] = \
        validate.validate_list_of_strings

# Attribute Map
RESOURCE_ATTRIBUTE_MAP = {
    "externalwhitelist": {
        'id': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:uuid_or_none': None},
            'is_visible': True,
            'default': None},
        'tenant_id': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 255},
            'default': '',
            'is_visible': True},
        'router_id': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:uuid_or_none': None},
            'is_visible': True,
            'default': None},
        'routes': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': [],
            'validate': {'type:list_of_strings': None}},
        'subnets': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': [],
            'validate': {'type:list_of_strings': None}},
    }
}


class External_whitelist(ExtensionDescriptor):
    """Consistency report generation"""

    @classmethod
    def get_name(cls):
        """Name for this extension"""
        return "external whitelist"

    @classmethod
    def get_alias(cls):
        """core_plugin class to load the extension"""
        return 'ac-external-whitelist'

    @classmethod
    def get_description(cls):
        """A small description about this extension"""
        return "Support connecting site with datacenter by external whitelist"

    @classmethod
    def get_namespace(cls):
        """The XML namespace for this extension"""
        return ""

    @classmethod
    def get_updated(cls):
        """Specify when was this extension last updated"""
        return "2017-04-18T21:10:50-00:00"

    @classmethod
    def get_resources(cls):
        """Returns Ext Resources."""
        allow_pagination, allow_sorting, plugin = \
            Sync_result.get_resource_before(EXTERNAL_WHITELIST)

        resources = []
        for resource_name in [EXTERNAL_WHITELIST]:
            collection_name = EXTERNAL_WHITELIST + 's'
            params = RESOURCE_ATTRIBUTE_MAP.get("externalwhitelist", {})
            member_actions = {'add_routes': 'PUT', 'remove_routes': 'PUT',
                              'add_subnets': 'PUT', 'remove_subnets': 'PUT'}
            controller = base.create_resource(
                collection_name, resource_name, plugin, params, allow_bulk=True,
                member_actions=member_actions, allow_sorting=allow_sorting,
                allow_pagination=allow_pagination)
            res = extensions.ResourceExtension(
                collection_name, controller, member_actions=member_actions,
                attr_map=params)
            resources.append(res)
        return resources

    @classmethod
    def get_plugin_interface(cls):
        """Get plugin interface"""
        return ExternalWhitelistBase

    def get_extended_resources(self, version):
        """Get extended resources"""
        if version != "2.0":
            return {}
        return RESOURCE_ATTRIBUTE_MAP

    def update_attributes_map(self, attributes, extension_attrs_map=None):
        """Update attributes map"""
        super(External_whitelist, self).update_attributes_map(
            attributes, extension_attrs_map=RESOURCE_ATTRIBUTE_MAP)


@six.add_metaclass(abc.ABCMeta)
class ExternalWhitelistBase(ServicePluginBase):
    """External Whitelist class"""

    def get_plugin_type(self):
        """Get plugin type"""
        pass

    def get_plugin_description(self):
        """Get plugin description"""
        pass

    def get_plugin_name(self):
        """Get plugin name"""
        pass

    @abc.abstractmethod
    def create_external_whitelist(self, context, external_whitelist):
        """the abstractmethod to create external whitelist"""
        pass

    @abc.abstractmethod
    def update_external_whitelist(self, context, external_whitelist_id,
                                  external_whitelist):
        """the abstractmethod to update external whitelist"""
        pass

    @abc.abstractmethod
    def delete_external_whitelist(self, context, external_whitelist_id):
        """the abstractmethod to delete external whitelist"""
        pass

    @abc.abstractmethod
    def get_external_whitelist(self, context, external_whitelist_id,
                               fields=None):
        """the abstractmethod to get external whitelist"""
        pass

    @abc.abstractmethod
    def add_routes(self, context, external_whitelist_id, external_whitelist):
        """the abstractmethod to add routes for external whitelist"""
        pass

    @abc.abstractmethod
    def remove_routes(self, context, external_whitelist_id, external_whitelist):
        """the abstractmethod to remove routes for external whitelist"""
        pass

    @abc.abstractmethod
    def add_subnets(self, context, external_whitelist_id, external_whitelist):
        """the abstractmethod to add subnets for external whitelist"""
        pass

    @abc.abstractmethod
    def remove_subnets(self, context, external_whitelist_id,
                       external_whitelist):
        """the abstractmethod to remove subnets for external whitelist"""
        pass

    @abc.abstractmethod
    def get_external_whitelists(self, context, filters=None, fields=None):
        """the abstractmethod to get external whitelists"""
        pass
