#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2020 Huawei Technologies Co. Ltd. All rights reserved.
"""flow mirror extension"""

import abc
import six
import netaddr
from neutron import manager
from neutron.api import extensions
from neutron.api.v2 import base
from neutron.api.v2 import resource as wsgi_resource
from oslo_config import cfg

try:
    from neutron_lib.plugins import directory
except ImportError:
    pass

try:
    from neutron_lib.api import validators
except ImportError:
    validators = None
try:
    from neutron.api.v2 import attributes
except ImportError:
    attributes = None

try:
    from neutron.api.v2.base import FAULT_MAP as neutronfaultmap
except ImportError:
    from neutron_lib.api.faults import FAULT_MAP as neutronfaultmap

try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor

try:
    from neutron.common.exceptions import NotFound, InvalidInput
except ImportError:
    from neutron_lib.exceptions import NotFound, InvalidInput

try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase
import networking_huawei.drivers.ac.common.constants as ac_constants
from networking_huawei.drivers.ac.extensions import flowmirror \
    as flow_mirror_extension
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import validate as ac_validator
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei._i18n import _LE

LOG = logging.getLogger(__name__)


class TapServiceNotFound(NotFound):
    """Tap Service not found class"""
    msg = "TapService %(tap_service_id)s could not be found."


class TapFlowNotFound(NotFound):
    """Tap Service not found class"""
    msg = "TapFlow %(tap_flow_id)s could not be found."


class InvalidRemoteIP(InvalidInput):
    """remote ip is invalid"""
    message = " create Tap Service error, remote ip is invalid."


class InvalidTapFlowSourcePortException(InvalidInput):
    """ Invalid tap_flow source port"""
    message = "Invalid tap_flow source port"


class InvalidTapServiceRemoteIpException(InvalidInput):
    """ Invalid tap_service remote ip"""
    message = "Invalid tap_service remote ip"


class InvalidTapServiceIdAndTapServiceNameException(InvalidInput):
    """ Invalid tap service id"""
    message = "tap_flow_tap_service_id and " \
              "tap_flow_sdn_tapservice_name both or No one"


class InvalidTapFlowDestinationIpPrefixException(InvalidInput):
    """ Invalid tap flow destination ip prefix"""
    message = "Invalid tap flow destination ip prefix"


class FlowMirrorExceptionFromController(Exception):
    """ FlowMirror Exception from Controller """
    message = "FlowMirror Exception from Controller"


extensions.append_api_extensions_path(flow_mirror_extension.__path__)

FLOWMIRROR = 'flow-mirror'
TAPSERVCIES = 'tap_services'
TAPFLOWS = 'tap_flows'
OPS_VERSION = ncu.get_ops_version()


def transform_destination_ip_prefix(destination_ip_prefix):
    """ transform destination ip prefix """
    try:
        return str(netaddr.IPNetwork(destination_ip_prefix).cidr)
    except netaddr.core.AddrFormatError:
        LOG.error(_LE('[AC] destination ip prefix : %s, is not correct.'),
                  destination_ip_prefix)
        raise InvalidTapFlowDestinationIpPrefixException


def create_resource_flowmirror(collection, resource, plugin, params,
                               allow_bulk=False, member_actions=None,
                               parent=None, allow_pagination=False,
                               allow_sorting=False):
    """create resource flowmirror"""
    controller = base.Controller(plugin, collection, resource, params,
                                 allow_bulk, member_actions=member_actions,
                                 parent=parent,
                                 allow_pagination=allow_pagination,
                                 allow_sorting=allow_sorting)

    return wsgi_resource.Resource(controller, neutronfaultmap, None, None,
                                  dict(create=201, delete=200))


def validate_tap_flow_info(tap_flow_source_port,
                           tap_flow_tap_service_id,
                           tap_flow_sdn_tapservice_name,
                           tap_flow_info):
    """ validate tap flow info """
    tap_flow_destination_v4ip_prefix = \
        tap_flow_info.get('destination_v4ip_prefix')
    tap_flow_destination_v6ip_prefix = \
        tap_flow_info.get('destination_v6ip_prefix')
    if not tap_flow_source_port:
        LOG.error(_LE('[AC] tap_flow_source_port is empty.'))
        raise InvalidTapFlowSourcePortException
    if tap_flow_tap_service_id and tap_flow_sdn_tapservice_name:
        LOG.error(_LE('[AC] tap_flow_tap_service_id and '
                      'tap_flow_sdn_tapservice_name choose one.'))
        raise InvalidTapServiceIdAndTapServiceNameException
    if not tap_flow_tap_service_id and not tap_flow_sdn_tapservice_name:
        LOG.error(_LE('[AC] tap_flow_tap_service_id and '
                      'tap_flow_sdn_tapservice_name choose one.'))
        raise InvalidTapServiceIdAndTapServiceNameException
    if tap_flow_destination_v4ip_prefix:
        tap_flow_destination_v4ip_prefix = \
            transform_destination_ip_prefix(tap_flow_destination_v4ip_prefix)
    if tap_flow_destination_v6ip_prefix:
        tap_flow_destination_v6ip_prefix = \
            transform_destination_ip_prefix(tap_flow_destination_v6ip_prefix)
    return tap_flow_destination_v4ip_prefix, tap_flow_destination_v6ip_prefix


def validate_tap_service_info(tap_service_info):
    """ validate tap service info """
    if tap_service_info.get('remote_v4ip') and \
            not netaddr.valid_ipv4(tap_service_info.get('remote_v4ip')):
        LOG.error(_LE('[AC] Remote ip address : %s, is not ipv4.'),
                  tap_service_info.get('remote_v4ip'))
        raise InvalidTapServiceRemoteIpException

    if tap_service_info.get('remote_v6ip') and \
            not netaddr.valid_ipv6(tap_service_info.get('remote_v6ip')):
        LOG.error(_LE('[AC] Remote ip address : %s, is not ipv6.'),
                  tap_service_info.get('remote_v6ip'))
        raise InvalidTapServiceRemoteIpException

    if not tap_service_info.get('remote_v4ip') and \
            not tap_service_info.get('remote_v6ip'):
        raise InvalidRemoteIP()


if validators and hasattr(validators, 'add_validator') and \
        'type:ip_address_list_or_nodata' not in validators.validators:
    validators.add_validator('ip_address_list_or_nodata',
                             ac_validator.validate_source_ips)

if attributes and hasattr(attributes, 'validators') and \
        'type:ip_address_list_or_nodata' not in attributes.validators:
    attributes.validators['type:ip_address_list_or_nodata'] = \
        ac_validator.validate_source_ips

# Attribute Map
RESOURCE_ATTRIBUTE_MAP = {
    TAPSERVCIES: {
        'id': {
            'allow_post': False, 'allow_put': True, 'is_visible': True,
            'primary_key': True, 'validate': {'type:uuid_or_none': None}},
        'name': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 255}, 'is_visible': True},
        'description': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 255}, 'is_visible': True},
        'tenant_id': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:uuid': 48}, 'is_visible': True},
        'port_id': {
            'allow_post': True, 'allow_put': True, 'is_visible': True,
            'validate': {'type:string_or_none': 48}, 'default': ''},
        'remote_v4ip': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 32}, 'is_visible': True,
            'default': ''},
        'remote_v6ip': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': '',
            'validate': {'type:string': 48}},
    },
    TAPFLOWS: {
        'id': {
            'allow_post': False, 'allow_put': True,
            'validate': {'type:uuid_or_none': None},
            'is_visible': True, 'primary_key': True},
        'name': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 255},
            'is_visible': True},
        'description': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 255},
            'is_visible': True},
        'tenant_id': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:uuid': 48},
            'is_visible': True},
        'source_port': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True,
            'validate': {'type:uuid': 48}},
        'destination_v4ip_prefix': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': 32},
            'is_visible': True,
            'default': ''},
        'destination_v6ip_prefix': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': '',
            'validate': {'type:string': 48}},
        'tap_service_id': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': None,
            'validate': {'type:uuid_or_none': None}},
        'direction': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True,
            'validate': {'type:string': 5}},
        'sdn_tapservice_name': {
            'allow_post': True, 'allow_put': True,
            'is_visible': True, 'default': '',
            'validate': {'type:string': 255}},
    }
}


class Flow_mirror(ExtensionDescriptor):
    """Consistency report generation"""

    @classmethod
    def get_name(cls):
        """ get name """
        # Name for this extension
        return "Flow_mirror"

    @classmethod
    def get_alias(cls):
        """ get alias """
        # core_plugin class to load the extension
        return 'huawei-flow-mirror'

    @classmethod
    def get_description(cls):
        """ get description """
        # A small description about this extension
        return "Support create a mechanism to forward traffic"

    @classmethod
    def get_namespace(cls):
        """get namespace"""
        # The XML namespace for this extension
        return ""

    @classmethod
    def get_updated(cls):
        """get updated"""
        # Specify when was this extension last updated
        return "2019-08-19T21:10:50-00:00"

    @classmethod
    def get_resources(cls):
        """Returns Ext Resources."""
        if OPS_VERSION in ac_constants.OPS_VERSION_O_PQRTW_6_21:
            allow_pagination = True
            allow_sorting = True
            plugin = directory.get_plugin(FLOWMIRROR)
        else:
            plugin = \
                manager.NeutronManager.get_service_plugins()[FLOWMIRROR]
            allow_pagination = cfg.CONF.allow_pagination
            allow_sorting = cfg.CONF.allow_sorting
        resources = []
        for resource_name in ["tap_service", "tap_flow"]:
            collection_name = resource_name + "s"
            params = RESOURCE_ATTRIBUTE_MAP.get(resource_name + 's', dict())
            member_actions = {}
            controller = create_resource_flowmirror(
                collection_name, resource_name, plugin, params, allow_bulk=True,
                member_actions=member_actions, allow_sorting=allow_sorting,
                allow_pagination=allow_pagination)
            resources.append(extensions.ResourceExtension(
                collection_name, controller, member_actions=member_actions,
                attr_map=params))

        return resources

    @classmethod
    def get_plugin_interface(cls):
        """get plugin interface"""
        return FlowMirrorBase

    def update_attributes_map(self, attr):
        """ update attributes map """
        super(Flow_mirror, self).update_attributes_map(
            attr, extension_attrs_map=RESOURCE_ATTRIBUTE_MAP)

    @classmethod
    def get_extended_resources(cls, version):
        """ get extended resource """
        if version == "2.0":
            return RESOURCE_ATTRIBUTE_MAP
        return {}


@six.add_metaclass(abc.ABCMeta)
class FlowMirrorBase(ServicePluginBase):
    """FlowMirrorBase class"""

    def get_plugin_type(self):
        """ get plugin type """
        pass

    def get_plugin_description(self):
        """ get plugin description """
        pass

    def get_plugin_name(self):
        """ get plugin name """
        pass

    @abc.abstractmethod
    def create_tap_service(self, context, tap_service):
        """the abstractmethod to create a Tap Service"""
        pass

    @abc.abstractmethod
    def create_tap_flow(self, context, tap_flow):
        """the abstractmethod to create a Tap Flow"""
        pass

    @abc.abstractmethod
    def delete_tap_service(self, context, tap_service_id):
        """the abstractmethod to delete a Tap Service"""
        pass

    @abc.abstractmethod
    def delete_tap_flow(self, context, tap_flow_id):
        """the abstractmethod to delete a Tap Flow"""
        pass

    @abc.abstractmethod
    def get_tap_service(self, context, tap_service_id, fields=None):
        """the abstractmethod to list a Tap Service"""
        pass

    @abc.abstractmethod
    def get_tap_flow(self, context, tap_flow_id, fields=None):
        """the abstractmethod to list a Tap Flow"""
        pass

    @abc.abstractmethod
    def get_tap_services(self, context, filters=None, fields=None):
        """the abstractmethod to list all Tap Services"""
        pass

    @abc.abstractmethod
    def get_tap_flows(self, context, filters=None, fields=None):
        """the abstractmethod to list all Tap Flows"""
        pass

    @abc.abstractmethod
    def update_tap_service(self, context, tap_service_id, tap_service):
        """update Tap Service,only for dry run"""
        pass

    @abc.abstractmethod
    def update_tap_flow(self, context, tap_flow_id, tap_flow):
        """update Tap Flow,only for dry run"""
        pass
