#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""portbinding extension"""

try:
    from neutron.db import common_db_mixin
except ImportError:
    from networking_huawei.drivers.ac.common import common_db_mixin
from neutron.api import extensions
try:
    from neutron.api.v2.attributes import ATTR_NOT_SPECIFIED
except ImportError:
    from neutron_lib.constants import ATTR_NOT_SPECIFIED
try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor
from neutron.plugins.ml2 import models
from networking_huawei._i18n import _LE, _LI
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_ml2_api as api
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.extensions import portbinding
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging

extensions.append_api_extensions_path(portbinding.__path__)
LOG = logging.getLogger(__name__)

BINDING_LEVEL = 'binding_level'
BINDING_LEVELS = '%ss' % BINDING_LEVEL
DESCRIPTION = 'extend binding levels for port'

EXTENDED_ATTRIBUTES_2_0 = {
    'ports': {
        BINDING_LEVELS: {'allow_post': False, 'allow_put': False,
                         'default': ATTR_NOT_SPECIFIED,
                         'enforce_policy': True,
                         'is_visible': True},
    }
}


class Portbinding(ExtensionDescriptor):
    """Portbinding extension"""
    @classmethod
    def get_name(cls):
        """get name"""
        return BINDING_LEVEL

    @classmethod
    def get_alias(cls):
        """get alias"""
        return BINDING_LEVEL

    @classmethod
    def get_description(cls):
        """get description"""
        return DESCRIPTION

    @classmethod
    def get_namespace(cls):
        """get namespace"""
        return ''

    @classmethod
    def get_updated(cls):
        """get updated"""
        return '2018-02-02T12:00:00-00:00'

    def get_extended_resources(self, version):
        """get extended resources"""
        if version == "2.0":
            return EXTENDED_ATTRIBUTES_2_0
        return {}


class PortBindingExtensionDriver(api.ExtensionDriver,
                                 common_db_mixin.CommonDbMixin):
    """PortBinding Extension Driver"""
    _supported_extension_alias = BINDING_LEVEL

    def initialize(self):
        """initialize"""
        self.ops_version = ncu.get_ops_version()
        LOG.info(_LI('[AC][PortBindingExtensionDriver] initialization '
                     'complete'))

    @property
    def extension_alias(self):
        """extension alias"""
        return self._supported_extension_alias

    def extend_port_dict(self, session, db_data, result):
        """extend port dict"""
        binding_levels = []
        with session.begin(subtransactions=True):
            try:
                port_binding_query = session.query(
                    models.PortBindingLevel, ncu.NetworkSegment)
                port_binding_query = port_binding_query.join(ncu.NetworkSegment)
                port_bindings = port_binding_query.filter(
                    models.PortBindingLevel.port_id == db_data['id']
                ).all()
                for (binding, segment) in port_bindings:
                    binding_levels.append({
                        'level': binding.level,
                        'driver': binding.driver,
                        'network_type': segment.network_type,
                        'physical_network': segment.physical_network or '',
                        'segmentation_id': segment.segmentation_id,
                    })
            except Exception as ex:
                LOG.error(_LE('[AC][PortBindingExtensionDriver] extend port '
                              'dict failed: %s'), ex)
        result['binding_levels'] = binding_levels
