#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""snat"""

import abc
import six
from neutron.api import extensions
from neutron.api.v2 import base
try:
    from neutron_lib.exceptions import InvalidInput
except ImportError:
    from neutron.common.exceptions import InvalidInput
try:
    from neutron_lib.api.validators import validate_dict_or_nodata
except ImportError:
    from neutron.api.v2.attributes import _validate_dict_or_nodata as \
        validate_dict_or_nodata
try:
    from neutron_lib.api import validators
except ImportError:
    validators = None
try:
    from neutron.api.v2 import attributes
except ImportError:
    attributes = None
try:
    from neutron.services.service_base import ServicePluginBase
except ImportError:
    from neutron_lib.services.base import ServicePluginBase
try:
    from neutron.api.extensions import ExtensionDescriptor
except ImportError:
    from neutron_lib.api.extensions import ExtensionDescriptor
try:
    from neutron.common.exceptions import NeutronException
except ImportError:
    from neutron_lib.exceptions import NeutronException

from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.extensions import snat as extension


SNAT = 'snat'
SNATS = '%ss' % SNAT
NAME_MAX_LEN = 255
TENANT_ID_MAX_LEN = 255

extensions.append_api_extensions_path(extension.__path__)


class ResourceNotFound(NeutronException):
    """ResourceNotFound"""
    message = "%(rs)s %(id)s could not be found."


class TenantIdConflict(NeutronException):
    """TenantIdConflict"""
    message = "Tenant id %(tid)s is different from router %(rid)s."


class SnatIpAddressNotSpecified(NeutronException):
    """SnatIpAddressNotSpecified"""
    message = "SNAT IP address is not specified in NAT44."


class SnatIpAddressSpecified(NeutronException):
    """SnatIpAddressSpecified"""
    message = "SNAT IP address can not be specified in NAT64."


class SnatIpPoolNotSpecified(NeutronException):
    """SnatIpPoolNotSpecified"""
    message = "SNAT IP pool is not specified in NAT64."


class SnatIpPoolSpecified(NeutronException):
    """SnatIpPoolSpecified"""
    message = "SNAT IP pool can not be specified in NAT44."


class SnatIpPoolOverLength(NeutronException):
    """SnatIpPoolOverLength"""
    message = "SNAT IP pool length is over 8."


class SnatIpPoolParamNotSpecified(NeutronException):
    """SnatIpPoolParamNotSpecified"""
    message = "SNAT IP pool %(param)s is not specified."


class SnatIpPoolInvalid(NeutronException):
    """SnatIpPoolInvalid"""
    message = "SNAT IP pool %(ip_pool)s is invalid."


class SnatIpPoolConflict(NeutronException):
    """SnatIpPoolConflict"""
    message = "SNAT IP pool %(ip_pool)s conflict."


class SnatOriginalCidrsInvalid(NeutronException):
    """SnatOriginalCidrsInvalid"""
    message = "SNAT original cidrs %(cidr)s is invalid."


class SnatOriginalCidrsConflict(NeutronException):
    """SnatOriginalCidrsConflict"""
    message = "SNAT original cidrs %(cidr)s conflict."


def _validate_list_of_dicts(data, valid_values=None):
    if not isinstance(data, list):
        msg = "'%s' is not a list" % data
        raise InvalidInput(error_message=msg)

    for item in data:
        msg = validate_dict_or_nodata(item, valid_values)
        if msg:
            raise InvalidInput(error_message=msg)


if validators and hasattr(validators, 'add_validator') and \
        'type:list_of_dicts' not in validators.validators:
    validators.add_validator('list_of_dicts', _validate_list_of_dicts)
if validators and hasattr(validators, 'add_validator') and \
        'type:list_of_strings' not in validators.validators:
    validators.add_validator('list_of_strings', validate.validate_list_of_strings)

if attributes and hasattr(attributes, 'validators') and \
        'type:list_of_dicts' not in attributes.validators:
    attributes.validators['type:list_of_dicts'] = _validate_list_of_dicts
if attributes and hasattr(attributes, 'validators') and \
        'type:list_of_strings' not in attributes.validators:
    attributes.validators['type:list_of_strings'] = validate.validate_list_of_strings


RESOURCE_ATTRIBUTE_MAP = {
    SNATS: {
        'id': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:uuid_or_none': None},
            'is_visible': True,
            'primary_key': True,
            'default': None
        },
        'name': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:string': NAME_MAX_LEN},
            'default': '',
            'is_visible': True,
        },
        'tenant_id': {
            'allow_post': True, 'allow_put': False,
            'required_by_policy': False,
            'validate': {'type:string': TENANT_ID_MAX_LEN},
            'is_visible': True,
        },
        'router_id': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:uuid': None},
            'is_visible': True,
        },
        'snat_network_id': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:uuid': None},
            'is_visible': True,
        },
        'snat_ip_address': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:ip_address_or_none': None},
            'default': None,
            'is_visible': True,
        },
        'snat_ip_pool': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:list_of_dicts': None},
            'default': [],
            'is_visible': True,
        },
        'original_cidrs': {
            'allow_post': True, 'allow_put': True,
            'validate': {'type:list_of_strings': None},
            'is_visible': True,
        },
        'type': {
            'allow_post': True, 'allow_put': False,
            'validate': {'type:range': [0, 1]},
            'default': 0,
            'is_visible': True,
        },
    }
}


class Snat(ExtensionDescriptor):
    """Snat extension description"""

    @classmethod
    def get_name(cls):
        """get name"""
        return SNAT

    @classmethod
    def get_alias(cls):
        """get_alias"""
        return SNAT

    @classmethod
    def get_description(cls):
        """get_description"""
        return SNAT

    @classmethod
    def get_namespace(cls):
        """get_namespace"""
        return ''

    @classmethod
    def get_updated(cls):
        """get_updated"""
        return '2018-02-22T12:00:00-00:00'

    @classmethod
    def get_resources(cls):
        """get_resources"""

        plugin = ncu.get_service_plugin()[SNAT]
        resources = []
        for resource_name in [SNAT]:
            collection_name = '{}s'.format(resource_name)
            params = RESOURCE_ATTRIBUTE_MAP.get(collection_name, dict())
            member_actions = {}
            controller = base.create_resource(
                collection_name, resource_name, plugin, params,
                allow_bulk=True,
                member_actions=member_actions,
                allow_pagination=True,
                allow_sorting=True)
            resource = extensions.ResourceExtension(
                collection_name, controller,
                member_actions=member_actions,
                attr_map=params)
            resources.append(resource)
        return resources


@six.add_metaclass(abc.ABCMeta)
class SnatBase(ServicePluginBase):
    """SnatBase"""

    def get_plugin_name(self):
        """get_plugin_name"""
        pass

    def get_plugin_type(self):
        """get_plugin_type"""
        pass

    def get_plugin_description(self):
        """get_plugin_description"""
        pass

    @abc.abstractmethod
    def create_snat(self, context, snat):
        """create_snat"""
        pass

    @abc.abstractmethod
    def update_snat(self, context, snat_id, snat):
        """update_snat"""
        pass

    @abc.abstractmethod
    def delete_snat(self, context, snat_id):
        """delete_snat"""
        pass

    @abc.abstractmethod
    def get_snat(self, context, snat_id, fields=None):
        """get_snat"""
        pass

    @abc.abstractmethod
    def get_snats(self, context, filters=None, fields=None):
        """get_snats"""
        pass
