#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""bgp route model"""

from oslo_log import log as logging

from networking_huawei._i18n import _LE
from networking_huawei.drivers.ac.common import constants as ac_const

LOG = logging.getLogger(__name__)


class ACBgpRouteModel(object):
    """ACBgpRouteModel class """

    @staticmethod
    def ac_model_format(bgp_route):
        """Convert data to AC model"""

        try:
            bgp_route_info = {
                'uuid': bgp_route['id'],
                'name': bgp_route['name'],
                'tenant-id': bgp_route['tenant_id'],
                'router-id': bgp_route['router_id'],
                'peer-ip-address': bgp_route['peer_ip_address'],
                'peer-as-number': bgp_route['peer_as_number']
            }
            if bgp_route.get('description'):
                bgp_route_info['description'] = bgp_route['description']

            if bgp_route.get('router_interface_id'):
                bgp_route_info['router-interface-id'] = \
                    bgp_route['router_interface_id']
            if bgp_route.get('ip_version'):
                bgp_route_info['ip-version'] = bgp_route['ip_version']
                bgp_route_info['ip-version'] = \
                    ac_const.NEUTRON_CONSTANTS +\
                    'ip-version-v' + str(bgp_route['ip_version'])
            if 'bfd' in bgp_route and bgp_route['bfd'] is not None:
                bgp_route_info['bfd'] = bgp_route['bfd']
            if bgp_route.get('session_attribute'):
                bgp_route_info = ACBgpRouteModel._process_para_session_attribute(
                    bgp_route, bgp_route_info)
            if bgp_route.get('source_ips'):
                bgp_route_info['source-ips'] = bgp_route['source_ips']
            if 'peer_type' in bgp_route and bgp_route['peer_type'] is not None:
                bgp_route_info['peer-type'] = bgp_route['peer_type']
        except Exception as value:
            LOG.error(_LE("[AC]Get AC model data error: %s."),
                      value)
            raise value
        return bgp_route_info

    @staticmethod
    def _process_para_session_attribute(bgp_route, bgp_route_info):
        bgp_route_dict = bgp_route_info
        session_attribute = bgp_route['session_attribute']
        bgp_route_dict['session-attribute'] = {}
        if session_attribute.get('keepalive_time') or \
                session_attribute.get('keepalive_time') == 0:
            bgp_route_dict['session-attribute']['keepalive-time'] = \
                bgp_route['session_attribute']['keepalive_time']
        hold_time = session_attribute.get('hold_time')
        if hold_time or hold_time is 0:
            bgp_route_dict['session-attribute']['hold-time'] = \
                bgp_route['session_attribute']['hold_time']
        if session_attribute.get('auth'):
            auth = bgp_route['session_attribute']['auth']
            bgp_route_dict['session-attribute']['auth'] = {}
            if auth.get('auth_type'):
                bgp_route_dict['session-attribute']['auth']['auth-type'] = \
                    bgp_route['session_attribute']['auth']['auth_type']
            if auth.get('password_type'):
                bgp_route_dict['session-attribute']['auth']['password-type'] = \
                    bgp_route['session_attribute']['auth']['password_type']
            if auth.get('password_text'):
                bgp_route_dict['session-attribute']['auth']['password-text'] = \
                    bgp_route['session_attribute']['auth']['password_text']
        if 'suppress' in session_attribute \
                and session_attribute['suppress'] is not None:
            bgp_route_dict['session-attribute']['suppress'] = \
                bgp_route['session_attribute']['suppress']
        return bgp_route_dict
