#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""dnat for neutron client"""

from networking_huawei._i18n import _
from neutronclient.common import extension
from neutronclient.neutron import v2_0 as neutron_v20

ATTRIBUTES = ['tenant_id', 'floating_ip_id', 'router_id', 'port_id',
              'fixed_ip_address', 'protocol', 'floating_ip_port',
              'fixed_ip_port', 'status']


class Base(extension.NeutronClientExtension):
    """Base extension"""
    resource = 'dnat'
    resource_plural = '%ss' % resource
    object_path = '/%s' % resource_plural
    resource_path = '/%s/%%s' % resource_plural
    versions = ['2.0']
    allow_names = False


class CreateDNAT(extension.ClientExtensionCreate, Base):
    """Create a DNAT for a given floating ip."""
    shell_command = 'dnat-create'

    def add_known_arguments(self, parser):
        """add known arguments"""
        parser.add_argument(
            'floating_ip_id', metavar='FLOATINGIP_ID',
            help=_('Floating IP ID of the DNAT to create.')
        )
        parser.add_argument(
            '--port-id', dest='port_id', action='store',
            help=_('Port ID of the DNAT to create.')
        )
        parser.add_argument(
            '--fixed-ip-address', dest='fixed_ip_address', action='store',
            help=_('Fixed IP of the DNAT to create.')
        )
        parser.add_argument(
            '--protocol', dest='protocol', action='store',
            help=_('Protocol of the DNAT to create.')
        )
        parser.add_argument(
            '--floating-ip-port', dest='floating_ip_port', action='store',
            help=_('Floating IP port of the DNAT to create.')
        )
        parser.add_argument(
            '--fixed-ip-port', dest='fixed_ip_port', action='store',
            help=_('Fixed IP port of the DNAT to create.')
        )

    def args2body(self, parsed_args):
        """args to body"""
        body = {}
        neutron_client = self.get_client()
        neutron_v20.find_resource_by_id(
            neutron_client, 'floatingip', parsed_args.floating_ip_id)
        neutron_v20.find_resource_by_id(
            neutron_client, 'port', parsed_args.port_id)
        neutron_v20.update_dict(parsed_args, body, ATTRIBUTES)
        return {self.resource: body}


class DeleteDNAT(extension.ClientExtensionDelete, Base):
    """Delete a given DNAT."""
    shell_command = 'dnat-delete'


class ListDNAT(extension.ClientExtensionList, Base):
    """List DNATs that belong to a given tenant."""
    shell_command = 'dnat-list'
    list_columns = ['id', 'floating_ip_id', 'port_id', 'fixed_ip_address',
                    'protocol', 'floating_ip_port', 'fixed_ip_port']
    pagination_support = True
    sorting_support = True


class ShowDNAT(extension.ClientExtensionShow, Base):
    """Show information of a given DNAT."""
    shell_command = "dnat-show"
