#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright 2020 Huawei Technologies Co. Ltd. All rights reserved.
"""Health check"""

from __future__ import print_function
import re
import os
import copy
import logging
import subprocess
import sys

import six
import requests
import netaddr
from oslo_config import cfg
from oslo_serialization import jsonutils

try:
    from FSSecurity import crypt
except ImportError:
    import crypt
from neutron import manager

try:
    from neutron import context as neutron_context
except ImportError:
    try:
        from neutron_lib import context as neutron_context
    except ImportError:
        neutron_context = None

from networking_huawei.drivers.ac.common import config, security_util, \
    constants as ac_constants
from networking_huawei.drivers.ac.common.util import DataFilterUtil, \
    ConfigParas, check_father_process
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.client.https_adapter import HWHTTPSAdapter

LOGGER = logging.getLogger(__name__)
LOGGER.setLevel(logging.INFO)
LOG_FILE = os.path.join(ac_constants.OPS_PATH, 'health_check.log')
FILE_HANDLER = logging.FileHandler(LOG_FILE, mode='a')
FILE_HANDLER.setLevel(logging.DEBUG)
FORMATTER = logging.Formatter("%(asctime)s - %(filename)s[line:%(lineno)d]"
                              " - %(levelname)s: %(message)s")
FILE_HANDLER.setFormatter(FORMATTER)
LOGGER.propagate = False
LOGGER.addHandler(FILE_HANDLER)

try:
    cfg.CONF.register_opts(config.HUAWEI_AC_DRIVER_OPTS,
                           ac_constants.HUAWEI_CONFIG)
except Exception:
    LOGGER.error("cfg.CONF.register_opts failed.")
VXLAN_OPTS = [cfg.ListOpt('vni_ranges', default=[])]
try:
    cfg.CONF.register_opts(VXLAN_OPTS, "ml2_type_vxlan")
except Exception:
    LOGGER.error("cfg.CONF.register_opts failed.")


def manage_base_get_cmd(cmd):
    """Get cmd result"""
    pipe = os.popen(cmd)
    result = pipe.read().rstrip('\n')
    pipe.close()
    return result


def get_ac_plugin_path():
    """Get ac plugin file path"""
    neutron_path = manage_base_get_cmd(
        "python -c 'import os,neutron;"
        "print(os.path.abspath(neutron.__file__))'")
    neutron_index = neutron_path.find("/neutron")
    python_path = neutron_path[:neutron_index]
    plugin_path = os.path.join(python_path, 'networking_huawei/')
    py_version = sys.version_info
    if not os.path.exists(plugin_path):
        plugin_path = os.path.realpath(os.path.join(
            '/usr/local/lib/python%s.%s' % (py_version.major, py_version.minor),
            '/dist-packages/networking_huawei/'))
    return plugin_path


def check_ip(ip_address):
    """Check ac ip address format and exec ping operation"""
    result = [False, False]
    if netaddr.valid_ipv4(ip_address) or netaddr.valid_ipv6(ip_address):
        result[0] = True
    if netaddr.valid_ipv6(ip_address):
        cmd_ping6 = ['ping', '-6', '-c', '3', ip_address]
        if not subprocess.call(cmd_ping6):
            result[1] = True
    else:
        cmd_ping4 = ['ping', '-c', '3', ip_address]
        if not subprocess.call(cmd_ping4):
            result[1] = True
    return result


def check_req_result(req):
    """Check result from ac while get token"""
    if requests.codes.ok <= req.status_code < requests.codes.multiple_choices:
        if req.content:
            result_data = jsonutils.loads(req.content)
            return result_data
    elif req.status_code == requests.codes.unauthorized:
        return 'wrong ac_auth_username or ac_auth_password, ' \
               'please check the ac account!'
    elif req.status_code == requests.codes.internal_server_error:
        return 'Internal server error, please check AC account status!'
    return 'something wrong with Agile Controller,please check Agile ' \
           'Controller. '


def check_neutron_server(ops_version, is_tools_check):
    """Check neutron server"""
    if not is_tools_check and ops_version in [ac_constants.FSP_21_0]:
        return 'neutron-server', 'ok', ''
    server_inf = manage_base_get_cmd("ps -ef | grep '^neutron*'").split('\n')
    server_inf.extend(
        manage_base_get_cmd("ps -ef | grep 'neutron-server' | "
                            "grep '^opensta*'").split('\n'))
    server_inf = [_f for _f in server_inf if _f]
    if not server_inf:
        LOGGER.error('[HEALTH]check neutron-server:failed, '
                     'neutron-server is not running.')
        return 'neutron-server', 'failed', 'neutron-server is not running.'
    len_ppid1 = check_father_process(server_inf)
    if len(len_ppid1) == 1:
        LOGGER.info('[HEALTH]check neutron-server:ok')
        return 'neutron-server', 'ok', ''
    LOGGER.warning('[HEALTH]check neutron-server:failed, '
                   'there are more than 1 father process of '
                   'neutron-server')
    return 'neutron-server', 'failed', \
           'there are more than 1 father process of neutron-server.'


def check_thread_time(gap, std):
    """Check thread time"""
    if len(gap.split('.')[0]) > 4:
        return False
    if gap and float(gap) > std:
        return False
    return True


def check_normal_sync():
    """Check normal sync"""
    neutron_log = ac_constants.FSP_LOG if os.path.exists(
        ac_constants.FSP_LOG) else ac_constants.OPS_LOG
    sync_time = manage_base_get_cmd(
        "grep 'normal sync heart beat time stamp' %s | tail -1" %
        neutron_log).split('stamp: ')[-1]
    monitor_time = manage_base_get_cmd(
        "grep 'normal sync monitor time stamp' %s | tail -1" %
        neutron_log).split('stamp:')[-1]
    now_time = manage_base_get_cmd("date +'%s'")
    sync_gap = manage_base_get_cmd(
        "echo '%s - %s'| bc -l" % (now_time, sync_time))
    monitor_gap = manage_base_get_cmd("echo '%s - %s'| bc -l" %
                                      (now_time, monitor_time))
    if not check_thread_time(sync_gap, 150):
        LOGGER.error('[HEALTH]check normal_sync thread:failed, '
                     'normal_sync thread is dead')
        return 'normal_sync thread', 'failed', 'normal_sync thread is dead.'
    if not check_thread_time(monitor_gap, 450):
        LOGGER.error('[HEALTH]check normal_sync thread:failed, '
                     'monitor thread is dead.')
        return 'normal_sync thread', 'failed', 'monitor thread is dead.'
    LOGGER.info('[HEALTH]check normal_sync thread:ok')
    return 'normal_sync thread', 'ok', ''


def check_plugin():
    """Check plugin whether is installed"""
    plugin_path = get_ac_plugin_path()
    if os.path.exists(plugin_path):
        LOGGER.info("[HEALTH]check whether the plug-in "
                    "is installed and deployed plugin: ok")
        return 'plugin', 'ok'
    LOGGER.error("[HEALTH]check whether the plug-in is installed "
                 "and deployed plugin: failed, plug_in is not "
                 "installed and deployed.")
    return 'plugin', 'failed'


def get_cert_file():
    """Get cert file to get token from ac plugin file"""
    plugin_path = get_ac_plugin_path()
    cert_path = os.path.join(plugin_path, 'drivers/ac/client/')
    cert_file = os.path.join(cert_path, 'ssl_cacert.pem')
    if os.path.exists(cert_file):
        LOGGER.info('[HEALTH]Get cert file :%s', cert_file)
        return cert_file
    LOGGER.error('[HEALTH] %s does not exist', cert_file)
    return ""


def get_plugin_version():
    """Get plugin version from ac plugin file"""
    plugin_path = get_ac_plugin_path()
    version_path = os.path.join(plugin_path, 'version.txt')

    try:
        with open(version_path, 'r') as r_file:
            version_info = r_file.read()
    except Exception as ex:
        return 'plugin_version', 'failed', str(ex)
    if version_info:
        version = ((version_info.split('\n')[0]).strip()).split(' ')[2]
        return 'plugin_version', 'configured', version
    return 'plugin_version', 'not configured', ''


def get_token(host, username, password):
    """Get token from ac"""
    headers = {"Content-type": "application/json",
               "Accept": "application/json", "Accept-Language": "en-US"}
    auth_url = "%s%s:%s%s" % (ac_constants.HTTPS_HEADER, host,
                              ac_constants.rest_server_port,
                              ac_constants.AUTH_BASE_URL)
    auth = {"userName": username, "password": password}
    auth_data = jsonutils.dumps(auth)
    try:
        cert_file = get_cert_file() or True
        requests.packages.urllib3.disable_warnings()
        with requests.session() as session:
            adapter = HWHTTPSAdapter(cert_file=None, key_file=None,
                                     password=None,
                                     pool_connections=100, pool_maxsize=100)
            session.mount("https://", adapter)
            req = session.request('POST',
                                  url=auth_url,
                                  headers=headers,
                                  data=auth_data,
                                  verify=cert_file,
                                  timeout=20)
            LOGGER.info('[HEALTH]Send request get return :%s', req)
            return check_req_result(req)
    except Exception as ex:
        LOGGER.error("[HEALTH]Send request failed for :%s", ex)
        return 'sending checking request failed.'


def display_title(check_num, check_title):
    """Display title"""
    print('[' + check_num + ']' + check_title)


def display_title1():
    """Display title1"""
    display_dividing()
    print("   |%-30s|%-40s|" % ('item', 'value'))
    display_dividing()


def display_dividing():
    """Display dividing"""
    print('   +' + '-' * 30 + '+' + '-' * 40 + '+')


def display_plugin_install(res):
    """Display plugin install status"""
    if res[1] == 'ok':
        print('   \033[1;32mOk\033[0m')
    else:
        print('   \033[1;31mFail\033[0m')
        print('       \033[1;31mError:plugin has not installed.\033[0m')


def network_black_and_white_list(result):
    """Check network black and white list"""
    if result[1].startswith('       \033[1;31mError:'):
        print('%s\033[1;31m%20s\033[0m' % ('       ' + result[0], 'Fail'))
        print('       ' + '-' * 60)
        print(result[1])
    elif result[1] == 'failed':
        print('%s\033[1;31m%20s\033[0m' % ('       ' + result[0], 'Fail'))
        print('       ' + '-' * 60)
        print('       \033[1;31m%s\033[0m' % ('Error:' + result[2]))
    elif result[1] == 'not configured.':
        print('%s\033[1;31m%20s\033[0m' % ('       ' + result[0], 'Fail'))
        print('       ' + '-' * 60)
        print('       \033[1;33m%s\033[0m' % ('Warning:' + result[1]))
    elif result[1] == 'ok':
        print('%s\033[1;32m%20s\033[0m' % ('       ' + result[0], 'Ok'))
        print('       ' + '-' * 60)
        print('       ' + result[2])


def check_configure_num(configname, path):
    """Check config num"""
    with open(path) as config_file:
        num = 0
        while True:
            params = config_file.readline()
            if configname in params:
                num = num + 1
            if not params:
                break
        if num > 1:
            LOGGER.error('[HEALTH]check black and white list: failed, '
                         'there are more than 1 %s', configname)
            return configname, '       \033[1;31mError:there are more ' \
                               'than one %s\033[0m' % configname
        elif num == 0:
            LOGGER.error('[HEALTH]check black and white list: failed, '
                         'there is no %s', configname)
            return configname, '       \033[1;31mError:there is' \
                               ' no %s\033[0m' % configname
        return num


def _process_configname(value, configname):
    """process config name"""
    name = value.split(',')
    for item in name:
        item1 = re.compile(r'^[a-zA-Z0-9][\w]+[a-zA-Z0-9]$',
                           re.S)
        item2 = re.compile(r'([_])\1+')
        if item == '' or re.findall(item1, item) == [] or \
                re.findall(item2, item) != []:
            LOGGER.error('[HEALTH]check %s : failed, %s'
                         ' contains illegal characters '
                         'in value.', configname, configname)
            msg = '%s contains illegal characters in value.' % configname
            return configname, 'failed', msg
    LOGGER.info('[HEALTH]check %s: ok', configname)
    return configname, 'ok', value


def check_network_black_and_white_list(configname, path):
    """Check network black and white list from config file"""
    res = check_configure_num(configname, path)
    if res != 1:
        return res
    with open(path, 'r') as config_file:
        for values in config_file.readlines():
            if not values.__contains__(configname):
                continue
            result1 = re.sub(r'#', '', values)
            result2 = re.sub(r'\n', '', result1)
            r2_list = result2.split(' =')[1]
            value = r2_list.lstrip(' ')
            if value != '':
                _process_configname(value, configname)
        LOGGER.warning('[HEALTH]check %s: failed, %s '
                       'not configured', configname, configname)
        return configname, 'not configured.', configname


def ext_net_not_create(ext_net_name, ext_gw_name):
    """Get result while external network net created in neutron or ac"""
    results = ''
    if not ext_net_name and not ext_gw_name:
        LOGGER.warning('[HEALTH]No external net on the cloud platform, '
                       'and no external gateway created on ac.')
        results = 'Warning:there is no external net on the cloud ' \
                  'platform, and no external gateway created on ac.'
    elif not ext_net_name and ext_gw_name:
        LOGGER.warning('[HEALTH]No external net on the cloud platform.')
        results = 'Warning:there is no external net on the cloud platform.'
    elif ext_net_name and not ext_gw_name:
        LOGGER.warning('[HEALTH]No external gateway created on ac.')
        results = 'Warning:there is no external gateway created on ac.'
    return results


def check_ext_net(ext_net_name, ext_gw_name):
    """Check external network items"""
    results = ''
    ext1 = set()
    if type(ext_gw_name).__name__ != 'list' and \
            ext_gw_name.startswith('Error'):
        results = ext_gw_name
        return results
    if type(ext_net_name).__name__ != 'list' and \
            ext_net_name.startswith('Error'):
        results = ext_net_name
        return results
    if not ext_net_name or not ext_gw_name:
        results = ext_net_not_create(ext_net_name, ext_gw_name)
        return results
    for gw_name in ext_gw_name:
        for net_name in ext_net_name:
            if net_name.startswith(gw_name):
                ext1.add(net_name)
        ext2 = set(ext_net_name)
        result = ext2 - ext1
        results = ','.join(result)
    return results


def display_ext_net(results):
    """Display external network items"""
    if results == '':
        LOGGER.info('[HEALTH]check ext_gw and ext_net: ok')
        print("   \033[1;32mOk\033[0m")
    else:
        LOGGER.warning('[HEALTH]check ext_gw and ext_net: failed, '
                       'no match ext_gw: %s', results)
        print("   \033[1;31mFail\033[0m")
        if results.startswith('Warning:'):
            print('       \033[1;33m' + results + '.\033[0m')
        elif results.startswith('Error:'):
            print('       \033[1;31m' + results + '\033[0m')
        else:
            print('       \033[1;33mWarning:no match external network:'
                  + results + '.\033[0m')


def check_and_display_results(result1, result2):
    """Check and display results"""
    if result1[1] == 'ok' and result2[1] == 'ok':
        print('   \033[1;32mOk\033[0m')
    else:
        print('   \033[1;31mFail\033[0m')
        if result1[1] == 'failed':
            print('       \033[1;31m' + 'Error:' + result1[2] + '\033[0m')
        if result2[1] == 'failed':
            print('       \033[1;31m' + 'Error:' + result2[2] + '\033[0m')


def display_required_filled_general(result):
    """Display required filled general"""
    if result[1] == 'configured':
        print("   |%-30s|%-40s|" % (result[0], result[2]))
    else:
        print("   |\033[1;33m%-30s\033[0m|%-40s|" % (result[0], ''))
    display_dividing()


def display_default_config(result):
    """Display default config"""
    if result[1] == 'ok':
        print("   |%-30s|%-40s|" % (result[0], result[2]))
    elif result[1] == 'failed':
        print("   |\033[1;33m%-30s\033[0m|%-40s|" % (result[0], result[2]))
    else:
        print("   |\033[1;33m%-30s\033[0m|\033[1;33m%-40s\033[0m|" %
              (result[0], 'not default'))
    display_dividing()


def display_vhost_user(result):
    """Display vhost user info"""
    if result[1] == 'ok':
        print('   \033[1;32mOk\033[0m')
    if result[1] == 'failed':
        print('   \033[1;31mFail\033[0m')
        print('       \033[1;31m%s\033[0m' %
              ('Error:the value of vhost_user is not '
               'consistent with the selected scenario.'))


def get_cpu_and_mem(cpu_info, user):
    """Get cpu and memory"""
    cpu_used = mem_used = 0
    for info in cpu_info:
        if not info.startswith(user):
            continue
        cpu_used = [i for i in info.split(' ') if i][2]
        mem_used = [i for i in info.split(' ') if i][5]
        break
    return float(cpu_used), float(mem_used)


def calc_cpu_and_mem(server_infs):
    """Calculate and check the memory usage"""
    if server_infs == ['']:
        LOGGER.error('[HEALTH]obtain cpu and mem being '
                     'used by neutron-server: failed, '
                     'neutron-server is not running')
        return 'Error:neutron-server is not running.'

    user = server_infs[0].split(' ')[0]
    cpu_total = 0
    mem_total = 0
    for server_inf in server_infs:
        res = server_inf.split(' ')
        pid = [i for i in res if i][1]
        cpu_info = manage_base_get_cmd("ps -aux | grep %s" % pid).split('\n')
        cpu_used, mem_used = get_cpu_and_mem(cpu_info, user)
        cpu_total = cpu_total + cpu_used
        mem_total = mem_total + mem_used
    LOGGER.info('[HEALTH]obtain cpu and mem being used by '
                'neutron-server: ok, cpu being used by neutron-server %s'
                ' ,mem being used by neutron-server %s',
                str(cpu_total) + '%', str(mem_total) + 'K.')
    return cpu_total, mem_total


def deal_file(source_result):
    """Deal file"""
    plugin_file = []
    plugin_dir = []
    if source_result != '':
        dir_infos = source_result.split('\n')
        dir_infos = list(set(dir_infos))
        for dir_info in dir_infos:
            dirs = [i for i in dir_info.split(' ') if i][-1]
            plugin_dir.append(dirs.lstrip('./'))
            plugin_file.append(dirs.split('/')[-1])
        return plugin_file, plugin_dir
    return source_result


def display_network_black_and_white_list():
    """Display network black and white list"""
    black_result = check_network_black_and_white_list(
        'network_black_list', ac_constants.HUAWEI_CONFIG_FILE)
    white_result = check_network_black_and_white_list(
        'network_white_list', ac_constants.HUAWEI_CONFIG_FILE)
    if black_result[1] == 'ok' and white_result[1] == 'ok':
        print('   \033[1;32mOk\033[0m')
    else:
        print('   \033[1;31mFail\033[0m')
    network_black_and_white_list(black_result)
    print('       ' + '-' * 60)
    network_black_and_white_list(white_result)
    print('       ' + '-' * 60)


def is_fsp(ops_version):
    """Check whether is FusionSphere version"""
    if re.match('^FusionSphere.+', ops_version):
        return True
    return False


def after_fsp_6_3_0(ops_version):
    """Check FusionSphere version whether is after 6.3.0"""
    if re.match('^FusionSphere.+', ops_version) and ops_version \
            not in ['FusionSphere6.1', 'FusionSphere6.3.0']:
        return True
    return False


def decrpt_keystone_passwd(keystone_passwd, ops_version):
    """Decrpt keystone password by ops version"""
    if after_fsp_6_3_0(ops_version):
        password_real = crypt.decrypt(keystone_passwd)
    else:
        password_real = six.text_type(
            security_util.decrypt_data(keystone_passwd,
                                       data_type=2))
    return password_real


class ManageBase(object):
    """Manage base class"""

    def __init__(self):
        """Init function"""
        self.huawei_config = ConfigParas(ac_constants.HUAWEI_CONFIG_FILE)
        self.ml2_config = ConfigParas(ac_constants.ML2_CONFIG_FILE)
        self.ops_version = self.huawei_config.get_config_detail(
            ac_constants.HUAWEI_CONFIG, 'OPS_version', default='Liberty')
        self.data_filter = DataFilterUtil()
        self.keystone_values = {}
        self.default_values = {}

    def check_required_filled_general(self, group_name, config_name):
        """Check required param values"""
        detail = self.huawei_config.get_config_detail(group_name, config_name)
        if detail:
            LOGGER.info("[HEALTH]check mandatory items, result:%s",
                        {'config_name': config_name, 'detail': detail})
            return config_name, 'configured', detail
        LOGGER.warning("[HEALTH]check mandatory items, result:%s",
                       {'config_name': config_name,
                        'detail': 'not configured'})
        return config_name, 'failed', 'not configured.'

    def check_optional_filled_general(self, group_name, config_name):
        """Check optional param values"""
        detail = self.huawei_config.get_config_detail(group_name, config_name)
        if detail:
            LOGGER.info("[HEALTH]check optional items, result:%s",
                        {'config_name': config_name, 'detail': detail})
            return config_name, 'configured', detail
        LOGGER.warning("[HEALTH]check optional items, result:%s",
                       {'config_name': config_name,
                        'detail': 'not configured'})
        return config_name, 'not configured', 'not configured.'

    def check_ac_ip(self):
        """Get ac ip address"""
        rpc_server_ips = re.sub(r'\s', '', self.huawei_config.get_config_detail(
            'huawei_ac_agent_config', 'rpc_server_ip'))
        if not rpc_server_ips:
            return 'rpc_server_ip', 'failed', 'not configured.'
        return 'rpc_server_ip', 'configured', rpc_server_ips

    def check_rpc_server_ip(self):
        """Get rpc server ip address and return"""
        rpc_server_ips = re.sub(r'\s', '', self.huawei_config.get_config_detail(
            'huawei_ac_agent_config', 'rpc_server_ip'))
        if not rpc_server_ips:
            LOGGER.error('[HEALTH]check rpc_server_ip, result:%s',
                         {'rpc_server_ip': rpc_server_ips,
                          'detail': 'not configured'})
            return 'rpc_server_ip', 'failed', 'not configured.'
        for ac_ip in rpc_server_ips.split(','):
            if not check_ip(ac_ip)[0]:
                LOGGER.error('[HEALTH]check rpc_server_ip, result:%s',
                             {'rpc_server_ip': ac_ip,
                              'detail': 'not correct format'})
                return 'rpc_server_ip', 'failed', \
                       rpc_server_ips + ' is not correct format.'
            if check_ip(ac_ip)[1]:
                LOGGER.info('[HEALTH]check rpc_server_ip, result:%s',
                            {'rpc_server_ip': ac_ip, 'detail': 'ok'})
                return 'rpc_server_ip', 'ok', rpc_server_ips
        LOGGER.error('[HEALTH]check rpc_server_ip, result:%s',
                     {'rpc_server_ip': rpc_server_ips,
                      'detail': 'can not ping'})
        return 'rpc_server_ip', 'failed', rpc_server_ips + ' can not ping.'

    def check_host_ip(self):
        """Get host ip address and return"""
        host_ip = self.huawei_config.get_config_detail(
            'huawei_ac_agent_config', 'host_ip')
        if not host_ip:
            LOGGER.warning('[HEALTH]check host_ip, result:%s',
                           {'host_ip': host_ip, 'detail': 'not configured'})
            return 'host_ip', 'failed', 'not configured.'
        if netaddr.valid_ipv4(host_ip) or netaddr.valid_ipv6(host_ip):
            LOGGER.info('[HEALTH]check host_ip, result:%s',
                        {'host_ip': host_ip, 'detail': 'ok'})
            return 'host_ip', 'configured', host_ip
        LOGGER.error('[HEALTH]check host_ip, result:%s',
                     {'host_ip': host_ip, 'detail': 'not in correct format'})
        return 'host_ip', 'failed', 'not in correct format.'

    def check_vni_ranges(self):
        """Check vni ranges"""
        vni_ranges = self.ml2_config.get_config_detail(
            'ml2_type_vxlan', 'vni_ranges')
        if not vni_ranges or vni_ranges == []:
            return 'host_ip', 'failed', 'not configured.'
        return 'vni_ranges', 'configured', vni_ranges

    def check_vhost_user(self, scene_num, true_num):
        """Check vhost user"""
        vhost_user = self.huawei_config.get_config_detail(
            ac_constants.HUAWEI_CONFIG, 'vhost_user', 'false')
        if (scene_num in true_num and vhost_user.lower() == 'true') or \
                (scene_num not in true_num and vhost_user.lower() == 'false'):
            LOGGER.info('[HEALTH]check vhost_user,result:%s',
                        {'vhost_user': vhost_user, 'detail': 'ok'})
            return 'vhost_user', 'ok', vhost_user
        LOGGER.warning('[HEALTH]check vhost_user,result:%s',
                       {'vhost_user': vhost_user,
                        'detail': 'not consistent with '
                                  'the selected scenario'})
        return 'vhost_user', 'failed', vhost_user

    def check_ac_passwd(self):
        """Check ac username and password in config file"""
        username = self.huawei_config.get_config_detail(
            ac_constants.HUAWEI_CONFIG, 'ac_auth_username')
        password = self.huawei_config.get_config_detail(
            ac_constants.HUAWEI_CONFIG, 'ac_auth_password')
        rpc_server_ips = re.sub(r'\s', '', self.huawei_config.get_config_detail(
            'huawei_ac_agent_config', 'rpc_server_ip')).split(',')
        if not username:
            return 'ac_auth_username', 'failed', \
                   'ac_auth_username not configured.'
        if not password:
            return 'ac_auth_password', 'failed', \
                   'ac_auth_password not configured.'
        if after_fsp_6_3_0(self.ops_version):
            password_real = crypt.decrypt(password)
        else:
            password_real = six.text_type(security_util.decrypt_data(password),
                                          errors='ignore')
        token_result = ''
        for host in rpc_server_ips:
            if host and netaddr.valid_ipv6(host):
                host = "[" + str(host) + "]"
            token_result = get_token(host, username, password_real)
            if token_result and isinstance(token_result, dict) \
                    and token_result.get('data') \
                    and token_result['data'].get('token_id'):
                LOGGER.info('[HEALTH]check ac_auth_password '
                            'and ac_auth_username: ok')
                return 'ac_auth_password', 'ok', password
        LOGGER.error('[HEALTH]check ac_auth_password '
                     'and ac_auth_username failed')
        return 'ac_auth_password', 'failed', token_result

    def get_keystone_params(self):
        """Get keystone params from config file"""
        keystone_params = {}
        try:
            keystone_params['auth_url'] = self.huawei_config.config.get(
                ac_constants.HUAWEI_CONFIG, 'auth_url')
        except six.moves.configparser.NoOptionError:
            if self.ops_version in [ac_constants.OPS_R, ac_constants.OPS_W]:
                keystone_params['auth_url'] = 'http://controller:5000/v3'
            else:
                keystone_params['auth_url'] = 'http://controller:35357'
        for param in ac_constants.KEYSTONE_DEFAULT_PARAMS:
            value = self.huawei_config.get_config_detail(
                ac_constants.HUAWEI_CONFIG, param,
                ac_constants.KEYSTONE_DEFAULT_PARAMS.get(param))
            keystone_params[param] = value
        return keystone_params

    def check_keystone_params(self):
        """Check keystone params by keystone client"""
        keystone_params = self.get_keystone_params()
        keystone_info = {}
        for param, param_value in keystone_params.items():
            if param in ['keystone_tenant', 'keystone_user', 'keystone_passwd']:
                if not param_value:
                    LOGGER.warning('[HEALTH]check param failed :%s,not configured', param)
                    return param, 'failed', str(param) + ' not configured.'
                if param == 'keystone_tenant':
                    keystone_info['tenant_name'] = param_value
                elif param == 'keystone_user':
                    keystone_info['user_name'] = param_value
                elif param == 'keystone_passwd':
                    keystone_info['password'] = decrpt_keystone_passwd(param_value, self.ops_version)
            else:
                keystone_info[param] = param_value
        try:
            if keystone_info.get('auth_url').endswith('v3'):
                ACKeyStoneIf.get_keystone_v3(keystone_info)
            else:
                ACKeyStoneIf.get_keystone_v2(keystone_info)
        except Exception as ex:
            LOGGER.error('[HEALTH]failed to get the keystone authentication for: %s', ex)
            return 'keystone', 'failed', str(ex)

        LOGGER.info('[HEALTH]check keystone_password:ok')
        return 'keystone', 'ok', 'check keystone params ok.'

    def display_keystone_info(self):
        """Display keystone info and status"""
        check_keystone = self.check_keystone_params()
        if check_keystone[1] == 'ok':
            print('   \033[1;32mOk\033[0m')
        else:
            print('   \033[1;31mFail\033[0m')
            print('       \033[1;31m%s\033[0m\033[1;31m%s\033[0m' % (
                'Error:', check_keystone[2]))

    def check_default_config(self, param):
        """Check default config from plugin config file"""
        param_value = ac_constants.CHECK_DEFAULT_VALUE_PARAMS.get(param)
        default = param_value.get('default')
        config_value = self.huawei_config.get_config_detail(
            ac_constants.HUAWEI_CONFIG, param, default)

        if param_value.get('values') == ['true', 'false']:
            config_value = str(config_value).lower()
        if config_value in param_value.get('values'):
            LOGGER.info("[HEALTH]default items, result:%s",
                        {'config_name': param, 'detail': config_value})
            return param, 'ok', config_value
        LOGGER.error("[HEALTH]default items, result:%s",
                     {'config_name': param, 'detail': 'value is incorrect'})
        error_value = '' + str(config_value) + '(The value is incorrect)'
        return param, 'failed', error_value

    def check_token(self):
        """Check token from ac request"""
        username = self.huawei_config.get_config_detail(
            ac_constants.HUAWEI_CONFIG, 'ac_auth_username')
        password = self.huawei_config.get_config_detail(
            ac_constants.HUAWEI_CONFIG, 'ac_auth_password')
        rpc_server_ips = re.sub(r'\s', '', self.huawei_config.get_config_detail(
            'huawei_ac_agent_config', 'rpc_server_ip')).split(',')
        if not password:
            return 'ac_auth_password', 'failed', 'not configured.'
        if after_fsp_6_3_0(self.ops_version):
            password_real = crypt.decrypt(password)
        else:
            password_real = six.text_type(security_util.decrypt_data(password),
                                          errors='ignore')
        for host in rpc_server_ips:
            if host and netaddr.valid_ipv6(host):
                host = "[" + str(host) + "]"
            token_result = get_token(host, username, password_real)
            if token_result and isinstance(token_result, dict) \
                    and token_result.get('data') \
                    and token_result['data'].get('token_id'):
                return token_result['data']['token_id']
        return 'failed'

    def get_external_gateway(self):
        """Get external gateway info from ac by northbound interface"""
        rpc_server_ips = re.sub(r'\s', '', self.huawei_config.get_config_detail(
            'huawei_ac_agent_config', 'rpc_server_ip')).split(',')
        token = self.check_token()
        for host in rpc_server_ips:
            if host and netaddr.valid_ipv6(host):
                host = "[" + str(host) + "]"
            headers = {"Content-type": "application/json",
                       "Accept": "application/json",
                       "X-ACCESS-TOKEN": token}
            auth_url = "%s%s%s%s%s" % (ac_constants.HTTPS_HEADER,
                                       host, ":",
                                       ac_constants.rest_server_port,
                                       ac_constants.EXTERNAL_GATEWAYS_URL)

            try:
                cert_file = get_cert_file() or True
                requests.packages.urllib3.disable_warnings()
                with requests.session() as session:
                    adapter = HWHTTPSAdapter(
                        cert_file=None, key_file=None, password=None,
                        pool_connections=100, pool_maxsize=100)
                    session.mount("https://", adapter)
                    req = session.request('GET',
                                          url=auth_url,
                                          headers=headers,
                                          verify=cert_file,
                                          timeout=105)
                    LOGGER.info("[HEALTH]Get external net from ac :%s",
                                str(req.content))
                return jsonutils.loads(req.content)
            except Exception as ex:
                LOGGER.error("[HEALTH]sending request failed :%s", ex)
                return 'sending getting request failed.'

    def get_external_gateway_name(self):
        """Get external gateway names"""
        external_gateways = self.get_external_gateway()
        external_gateway_name = []
        if isinstance(external_gateways, dict) and \
                'errcode' not in external_gateways:
            if not external_gateways or \
                    external_gateways.get('externalGateway', []) == []:
                return external_gateway_name
            for external_gateway in \
                    external_gateways.get('externalGateway', []):
                name = external_gateway.get('name', '')
                if name:
                    external_gateway_name.append(name)
            LOGGER.info("[HEALTH]Get external net name from ac :%s",
                        external_gateway_name)
            return external_gateway_name
        else:
            return "Error:sending getting request failed."

    def get_ext_net_from_db(self):
        """Get external networks from neutron db"""
        results = []
        admin_context = neutron_context.get_admin_context()
        filters = {
            'router:external': [True]
        }
        fields = ['name', 'id']
        if self.ops_version in ac_constants.OPS_VERSION_O_PQRTW_6_21:
            try:
                from neutron_lib.plugins import directory
                external_networks = directory.get_plugin().get_networks(
                    admin_context, filters=filters, fields=fields)
            except ImportError:
                LOGGER.error('import neutron_lib.plugins.directory failed')
        else:
            external_networks = manager.NeutronManager.get_plugin(). \
                get_networks(admin_context, filters=filters, fields=fields)
        networks = copy.deepcopy(external_networks)
        for network in networks:
            if self.data_filter.not_in_white_or_in_black_list(
                    admin_context, network, ac_constants.NW_HW_NETWORKS,
                    None):
                results.append(network.get('name'))
        LOGGER.info("[HEALTH]Get external networks from neutron :%s", results)
        return results

    def check_neutronserver_cpu_and_mem(self):
        """Check the neutron-server is ok"""
        if is_fsp(self.ops_version):
            server_infs = manage_base_get_cmd(
                "ps -ef | grep 'neutron-server' | "
                "grep '^opensta+*'").split('\n')
            res = calc_cpu_and_mem(server_infs)
        else:
            server_infs = manage_base_get_cmd(
                "ps -ef | grep 'neutron-server' | "
                "grep '^neutron*'").split('\n')
            res = calc_cpu_and_mem(server_infs)
        return res

    def display_neutronserver_cpu_and_mem_used(self):
        """Display neutron server cpu and memery used"""
        res = self.check_neutronserver_cpu_and_mem()
        if res.__len__() == 2:
            print("   \033[1;32mOk\033[0m")
            print(
                '       cpu being used by neutron-server:' + str(res[0]) + '%')
            print(
                '       mem being used by neutron-server:' + str(res[1]) + 'K')
        else:
            print('   \033[1;31mFail\033[0m')
            print('       \033[1;31m%s\033[0m' % str(res))

    def check_file_user_and_group(self):
        """Check file user and the file permission is correct"""
        path = get_ac_plugin_path()
        if is_fsp(self.ops_version):
            source_result = manage_base_get_cmd(
                "cd " + path +
                "\nfind . -not -user openstack -ls | grep -v drwx"
                "\nfind . -not -group openstack -ls | grep -v drwx")
            res = deal_file(source_result)
        else:
            source_result = manage_base_get_cmd(
                "cd " + path +
                "\nfind . -not -user neutron -ls | grep -v drwx"
                "\nfind . -not -group neutron -ls | grep -v drwx")
            res = deal_file(source_result)
        return res

    def display_file_user_and_group(self):
        """Display file user and group while file is not correct"""
        res = self.check_file_user_and_group()
        path = get_ac_plugin_path()
        if res == '':
            LOGGER.info('[HEALTH]check files user and group: ok')
            print("   \033[1;32mOk\033[0m")
        else:
            LOGGER.warning('[HEALTH]check files user and group: failed')
            print('   \033[1;31mFail\033[0m')
            for i in six.moves.range(len(res[0])):
                LOGGER.warning('[HEALTH]files info %s%s%s',
                               res[0][i], path, res[1][i])
                print('       \033[1;31m%s\033[0m' %
                      ('Error:the file %s has been changed, Path:%s%s.' %
                       (res[0][i], path, res[1][i])))


def check_config_value(manage_base):
    yield get_plugin_version()
    for call_fun in (ManageBase.check_ac_ip, ManageBase.check_host_ip, ManageBase.check_vni_ranges):
        yield call_fun(manage_base)


class HealthCheck(object):
    """HealthCheck class"""

    def __init__(self):
        """Init function"""
        self.manage_base = ManageBase()
        self.results = []

    def check_config_items(self, scene):
        """Check config items"""
        result = {}

        for elem in check_config_value(self.manage_base):
            result[elem[0]] = elem[2]

        for config_name in ac_constants.CHECK_MANDATORY_PARAMS:
            data = self.manage_base.check_required_filled_general(
                ac_constants.HUAWEI_CONFIG, config_name)
            result[data[0]] = data[2]

        for config_name in ac_constants.CHECK_DEFAULT_VALUE_PARAMS:
            data = self.manage_base.check_default_config(config_name)
            result[data[0]] = data[2]

        if scene in [3, 4]:
            for config_name in ac_constants.CHECK_FSP_OTHER_ITEMS:
                data = self.manage_base.check_optional_filled_general(
                    ac_constants.HUAWEI_CONFIG, config_name)
                result[data[0]] = data[2]

        return result

    def check_comm_with_ac(self):
        """Check whether the communication with the AC is normal"""
        data1 = self.manage_base.check_rpc_server_ip()
        data2 = self.manage_base.check_ac_passwd()
        result = self.get_result(data1, data2)
        return result

    def check_keystone(self):
        """Check whether the Keystone service is normal"""
        data = self.manage_base.check_keystone_params()
        if data[1] == 'ok':
            return 'OK'
        return 'ERROR:detail:' + data[2]

    def check_black_and_white_list(self):
        """Check whether the network blacklist and whitelist are normal"""
        data1 = check_network_black_and_white_list(
            'network_black_list', ac_constants.HUAWEI_CONFIG_FILE)
        data2 = check_network_black_and_white_list(
            'network_white_list', ac_constants.HUAWEI_CONFIG_FILE)
        result = self.get_result(data1, data2)
        return result

    def check_process(self):
        """Check whether the neutron-server process is running properly"""
        data1 = check_neutron_server(self.manage_base.ops_version, False)
        data2 = check_normal_sync()
        result = self.get_result(data1, data2)
        return result

    def check_external_network(self):
        """Check whether the neutron matches the external net of the ac"""
        data = check_ext_net(
            self.manage_base.get_ext_net_from_db(),
            self.manage_base.get_external_gateway_name())
        if data == '':
            result = 'OK'
        else:
            if data.startswith('Warning:') or data.startswith('Error:'):
                result = data.replace('Warning:', 'WARNING:detail:'). \
                    replace('Error:', 'ERROR:detail:')
            else:
                result = 'WARNING:detail:no match external network:' + data

        return result

    def check_vhost_user(self, scene):
        """Check vhost user param"""
        data = self.manage_base.check_vhost_user(scene, [2, 4])
        if data[1] == 'ok':
            return 'OK'
        return 'ERROR:detail:the value of vhost_user is not consistent ' \
               'with the selected scenario.'

    def check_cpu_and_memory(self):
        """Check whether the CPU and memory usage of Neutron is normal"""
        data = self.manage_base.check_neutronserver_cpu_and_mem()
        if data.__len__() == 2:
            result = 'OK:detail:cpu being used by neutron-server:' \
                     + str(data[0]) + '% and mem being used by ' \
                                      'neutron-server:' \
                     + str(data[1]) + 'K'
        else:
            result = 'ERROR:detail' + str(data)
        return result

    def check_ac_pugin_file(self):
        """Check whether the AC plug-in file is normal"""
        data = self.manage_base.check_file_user_and_group()
        path = get_ac_plugin_path()

        if data == '':
            result = 'OK'
        else:
            result = 'ERROR:detail:'
            for i in six.moves.range(len(data[0])):
                result += ('the file ' + data[0][i] +
                           ' has been changed, Path:' + path + data[1][i] + '.')
        return result

    def check_ac_plugin(self, scene):
        """Check whether the AC plug-in is normal"""
        result = {}
        scene_num = ac_constants.HEALTH_CHECK_SCENE.get(scene)
        LOGGER.info("Start to execute health check by scene :%s", scene_num)
        result['check_config_items'] = self.check_config_items(scene_num)
        result['check_comm_with_ac'] = self.check_comm_with_ac()
        result['check_keystone'] = self.check_keystone()
        result['check_black_white_list'] = self.check_black_and_white_list()
        result['check_process'] = self.check_process()
        result['check_external_network'] = self.check_external_network()
        result['check_vhost_user'] = self.check_vhost_user(scene_num)
        result['check_cpu_and_memory'] = self.check_cpu_and_memory()
        result['check_ac_pugin_file'] = self.check_ac_pugin_file()
        LOGGER.info("Get health check result :%s", result)
        return result

    @classmethod
    def get_result(cls, data1, data2):
        """Get result"""
        result = 'OK'
        error_message = ''
        for data in [data1, data2]:
            if data[1] == 'failed':
                error_message += data[2]
        if error_message:
            result = 'ERROR:detail:' + error_message
        return result
