#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""bgp router plugin"""

import copy
from oslo_log import log as logging

from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.extensions.bgp_route import \
    bgp_route as bgp_r
from networking_huawei.drivers.ac.model.bgp_route_model import ACBgpRouteModel
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei.drivers.ac.db.bgp_route import bgp_route_db
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler


LOG = logging.getLogger(__name__)


class HuaweiBgpRoutePlugin(bgp_r.BgpRouteBase,
                           bgp_route_db.BgpRouteDbMixin):
    """Implementation of the Huawei AC Bgp route Service Plugin."""
    supported_extension_aliases = ['bgp-route']

    def __init__(self):
        LOG.info(_LI("[AC] Init huawei bgp route plugin."))
        super(HuaweiBgpRoutePlugin, self).__init__()
        self.ac_reliability = ACReliabilityAPI(ac_constants.NW_HW_BGP_ROUTE)
        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for huawei bgp route plugin."))

    def get_plugin_type(self):
        """get plugin type"""
        return ac_constants.NW_HW_BGP_ROUTE

    def get_plugin_description(self):
        """get plugin description"""
        return 'Huawei bgp route service plugin'

    def create_bgp_route(self, context, bgp_route):
        """create bgp route"""
        bgp_route_log = copy.deepcopy(bgp_route)['bgp_route']
        bgp_route_log = validate.validate_log_record(
            bgp_route_log, ac_constants.NW_HW_BGP_ROUTE)
        LOG.info(_LI("[AC] Begin to create bgp route: %s "),
                 bgp_route_log)
        ac_osprofiler.record_chain_start("create bgp route start")

        bgp_route_dict = super(HuaweiBgpRoutePlugin, self).\
            create_db_bgp_route(context, bgp_route)
        bgp_route_log = copy.deepcopy(bgp_route_dict)
        bgp_route_log = validate.validate_log_record(
            bgp_route_log, ac_constants.NW_HW_BGP_ROUTE)
        LOG.info(_LI("[AC] Create bgp route record in Neutron DB "
                     "successful: %s "), bgp_route_log)
        ac_osprofiler.record_chain_start("bgp route id:" +
                                         bgp_route_dict['id'])
        bgp_route_info = ACBgpRouteModel.ac_model_format(bgp_route_dict)

        try:
            LOG.debug("Start to send the request of create bgp route")
            self.ac_reliability.\
                update_plugin_record(context,
                                     bgp_route_info['uuid'],
                                     bgp_route_info,
                                     ac_constants.NW_HW_CREATE_BGP_ROUTE)
        except Exception as ex:
            LOG.error(_LE("[AC] Huawei AC create bgp route failed. "
                          "Roll back: delete bgp route in Neutron DB. "
                          "catch: %s"), ex)
            ac_osprofiler.record_chain_exception_end("create bgp route fail")
            super(HuaweiBgpRoutePlugin, self).\
                delete_db_bgp_route(context, bgp_route_info['uuid'])
            raise ex

        LOG.info(_LI("[AC] Huawei AC create bgp route successful."))
        ac_osprofiler.record_chain_end_with_reason("create bgp route success")
        bgp_route_display = validate.validate_display_record(bgp_route_dict)
        return bgp_route_display

    def delete_bgp_route(self, context, bgp_route_id):
        """delete bgp route"""
        LOG.info(_LI("[AC] Begin to delete bgp route: %s"), bgp_route_id)
        ac_osprofiler.record_chain_start("delete bgp route start,ID:%s"
                                         % bgp_route_id)
        try:
            LOG.debug("Start to send the request of delete bgp route")
            self.ac_reliability.\
                update_plugin_record(context,
                                     bgp_route_id,
                                     {},
                                     ac_constants.NW_HW_DELETE_BGP_ROUTE)
        except Exception as value:
            LOG.error(_LE("[AC] AC delete bgp route "
                          "failed for %s."), value)
            ac_osprofiler.record_chain_exception_end("delete bgp route fail")
            raise value

        LOG.info(_LI("[AC] AC delete bgp route successful."))

        try:
            super(HuaweiBgpRoutePlugin, self).\
                delete_db_bgp_route(context, bgp_route_id)
        except Exception as value:
            LOG.error(_LE("[AC] Failed to delete bgp route "
                          "in Neutron DB for %s."), value)
            raise value
        ac_osprofiler.record_chain_end_with_reason("delete bgp route success")
        LOG.info(_LI("[AC] Delete bgp route record in "
                     "Neutron DB successful."))

    def update_bgp_route(self, context, bgp_route_id, bgp_route):
        """update bgp route"""
        bgp_route_log = copy.deepcopy(bgp_route)['bgp_route']
        bgp_route_log = validate.validate_log_record(
            bgp_route_log, ac_constants.NW_HW_BGP_ROUTE)
        LOG.info(_LI("[AC] Begin to update bgp route: %s as: %s"),
                 bgp_route_id, bgp_route_log)
        ac_osprofiler.record_chain_start("update bgp route start,ID:%s"
                                         % bgp_route_id)
        original_bgp_route = super(HuaweiBgpRoutePlugin, self).\
            get_db_bgp_route(context, bgp_route_id)
        bgp_route_log = copy.deepcopy(original_bgp_route)
        bgp_route_log = validate.validate_log_record(
            bgp_route_log, ac_constants.NW_HW_BGP_ROUTE)
        LOG.debug("[AC] Get original bgp route in Neutron DB, %s",
                  bgp_route_log)
        updated_bgp_route = super(HuaweiBgpRoutePlugin, self).\
            update_db_bgp_route(context, bgp_route_id, bgp_route)
        bgp_route_log = copy.deepcopy(updated_bgp_route)
        bgp_route_log = validate.validate_log_record(
            bgp_route_log, ac_constants.NW_HW_BGP_ROUTE)
        LOG.info(_LI("[AC] Updated bgp route record in Neutron DB "
                     "successful: %s"), bgp_route_log)

        new_bgp_route_info = ACBgpRouteModel.\
            ac_model_format(updated_bgp_route)
        try:
            LOG.debug("Start to send the request of update bgp route")
            self.ac_reliability.\
                update_plugin_record(context,
                                     new_bgp_route_info['uuid'],
                                     new_bgp_route_info,
                                     ac_constants.NW_HW_UPDATE_BGP_ROUTE)
        except Exception as value:
            LOG.error(_LE("[AC] AC update bgp route failed for %s. "
                          "Roll back to the previous bgp route "
                          "info."), value)
            ac_osprofiler.record_chain_exception_end("update bgp route fail")
            super(HuaweiBgpRoutePlugin, self).\
                update_db_bgp_route(
                    context, bgp_route_id, {'bgp_route': original_bgp_route})
            raise value

        LOG.info(_LI("[AC] Huawei AC update bgp route successful."))
        ac_osprofiler.record_chain_end_with_reason("update bgp route success")
        bgp_route_display = validate.validate_display_record(updated_bgp_route)
        return bgp_route_display

    def get_bgp_route(self, context, bgp_route_id, fields=None):
        """get bgp route"""
        LOG.info(_LI("[AC] Begin to get bgp route : %s"), bgp_route_id)
        rest_info = super(HuaweiBgpRoutePlugin, self). \
            get_db_bgp_route(context, bgp_route_id,
                             fields=fields, need_decrypt=False)
        LOG.info(_LI("[AC] End to get bgp route."))
        bgp_route_display = validate.validate_display_record(rest_info)
        return bgp_route_display

    def get_bgp_routes(self, context, filters=None, fields=None):
        """get bgp routes"""
        LOG.info(_LI("[AC] Begin to get bgp routes."))
        bgp_routes = []
        bgp_route_list_tmp = super(HuaweiBgpRoutePlugin, self). \
            get_db_bgp_routes(context,
                              filters=filters,
                              fields=fields, need_decrypt=False)
        for bgp_route in bgp_route_list_tmp:
            bgp_route_display = validate.validate_display_record(bgp_route)
            bgp_routes.append(bgp_route_display)
        return bgp_routes
