#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Bgpneighbor Plugin"""

from oslo_log import log as logging

from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.extensions.bgpneighbor import \
    bgpneighbor as bgp_n
from networking_huawei.drivers.ac.model.bgp_route_model import ACBgpRouteModel
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei.drivers.ac.db.bgp_route import bgpneighbor_db
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler


LOG = logging.getLogger(__name__)

class HuaweiBgpNeighborPlugin(bgp_n.BgpneighborBase,
                              bgpneighbor_db.BgpNeighborDbMixin):
    """Implementation of the Huawei AC Bgpneighbor Service Plugin."""
    supported_extension_aliases = ['bgpneighbor']

    def __init__(self):
        LOG.info(_LI("[AC] Init huawei bgpneighbor plugin."))
        super(HuaweiBgpNeighborPlugin, self).__init__()
        self.ac_reliability = ACReliabilityAPI(ac_constants.NW_HW_BGP_ROUTE)
        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for huawei bgpneighbor plugin."))

    def get_plugin_type(self):
        """get plugin type"""
        return "bgpneighbor"

    def get_plugin_description(self):
        """get plugin description"""
        return 'Huawei bgpneighbor service plugin'

    def create_bgpneighbor(self, context, bgpneighbor):
        """create bgpneighbor"""
        LOG.info(_LI("[AC] Begin to create bgpneighbor: %s "),
                 bgpneighbor)
        ac_osprofiler.record_chain_start("create bgp route start")

        bgp_route_dict = super(HuaweiBgpNeighborPlugin, self).\
            create_db_bgpneighbor(context, bgpneighbor)
        LOG.info(_LI("[AC] Create bgpneighbor record in Neutron DB "
                     "successful: %s "), bgp_route_dict)
        ac_osprofiler.record_chain_start("bgp route id:" +
                                         bgp_route_dict['id'])
        bgp_route_info = ACBgpRouteModel.ac_model_format(bgp_route_dict)
        try:
            LOG.debug("Start to send the request of create bgpneighbor")
            self.ac_reliability.\
                update_plugin_record(context,
                                     bgp_route_info['uuid'],
                                     bgp_route_info,
                                     ac_constants.NW_HW_CREATE_BGP_ROUTE)
        except Exception as ex:
            LOG.error(_LE("[AC] Huawei AC create bgpneighbor failed. "
                          "Roll back: delete bgpneighbor in Neutron DB. "
                          "catch: %s"), ex)
            ac_osprofiler.record_chain_exception_end("create bgpneighbor fail")
            super(HuaweiBgpNeighborPlugin, self).\
                delete_db_bgpneighbor(context, bgp_route_info['uuid'])
            raise

        LOG.info(_LI("[AC] Huawei AC create bgpneighbor successful."))
        ac_osprofiler.record_chain_end_with_reason("create bgpneighbor success")

        return self._make_bgpneighbor_dict(bgp_route_dict)

    def delete_bgpneighbor(self, context, bgpneighbor_id):
        """delete bgpneighbor"""
        LOG.info(_LI("[AC] Begin to delete bgpneighbor: %s"), bgpneighbor_id)
        ac_osprofiler.record_chain_start("delete bgpneighbor start,ID:"
                                         + bgpneighbor_id)
        try:
            LOG.debug("Start to send the request of delete bgpneighbor")
            self.ac_reliability.\
                update_plugin_record(context, bgpneighbor_id, {},
                                     ac_constants.NW_HW_DELETE_BGP_ROUTE)
        except Exception as value:
            LOG.error(_LE("[AC] AC delete bgpneighbor "
                          "failed for %s."), value)
            ac_osprofiler.record_chain_exception_end("delete bgpneighbor fail")
            raise

        LOG.info(_LI("[AC] AC delete bgpneighbor successful."))

        try:
            super(HuaweiBgpNeighborPlugin, self).\
                delete_db_bgpneighbor(context, bgpneighbor_id)
        except Exception as value:
            LOG.error(_LE("[AC] Failed to delete bgpneighbor "
                          "in Neutron DB for %s."), value)
            raise
        ac_osprofiler.record_chain_end_with_reason("delete bgp route success")
        LOG.info(_LI("[AC] Delete bgpneighbor record in "
                     "Neutron DB successful."))

    def update_bgpneighbor(self, context, bgpneighbor_id, bgpneighbor):
        """update bgpneighbor"""
        LOG.info(_LI("[AC] Begin to update bgpneighbor: %s as: %s"),
                 bgpneighbor_id, bgpneighbor)
        ac_osprofiler.record_chain_start("update bgpneighbor start,ID:" +
                                         bgpneighbor_id)
        original_bgp_route = super(HuaweiBgpNeighborPlugin, self).\
            get_db_bgpneighbor(context, bgpneighbor_id)
        LOG.debug("[AC] Get original bgpneighbor in Neutron DB, %s",
                  original_bgp_route)
        updated_bgp_route = super(HuaweiBgpNeighborPlugin, self).\
            update_db_bgpneighbor(context, bgpneighbor_id, bgpneighbor)
        LOG.info(_LI("[AC] Updated bgpneighbor record in Neutron DB "
                     "successful: %s"), updated_bgp_route)

        new_bgp_route_info = ACBgpRouteModel.\
            ac_model_format(updated_bgp_route)
        try:
            LOG.debug("Start to send the request of update bgpneighbor")
            self.ac_reliability.\
                update_plugin_record(context,
                                     new_bgp_route_info['uuid'],
                                     new_bgp_route_info,
                                     ac_constants.NW_HW_UPDATE_BGP_ROUTE)
        except Exception as value:
            LOG.error(_LE("[AC] AC update bgpneighbor failed for %s. "
                          "Roll back to the previous bgpneighbor "
                          "info."), value)
            ac_osprofiler.record_chain_exception_end("update bgpneighbor fail")
            super(HuaweiBgpNeighborPlugin, self).\
                update_db_bgpneighbor(context, bgpneighbor_id,
                                      {'bgpneighbor': original_bgp_route})
            raise

        LOG.info(_LI("[AC] Huawei AC update bgpneighbor successful."))
        ac_osprofiler.record_chain_end_with_reason("update bgpneighbor success")

        return self._make_bgpneighbor_dict(updated_bgp_route)

    def get_bgpneighbor(self, context, bgpneighbor_id, fields=None):
        """get single bgpneighbor"""
        LOG.info(_LI("[AC] Begin to get bgpneighbor : %s"), bgpneighbor_id)
        rest_info = super(HuaweiBgpNeighborPlugin, self). \
            get_db_bgpneighbor(context, bgpneighbor_id, fields=fields)
        LOG.info(_LI("[AC] End to get bgpneighbor."))
        return rest_info

    def get_bgpneighbors(self, context, filters=None, fields=None,
                         sorts=None, limit=None, marker_obj=None,
                         page_reverse=False):
        """get bgpneighbors"""
        LOG.info(_LI("[AC] Begin to get bgpneighbors."))

        return super(HuaweiBgpNeighborPlugin, self). \
            get_db_bgpneighbors(context, filters=filters, fields=fields,
                                sorts=sorts, limit=limit,
                                marker_obj=marker_obj,
                                page_reverse=page_reverse)
