#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""External whitelist plugin"""

from oslo_log import log as logging

from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.extensions.external_whitelist \
    import external_whitelist as ext_white
from networking_huawei.drivers.ac.client.service import ACReSTService
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler

LOG = logging.getLogger(__name__)

class HuaweiExternalWhitelistPlugin(ext_white.ExternalWhitelistBase):
    """Implementation of the Huawei AC external whitelist Service Plugin."""
    supported_extension_aliases = ['ac-external-whitelist']

    def __init__(self):
        LOG.info(_LI("[AC] Init huawei external whitelist plugin."))
        super(HuaweiExternalWhitelistPlugin, self).__init__()
        self.ac_reliability = ACReliabilityAPI(
            ac_constants.NW_HW_EXTERNAL_WHITELIST)
        self.rest_service = ACReSTService()
        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for huawei external whitelist plugin."))

    def get_plugin_type(self):
        """Type for this plugin"""
        return "external_whitelist"

    def get_plugin_description(self):
        """A small description about this plugin"""
        return 'Huawei external whitelist service plugin'

    def create_external_whitelist(self, context, external_whitelist):
        """Create external whitelist"""
        LOG.info(_LI("[AC] Begin to create external whitelist: %s "),
                 external_whitelist)
        ac_osprofiler.record_chain_start("create external whitelist start")

        external_whitelist_info = external_whitelist.get("external_whitelist")
        router_id = external_whitelist_info.get("router_id")
        router_filter = {"router_id": [router_id]}
        external_whitelist_id = external_whitelist_info.get("id")
        if not external_whitelist_id:
            external_whitelist_id = "external_whitelist_id"
            external_whitelist_info.pop("id")
        try:
            LOG.debug("Start to send the request of create external whitelist")
            self.ac_reliability.update_plugin_record(
                context, external_whitelist_id, external_whitelist_info,
                ac_constants.NW_HW_CREATE_EXTERNAL_WHITELIST)

        except Exception as ex:
            LOG.error(_LE("[AC] Huawei AC create external whitelist failed. "
                          "catch: %s"), ex)
            raise

        LOG.info(_LI("[AC] Huawei AC create external whitelist successful."))
        ac_osprofiler.record_chain_end_with_reason("create external "
                                                   "whitelist success")
        return self.get_external_whitelists(context, router_filter)[0]

    def delete_external_whitelist(self, context, external_whitelist_id):
        """Delete external whitelist"""
        LOG.info(_LI("[AC] Begin to delete external whitelist: %s"),
                 external_whitelist_id)
        ac_osprofiler.record_chain_start("delete external whitelist "
                                         "start,ID:" + external_whitelist_id)
        try:
            LOG.debug("Start to send the request of delete external whitelist")
            self.ac_reliability.update_plugin_record(
                context, external_whitelist_id, {},
                ac_constants.NW_HW_DELETE_EXTERNAL_WHITELIST)

        except Exception as value:
            LOG.error(_LE("[AC] AC delete external whitelist "
                          "failed for %s."), value)
            ac_osprofiler.record_chain_exception_end("delete external whitelist"
                                                     " fail")
            raise

        LOG.info(_LI("[AC] AC delete external whitelist successful."))
        ac_osprofiler.record_chain_end_with_reason("delete external whitelist "
                                                   "success")

    def update_external_whitelist(self, context, external_whitelist_id,
                                  external_whitelist):
        """Update external whitelist"""
        LOG.info(_LI("[AC] Begin to update external whitelist:%s as:%s"),
                 external_whitelist_id, external_whitelist)
        ac_osprofiler.record_chain_start("update external whitelist start, ID:"
                                         + external_whitelist_id)
        external_whitelist_info = external_whitelist.get("external_whitelist")

        try:
            LOG.debug("Start to send the request of update external whitelist")
            self.ac_reliability.update_plugin_record(
                context, external_whitelist_id, external_whitelist_info,
                ac_constants.NW_HW_UPDATE_EXTERNAL_WHITELIST)

        except Exception as value:
            LOG.error(_LE("[AC] AC update external whitelist failed "
                          "for %s. "), value)
            ac_osprofiler.record_chain_exception_end("update external "
                                                     "whitelist fail")
            raise

        LOG.info(_LI("[AC] Huawei AC update external whitelist successful."))
        ac_osprofiler.record_chain_end_with_reason("update external "
                                                   "whitelist success")
        return self.get_external_whitelist(context, external_whitelist_id)

    def get_external_whitelist(self, context, external_whitelist_id,
                               fields=None):
        """Get single external whitelist"""
        LOG.info('context=%s,fields=%s', context, fields)
        LOG.info(_LI("[AC] Begin to get external whitelist : %s"),
                 external_whitelist_id)
        try:
            response = self.rest_service.get_ac_external_whitelist_info(
                external_whitelist_id)
        except Exception as value:
            ac_osprofiler.record_chain_exception_end("get external "
                                                     "whitelist fail")
            LOG.error(_LE("[AC] Get external whitelist %s failed: %s."),
                      external_whitelist_id, value)
            raise
        LOG.info(_LI("[AC] Success to get external whitelist: %s"), response)
        ac_osprofiler.record_chain_end_with_reason("Get external "
                                                   "whitelist success")
        return response

    def get_external_whitelists(self, context, filters=None, fields=None):
        """Get external whitelists"""
        LOG.info('context=%s,fields=%s', context, fields)
        LOG.info(_LI("[AC] Begin to get external whitelist with filter: %s"),
                 filters)
        router_id = filters.get("router_id")
        if router_id:
            router_id = router_id[0]
        try:
            response = self.rest_service.get_ac_external_whitelist_info(
                router_id=router_id)
        except Exception as value:
            ac_osprofiler.record_chain_exception_end("get external "
                                                     "whitelist fail")
            LOG.error(_LE("[AC] Get external whitelist of router %s failed: %s."
                          ""), router_id, value)
            raise
        LOG.info(_LI("[AC] Success to get external whitelist %s of router: %s"),
                 response, router_id)
        ac_osprofiler.record_chain_end_with_reason("Get external "
                                                   "whitelist success")
        return response

    def add_routes(self, context, external_whitelist_id, external_whitelist):
        """Add routes for external whitelist"""
        LOG.info(_LI("[AC] Begin to add routes for external whitelist:"
                     "%s as:%s"), external_whitelist_id, external_whitelist)
        ac_osprofiler.record_chain_start("add routes external whitelist "
                                         "start, ID:" + external_whitelist_id)
        external_whitelist_info = external_whitelist.get("external_whitelist")
        try:
            LOG.debug("Start to send the request to add routes for "
                      "external whitelist")
            self.ac_reliability.update_plugin_record(
                context, external_whitelist_id, external_whitelist_info,
                ac_constants.NW_HW_ADDROUTES_EXTERNAL_WHITELIST)

        except Exception as value:
            LOG.error(_LE("[AC] Add routes for external whitelist failed:%s."),
                      value)
            ac_osprofiler.record_chain_exception_end("Add routes for external "
                                                     "whitelist fail")
            raise
        LOG.info(_LI("[AC] Add routes for external whitelist success."))
        ac_osprofiler.record_chain_end_with_reason("Add routes for external "
                                                   "whitelist success")
        return {"external_whitelist":
                    self.get_external_whitelist(context, external_whitelist_id)}

    def remove_routes(self, context, external_whitelist_id, external_whitelist):
        """Remove routes for external whitelist"""
        LOG.info(_LI("[AC] Begin to remove routes for external whitelist:%s "
                     "as:%s"), external_whitelist_id, external_whitelist)
        ac_osprofiler.record_chain_start("remove routes external whitelist "
                                         "start,ID:" + external_whitelist_id)
        external_whitelist_info = external_whitelist.get("external_whitelist")
        try:
            LOG.debug("Start to send the request to remove routes for "
                      "external whitelist")
            self.ac_reliability.update_plugin_record(
                context, external_whitelist_id, external_whitelist_info,
                ac_constants.NW_HW_REMOVEROUTES_EXTERNAL_WHITELIST)

        except Exception as value:
            LOG.error(_LE("[AC] AC remove routes for external whitelist failed "
                          "for %s. "), value)
            ac_osprofiler.record_chain_exception_end(
                "remove routes for external whitelist fail")
            raise
        LOG.info(_LI("[AC] Remove routes for external whitelist success."))
        ac_osprofiler.record_chain_end_with_reason(
            "Remove routes for external whitelist success")
        return {"external_whitelist":
                    self.get_external_whitelist(context, external_whitelist_id)}

    def add_subnets(self, context, external_whitelist_id, external_whitelist):
        """Add subnets for external whitelist"""
        LOG.info(_LI("[AC] Begin to add subnets for external whitelist:"
                     "%s as:%s"), external_whitelist_id, external_whitelist)
        ac_osprofiler.record_chain_start(
            "add subnets external whitelist start,ID:" + external_whitelist_id)
        external_whitelist_info = external_whitelist.get("external_whitelist")
        try:
            LOG.debug("Start to send the request to add subnets for "
                      "external whitelist")
            self.ac_reliability.update_plugin_record(
                context, external_whitelist_id, external_whitelist_info,
                ac_constants.NW_HW_ADDSUBNETS_EXTERNAL_WHITELIST)
        except Exception as value:
            LOG.error(_LE("[AC]AC add subnets for external whitelist for %s."),
                      value)
            ac_osprofiler.record_chain_exception_end("add subnets for external "
                                                     "whitelist fail")
            raise
        LOG.info(_LI("[AC] Add subnets for external whitelist success."))
        ac_osprofiler.record_chain_end_with_reason("dd subnets for external "
                                                   "whitelist success")
        return {"external_whitelist":
                    self.get_external_whitelist(context, external_whitelist_id)}

    def remove_subnets(self, context, external_whitelist_id,
                       external_whitelist):
        """Remove subnets for external whitelist"""
        LOG.info(_LI("[AC] Begin to remove subnets for external whitelist:%s "
                     "as:%s"), external_whitelist_id, external_whitelist)
        ac_osprofiler.record_chain_start("remove subnets external whitelist "
                                         "start,ID:" + external_whitelist_id)
        external_whitelist_info = external_whitelist.get("external_whitelist")
        try:
            LOG.debug("Start to send the request to remove subnets for "
                      "external whitelist")

            self.ac_reliability.update_plugin_record(
                context, external_whitelist_id, external_whitelist_info,
                ac_constants.NW_HW_REMOVESUBNETS_EXTERNAL_WHITELIST)

        except Exception as value:
            LOG.error(_LE("[AC] AC remove subnets for external whitelist "
                          "for %s."), value)
            ac_osprofiler.record_chain_exception_end("remove subnets for "
                                                     "external whitelist fail")
            raise
        LOG.info(_LI("[AC] Remove subnets for external whitelist success."))
        ac_osprofiler.record_chain_end_with_reason("Remove subnets for external"
                                                   " whitelist success")
        return {"external_whitelist":
                    self.get_external_whitelist(context, external_whitelist_id)}
