#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""fwaas plugin file"""

import os
from distutils.sysconfig import get_python_lib
import six
import netaddr
from oslo_config import cfg
try:
    from neutron.api.v2.attributes import ATTR_NOT_SPECIFIED
except ImportError:
    from neutron_lib.constants import ATTR_NOT_SPECIFIED

try:
    from neutron import context as neutron_context
except ImportError:
    from neutron_lib import context as neutron_context

try:
    from neutron_fwaas.db.firewall.firewall_db_poly import \
        FirewallPolyDbMixin as fw_db
except ImportError:
    try:
        from neutron_fwaas.db.firewall.firewall_db import \
            Firewall_db_mixin as fw_db
    except ImportError:
        from neutron.db.firewall.firewall_db import Firewall_db_mixin as fw_db
try:
    # import firewall db for openstack
    from neutron_fwaas.db.firewall import firewall_db
except ImportError:
    # import firewall db for fusionsphere
    from neutron.db.firewall import firewall_db
try:
    # import firewall router insertion db for openstack
    from neutron_fwaas.db.firewall import firewall_router_insertion_db
except ImportError:
    # import firewall router insertion db for fusionsphere
    from neutron.db.firewall import firewall_router_insertion_db

from networking_huawei._i18n import _LE
from networking_huawei._i18n import _LI
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

from networking_huawei.drivers.ac.model.firewall_model \
    import ACFirewallModel
from networking_huawei.drivers.ac.model.firewall_rule_model \
    import ACFirewallRuleModel
from networking_huawei.drivers.ac.model.firewall_policy_model \
    import ACFirewallPolicyModel
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI

from networking_huawei.drivers.ac.db.lock import lockedobjects_db as lock_db
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.db.ac_proc_status.ac_proc_status_db \
    import ACProStatusDbMixin

from networking_huawei.drivers.ac.db.firewall_rule.firewall_rule_db \
    import ACFirewallruleAddrsDbMixin


LOG = ncu.ac_log.getLogger(__name__)
FIREWALL_POLICY_FILE = "neutron_fwaas/extensions/firewall_poly.py"


class HuaweiACFwaasPlugin(
        fw_db, firewall_router_insertion_db.FirewallRouterInsertionDbMixin):
    """Implementation of the Huawei AC Fwaas Service Plugin."""

    supported_extension_aliases = ["fwaas", "fwaasrouterinsertion"]

    lib_path = get_python_lib()
    if os.path.exists(os.path.join(get_python_lib(), FIREWALL_POLICY_FILE)):
        supported_extension_aliases.append("ext_fwaas")

    path_prefix = "/fw"

    __native_pagination_support = True
    __native_sorting_support = True

    def __init__(self):
        LOG.info(_LI("[AC]Init Huawei fwaas plugin."))
        super(HuaweiACFwaasPlugin, self).__init__()

        self.ops_version = ncu.get_ops_version()
        self.fw_driver = HuaweiACFwaasDriver()
        self.ac_proc_db_func = ACProStatusDbMixin()
        self.firewall_rule_addrs_func = ACFirewallruleAddrsDbMixin()
        try:
            firewall_db.subscribe()
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to load db subscribe in '
                          'huawei fwaas plugin: %s'), ex)
        LOG.info(_LI("[AC]Init Huawei fwaas plugin is successful."))

    def _get_routers_for_create_firewall(self, tenant_id, context, firewall):
        router_ids = firewall['firewall'].pop('router_ids', None)
        if router_ids == ATTR_NOT_SPECIFIED:
            l3_plugin = ncu.get_service_plugin()['L3_ROUTER_NAT']
            ctx = neutron_context.get_admin_context()
            routers = l3_plugin.get_routers(ctx)
            router_ids = []
            for router in routers:
                if router['tenant_id'] == tenant_id:
                    router_ids.append(router['id'])
            self.validate_firewall_routers_not_in_use(context, router_ids)
            return router_ids
        if not router_ids:
            return []
        self.validate_firewall_routers_not_in_use(context, router_ids)
        return router_ids

    def _ensure_update_firewall(self, context, firewall_id):
        fwall = self.get_firewall(context, firewall_id)
        try:
            # import firewall extensions for openstack and fusionsphere
            from neutron_fwaas.extensions.firewall import \
                FirewallInPendingState
        except ImportError:
            # import firewall extensions for fusionsphere
            from neutron_lib.exceptions.firewall_v1 import \
                FirewallInPendingState
        if fwall['status'] in [ac_const.NEUTRON_STATUS_PENDING_CREATE,
                               ac_const.NEUTRON_STATUS_PENDING_UPDATE,
                               ac_const.NEUTRON_STATUS_PENDING_DELETE]:
            raise FirewallInPendingState(
                firewall_id=firewall_id, pending_state=fwall['status'])

    def _ensure_update_firewall_policy(self, context, firewall_policy_id):
        firewall_policy = self.get_firewall_policy(context, firewall_policy_id)
        LOG.debug("Current firewall policy: %s", firewall_policy)
        if firewall_policy and 'firewall_list' in firewall_policy:
            for firewall_id in firewall_policy['firewall_list']:
                self._ensure_update_firewall(context, firewall_id)

    def _ensure_update_firewall_rule(self, context, firewall_rule_id):
        fw_rule = self.get_firewall_rule(context, firewall_rule_id)
        LOG.debug("Current firewall rule: %s", fw_rule)
        if 'firewall_policy_id' in fw_rule and fw_rule['firewall_policy_id']:
            self._ensure_update_firewall_policy(context,
                                                fw_rule['firewall_policy_id'])

    def get_firewall_rules(self, context, filters=None, fields=None,
                           sorts=None, limit=None, marker=None,
                           page_reverse=False, offset=None):
        """
        get_firewall_rules
        """
        if ncu.IS_FSP and self.ops_version not in [ac_const.FSP_6_1, ac_const.FSP_21_0]:
            firewall_rule_list = super(HuaweiACFwaasPlugin, self). \
                get_firewall_rules(context, filters=filters, fields=fields,
                                   sorts=sorts, limit=limit, marker=marker,
                                   page_reverse=page_reverse, offset=offset)
        else:
            firewall_rule_list = super(HuaweiACFwaasPlugin, self). \
                get_firewall_rules(context, filters, fields)
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            for firewall_rule in firewall_rule_list:
                firewall_rule_status = self.ac_proc_db_func.\
                    get_db_ac_proc_status(firewall_rule['id'])
                if firewall_rule_status:
                    firewall_rule['ac_process_state'] = firewall_rule_status.\
                        get('ac_proc_status', '')
                else:
                    firewall_rule['ac_process_state'] = ''

        if self._check_attribute_extension_open() and \
                self.ops_version in [ac_const.OPS_M, ac_const.OPS_K,
                                     ac_const.OPS_P]:
            for firewall_rule in firewall_rule_list:
                firewall_rule_addrs = self.firewall_rule_addrs_func.\
                    get_db_ac_firewall_rule(firewall_rule['id'])
                if firewall_rule_addrs:
                    firewall_rule['source_ports'] = \
                        firewall_rule_addrs.get('source_ports', '')
                    firewall_rule['destination_ports'] = \
                        firewall_rule_addrs.get('destination_ports', '')
                    firewall_rule['source_ip_addresses'] = \
                        firewall_rule_addrs.get('source_ip_addresses', '')
                    firewall_rule['destination_ip_addresses'] = \
                        firewall_rule_addrs.get('destination_ip_addresses', '')

        return firewall_rule_list

    def get_firewall_rule(self, context, filters=None, fields=None):
        """
        get_firewall_rule
        """
        res = super(HuaweiACFwaasPlugin, self).\
            get_firewall_rule(context, filters, fields)
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            firewall_rule_status = self.ac_proc_db_func.\
                get_db_ac_proc_status(res['id'])
            if firewall_rule_status:
                res['ac_process_state'] = firewall_rule_status.\
                    get('ac_proc_status', '')
            else:
                res['ac_process_state'] = ''

        if self._check_attribute_extension_open() and \
                self.ops_version in [ac_const.OPS_M, ac_const.OPS_K,
                                     ac_const.OPS_P]:
            firewall_rule_addresses = self.firewall_rule_addrs_func.\
                get_db_ac_firewall_rule(res['id'])
            if firewall_rule_addresses:
                res['source_ports'] = \
                    firewall_rule_addresses.get('source_ports', '')
                res['destination_ports'] = \
                    firewall_rule_addresses.get('destination_ports', '')
                res['source_ip_addresses'] = \
                    firewall_rule_addresses.get('source_ip_addresses', '')
                res['destination_ip_addresses'] = \
                    firewall_rule_addresses.get('destination_ip_addresses', '')

        return res

    def get_firewall_policies(self, context, filters=None, fields=None,
                              sorts=None, limit=None, marker=None,
                              page_reverse=False, offset=None):
        """
        get_firewall_policies
        """
        fw_policy_list = super(HuaweiACFwaasPlugin, self).\
            get_firewall_policies(context, filters, fields)
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            for fw_policy in fw_policy_list:
                fw_policy_status = self.ac_proc_db_func.\
                    get_db_ac_proc_status(fw_policy['id'])
                if fw_policy_status:
                    fw_policy['ac_process_state'] = fw_policy_status.\
                        get('ac_proc_status', '')
                else:
                    fw_policy['ac_process_state'] = ''
        return fw_policy_list

    def get_firewall_policy(self, context, filters=None, fields=None):
        """
        get_firewall_policy
        """
        res = super(HuaweiACFwaasPlugin, self).\
            get_firewall_policy(context, filters, fields)
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            fw_policy_status = self.ac_proc_db_func.\
                get_db_ac_proc_status(res['id'])
            if fw_policy_status:
                res['ac_process_state'] = fw_policy_status.\
                    get('ac_proc_status', '')
            else:
                res['ac_process_state'] = ''
        return res

    def get_firewalls(self, context, filters=None, fields=None,
                      sorts=None, limit=None, marker=None,
                      page_reverse=False, offset=None):
        """
        get_firewalls
        """
        fw_list = super(HuaweiACFwaasPlugin, self). \
            get_firewalls(context, filters, fields)
        for fw in fw_list:
            fw_current_rtrs = self.get_firewall_routers(context, fw['id'])
            fw['router_ids'] = fw_current_rtrs
        return fw_list

    def get_firewall(self, context, id, fields=None):
        """
        get_firewall
        """
        res = super(HuaweiACFwaasPlugin, self).get_firewall(context, id, fields)
        fw_current_rtrs = self.get_firewall_routers(context, id)
        res['router_ids'] = fw_current_rtrs
        return res

    def _make_firewall_dict_with_rules(self, context, firewall_id):
        firewall = self.get_firewall(context, firewall_id)
        fw_policy_id = firewall['firewall_policy_id']
        if fw_policy_id:
            fw_policy = self.get_firewall_policy(context, fw_policy_id)
            firewall['firewall_rule_list'] = [self.get_firewall_rule(
                context, rule_id) for rule_id in fw_policy['firewall_rules']]
        else:
            firewall['firewall_rule_list'] = []
        return firewall

    def create_firewall(self, context, firewall):
        """
        create_firewall
        """
        LOG.info(_LI("[AC]Begin to create firewall: %s "), firewall)

        if self.ops_version in \
                [ac_const.OPS_K, ac_const.OPS_L]:
            tenant_id = self._get_tenant_id_for_create(
                context, firewall['firewall'])
            fw_new_rtrs = self._get_routers_for_create_firewall(
                tenant_id, context, firewall)
        else:
            fw_new_rtrs = self._get_routers_for_create_firewall(
                firewall['firewall']['tenant_id'], context, firewall)
        fw = super(HuaweiACFwaasPlugin, self).create_firewall(
            context, firewall, ac_const.NEUTRON_STATUS_PENDING_CREATE)

        if not fw_new_rtrs:
            fw['router_ids'] = []
        else:
            fw['router_ids'] = fw_new_rtrs

            fw_with_rules = (self._make_firewall_dict_with_rules(
                context, fw['id']))
            fw_with_rtrs = {'fw_id': fw['id'], 'router_ids': fw_new_rtrs}
            self.set_routers_for_firewall(context, fw_with_rtrs)
            fw_with_rules['del-router-ids'] = []
            fw_with_rules['add-router-ids'] = fw_new_rtrs

        LOG.info(_LI("[AC]Create firewall in Neutron DB successful: %s "), fw)
        ac_osprofiler.record_chain_start("create firewall start,ID:" +
                                         fw['id'])

        try:
            self.fw_driver.create_firewall(context, fw)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."),
                      cause)
            ac_osprofiler.record_chain_exception_end("create firewall fail")
            fw = super(HuaweiACFwaasPlugin, self). \
                delete_firewall(context, fw['id'])
            LOG.info(_LI("[AC]Create firewall delete DB: %s "), fw)

            raise cause

        LOG.info(_LI("[AC]Create firewall is successful."))
        ac_osprofiler.record_chain_end_with_reason("create firewall success")
        return fw

    def delete_firewall(self, context, id):
        """
        delete_firewall
        """
        LOG.info(_LI("[AC]Begin to delete firewall: %s "), id)
        ac_osprofiler.record_chain_start("delete firewall start,ID:" + id)
        try:
            self.fw_driver.delete_firewall(context, id)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."),
                      cause)

            ac_osprofiler.record_chain_exception_end("delete firewall fail")
            raise cause

        fw_with_rules = (
            self._make_firewall_dict_with_rules(context, id))
        fw_with_rules['del-router-ids'] = self.get_firewall_routers(
            context, id)
        fw_with_rules['add-router-ids'] = []

        super(HuaweiACFwaasPlugin, self).delete_firewall(context, id)

        ac_osprofiler.record_chain_end_with_reason("delete firewall success")
        LOG.info(_LI("[AC]Delete firewall is successful."))

    def update_firewall(self, context, id, firewall):
        """
        update_firewall
        """
        LOG.info(_LI("[AC]Begin to update firewall: %s "), firewall)
        ac_osprofiler.record_chain_start("update firewall start,ID:" + id)
        original_fw = super(HuaweiACFwaasPlugin, self).get_firewall(context, id)

        self._ensure_update_firewall(context, id)
        router_ids = firewall['firewall'].pop('router_ids', None)
        fw_current_rtrs = self.get_firewall_routers(context, id)

        if router_ids is not None:
            if router_ids == []:
                fw_new_rtrs = []
            else:
                self.validate_firewall_routers_not_in_use(
                    context, router_ids, id)
                fw_new_rtrs = router_ids
        else:
            fw_new_rtrs = self.get_firewall_routers(context, id)

        if fw_new_rtrs == router_ids:
            self.update_firewall_routers(
                context, {'fw_id': id, 'router_ids': fw_new_rtrs})

        firewall['firewall']['status'] = ac_const.NEUTRON_STATUS_PENDING_UPDATE
        fw = super(HuaweiACFwaasPlugin, self).update_firewall(context, id, firewall)

        if not fw_new_rtrs and not fw_current_rtrs:
            fw['router_ids'] = []
        else:
            fw['router_ids'] = fw_new_rtrs

            fw_with_rules = (
                self._make_firewall_dict_with_rules(context, fw['id']))

            fw_with_rules['add-router-ids'] = fw_new_rtrs
            fw_with_rules['del-router-ids'] = list(
                set(fw_current_rtrs).difference(set(fw_new_rtrs)))

            fw_with_rules['last-router'] = not fw_new_rtrs

            LOG.info("update firewall: Add routers: %s, Del routers: %s",
                     fw_with_rules['add-router-ids'],
                     fw_with_rules['del-router-ids'])

        LOG.info(_LI("[AC]Update firewall is successful: %s "), fw)

        try:
            self.fw_driver.update_firewall(context, id, fw)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."), cause)
            ac_osprofiler.record_chain_exception("update firewall fail")
            original_fw['status'] = ac_const.NEUTRON_STATUS_ERROR
            super(HuaweiACFwaasPlugin, self).update_firewall(
                context, id, {'firewall': original_fw})

            raise cause

        LOG.info(_LI("[AC]Update firewall is successful."))
        ac_osprofiler.record_chain_end_with_reason("update firewall success")
        return fw

    def create_firewall_policy(self, context, firewall_policy):
        """
        create_firewall_policy
        """
        LOG.info(_LI("[AC]Begin to create firewall policy: %s "),
                 firewall_policy)

        fw = super(HuaweiACFwaasPlugin, self).create_firewall_policy(
            context, firewall_policy)
        LOG.info(_LI("[AC]Create firewall policy in Neutron "
                     "DB successful: %s "), fw)
        # create ac process status in db
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {
                'id': fw['id'],
                'type': 'firewall_policy',
            }
            ac_status_dict = self.ac_proc_db_func.create_db_ac_proc_status(
                device_dict)
            LOG.info(_LI("[AC] Creating firewall-policy, Create ac process "
                         "status in Neutron DB successfully: %s "),
                     ac_status_dict)

        ac_osprofiler.record_chain_start(
            "create firewall policy start,ID:" + fw['id'])
        try:
            self.fw_driver.create_firewall_policy(context, fw)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."),
                      cause)

            # if fail to create firewall-policy, delete ac process status in db
            if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
                self.ac_proc_db_func.delete_db_ac_proc_status(fw['id'])
                LOG.info(_LI("[AC] Creating firewall-policy failed, delete ac "
                             "process status in Neutron DB."))
            ac_osprofiler.record_chain_exception_end("create firewall policy "
                                                     "fail")
            fw_rollback = super(HuaweiACFwaasPlugin, self). \
                delete_firewall_policy(context, fw['id'])
            LOG.info(_LI("[AC]Create firewall policy delete DB: %s "),
                     fw_rollback)

            raise cause

        LOG.info(_LI("[AC]Create firewall policy is successful."))
        # if succeed to create firewall-policy, update ac process status in db
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {
                'id': fw['id'],
                'type': 'firewall_policy',
                'ac_proc_status': 'ACTIVE'
            }
            ac_status_dict = self.ac_proc_db_func.update_db_ac_proc_status(
                device_dict)
            LOG.info(_LI("[AC] Creating firewall-policy successfully, "
                         "update ac process status in Neutron DB: %s "),
                     ac_status_dict)
            fw['ac_process_state'] = ac_status_dict.get('ac_proc_status', '')
        ac_osprofiler.record_chain_end_with_reason("create firewall policy "
                                                   "success")
        return fw

    def delete_firewall_policy(self, context, id):
        """
        delete_firewall_policy
        """
        LOG.info(_LI("[AC]Begin to delete firewall policy: %s "), id)
        ac_osprofiler.record_chain_start("delete firewall policy start,ID:" +
                                         id)
        # before delete firewall-policy, update ac process status to 'pending'
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {
                'id': id,
                'type': 'firewall_policy',
                'ac_proc_status': 'PENDING'
            }
            ac_status_dict = self.ac_proc_db_func.update_db_ac_proc_status(
                device_dict)
            LOG.info(_LI("[AC] Deleting firewall-policy, Update ac process "
                         "status in Neutron DB successfully: %s "),
                     ac_status_dict)
        try:
            self.fw_driver.delete_firewall_policy(context, id)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."),
                      cause)
            # if fail to delete firewall-policy,
            # update ac process status to 'error'
            if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
                device_dict = {
                    'id': id,
                    'type': 'firewall_policy',
                    'ac_proc_status': 'ERROR'
                }
                ac_status_dict = self.ac_proc_db_func.update_db_ac_proc_status(
                    device_dict)
                LOG.info(_LI("[AC] Deleting firewall-policy failed, Update ac "
                             "process status in Neutron DB: %s "),
                         ac_status_dict)
            ac_osprofiler.record_chain_exception_end("update firewall policy "
                                                     "fail")
            raise cause

        fw = super(HuaweiACFwaasPlugin, self).delete_firewall_policy(
            context, id)

        LOG.info(_LI("[AC]Delete firewall policy is successful."))
        # if succeed to delete firewall-policy,
        # delete ac process status record in db
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            self.ac_proc_db_func.delete_db_ac_proc_status(id)
            LOG.info(_LI("[AC] Deleting firewall-policy successfully, delete "
                         "ac process status in Neutron DB."))
        ac_osprofiler.record_chain_end_with_reason("delete firewall policy "
                                                   "success")
        return fw

    def update_firewall_policy(self, context, id, firewall_policy):
        """
        update_firewall_policy
        """
        LOG.info(_LI("[AC]Begin to update firewall policy: %s "),
                 firewall_policy)
        ac_osprofiler.record_chain_start("update firewall policy start,ID:" +
                                         id)
        original_fw_policy = self.get_firewall_policy(context, id)

        self._ensure_update_firewall_policy(context, id)
        # before update firewall-policy, update ac process status to 'pending'
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {'id': id, 'type': 'firewall_policy',
                           'ac_proc_status': 'PENDING'}
            ac_status_dict = self.ac_proc_db_func.update_db_ac_proc_status(
                device_dict)
            LOG.info(_LI("[AC] Updating firewall-policy, Update ac process "
                         "status in Neutron DB successfully: %s "),
                     ac_status_dict)
        fw = super(HuaweiACFwaasPlugin, self).update_firewall_policy(
            context, id, firewall_policy)

        LOG.info(_LI("[AC]Update firewall policy is successful: %s "), fw)

        try:
            self.fw_driver.update_firewall_policy(context, id, fw)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."), cause)
            ac_osprofiler.record_chain_exception_end("update firewall policy "
                                                     "fail")
            # if fail to update firewall-policy,
            # update ac process status to 'error'
            if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
                device_dict = {'id': id, 'type': 'firewall_policy',
                               'ac_proc_status': 'ERROR'}
                ac_status_dict = self.ac_proc_db_func.update_db_ac_proc_status(
                    device_dict)
                LOG.info(_LI("[AC] Updating firewall-policy failed, Update ac "
                             "process status in Neutron DB successful: %s "),
                         ac_status_dict)
            fw = super(HuaweiACFwaasPlugin, self).update_firewall_policy(
                context, id, {'firewall_policy': original_fw_policy})

            raise cause

        LOG.info(_LI("[AC]Update firewall policy is successful."))
        # if succeed to update firewall-policy,
        # update ac process status to 'active'
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {'id': id, 'type': 'firewall_policy',
                           'ac_proc_status': 'ACTIVE'}
            ac_status_dict = self.ac_proc_db_func.update_db_ac_proc_status(
                device_dict)
            LOG.info(_LI("[AC] Updating firewall-policy successfully, "
                         "update ac process status in Neutron DB: %s "),
                     ac_status_dict)
            fw['ac_process_state'] = ac_status_dict.get('ac_proc_status', '')
        ac_osprofiler.record_chain_end_with_reason("update firewall policy "
                                                   "success")
        return fw

    @classmethod
    def has_extension_attributes(cls, firewall_rule):
        """
        has_extension_attributes
        """
        extension_attributes = ['source_ip_addresses',
                                'destination_ip_addresses',
                                'source_ports',
                                'destination_ports']
        has_extension_attributes = False
        rules = firewall_rule['firewall_rule']
        for extension_attribute in extension_attributes:
            if rules.get(extension_attribute, None):
                return True
        return has_extension_attributes

    @staticmethod
    def get_ip_address_str(ip_addresses):
        """get_ip_address_str"""
        ip_list = ''
        for ip_address in ip_addresses.split(','):
            if not ip_address.strip():
                continue
            if ip_list:
                ip_list = ip_list + ',' + ip_address.strip()
            if not ip_list:
                ip_list = ip_address.strip()

        return ip_list

    def check_fwr_src_dst_ips_str(self, firewall_rule):
        """
        check_fwr_src_dst_ips_str
        """
        src_ips = firewall_rule['firewall_rule']. \
            get('source_ip_addresses')
        if src_ips:
            if isinstance(src_ips, list):
                firewall_rule['firewall_rule']['source_ip_addresses'] = \
                    [src_ip.replace(',', '') for src_ip in src_ips]
            else:
                firewall_rule['firewall_rule']['source_ip_addresses'] = \
                    self.get_ip_address_str(src_ips)

        dst_ips = firewall_rule['firewall_rule']. \
            get('destination_ip_addresses')
        if dst_ips:
            if isinstance(dst_ips, list):
                firewall_rule['firewall_rule']['destination_ip_addresses'] = \
                    [dst_ip.replace(',', '') for dst_ip in dst_ips]
            else:
                firewall_rule['firewall_rule']['destination_ip_addresses'] = \
                    self.get_ip_address_str(dst_ips)
        return firewall_rule

    def _create_firewall_rule(self, context, firewall_rule, firewall):
        if self.ops_version in [ac_const.OPS_M, ac_const.OPS_K,
                                ac_const.OPS_P] and \
                self._check_attribute_extension_open() and \
                self.has_extension_attributes(firewall_rule):
            try:
                firewall_rule_addresses, ip_version = \
                    self.firewall_rule_addrs_func.create_db_ac_firewall_rule(
                        firewall_rule, firewall['id'])
            except Exception as cause:
                super(HuaweiACFwaasPlugin, self).delete_firewall_rule(
                    context, firewall['id'])

                self.firewall_rule_addrs_func.delete_db_ac_firewall_rule(
                    firewall['id'])

                raise cause

            firewall['ip_version'] = ip_version
            firewall['source_ports'] = firewall_rule_addresses.get(
                'source_ports', '')
            firewall['destination_ports'] = firewall_rule_addresses.get(
                'destination_ports', '')
            firewall['source_ip_addresses'] = firewall_rule_addresses.get(
                'source_ip_addresses', '')
            firewall['destination_ip_addresses'] = firewall_rule_addresses.get(
                'destination_ip_addresses', '')

    def create_firewall_rule(self, context, firewall_rule):
        """
        create_firewall_rule
        """
        firewall_rule = self._check_fw_rule_ip_version(firewall_rule)
        if not ncu.IS_FSP:
            firewall_rule = self.check_fwr_src_dst_ips_str(firewall_rule)

        LOG.info(_LI("[AC]Begin to create firewall rule: %s "), firewall_rule)

        firewall = super(HuaweiACFwaasPlugin, self).create_firewall_rule(
            context, firewall_rule)

        self._create_firewall_rule(context, firewall_rule, firewall)

        # create ac process status in db
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {'id': firewall['id'], 'type': 'firewall_rule'}
            ac_status_dict = self.ac_proc_db_func. \
                create_db_ac_proc_status(device_dict)
            LOG.info(_LI("[AC] Creating firewall-rule, Create ac process "
                         "status in Neutron DB successfully: %s "),
                     ac_status_dict)
        try:
            ac_osprofiler.record_chain_start(
                "create firewall rule start,ID:" + firewall['id'])
            self.fw_driver.create_firewall_rule(context, firewall)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."), cause)
            # if fail to create firewall-rule, delete ac process status in db
            if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
                self.ac_proc_db_func.delete_db_ac_proc_status(firewall['id'])
                LOG.info(_LI("[AC] Creating firewall-rule failed, delete ac "
                             "process status in Neutron DB."))

            firewall = super(HuaweiACFwaasPlugin, self).delete_firewall_rule(
                context, firewall['id'])
            if self.ops_version in \
                    [ac_const.OPS_M, ac_const.OPS_K, ac_const.OPS_P] \
                    and self._check_attribute_extension_open():
                if self._check_attribute_extension_open()\
                        and self.has_extension_attributes(firewall_rule)\
                        and firewall:
                    self.firewall_rule_addrs_func.delete_db_ac_firewall_rule(
                        firewall['id'])

            LOG.info(_LI("[AC]Create firewall rule delete DB: %s "), firewall)

            ac_osprofiler.record_chain_exception_end(
                "create firewall rule fail")
            raise cause

        ac_osprofiler.record_chain_end_with_reason(
            "create firewall rule success")
        # if succeed to create firewall-rule, update ac process status in db
        self._update_ac_status(firewall)
        LOG.info(_LI("[AC]Create firewall rule is successful."))
        return firewall

    def _update_ac_status(self, firewall):
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {
                'id': firewall['id'],
                'type': 'firewall_rule',
                'ac_proc_status': 'ACTIVE'
            }
            ac_status_dict = self.ac_proc_db_func. \
                update_db_ac_proc_status(device_dict)
            LOG.info(_LI("[AC] Creating firewall-rule successfully, "
                         "update ac process status in Neutron DB: %s "),
                     ac_status_dict)
            firewall['ac_process_state'] = \
                ac_status_dict.get('ac_proc_status', '')

    def delete_firewall_rule(self, context, rule_id):
        """
        delete_firewall_rule
        """
        LOG.info(_LI("[AC]Begin to delete firewall rule: %s "), rule_id)

        ac_osprofiler.record_chain_start(
            "delete firewall rule start,ID:" + rule_id)

        # before delete firewall-rule, update ac process status to 'pending'
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {'id': rule_id, 'type': 'firewall_rule',
                           'ac_proc_status': 'PENDING'}
            ac_status_dict = self.ac_proc_db_func. \
                update_db_ac_proc_status(device_dict)
            LOG.info(_LI("[AC] Deleting firewall-rule, Update ac process "
                         "status in Neutron DB successfully: %s "),
                     ac_status_dict)
        try:
            self.fw_driver.delete_firewall_rule(context, rule_id)
            if self.ops_version in \
                    [ac_const.OPS_M, ac_const.OPS_K, ac_const.OPS_P] \
                    and self._check_attribute_extension_open():
                self.firewall_rule_addrs_func.delete_db_ac_firewall_rule(
                    rule_id)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."), cause)
            # if fail to delete firewall-rule,
            # update ac process status to 'error'
            if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
                device_dict = {'id': rule_id, 'type': 'firewall_rule',
                               'ac_proc_status': 'ERROR'}
                ac_status_dict = self.ac_proc_db_func. \
                    update_db_ac_proc_status(device_dict)
                LOG.info(_LI("[AC] Deleting firewall-rule failed, Update ac "
                             "process status in Neutron DB: %s "),
                         ac_status_dict)
            ac_osprofiler.record_chain_exception_end("update firewall rule "
                                                     "fail")
            raise cause

        firewall = super(HuaweiACFwaasPlugin, self). \
            delete_firewall_rule(context, rule_id)

        LOG.info(_LI("[AC]Delete firewall rule is successful."))
        # if succeed to delete firewall-rule,
        # delete ac process status record in db
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            self.ac_proc_db_func.delete_db_ac_proc_status(rule_id)
            LOG.info(_LI("[AC] Deleting firewall-rule successfully, delete "
                         "ac process status in Neutron DB."))
        ac_osprofiler.record_chain_end_with_reason("delete firewall rule "
                                                   "success")
        return firewall

    def _pre_update_firewall_rule(self, context, rule_id, firewall_rule):
        firewall_rule = self._check_fw_rule_ip_version(firewall_rule)
        LOG.info(_LI("[AC]Begin to update firewall rule: %s "), firewall_rule)
        ac_osprofiler.record_chain_start(
            "update firewall rule start,ID:" + rule_id)
        original_fw_rule = self.get_firewall_rule(context, rule_id)

        self._ensure_update_firewall_rule(context, rule_id)

        new_fw_rule = firewall_rule['firewall_rule']
        return firewall_rule, original_fw_rule, new_fw_rule

    @classmethod
    def _check_attribute_extension_open(cls):
        return 'huawei_ac_firewall_rule' in cfg.CONF.ml2.extension_drivers

    def update_firewall_rule(self, context, rule_id, firewall_rule):
        """
        update_firewall_rule
        """
        firewall_rule, original_fw_rule, new_fw_rule = self.\
            _pre_update_firewall_rule(context, rule_id, firewall_rule)

        if self._check_attribute_extension_open() and \
                self.ops_version in [ac_const.OPS_M, ac_const.OPS_K,
                                     ac_const.OPS_P]:
            has_extension_attributes = self.has_extension_attributes(
                firewall_rule)
            new_firewall_rule = self.firewall_rule_addrs_func.\
                get_db_ac_firewall_rule(rule_id)
            if new_firewall_rule or has_extension_attributes:
                fw_rule = new_fw_rule
                new_fw_rule = self.firewall_rule_addrs_func.\
                    update_db_ac_firewall_rule(firewall_rule, original_fw_rule)
                for key, value in six.iteritems(fw_rule):
                    new_fw_rule[key] = value

        # before update firewall-rule, update ac process status to 'pending'
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {'id': rule_id, 'type': 'firewall_rule',
                           'ac_proc_status': 'PENDING'}
            ac_status_dict = self.ac_proc_db_func. \
                update_db_ac_proc_status(device_dict)
            LOG.info(_LI("[AC] Updating firewall-rule, Update ac process "
                         "status in Neutron DB successfully: %s "),
                     ac_status_dict)

        firewall = super(HuaweiACFwaasPlugin, self).update_firewall_rule(
            context, rule_id, {'firewall_rule': new_fw_rule})

        if self._check_attribute_extension_open() and \
                self.ops_version in [ac_const.OPS_M, ac_const.OPS_K,
                                     ac_const.OPS_P]:
            firewall = self._get_firewall_dict(firewall, new_fw_rule)

        try:
            LOG.info(_LI("[AC] Updating firewall-rule,fw is : %s "), firewall)
            self.fw_driver.update_firewall_rule(context, rule_id, firewall)

        except Exception as cause:
            self._deal_exc_in_update_fwr(context, rule_id,
                                         original_fw_rule, cause)
            raise cause
        ac_osprofiler.record_chain_end_with_reason("update firewall rule "
                                                   "success")
        # waiting for sync msg
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            ac_status_dict = self.ac_proc_db_func.get_db_ac_proc_status(id)
            firewall['ac_process_state'] = \
                ac_status_dict.get('ac_proc_status', '')
        LOG.info(_LI("[AC]Update firewall rule is successful."))
        return firewall

    @classmethod
    def _get_firewall_dict(cls, firewall, new_fw_rule):
        firewall['source_ports'] = new_fw_rule.get('source_ports', None)
        firewall['destination_ports'] = \
            new_fw_rule.get('destination_ports', None)
        firewall['source_ip_addresses'] = \
            new_fw_rule.get('source_ip_addresses', None)
        firewall['destination_ip_addresses'] = \
            new_fw_rule.get('destination_ip_addresses', None)
        return firewall

    def _deal_exc_in_update_fwr(self, context, rule_id,
                                original_fw_rule, cause):
        LOG.error(_LE("[AC] The exception is :%s."), cause)
        ac_osprofiler.record_chain_exception_end(
            "delete firewall rule fail")
        # if fail to update firewall-rule,
        # update ac process status to 'error'
        if ncu.need_ac_status(ac_const.NW_HW_AC_STATUS_FIREWALL_POLICY):
            device_dict = {'id': rule_id, 'type': 'firewall_rule',
                           'ac_proc_status': 'ERROR'}
            ac_status_dict = self.ac_proc_db_func. \
                update_db_ac_proc_status(device_dict)
            LOG.info(_LI("[AC] Updating firewall-rule failed, Update ac "
                         "process status in Neutron DB successful: %s "),
                     ac_status_dict)
        super(HuaweiACFwaasPlugin, self). \
            update_firewall_rule(context, rule_id,
                                 {'firewall_rule': original_fw_rule})

        if self._check_attribute_extension_open() and \
                self.ops_version in [ac_const.OPS_M, ac_const.OPS_K,
                                     ac_const.OPS_P]:
            self.firewall_rule_addrs_func.rollback_db_ac_firewall_rule(
                original_fw_rule)

    def _check_fw_rule_ip_version(self, firewall_rule):
        if self.ops_version in [ac_const.OPS_K, ac_const.OPS_L]:
            source_ip_address = firewall_rule['firewall_rule'].get(
                'source_ip_address', None)
            if source_ip_address:
                firewall_rule['firewall_rule']['ip_version'] = netaddr. \
                    IPNetwork(source_ip_address).version

            destination_ip_address = firewall_rule['firewall_rule'].get(
                'destination_ip_address', None)
            if destination_ip_address:
                firewall_rule['firewall_rule']['ip_version'] = netaddr. \
                    IPNetwork(destination_ip_address).version
        return firewall_rule

    @lock_db.wrap_db_lock(lock_db.RESOURCE_FW_POLICY)
    def insert_rule(self, context, id, rule_info):
        """
        insert_rule
        """
        LOG.info(_LI("[AC]Begin to insert firewall rule: %s "), rule_info)
        ac_osprofiler.record_chain_start("insert firewall rule start,ID:" + id)
        if 'insert_before' not in rule_info:
            rule_info['insert_before'] = ''

        if 'insert_after' not in rule_info:
            rule_info['insert_after'] = ''

        rest_info = {}
        if 'insert_before' in rule_info \
                and rule_info['insert_before'] is not None:
            rest_info['insertBefore'] = rule_info['insert_before']

        if 'insert_after' in rule_info \
                and rule_info['insert_after'] is not None:
            rest_info['insertAfter'] = rule_info['insert_after']
        rest_info['firewallRuleId'] = rule_info['firewall_rule_id']
        rest_info['id'] = id

        if not rule_info['insert_before'] and not rule_info['insert_after']:
            filters = {}
            filters['firewall_policy_id'] = [id]
            filters['position'] = ['1']
            firewall_rules = self.get_firewall_rules(context, filters, None)
            if firewall_rules:
                firewall_rule_first = firewall_rules[0]
                rest_info['insertBefore'] = firewall_rule_first['id']

        fw_policy = super(HuaweiACFwaasPlugin, self).insert_rule(
            context, id, rule_info)

        try:
            self.fw_driver.insert_firewall_rule(context, id, rest_info)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."),
                      cause)
            super(HuaweiACFwaasPlugin, self).remove_rule(context, id, rule_info)
            ac_osprofiler.record_chain_exception_end("insert firewall rule "
                                                     "fail")
            raise cause

        LOG.info(_LI("[AC]Insert firewall rule is successful."))
        ac_osprofiler.record_chain_end_with_reason("insert firewall rule "
                                                   "success")
        return fw_policy

    @lock_db.wrap_db_lock(lock_db.RESOURCE_FW_POLICY)
    def remove_rule(self, context, id, rule_info):
        """
        remove_rule
        """
        LOG.info(_LI("[AC]Begin to remove firewall rule: %s "), rule_info)
        ac_osprofiler.record_chain_start("delete firewall rule start,ID:" + id)

        original_fw_policy = self.get_firewall_policy(context, id)

        rest_info = {}
        rest_info['firewallRuleId'] = rule_info['firewall_rule_id']
        rest_info['id'] = id
        rest_info['insertBefore'] = rule_info.get('insert_before', '')
        rest_info['insertAfter'] = rule_info.get('insert_after', '')

        fw_policy = super(HuaweiACFwaasPlugin, self).remove_rule(
            context, id, rule_info)

        try:
            self.fw_driver.remove_firewall_rule(context, id, rest_info)

        except Exception as cause:
            LOG.error(_LE("[AC] The exception is :%s."),
                      cause)
            super(HuaweiACFwaasPlugin, self).update_firewall_policy(
                context, id, {'firewall_policy': original_fw_policy})
            raise cause

        LOG.info(_LI("[AC]Remove firewall rule is successful."))
        return fw_policy


class HuaweiACFwaasDriver(object):
    """Huawei AC Fwaas Service Driver."""

    def __init__(self):
        LOG.info(_LI("[AC]Init Huawei fwaas driver."))
        self.reliability_api = ACReliabilityAPI(ac_const.NW_HW_FWAAS)
        LOG.info(_LI("[AC]Init Huawei fwaas driver successfully."))

    def __rest_request__(self, context, res_id, entry_info, operation):
        self.reliability_api.update_plugin_record(
            context, res_id, entry_info, operation)

    def create_firewall(self, context, obj_dict):
        """
        create_firewall
        """
        LOG.debug("[AC]FW driver create firewall: %s", obj_dict)

        obj_info = ACFirewallModel.ac_model_format(obj_dict)

        LOG.debug("[AC]FW driver transform firewall successful: %s", obj_info)

        self.__rest_request__(context, obj_dict['id'], obj_info,
                              ac_const.NW_HW_CREATE_FIREWALL)

        LOG.debug("[AC]FW driver create firewall is successful.")

    def delete_firewall(self, context, id):
        """delete firewall by id"""
        LOG.debug("[AC]FW driver delete firewall: %s", id)
        self.__rest_request__(context, id, {},
                              ac_const.NW_HW_DELETE_FIREWALL)

        LOG.debug("[AC]FW driver delete firewall is successful.")

    def update_firewall(self, context, id, obj_dict):
        """
        update_firewall
        """
        LOG.debug("[AC]FW driver update firewall: %s", obj_dict)

        obj_info = ACFirewallModel.ac_model_format(obj_dict)

        LOG.debug("[AC]FW driver transform firewall successful: %s", obj_info)

        self.__rest_request__(context, id, obj_info,
                              ac_const.NW_HW_UPDATE_FIREWALL)

        LOG.debug("[AC]FW driver update firewall is successful.")

    def create_firewall_policy(self, context, obj_dict):
        """
        create_firewall_policy
        """
        LOG.debug("[AC]FW driver create firewall policy: %s", obj_dict)

        obj_info = ACFirewallPolicyModel.ac_model_format(obj_dict)

        LOG.debug(
            "[AC]FW driver transform firewall policy successful: %s", obj_info)

        self.__rest_request__(context, obj_dict['id'], obj_info,
                              ac_const.NW_HW_CREATE_FIREWALL_POLICY)

        LOG.debug("[AC]FW driver create firewall policy is successful.")

    def delete_firewall_policy(self, context, id):
        """
        delete_firewall_policy
        """
        LOG.debug("[AC]FW driver delete firewall policy: %s", id)

        self.__rest_request__(context, id, {},
                              ac_const.NW_HW_DELETE_FIREWALL_POLICY)

        LOG.debug("[AC]FW driver delete firewall policy is successful.")

    def update_firewall_policy(self, context, id, obj_dict):
        """
        update_firewall_policy
        """
        LOG.debug("[AC]FW driver update firewall policy: %s", obj_dict)

        obj_info = ACFirewallPolicyModel.ac_model_format(obj_dict)

        LOG.debug(
            "[AC]FW driver transform firewall policy successful: %s", obj_info)

        self.__rest_request__(context, id, obj_info,
                              ac_const.NW_HW_UPDATE_FIREWALL_POLICY)

        LOG.debug("[AC]FW driver update firewall policy is successful.")

    def create_firewall_rule(self, context, obj_dict):
        """
        create_firewall_rule
        """
        LOG.debug("[AC]FW driver create firewall rule: %s", obj_dict)

        obj_info = ACFirewallRuleModel.ac_model_format(obj_dict)

        LOG.debug(
            "[AC]FW driver transform firewall rule successful: %s", obj_info)

        self.__rest_request__(context, obj_dict['id'], obj_info,
                              ac_const.NW_HW_CREATE_FIREWALL_RULE)

        LOG.debug("[AC]FW driver create firewall rule is successful.")

    def delete_firewall_rule(self, context, id):
        """
        delete_firewall_rule
        """
        LOG.debug("[AC]FW driver delete firewall rule: %s", id)

        self.__rest_request__(context, id, {},
                              ac_const.NW_HW_DELETE_FIREWALL_RULE)

        LOG.debug("[AC]FW driver delete firewall rule is successful.")

    def update_firewall_rule(self, context, id, obj_dict):
        """
        update_firewall_rule
        """
        LOG.debug("[AC]FW driver update firewall rule: %s", obj_dict)

        obj_info = ACFirewallRuleModel.ac_model_format(obj_dict)

        LOG.debug(
            "[AC]FW driver transform firewall rule successful: %s", obj_info)

        self.__rest_request__(context, id, obj_info,
                              ac_const.NW_HW_UPDATE_FIREWALL_RULE)

        LOG.debug("[AC]FW driver update firewall rule is successful.")

    def insert_firewall_rule(self, context, id, obj_dict):
        """
        insert_firewall_rule
        """
        LOG.info("[AC]FW driver insert firewall rule: %s", obj_dict)

        obj_dict['tenant-name'] = context.tenant_name
        obj_info = ACFirewallPolicyModel.ac_model_format_act_rule(obj_dict)
        LOG.info(
            "[AC]FW driver transform firewall rule successful: %s", obj_info)

        self.__rest_request__(context, id, obj_info,
                              ac_const.NW_HW_INSERT_FIREWALL_RULE)

        LOG.debug("[AC]FW driver insert firewall rule is successful.")

    def remove_firewall_rule(self, context, id, obj_dict):
        """
        remove_firewall_rule
        """
        LOG.debug("[AC]FW driver remove firewall rule: %s", obj_dict)

        obj_dict['tenant-name'] = context.tenant_name
        obj_info = ACFirewallPolicyModel.ac_model_format_act_rule(obj_dict)
        LOG.debug(
            "[AC]FW driver transform firewall rule successful: %s", obj_info)

        self.__rest_request__(context, id, obj_info,
                              ac_const.NW_HW_REMOVE_FIREWALL_RULE)

        LOG.debug("[AC]FW driver remove firewall rule is successful.")
