#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ac gbp driver"""

import threading
from neutron.common import log

from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.model.target_group_model import \
    ACTargetGroupModel
from networking_huawei.drivers.ac.model.policy_classifier_model import \
    ACPolicyClassifierModel
from networking_huawei.drivers.ac.model.policy_rule_model import \
    ACPolicyRuleModel
from networking_huawei.drivers.ac.model.policy_rule_set_model import \
    ACPolicyRuleSetModel
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei._i18n import _LE
from networking_huawei._i18n import _LI
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.db.dbif import ACdbInterface

from gbpservice.neutron.services.grouppolicy import (
    group_policy_driver_api as api)
from gbpservice.neutron.db.grouppolicy import group_policy_db as gbp_db

LOG = logging.getLogger(__name__)
GLOBAL_DATA = threading.local()


class HuaweiACGBPDriver(api.PolicyDriver):
    """Huawei AC GBP Driver"""

    @log.log
    def initialize(self):
        """initialize"""
        self.reliability_api = ACReliabilityAPI(ac_constants.NW_HW_GBP)

    @log.log
    def create_policy_target_group_postcommit(self, context):
        """create policy target group postcommit"""
        try:
            LOG.info(_LI("[AC]Request AC to create target group: %s."),
                     context.current)

            target_group_to_ac = ACTargetGroupModel.ac_model_format(
                context.current)
            self.__rest_request__(context, context.current['id'],
                                  target_group_to_ac,
                                  ac_constants.NW_HW_CREATE_POLICY_TRAGE_TGROUP)

        except Exception as ex:
            LOG.error(_LE("[AC]Create policy target group: %s failed."
                          "exceptions,raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    @log.log
    def update_policy_target_group_precommit(self, context):
        """update policy target group precommit"""

        try:
            LOG.info(_LI("[AC]Request AC to update target group: %s."),
                     context.current)
            if hasattr(GLOBAL_DATA, 'req') and GLOBAL_DATA.req and \
                    context._plugin_context.request_id in GLOBAL_DATA.req:
                LOG.info(_LE("[AC] ignore the update request when delete "
                             "policy target group ."))
                GLOBAL_DATA.req.remove(context._plugin_context.request_id)
                return

            target_group_to_ac = ACTargetGroupModel.ac_model_format(
                context.current)
            self.__rest_request__(context, context.current['id'],
                                  target_group_to_ac,
                                  ac_constants.NW_HW_UPDATE_POLICY_TRAGE_TGROUP)

        except Exception as ex:
            LOG.error(_LE("[AC]Update policy target group: %s failed."
                          "exceptions,raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    @log.log
    def delete_policy_target_group_precommit(self, context):
        """delete policy target group precommit"""
        GLOBAL_DATA.req = []
        GLOBAL_DATA.req.append(context._plugin_context.request_id)
        try:
            LOG.info(_LI("[AC]Request AC to delete target group: %s."),
                     context.current)

            target_group_to_ac = ACTargetGroupModel.ac_model_format(
                context.current)
            self.__rest_request__(context, context.current['id'],
                                  target_group_to_ac,
                                  ac_constants.NW_HW_DELETE_POLICY_TRAGE_TGROUP)

        except Exception as ex:
            LOG.error(_LE("[AC]Delete policy target group: %s failed."
                          "exceptions,raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    @log.log
    def create_policy_classifier_postcommit(self, context):
        """create policy classifier postcommit"""
        try:
            LOG.info(_LI("[AC]Request AC to create classifier: %s."),
                     context.current)
            tenant_name = ACKeyStoneIf.get_tenant_name(context.current,
                                                       context._plugin_context)
            classifier_to_ac = ACPolicyClassifierModel.ac_model_format(
                context.current, tenant_name)
            self.__rest_request__(context, context.current['id'],
                                  classifier_to_ac,
                                  ac_constants.NW_HW_CREATE_POLICY_CLASSIFIER)

        except Exception as ex:
            LOG.error(_LE("[AC]Create policy classifier: %s failed."
                          "exceptions,raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    @log.log
    def update_policy_classifier_postcommit(self, context):
        """update policy classifier postcommit"""
        try:
            LOG.info(_LI("[AC]Request AC to update classifier: %s."),
                     context.current)
            tenant_name = ACKeyStoneIf.get_tenant_name(context.current,
                                                       context._plugin_context)
            classifier_to_ac = ACPolicyClassifierModel.ac_model_format(
                context.current, tenant_name)
            self.__rest_request__(context, context.current['id'],
                                  classifier_to_ac,
                                  ac_constants.NW_HW_UPDATE_POLICY_CLASSIFIER)

        except Exception as ex:
            LOG.error(_LE("[AC]Update policy classifier: %s failed."
                          "exceptions,raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    @log.log
    def delete_policy_classifier_precommit(self, context):
        """delete policy classifier precommit"""
        try:
            LOG.info(_LI("[AC]Request AC to delete classifier: %s."),
                     context.current)

            classifier_to_ac = ACPolicyClassifierModel.ac_model_format(
                context.current, "")
            self.__rest_request__(context, context.current['id'],
                                  classifier_to_ac,
                                  ac_constants.NW_HW_DELETE_POLICY_CLASSIFIER)

        except Exception as ex:
            LOG.error(_LE("[AC]Delete policy classifier: %s failed."
                          "exceptions,raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    @log.log
    def create_policy_rule_postcommit(self, context):
        """create policy rule postcommit"""
        self._create_or_update_policy_rule(context, True)

    def _create_or_update_policy_rule(self, context, is_create_or_update):
        """fix:重复代码"""
        if is_create_or_update:
            action = 'create'
            operation = ac_constants.NW_HW_CREATE_POLICY_RULE
        else:
            action = 'update'
            operation = ac_constants.NW_HW_UPDATE_POLICY_RULE
        LOG.info(_LI("[AC]Request AC to %s policy rule: %s."), action,
                 context.current)
        try:
            policy_types = []
            if 'policy_actions' in context.current \
                    and context.current['policy_actions'] is not None:
                policy_action_ids = context.current['policy_actions']
                policy_types = self._get_actions_by_ids(policy_action_ids)
            policy_rule_to_ac = ACPolicyRuleModel.ac_model_format(
                context.current, policy_types)
            self.__rest_request__(context, context.current['id'],
                                  policy_rule_to_ac, operation)
        except Exception as ex:
            LOG.error(_LE("[AC]%s policy rule: %s failed.exceptions,raise "
                          "exception to plugin to rollback."), action.title(),
                      context.current['id'])
            LOG.error(_LE("[AC]The exception is :%s."), ex)
            raise

    @log.log
    def update_policy_rule_postcommit(self, context):
        """update policy rule postcommit"""
        self._create_or_update_policy_rule(context, False)

    @log.log
    def delete_policy_rule_precommit(self, context):
        """delete policy rule precommit"""
        try:
            LOG.info(_LI("[AC]Request AC to delete policy rule: %s."),
                     context.current)

            policy_rule_to_ac = ACPolicyRuleModel.ac_model_format(
                context.current, [])
            self.__rest_request__(context, context.current['id'],
                                  policy_rule_to_ac,
                                  ac_constants.NW_HW_DELETE_POLICY_RULE)

        except Exception as ex:
            LOG.error(_LE("[AC]Update policy rule: %s failed.exceptions,raise "
                          "exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    @log.log
    def create_policy_rule_set_postcommit(self, context):
        """create policy rule set postcommit"""
        try:
            LOG.info(_LI("[AC]Request AC to create policy rule set: %s."),
                     context.current)

            policy_rule_set_to_ac = ACPolicyRuleSetModel.ac_model_format(
                context.current)
            self.__rest_request__(context, context.current['id'],
                                  policy_rule_set_to_ac,
                                  ac_constants.NW_HW_CREATE_POLICY_RULE_SET)

        except Exception as ex:
            LOG.error(_LE("[AC]Create policy rule set: %s failed.exceptions,"
                          "raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    @log.log
    def update_policy_rule_set_postcommit(self, context):
        """update policy rule set postcommit"""
        try:
            LOG.info(_LI("[AC]Request AC to update policy rule set: %s."),
                     context.current)

            policy_rule_set_to_ac = ACPolicyRuleSetModel.ac_model_format(
                context.current)
            self.__rest_request__(context, context.current['id'],
                                  policy_rule_set_to_ac,
                                  ac_constants.NW_HW_UPDATE_POLICY_RULE_SET)

        except Exception as ex:
            LOG.error(_LE("[AC]Update policy rule set: %s failed.exceptions,"
                          "raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    @log.log
    def delete_policy_rule_set_precommit(self, context):
        """delete policy rule set precommit"""
        try:
            LOG.info(_LI("[AC]Request AC to delete policy rule set: %s."),
                     context.current)

            policy_rule_set_to_ac = ACPolicyRuleSetModel.ac_model_format(
                context.current)
            self.__rest_request__(context, context.current['id'],
                                  policy_rule_set_to_ac,
                                  ac_constants.NW_HW_DELETE_POLICY_RULE_SET)

        except Exception as ex:
            LOG.error(_LE("[AC]Delete policy rule set: %s failed.exceptions, "
                          "raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            raise

    def __rest_request__(self, context, res_id, entry_info, operation):
        self.reliability_api.update_plugin_record(context,
                                                  res_id,
                                                  entry_info,
                                                  operation)

    @classmethod
    def _get_actions_by_ids(cls, action_ids):
        session = ACdbInterface().get_session()
        actions = []
        for action_id in action_ids:
            action = session.query(gbp_db.PolicyAction).filter_by(
                id=action_id).first()
            actions.append(action["action_type"])
        return actions
