#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ml2 driver"""

import ast
import copy
import datetime
import os
import threading
import time
import traceback
from collections import namedtuple

import eventlet
import netaddr
from neutron.db import db_base_plugin_v2
from neutron.db import models_v2
from neutron.db.models_v2 import IPAllocation
from neutron.plugins.ml2 import db as ml2db
from neutron.plugins.ml2 import models
from oslo_config import cfg
from oslo_serialization import jsonutils
from six.moves import range

from networking_huawei._i18n import _LE
from networking_huawei._i18n import _LI
from networking_huawei.common import exceptions as ml2_exc
from networking_huawei.drivers.ac.ac_agent.ac_agent_api import ACAgentAPI
from networking_huawei.drivers.ac.ac_agent.rpc_agent import RpcAgent
from networking_huawei.drivers.ac.client.restclient import ACReSTClient
from networking_huawei.drivers.ac.client.service import ACReSTService, DryRunStateEnum
from networking_huawei.drivers.ac.common import constants as ac_cnst, neutron_version_util
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as ac_osprofiler
from networking_huawei.drivers.ac.common.neutron_compatible_util import ac_ml2_api as api
from networking_huawei.drivers.ac.common.util import ACCommonUtil, DataFilterUtil
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.db.ac_proc_status.ac_proc_status_db import ACProStatusDbMixin
from networking_huawei.drivers.ac.db.allocate_vlan.schema import AllocateVlanSchema
from networking_huawei.drivers.ac.db.exroute.exroute import RouterExRoute
from networking_huawei.drivers.ac.db.lock import lock_model as lock_models
from networking_huawei.drivers.ac.db.lock import lockedobjects_db as lock_db
from networking_huawei.drivers.ac.db.schema import ACPluginSchema, ACFailedResources
from networking_huawei.drivers.ac.extensions.exroute.l3_exroutes import ExRouteMultiBfdLeader
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.model.network_model import ACNetworkModel
from networking_huawei.drivers.ac.model.port_model import ACPortModel
from networking_huawei.drivers.ac.model.subnet_model import ACSubnetModel
from networking_huawei.drivers.ac.plugins.dry_run import dry_run_util
from networking_huawei.drivers.ac.plugins.flow_mirror.flow_mirror_plugin import HuaweiFlowMirrorPlugin
from networking_huawei.drivers.ac.plugins.ml2.driver_port_util import MechanismDriverPortUtil as ml2_port_util
from networking_huawei.drivers.ac.plugins.ml2.driver_security_util import MechanismDriverSecurityGroupUtil \
    as ml2_security_util
from networking_huawei.drivers.ac.plugins.ml2.driver_util import MechanismDriverUtil as ml2_util
from networking_huawei.drivers.ac.plugins.ml2.pubservice_driver import PublicService
from networking_huawei.drivers.ac.plugins.qos import ac_qos_extension_driver
from networking_huawei.drivers.ac.sync import http_heart_beat
from networking_huawei.drivers.ac.sync import util
from networking_huawei.drivers.ac.sync.message_reliability_api import ACReliabilityAPI
from networking_huawei.drivers.ac.sync.message_reliability_driver import ACMessageReliabilityDriver

try:
    from neutron.db import api as db_api
except ImportError:
    from neutron_lib.db import api as db_api
try:
    from neutron.common.exceptions import BadRequest, PortNotFound
except ImportError:
    from neutron_lib.exceptions import BadRequest, PortNotFound

try:
    from neutron.services.trunk import constants as trunkport_const

    TRUNK_CALLBACK = True
except ImportError:
    try:
        from neutron_lib.callbacks import resources as trunkport_const

        TRUNK_CALLBACK = True
    except ImportError:
        TRUNK_CALLBACK = False

LOG = ncu.ac_log.getLogger(__name__)

CREATE_SUBNET = threading.local()
DELETED_NETWORK = threading.local()
UPDATE_SUBNET = threading.local()
PRIMARY_INTERFACE = 'primary_interface'
VIF_TYPE_VHOST_USER = 'vhostuser'
DEVICE_OWNER_LB = 'neutron:LOADBALANCER'
NEUTRON_CONF = os.path.realpath('/etc/neutron/neutron.conf')

OPS_VERSIONS = {ac_cnst.OPS_O, ac_cnst.OPS_P, ac_cnst.OPS_Q, ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.FSP_6_5,
                ac_cnst.FSP_21_0, ac_cnst.OPS_W}
# 部分函数传参使用
IdArgs = namedtuple('IdArgs', ('tenant_id', 'network_id', 'subnet_id'))


def _check_port_during_deleting(context):
    obj_id = context.current['id'] + '_delete_port'
    result = context._plugin_context.session.query(
        lock_models.DBLockedObjects).filter_by(object_uuid=obj_id, lock=1).all()
    if result:
        return True
    return False


def _get_dry_run_state(data, data_type):
    LOG.info('[DryRun]data_type=%s,data=%s', data_type, data)
    if data is None:
        dry_run_state = DryRunStateEnum.NO_EXIST
    else:
        if dry_run_util.is_dry_run_data(data, data_type):
            dry_run_state = DryRunStateEnum.DRY_RUN_DATA
        else:
            dry_run_state = DryRunStateEnum.REAL_DATA
    return dry_run_state


class HuaweiACMechanismDriver(api.MechanismDriver):
    """huawei ml2 driver"""

    def __init__(self, *args, **kwargs):
        self.ac_msg_rel_driver = None
        self.flow_mirror_plugin = None
        self.reliability_api = None
        self.db_if = None
        self.ac_proc_db_func = None
        self.rpc_agent = None
        self.rest_service = None
        self.client = None
        self.db_base_plugin = None
        self.data_filter = None
        self.http_heart_beat = None
        self.physical_network = cfg.CONF.huawei_ac_config.physical_network
        self.vhost_user = cfg.CONF.huawei_ac_config.vhost_user
        self.port = ac_cnst.rest_server_port
        self.ops_version = ncu.get_ops_version()
        self.logical_hostnames = cfg.CONF.huawei_ac_config.logical_hostnames
        self.primary_interface = cfg.CONF.huawei_ac_config.primary_interface
        self.enable_ce1800v_dhcp = cfg.CONF.huawei_ac_config.enable_ce1800v_dhcp

        self.vif_details = None
        self.context_manager = None
        self.host = None
        self.url = None
        self.bind_port_url = None
        self.vcenter_network_info = None
        self.vcenter_host_info = None
        self.vcenter_cluster_info = None
        self.LB_host_mappings = None
        # 不知道父类初始化函数怎么写的，以防父类有特殊操作，最后调用父类初始化函数
        super(HuaweiACMechanismDriver, self).__init__(*args, **kwargs)

    def initialize(self):
        """Init huawei ml2 driver"""
        LOG.info(_LI("[AC]Init huawei ml2 driver."))
        if ncu.IS_FSP:
            ml2_util.get_cps_info()
            ml2_util.get_fs_ssl_cfg()
        dbif.ACdbInterface.get_neutron_read_dict()
        self._init_ml2_config()
        self.ac_msg_rel_driver = ACMessageReliabilityDriver()
        self.flow_mirror_plugin = HuaweiFlowMirrorPlugin()
        self.reliability_api = ACReliabilityAPI(ac_cnst.NW_HW_ML2)
        self.db_if = dbif.ACdbInterface()
        if self.ops_version in [ac_cnst.OPS_T, ac_cnst.OPS_W, ac_cnst.FSP_21_0]:
            from neutron_lib.api.definitions import portbindings
            self.vif_details = {portbindings.VIF_DETAILS_CONNECTIVITY: portbindings.CONNECTIVITY_L2}
        if hasattr(db_api, "context_manager"):
            self.context_manager = db_api.context_manager
        elif hasattr(db_api, "get_context_manager"):
            self.context_manager = db_api.get_context_manager()
        else:
            self.context_manager = None

        self.ac_proc_db_func = ACProStatusDbMixin()
        time.sleep(ac_cnst.DELAY_TIME_TO_INITIAL_RPC_AGENT)
        self.rpc_agent = RpcAgent()

        self.rest_service = ACReSTService()
        self.client = ACReSTClient()
        self.db_base_plugin = db_base_plugin_v2.NeutronDbPluginV2()
        if self.ops_version in [ac_cnst.FSP_6_5, ac_cnst.FSP_21_0]:
            from networking_huawei.drivers.ac.plugins.qos import ac_qos_driver
            ac_qos_driver.register()
        self.data_filter = DataFilterUtil()
        self.http_heart_beat = http_heart_beat.HttpHeatBeat2AC()
        self._register_security_group()
        ml2_port_util.register_port()
        # subscribe trunk precommit delete event
        if TRUNK_CALLBACK:
            ncu.registry.subscribe(self.update_parent_subports_in_precommit_delete_trunk, trunkport_const.TRUNK,
                                   ncu.events.PRECOMMIT_DELETE)
        # check if cmcc qos driver need to init, using for neutron_sync add cmcc qos
        ac_qos_extension_driver.check_init_cmcc_huawei_qos_driver()
        LOG.info(_LI("[AC]Initialization finished successfully for huawei ml2 driver."))

    def _init_ml2_config(self):
        self.physical_network = cfg.CONF.huawei_ac_config.physical_network
        self.vhost_user = cfg.CONF.huawei_ac_config.vhost_user
        host_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip.replace(' ', '').split(',')
        if len(host_list) == 1:
            self.host = host_list[0]
        else:
            self.host = ac_cnst.DEFAULT_AC_IP
        self.port = ac_cnst.rest_server_port
        self.url = '%s%s%s%s' % (ac_cnst.HTTPS_HEADER, self.host, ":", str(self.port))
        self.bind_port_url = self.url + ac_cnst.BIND_PORT_SUFFIX
        self.ops_version = ncu.get_ops_version()
        LOG.info(_LI("[AC] ops version is %s"), self.ops_version)
        self.vcenter_network_info = ml2_util.parse_vcenter_network_mappings(
            cfg.CONF.huawei_ac_config.vcenter_network_mappings)
        self.vcenter_host_info = ml2_util.parse_vcenter_host_mappings(
            cfg.CONF.huawei_ac_config.vcenter_host_mappings)
        self.vcenter_cluster_info = ml2_util.parse_vcenter_host_mappings(
            cfg.CONF.huawei_ac_config.vcenter_cluster_mappings)
        self.logical_hostnames = cfg.CONF.huawei_ac_config.logical_hostnames
        self.primary_interface = cfg.CONF.huawei_ac_config.primary_interface
        self.LB_host_mappings = ACPortModel.format_f5_mappings(
            cfg.CONF.huawei_ac_config.LB_host_mappings)
        LOG.debug("LB_host_mappings is %s" % self.LB_host_mappings)
        self.enable_ce1800v_dhcp = cfg.CONF.huawei_ac_config.enable_ce1800v_dhcp

    def _register_security_group(self):
        if cfg.CONF.huawei_ac_config.enable_security_group:
            ncu.registry.subscribe(ml2_security_util.create_security_group, ncu.resources.SECURITY_GROUP,
                                   ncu.events.AFTER_CREATE)
            ncu.registry.subscribe(ml2_security_util.update_security_group, ncu.resources.SECURITY_GROUP,
                                   ncu.events.AFTER_UPDATE)
            ncu.registry.subscribe(ml2_security_util.delete_security_group, ncu.resources.SECURITY_GROUP,
                                   ncu.events.AFTER_DELETE)
            ncu.registry.subscribe(ml2_security_util.validate_security_group_rule, ncu.resources.SECURITY_GROUP_RULE,
                                   ncu.events.BEFORE_CREATE)
            ncu.registry.subscribe(ml2_security_util.create_security_group_rule, ncu.resources.SECURITY_GROUP_RULE,
                                   ncu.events.AFTER_CREATE)
            ncu.registry.subscribe(ml2_security_util.delete_security_group_rule, ncu.resources.SECURITY_GROUP_RULE,
                                   ncu.events.AFTER_DELETE)

            security_group_sync_times = self.db_if.get_config_record('security_group_sync_times')
            if not security_group_sync_times:
                security_group_sync_times = self.db_if.create_config_record('security_group_sync_times', 0)
            if security_group_sync_times.value < cfg.CONF.huawei_ac_config.security_group_sync_times:
                try:
                    default_security_groups_thread = threading.Thread(
                        target=ml2_security_util.sync_default_security_groups)
                    default_security_groups_thread.start()
                    self.db_if.update_config_record(security_group_sync_times.key, security_group_sync_times.value + 1)
                except Exception as ex:
                    LOG.error(_LE('[AC] Failed to synchronise default security groups in huawei driver: %s'), ex)

    def _set_update_time(self, context, res_type):
        # if has the parameters create_at and update_at
        # update the timestamp in db when updating port
        if context.current.get('created_at') and context.current.get('updated_at'):
            current_time = ACCommonUtil.get_standard_current_time_utc()
            context.current['updated_at'] = '%s%s' % (current_time.strftime(ac_cnst.ISO8601_TIME_FORMAT),
                                                      'Z' if self.ops_version == ac_cnst.OPS_P else '')
            self.db_if.update_neutron_db_update_time(context._plugin_context.session, context.current['id'],
                                                     current_time, res_type)
        else:
            LOG.debug("No need to set update time for %s", context.current)

    def _is_vmware_port(self, context):
        current = context.current
        original = context.original
        port_id = current.get('id')

        host_id = current.get('binding:host_id', None)
        if host_id in self.vcenter_host_info:
            LOG.info(_LI('[AC] %s is in logical host mappings.'), port_id)
            return True

        try:
            port_vnic_type = current.get(ncu.VNIC_TYPE) if current else None
            if port_vnic_type == 'vcenter':
                LOG.info(_LI('[AC] %s vnic type is vcenter.'), port_id)
                return True
        except Exception as ex:
            LOG.error(_LE("[AC]%s vnic type get failed: %s"), port_id, ex)

        try:
            current_vif_details = current.get(ncu.VIF_DETAILS) if current else {}
            original_vif_details = original.get(ncu.VIF_DETAILS) if original else {}
            if (not current_vif_details) and (not original_vif_details):
                LOG.info(_LI('[AC]%s current and original vif details are both None'), port_id)
                return False

            current_port_vif_type = current_vif_details.get('port_vif_type')
            original_port_vif_type = original_vif_details.get('port_vif_type')
            if (not current_port_vif_type) and (not original_port_vif_type):
                LOG.info(_LI('[AC]%s current and original port vif type are both None'), port_id)
                return False

            if current_port_vif_type == 'vcenter':
                LOG.info(_LI('[AC]%s current port vif type is vcenter'), port_id)
                return True
            if original_port_vif_type == 'vcenter':
                LOG.info(_LI('[AC]%s original port vif type is vcenter'), port_id)
                return True

            LOG.info(_LI('[AC]%s current and original port vif type are not vcenter neither.'), port_id)
            return False

        except Exception as ex:
            LOG.error(_LE("[AC]%s port vif type get failed: %s"), port_id, ex)
            return False

    def create_network_precommit(self, context):
        """create network precommit"""
        self.reliability_api.check_neutron_sync(context, ac_cnst.OPER_CREATE, ac_cnst.NW_HW_NETWORKS,
                                                context.current['id'])
        if not self.data_filter.not_in_white_or_in_black_list(context._plugin_context, context.current,
                                                              ac_cnst.NW_HW_NETWORKS, context.current['id']):
            self.http_heart_beat.block_send_request_to_ac()

    @lock_db.wrap_db_lock(lock_db.DELETE_NETWORK_LOCK)
    def create_network_postcommit(self, context):
        """This function sends network create message to AC

        :param context: DB context for the network creation
        :return: None
        """
        try:
            ac_osprofiler.record_chain_start("create network start,ID: " + context.current['id'])
            LOG.info(_LI("[AC]Request AC to create network: %s."), context.current)

            if self.data_filter.not_in_white_or_in_black_list(context._plugin_context, context.current,
                                                              ac_cnst.NW_HW_NETWORKS, context.current['id']):
                return

            tenant_name = ACKeyStoneIf.get_tenant_name(context.current, context._plugin_context)
            network_info = ACNetworkModel.ac_model_format(context.current, tenant_name)
            self.__rest_request__(context, context.current['id'], network_info, ac_cnst.NW_HW_CREATE_NETWORK)

        except Exception as ex:
            LOG.error(_LE("[AC]Create network: %s failed. exceptions, raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            ac_osprofiler.record_chain_exception_end(str(ex))
            self.raise_ml2_exception(ex, 'network')
        ac_osprofiler.record_chain_end_with_reason("create network success")

    def _delete_dhcp_ports(self, net_id):
        LOG.info(_LI("[AC] Begin to delete dhcp ports for: %s"), net_id)
        admin_ctx = ncu.neutron_context.get_admin_context()
        with self.context_manager.reader.using(admin_ctx):
            cond = models_v2.Port.device_owner.in_([ncu.DEVICE_OWNER_DHCP])
            auto_delete_port_ids = []
            for elem in admin_ctx.session.query(models_v2.Port.id).filter_by(network_id=net_id).filter(cond):
                auto_delete_port_ids.append(elem.id)
        for port_id in auto_delete_port_ids:
            try:
                LOG.info(_LI("[AC] Begin to delete dhcp port: %s"), port_id)
                port = self.db_base_plugin.get_port(admin_ctx, port_id)
                port_info = ACPortModel.ac_model_format(port, str(port['tenant_id']))
                dry_run_state = _get_dry_run_state(port_info, ac_cnst.NW_HW_PORTS)
                self.reliability_api.update_plugin_record(admin_ctx, port_id, port_info, ac_cnst.NW_HW_DELETE_PORT,
                                                          dry_run_state=dry_run_state)
            except PortNotFound:
                LOG.debug("Ignoring PortNotFound when deleting port '%s'.The port has already been deleted.", port_id)

    @lock_db.wrap_db_lock(lock_db.DELETE_NETWORK_LOCK)
    def delete_network_precommit(self, context):
        """This function sends network delete message to AC

        :param context: DB context for the network delete
        :return: None
        """
        try:
            ac_osprofiler.record_chain_start("delete network: " + context.current['id'])
            LOG.info(_LI("[AC]Request AC to delete network:%s."), context.current)
            if self.data_filter.not_in_white_or_in_black_list(
                    context._plugin_context, context.current, ac_cnst.NW_HW_NETWORKS, context.current['id']):
                return

            self._check_network_in_use(context)
            if self.ops_version in [ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W, ac_cnst.FSP_21_0]:
                self._delete_dhcp_ports(context.current['id'])
            network_info = ACNetworkModel.ac_model_format(
                context.current, str(context.current['tenant_id']))

            self.__rest_request__(context, context.current['id'], network_info, ac_cnst.NW_HW_DELETE_NETWORK)

            if self.ops_version in OPS_VERSIONS.union({ac_cnst.OPS_N, }):
                DELETED_NETWORK.mapping = {context.current['id']: datetime.datetime.utcnow()}

            self._delete_network_allocate_vlan(context)
        except ml2_exc.ResourceReferenceException as ex_ref:
            self._delete_network_reference_exp(context, network_info, ex_ref)
        except Exception as ex:
            LOG.error(_LE("[AC]Delete network: %s failed. exceptions, raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            ac_osprofiler.record_chain_exception_end(str(ex))
            self.raise_ml2_exception(ex, ac_cnst.NW_HW_NETWORKS)
        ac_osprofiler.record_chain_end_with_reason("delete network success")

    def _delete_network_allocate_vlan(self, context):
        if 'vxlan_physnet_prefix' in context.current and self.context_manager:
            ctx = context._plugin_context
            allocations = ctx.session.query(AllocateVlanSchema).filter(
                AllocateVlanSchema.network_id == context.current['id']).all()
            with self.context_manager.writer.using(ctx):
                for allocation in allocations:
                    ctx.session.delete(allocation)

    def _delete_network_reference_exp(self, context, network_info, ex_ref):
        error_msg = "when delete network, AC return reference error, it may dhcp port ,try it again"
        retry_time = 0
        while retry_time < ac_cnst.NETWORK_DELETE_RETRY_TIME:
            try:
                ac_osprofiler.record_chain_exception_end(error_msg)
                LOG.info("[AC]" + error_msg)
                self.__rest_request__(context, context.current['id'], network_info, ac_cnst.NW_HW_DELETE_NETWORK)
                ac_osprofiler.record_chain_end_with_reason("delete network success")
                if self.ops_version in OPS_VERSIONS.union({ac_cnst.OPS_N, }):
                    DELETED_NETWORK.mapping = {context.current['id']: datetime.datetime.utcnow()}
                return
            except ml2_exc.ResourceReferenceException:
                retry_time = retry_time + 1
                time.sleep(ac_cnst.WAIT_AC_DELETE_DHCP_TIME)
            except Exception as ex:
                ex_ref = ex
                break
        LOG.error(_LE("[AC]Delete network: %s failed. exceptions: %s,raise exception to plugin to rollback."),
                  context.current['id'], ex_ref)
        ac_osprofiler.record_chain_exception_end(str(ex_ref))
        self.raise_ml2_exception(ex_ref, ac_cnst.NW_HW_NETWORKS)

    def _check_network_in_use(self, context):
        if ac_cnst.NW_HW_L2BR in ac_cnst.NW_HW_NEUTRON_SYNC_SUPPORT_RES:
            from networking_huawei.drivers.ac.db.l2br import l2br_db
            db_l2br = l2br_db.L2brDbMixin()
            l2brs = db_l2br.get_db_l2brs(context._plugin_context)
            network = context.current

            for l2br in l2brs:
                if ml2_util.check_l2br_vxlan(l2br, network) or ml2_util.check_l2br_vlan(l2br, network):
                    used_l2br_id = l2br['id']
                    error_msg = "Network is in use by l2br %s" % used_l2br_id
                    LOG.error(_LE(error_msg))
                    alarm_description = {'operation': ac_cnst.OPER_DELETE, 'res_type': ac_cnst.NW_HW_NETWORKS,
                                         'res_id': context.current['id'], 'reason': error_msg}
                    self.client.send_cloud_alarm(alarm_description, context._plugin_context.session)
                    ac_osprofiler.record_chain_exception_end(error_msg)
                    self.raise_ml2_exception(error_msg, 'network')

    def _not_need_update_network(self, context):
        if self.ops_version in OPS_VERSIONS.union({ac_cnst.OPS_N, }):
            if hasattr(DELETED_NETWORK, 'mapping') and context.current['id'] in DELETED_NETWORK.mapping:
                LOG.info(_LI('[AC] %s has been deleted, no need to update'), context.current['id'])
                ac_osprofiler.record_chain_end_with_reason("network has been deleted, no need to update")
                return True

            if not context.current.get('provider:network_type'):
                ac_osprofiler.record_chain_end_with_reason("network has no network type, no need to update")
                return True
        if self.data_filter.not_in_white_or_in_black_list(
                context._plugin_context, context.current, ac_cnst.NW_HW_NETWORKS, context.current['id']):
            return True
        return False

    def update_network_precommit(self, context):
        """This function is to check whether the qos of network is deleting

        :param context: DB context for the network update
        :return: None
        """
        self.reliability_api.check_neutron_sync(context, ac_cnst.OPER_UPDATE, ac_cnst.NW_HW_NETWORKS,
                                                context.current['id'])
        ac_osprofiler.record_chain_start("update network: " + context.current['id'])
        LOG.info(_LI("[AC]Check the dependent resources of network,the network info is %s."), context.current)

        if self._not_need_update_network(context):
            return

        plugin_context = context._plugin_context
        net_can_be_update, error_msg = self.data_filter.name_can_be_updated(
            plugin_context, context.current, context.original, ac_cnst.NW_HW_NETWORKS)
        if not net_can_be_update:
            self.raise_ml2_exception(error_msg, 'network')

        self._set_update_time(context, ac_cnst.NW_HW_NETWORKS)
        self.check_qos_dependent(plugin_context.session, context.current)
        try:
            LOG.info(_LI("[AC]Request AC to update network, network info is %s."), context.current)
            tenant_name = ACKeyStoneIf.get_tenant_name(context.current, plugin_context)
            network_info = ACNetworkModel.ac_model_format(context.current, tenant_name)
            self.__rest_request__(context, context.current['id'], network_info, ac_cnst.NW_HW_UPDATE_NETWORK)

            if ml2_util.network_qos_policy_changed(context):
                thread = threading.Thread(target=self._update_port_qos_thread, args=(context.current['id'],))
                thread.start()
        except Exception as ex:
            LOG.error(_LE("[AC]Update network: %s failed. exceptions, raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC]The exception is :%s."), ex)
            ac_osprofiler.record_chain_exception_end(str(ex))
            self.raise_ml2_exception(ex, 'network')
        ac_osprofiler.record_chain_end_with_reason("update network success")

    def _update_port_qos_thread(self, network_id):
        LOG.info(_LI("[AC] Begin to update %s port qos"), network_id)
        thread_pool = eventlet.GreenPool(size=5)
        admin_context = ncu.neutron_context.get_admin_context()
        core_plugin = ncu.get_core_plugin()
        filters = {'network_id': [network_id]}
        try:
            ports = core_plugin.get_ports(admin_context, filters=filters)
        except Exception as ex:
            LOG.error(_LE("[AC] %s get ports failed: %s"), network_id, ex)
            raise
        LOG.info(_LI("[AC] %s get ports: %s"), network_id, ports)
        for port in ports:
            thread_pool.spawn_n(self._update_port_qos, port)
        LOG.info(_LI("[AC] Update %s port qos end"))

    def _update_port_qos(self, port):
        LOG.info(_LI("[AC] Begin to update port qos: %s"), port)
        admin_context = ncu.neutron_context.get_admin_context()
        try:
            port_info = ACPortModel.ac_model_format(port, port['tenant_id'])
            self.reliability_api.update_plugin_record(admin_context, port['id'], port_info, ac_cnst.NW_HW_UPDATE_PORT)
        except Exception as ex:
            LOG.error(_LE("[AC] AC update port %s qos failed: %s"), port['id'], ex)
            raise

    def create_subnet_precommit(self, context):
        """create subnet precommit"""
        self.reliability_api.check_neutron_sync(context, ac_cnst.OPER_CREATE, ac_cnst.NW_HW_SUBNETS,
                                                context.current['id'])
        ac_osprofiler.record_chain_start("create subnet precommit, ID:" + context.current['id'])
        LOG.info(_LI("[AC]AC create subnet precommit: %s."), context.current)
        if self.data_filter.not_in_white_or_in_black_list(
                context._plugin_context, context.current, ac_cnst.NW_HW_SUBNETS, context.current['id']):
            return
        subnet_id = context.current['id']
        if self.ops_version in OPS_VERSIONS and self.db_if.create_config_record(subnet_id, 0, comment='subnet'):
            LOG.info(_LI('Create subnet :%s in config table'), subnet_id)
        self.http_heart_beat.block_send_request_to_ac()
        CREATE_SUBNET.mapping = {}
        if not hasattr(CREATE_SUBNET, 'need_update_ports'):
            CREATE_SUBNET.need_update_ports = []
        CREATE_SUBNET.mapping[context.current['id']] = datetime.datetime.utcnow()

    @lock_db.wrap_db_lock(lock_db.DELETE_SUBNET_LOCK)
    def create_subnet_postcommit(self, context):
        """This function sends subnet create message to AC

        :param context: DB context for the subnet creation
        :return: None
        """
        try:
            ac_osprofiler.record_chain_start("create subnet start,ID:" + context.current['id'])
            LOG.info(_LI("[AC]Request AC to create subnet: %s."), context.current)

            if (self.ops_version in OPS_VERSIONS) and self.db_if.get_config_record(key=context.current['id']):
                LOG.info(_LI('Delete subnet :%s from config table'), context.current['id'])
                self.db_if.delete_config_record(context.current['id'])

            plugin_context = context._plugin_context
            if self.data_filter.not_in_white_or_in_black_list(
                    plugin_context, context.current, ac_cnst.NW_HW_SUBNETS, context.current['id']):
                return

            subnet_info = ACSubnetModel.ac_model_format(context.current, str(context.current['tenant_id']))
            self.__rest_request__(context, context.current['id'], subnet_info, ac_cnst.NW_HW_CREATE_SUBNET)

            CREATE_SUBNET.mapping = {}
            if self.enable_ce1800v_dhcp and context.current.get('enable_dhcp', None):
                operation = 'CREATE_SUBNET'
                id_args = IdArgs(context.current.get('tenant_id'), context.current.get('network_id'),
                                 context.current.get('id'))
                self.create_or_update_dhcp_port(plugin_context, id_args, operation)
            self.update_ports_of_subnet(plugin_context, context.current['id'], CREATE_SUBNET.need_update_ports)
            CREATE_SUBNET.need_update_ports = []

        except Exception as ex:
            LOG.error(_LE("[AC]Create subnet: %s failed. exceptions, raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            ac_osprofiler.record_chain_exception_end(str(ex))
            self.raise_ml2_exception(ex, 'subnet')
        ml2_util.update_gateway_ports_when_subnet_create(context)
        ac_osprofiler.record_chain_end()

    @lock_db.wrap_db_lock(lock_db.DELETE_SUBNET_LOCK)
    def delete_subnet_precommit(self, context):
        """This function sends subnet delete message to AC

        :param context: DB context for the subnet delete
        :return: None
        """
        try:
            ac_osprofiler.record_chain_start("delete subnet start,ID:" + context.current['id'])
            LOG.info(_LI("[AC]Request AC to delete subnet: %s."), context.current)

            if self.data_filter.not_in_white_or_in_black_list(
                    context._plugin_context, context.current, ac_cnst.NW_HW_SUBNETS, context.current['id']):
                return
            subnet_info = ACSubnetModel.ac_model_format(context.current, str(context.current['tenant_id']))

            self.__rest_request__(context, context.current['id'], subnet_info, ac_cnst.NW_HW_DELETE_SUBNET)
        except Exception as ex:
            LOG.error(_LE("[AC]Delete subnet: %s failed. exceptions, raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            ac_osprofiler.record_chain_exception_end(str(ex))
            self.raise_ml2_exception(ex, 'subnet')
        ac_osprofiler.record_chain_end()

    def update_subnet_postcommit(self, context):
        """This function sends subnet update message to AC

        :param context: DB context for the subnet update
        :return: None
        """
        if self.data_filter.not_in_white_or_in_black_list(
                context._plugin_context, context.current, ac_cnst.NW_HW_SUBNETS, context.current['id']):
            return
        if self.enable_ce1800v_dhcp:
            LOG.info(_LI("Start to process dhcp port by subnet,the current info: %s, the original info:%s"),
                     context.current, context.original)
            subnet_id = context.current['id']
            ac_osprofiler.record_chain_start("update subnet start,ID:" + subnet_id)
            if hasattr(UPDATE_SUBNET, 'mapping') and subnet_id in UPDATE_SUBNET.mapping:
                LOG.info(_LI("The dhcp port has been rollback by subnet: %s"), subnet_id)
                return
            subnet_current_info = context.current
            subnet_original_info = context.original
            network_id = context.current.get('network_id', None)
            enable_dhcp_new = subnet_current_info.get('enable_dhcp', None)
            enable_dhcp_old = subnet_original_info.get('enable_dhcp', None)
            if not network_id or enable_dhcp_new == enable_dhcp_old:
                return
            UPDATE_SUBNET.mapping = {subnet_id: datetime.datetime.utcnow()}
            operation = 'UPDATE_SUBNET'
            try:
                id_args = IdArgs(context.current.get('tenant_id'), network_id, subnet_id)
                if enable_dhcp_old is False and enable_dhcp_new is True:
                    self.create_or_update_dhcp_port(context._plugin_context, id_args, operation)
                elif enable_dhcp_old is True and enable_dhcp_new is False:
                    self.delete_or_update_dhcp_port(context._plugin_context, id_args)
            except Exception as ex:
                LOG.error(_LE("[AC]Update subnet: %s failed.The exception is %s"), subnet_id, ex)
                LOG.info(_LI("Start to rollback subnet, the original info is :%s"), subnet_original_info)
                rollback_subnet = {'subnet': subnet_original_info}
                ncu.get_core_plugin().update_subnet(context._plugin_context, subnet_id, rollback_subnet)
                ac_osprofiler.record_chain_exception_end(str(ex))
                self.raise_ml2_exception(ex, 'subnet')

    def update_subnet_precommit(self, context):
        """This function is to update subnet update time

        :param context: DB context for the subnet update
        :return: None
        """
        self.reliability_api.check_neutron_sync(context, ac_cnst.OPER_UPDATE, ac_cnst.NW_HW_SUBNETS,
                                                context.current['id'])
        LOG.info(_LI("[AC] Check the dependent resources of subnet,the subnet info is %s."), context.current)
        ac_osprofiler.record_chain_start("update subnet start,ID:" + context.current['id'])

        if self.data_filter.not_in_white_or_in_black_list(
                context._plugin_context, context.current, ac_cnst.NW_HW_SUBNETS, context.current['id']):
            return
        self._set_update_time(context, ac_cnst.NW_HW_SUBNETS)

        try:
            LOG.info(_LI("[AC]Request AC to update subnet, subnet info is %s."), context.current)
            subnet_info = ACSubnetModel.ac_model_format(context.current, str(context.current['tenant_id']))
            self.__rest_request__(context, context.current['id'], subnet_info, ac_cnst.NW_HW_UPDATE_SUBNET)

        except Exception as ex:
            LOG.error(_LE("[AC]Update subnet: %s failed. exceptions, raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            ac_osprofiler.record_chain_exception_end(str(ex))
            self.raise_ml2_exception(ex, 'subnet')

    @lock_db.wrap_db_lock(lock_db.RESOURCE_DHCP_NETWORK)
    def create_or_update_dhcp_port(self, context, res_ids, operation):
        """This function is to create or update dhcp port by subnet"""
        subnet_id = res_ids.subnet_id
        LOG.info(_LI('Start to create or update dhcp port by subnet %s'), res_ids.subnet_id)
        success_port = {}
        try:
            ports = context.session.query(models_v2.Port).filter(
                models_v2.Port.network_id == res_ids.network_id,
                models_v2.Port.device_owner == ncu.DEVICE_OWNER_DHCP).all()
            LOG.info(_LI('Get all ports from db for create or update:%s'), ports)
            if not ports:
                attribute = {'network_id': res_ids.network_id, 'subnet_id': [{'subnet_id': subnet_id}],
                             'tenant_id': res_ids.tenant_id}
                for _ in range(2):
                    self.create_dhcp_port_by_subnet(context, attribute, success_port)
            self.process_dhcp_ports_update(context, ports, res_ids, success_port)
        except Exception as ex:
            LOG.error(_LE("Create or update dhcp port by subnet: %s failed.The exception is :%s"), subnet_id, ex)
            if not (operation == 'UPDATE_SUBNET' and success_port):
                self.raise_ml2_exception(ex, 'port')
            LOG.info(_LE('Start to rollback dhcp port :%s'), success_port)
            for (port_id, rollback_port) in success_port.items():
                if not rollback_port:
                    ncu.get_core_plugin().delete_port(context, port_id, l3_port_check=False)
                else:
                    dhcp_port = {"id": port_id, "port": {"network_id": res_ids.network_id, "fixed_ips": rollback_port,
                                                         "binding: host_id": ''}}
                    ncu.get_core_plugin().update_port(context, port_id, dhcp_port)

    def process_dhcp_ports_update(self, context, ports, res_ids, success_port):
        """update dhcp ports"""
        for port in ports:
            port_id = port.get('id', None)
            if not port_id:
                continue
            port_info = ncu.get_core_plugin().get_port(context, port_id)
            LOG.info(_LI('Get port info: %s'), port_info)
            fixed_ips = port_info.get('fixed_ips', [])
            new_fixed_ips = fixed_ips[:]
            if port_info.get('binding:host_id'):
                continue
            new_fixed_ips.append({'subnet_id': res_ids.subnet_id})
            attribute = {'network_id': res_ids.network_id, 'binding: host_id': '', 'fixed_ips': new_fixed_ips}
            self.update_dhcp_port_by_subnet(context, port_id, attribute)
            success_port[port_id] = fixed_ips

    def update_ports_of_subnet(self, context, subnet_id, port_list):
        """This function is Start to update ports by subnet"""
        LOG.info(_LI('Start to update ports %s by subnet %s'),
                 port_list, subnet_id)
        try:
            for port_id in port_list:
                port_info = ncu.get_core_plugin().get_port(context, port_id)
                LOG.info(_LI('Get port info: %s'), port_info)
                fixed_ips = port_info.get('fixed_ips', [])
                update_port = {"id": port_id, "port": {"fixed_ips": fixed_ips}}
                ncu.get_core_plugin().update_port(context, port_id, update_port)
        except Exception as ex:
            LOG.error(_LE("update ports by subnet: %s failed.The exception is :%s"), subnet_id, ex)
            self.raise_ml2_exception(ex, 'port')

    def process_fixed_ip_one_by_one(self, context, res_ids, fixed_ips, success_port):
        """process_fixed_ip_one_by_one"""
        new_fixed_ips = fixed_ips[:]
        for fixed_ip in fixed_ips:
            port_subnet_id = fixed_ip.get('subnet_id', None)
            if port_subnet_id != res_ids.get("subnet_id"):
                continue
            new_fixed_ips.remove(fixed_ip)
            attribute = {'network_id': res_ids.get("network_id"), 'binding: host_id': '', 'fixed_ips': new_fixed_ips}
            self.update_dhcp_port_by_subnet(context, res_ids.get("port_id"), attribute)
            success_port[res_ids.get("port_id")] = fixed_ips

    @lock_db.wrap_db_lock(lock_db.RESOURCE_DHCP_NETWORK)
    def delete_or_update_dhcp_port(self, context, res_ids):
        """This function is delete or update dhcp port by subnet"""
        subnet_id = res_ids.subnet_id
        network_id = res_ids.network_id
        LOG.info(_LI('Start to delete or update dhcp port by subnet %s'), subnet_id)
        success_port = {}
        try:
            ports = context.session.query(models_v2.Port).filter(
                models_v2.Port.network_id == network_id, models_v2.Port.device_owner == ncu.DEVICE_OWNER_DHCP).all()
            LOG.info(_LI('Get all ports from db for delete or update:%s'), ports)
            for port in ports:
                port_id = port.get('id', None)
                if not port_id:
                    LOG.debug("Can not get port id, continue")
                    continue
                port_info = ncu.get_core_plugin().get_port(context, port_id)
                LOG.info(_LI('Get port info: %s'), port_info)
                if port_info.get('binding:host_id', ''):
                    LOG.debug("The port %s is not ce1800v dhcp port", port_id)
                    continue
                fixed_ips = port_info.get('fixed_ips', [])
                self.process_fixed_ip_one_by_one(
                    context, {"network_id": network_id, "subnet_id": subnet_id, "port_id": port_id}, fixed_ips,
                    success_port)
        except Exception as ex:
            LOG.error(_LE("Delete or update dhcp port by subnet: %s failed.The exception is :%s"), subnet_id, ex)
            if success_port:
                LOG.info(_LE('Start to rollback dhcp port :%s'), success_port)
                for (port_id, rollback_port) in success_port.items():
                    dhcp_port = {"id": port_id,
                                 "port": {"network_id": network_id, "fixed_ips": rollback_port, "binding: host_id": ''}}
                    ncu.get_core_plugin().update_port(context, port_id, dhcp_port)
            self.raise_ml2_exception(ex, 'port')

    def create_dhcp_port_by_subnet(self, context, attribute, success_port):
        """This function is start to create dhcp port"""
        LOG.info(_LI('Start to create dhcp port, attribute: %s'), attribute)
        try:
            mac = ACCommonUtil.get_mac(self.ops_version)
            dhcp_port = ACAgentAPI.deal_dhcp_port(attribute, context, mac)
            success_port[dhcp_port['id']] = []
        except Exception as ex:
            LOG.error(_LE("[AC]Process creating dhcp port error,%s,traceback: %s"), str(ex), traceback.format_exc())
            self.raise_ml2_exception(ex, 'port')

    def update_dhcp_port_by_subnet(self, context, port_id, attribute):
        """This function is start to update dhcp port"""
        LOG.info(_LI('Start to update dhcp port, attribute: %s'), attribute)
        try:
            dhcp_port = {"id": port_id,
                         "port": {"network_id": attribute["network_id"], "fixed_ips": attribute['fixed_ips'],
                                  "binding: host_id": attribute['binding: host_id']}}
            ncu.get_core_plugin().update_port(context, port_id, dhcp_port)
        except Exception as ex:
            LOG.error(_LE("[AC]Process updating dhcp port error, %s,traceback: %s"), str(ex), traceback.format_exc())
            self.raise_ml2_exception(ex, 'port')

    def create_ac_proc_status(self, context):
        """add ac process status in db"""
        # add ac process status in db
        # in this step, ac process status is pending
        if ncu.need_ac_status(ac_cnst.NW_HW_AC_STATUS_PORT):
            device_dict = {'id': context.current['id'], 'type': 'port', }
            ac_status_dict = self.ac_proc_db_func.create_db_ac_proc_status(device_dict)
            LOG.info(_LI("[AC] Creating port, Create ac process status in Neutron DB successful: %s "), ac_status_dict)

    def filter_router_gw(self, context):
        """filter router gw"""
        device_owner = context.current.get('device_owner', None)
        device_id = context.current.get('device_id', None)
        network_id = context.current.get('network_id')
        if device_owner == ncu.DEVICE_OWNER_ROUTER_GW:
            if device_id and self._check_router_gw_exist(context, network_id, device_id):
                LOG.error("[AC]Router gw existed, don't need create again")
                alarm_description = {'operation': ac_cnst.OPER_CREATE, 'res_type': ac_cnst.NW_HW_PORTS,
                                     'res_id': context.current['id'],
                                     'reason': "Router gw existed, don't need create again"}
                self.client.send_cloud_alarm(alarm_description, context._plugin_context.session)
                ac_osprofiler.record_chain_exception_end("Router gw existed, don't need create again")
                self.raise_ml2_exception('router gw exits', 'port')

    def create_port_precommit(self, context):
        """create port precommit"""
        self.reliability_api.check_neutron_sync(context, ac_cnst.OPER_CREATE, ac_cnst.NW_HW_PORTS,
                                                context.current['id'])
        self.create_ac_proc_status(context)
        # check router gw port,if router bind mutiply gw port,throw exception
        ac_osprofiler.record_chain_start("create port start,ID:" + context.current['id'])
        self.filter_router_gw(context)

        self._add_primary_interface_to_port(context)
        if self.data_filter.not_in_white_or_in_black_list(
                context._plugin_context, context.current, ac_cnst.NW_HW_PORTS, context.current['id']):
            return

        if self.data_filter.port_contain_anti_affinity(context.current):
            return

        self.http_heart_beat.block_send_request_to_ac()

        if util.ACUtil.is_dhcp_port(context.current):
            if not self.enable_ce1800v_dhcp:
                network_id = context.current.get('network_id')
                host_id = context.current.get('binding:host_id')
                network = ncu.get_core_plugin().get_network(context._plugin_context, network_id)
                if host_id and network.get('provider:network_type') == ncu.TYPE_VXLAN:
                    self._get_bind_port_level(network_id, host_id, context)
            port_info = ACPortModel.ac_model_format(context.current, str(context.current['tenant_id']))
            self.__rest_request__(context, context.current['id'], port_info, ac_cnst.NW_HW_CREATE_PORT)

    def _add_primary_interface_to_port(self, context):
        if self.primary_interface and ml2_port_util.is_primary_port(context):
            LOG.info(_LI("[AC] The port need to add primary interface."))
            ac_osprofiler.record_chain_exception_end("add primary interface")
            if context._binding.vif_details:
                vif_details = jsonutils.loads(context._binding.vif_details)
            else:
                vif_details = {}
            vif_details.update({PRIMARY_INTERFACE: True})
            context._binding.vif_details = jsonutils.dumps(vif_details)
            context.current[ncu.VIF_DETAILS].update(vif_details)

    def _check_router_gw_exist(self, context, network_id, device_id):
        LOG.debug("[AC]Create gateway port, check if router gw exsit, router: %s", device_id)
        session = context._plugin_context.session
        resources = self.db_if.get_plugin_record_list(session, ACPluginSchema(res_type=ac_cnst.NW_HW_PORTS))
        for res in resources:
            res_data = (copy.deepcopy(res)).get('data')
            if res_data.get("device-id", None) and res_data.get("device-id") == device_id and res_data.get(
                    "network-id") == network_id:
                return True
        return False

    def _update_port_info(self, context, port_info):
        host_id = port_info.get('host-id', None)
        if host_id and host_id in self.logical_hostnames:
            port_info['host-id'] = ''
        if ml2_port_util.is_baremetal_port(context.current) and not context.current['device_owner'].startswith(
                'baremetal:external'):
            port_info['host-id'] = ""
            port_info['profile'] = '{}'
        return port_info

    def create_port_postcommit(self, context):
        """This function sends port create message to AC

        :param context: DB context for the port creation
        :return: None
        """
        try:
            LOG.info(_LI("[AC]create port: %s."), context.current)
            if util.ACUtil.is_dhcp_port(context.current):
                return

            if self.data_filter.not_in_white_or_in_black_list(
                    context._plugin_context, context.current, ac_cnst.NW_HW_PORTS, context.current['id']):
                return

            if self.data_filter.port_contain_anti_affinity(context.current):
                return

            if util.ACUtil.is_compute_port(context.current):
                allocation = self._allocate_vlan(context)
                if allocation:
                    context.current['binding:profile'].update({'plugin-allocate-vlan': allocation})

            high_priority_physnet = ml2_port_util.get_high_priority_physnet(context)
            if high_priority_physnet:
                context.current['binding:profile'].update({'physical_network': high_priority_physnet})

            port_info = ACPortModel.ac_model_format(context.current, str(context.current['tenant_id']))
            port_info = self._update_port_info(context, port_info)

            self.__rest_request__(context, context.current['id'], port_info, ac_cnst.NW_HW_CREATE_PORT)

            fixed_ips = context.current.get('fixed_ips', None)
            if ncu.IS_FSP and ACCommonUtil.is_port_contain_ipv4_address(fixed_ips):
                PublicService().create_fip(context)

        except Exception as ex:
            device_owners = ['network:router_interface', 'network:router_gateway', 'network:floatingip']
            if context.current['device_owner'] in device_owners:
                with context._plugin_context.session.begin(subtransactions=True):
                    plugin = db_base_plugin_v2.NeutronDbPluginV2()
                    port = plugin._get_port(context._plugin_context, context.current['id'])
                    port.update({'device_owner': ''})
                    context._plugin_context.session.flush()
            LOG.error(_LE("[AC]Create port: %s failed. raise exception: %s.to plugin to rollback."),
                      context.current['id'], ex)
            ac_osprofiler.record_chain_exception_end(str(ex))
            self.raise_ml2_exception(ex, 'port')
        ac_osprofiler.record_chain_end_with_reason("create port success")

    @classmethod
    def check_subnet_ready(cls, context):
        """check the subnet is ready or not"""
        for item in context.current['fixed_ips']:
            if hasattr(CREATE_SUBNET, 'mapping') and item['subnet_id'] in CREATE_SUBNET.mapping:
                LOG.info(_LI("[AC] the subnet %s is not ready, abort to update port %s."), item['subnet_id'],
                         context.current['id'])
                return False
            try:
                admin_context = ncu.neutron_context.get_admin_context()
                ncu.get_core_plugin().get_subnet(admin_context, item['subnet_id'])
            except Exception as e:
                LOG.info(_LI("[AC]the subnet %s is not ready, abort to update port %s.error:%s"), item['subnet_id'],
                         context.current['id'], traceback.format_exc())
                return False
        return True

    @classmethod
    def filter_port_update_dhcp(cls, context):
        """Filter port update dhcp"""
        if util.ACUtil.is_dhcp_port(context.current):
            return True
        return False

    def filter_port_update_no_subnet(self, context):
        """Filter subnet ready or not"""
        if not self.check_subnet_ready(context):
            if not hasattr(CREATE_SUBNET, 'need_update_ports'):
                CREATE_SUBNET.need_update_ports = []
            CREATE_SUBNET.need_update_ports.append(context.current['id'])
            return True
        return False

    def filter_port_update_white_black(self, context):
        """Filter white black list"""
        if self.data_filter.not_in_white_or_in_black_list(
                context._plugin_context, context.current, ac_cnst.NW_HW_PORTS, context.current['id']):
            return True
        return False

    def filter_port_update_affinity(self, context):
        """Filter port update affinity"""
        if self.data_filter.port_contain_anti_affinity(context.current):
            return True
        return False

    @classmethod
    def filter_port_update_except_status(cls, context):
        """filter port update except status"""
        current_profile = context.current['binding:profile']
        migrating_host = current_profile.get('migrating_host')
        if ml2_port_util.filter_port_update_status(context, migrating_host):
            return True
        return False

    @classmethod
    def filter_port_update_no_changes(cls, context):
        """Filter port with no change"""
        if context.current.get('device_owner', None) == 'trunk:subport' and ml2_port_util.check_subport_no_change(
                context.original, context.current):
            return True
        return False

    @classmethod
    def filter_port_update_gbp_sec(cls, context):
        """filter port update gbp sec"""
        service_plugins = ncu.get_service_plugin()
        if 'GROUP_POLICY' in service_plugins and ml2_security_util.security_group_no_change(
                context.original, context.current):
            LOG.info(_LI('[AC]port %s only update security group in gbp model,no need send to AC'),
                     context.current['id'])
            return True
        return False

    @classmethod
    def filter_port_update_vif(cls, context):
        """filter port update vif"""
        for key in set(context.original.keys()) - {'revision_number', 'created_at', 'updated_at', 'status',
                                                   'dns_assignment', 'binding:vif_details'}:
            if key == 'binding:vif_type' and context.original['binding:vnic_type'] != 'vcenter':
                continue
            if context.original[key] != context.current[key]:
                return False
        return True

    def filter_port_update_process(self, context):
        """filter the port update process"""
        update_port_filters = [self.filter_port_update_dhcp, self.filter_port_update_no_subnet,
                               self.filter_port_update_white_black, self.filter_port_update_affinity,
                               self.filter_port_update_except_status, self.filter_port_update_no_changes,
                               self.filter_port_update_gbp_sec, self.filter_port_update_vif]
        for func_filter in update_port_filters:
            if func_filter(context):
                return True
        return False

    @classmethod
    def _filter_vif_type(cls, vif_type, vnic_type):
        if vif_type and vif_type == ncu.VIF_TYPE_BINDING_FAILED and vnic_type == 'vcenter':
            return True
        return False

    def _need_update_migrate_host(self, current_profile):
        version_flg = self.ops_version in {ac_cnst.OPS_K, ac_cnst.OPS_N, ac_cnst.OPS_O,
                                           ac_cnst.OPS_P, ac_cnst.OPS_Q, ac_cnst.OPS_R,
                                           ac_cnst.OPS_T, ac_cnst.OPS_W, ac_cnst.FSP_21_0}
        if (ncu.IS_FSP or version_flg) and current_profile.get('migrating_to'):
            return True
        return False

    def update_port_postcommit(self, context):
        """This function sends port update message to AC

        :param context: DB context for the port update
        :return: None
        """
        if ml2_port_util.is_edge_dc_vm_port(context) and not ml2_port_util.compute_port_first_up(context):
            self.db_if.update_attribute_record(None, context.current.get('id'), ac_cnst.NW_HW_PORTS,
                                               updated_at=context.current.get('updated_at'),
                                               network_id=context.current.get('network_id'))
            return
        try:
            LOG.info(_LI("[AC]Request AC to update port: %s."), context.current)
            if self.filter_port_update_process(context):
                return

            if ml2_util.check_migrating_host(context):
                return

            self.update_binding_profile(context)

            port_info = ACPortModel.ac_model_format(context.current, str(context.current['tenant_id']))
            vif_type = port_info.get('vif-type', None)
            vnic_type = port_info.get('vnic-type', None)
            if self._filter_vif_type(vif_type, vnic_type):
                LOG.error(_LE("[AC] This is a vcenter port and vif type is binding failed, no need send to AC."))
                ml2_port_util.set_port_status(context, context.current['id'], ac_cnst.NEUTRON_STATUS_ERROR)
                ac_osprofiler.record_chain_end_with_reason(
                    "vcenter port and vif type is binding failed,no need send to AC,return")
                return
            port_info = self._hw_update_port_info(context, port_info)

            if _check_port_during_deleting(context):
                LOG.info("[AC]the port is expected to be deleted, skip updating in postcommit.")
                raise Exception("the port is expected to be deleted.")

            LOG.info(_LI("[AC]Request AC to update port, port_id: %s, port_info: %s"), context.current['id'], port_info)
            self.__rest_request__(context, context.current['id'], port_info, ac_cnst.NW_HW_UPDATE_PORT,
                                  dry_run_state=_get_dry_run_state(context.original, ac_cnst.NW_HW_PORTS))

            ml2_util.update_binding_profile_sdn_status(context, 'success')

            ml2_port_util.create_or_delete_fip_when_port_update(context)
            if ncu.IS_FSP and util.ACUtil.is_compute_port(context.current):
                self.update_floating_ip_address(context)

        except Exception as ex:
            self._handle_update_port_exp(context, ex)
        ac_osprofiler.record_chain_end_with_reason("update port success")

    def _handle_update_port_exp(self, context, ex):
        LOG.error(_LE("[AC]Update port: %s failed. exceptions,raise exception to plugin to rollback."),
                  context.current['id'])
        LOG.error(_LE("[AC] The exception is :%s. %s"), ex, traceback.format_exc())

        ml2_util.update_binding_profile_sdn_status(context, 'failed')
        self.db_if.create_or_update_failed_resource(
            ACFailedResources(id=context.current.get('id'), res_type=ac_cnst.NW_HW_PORTS,
                              operation=ac_cnst.NW_HW_UPDATE_PORT), True, context._plugin_context.session)

        ac_osprofiler.record_chain_exception_end(str(ex))
        self.raise_ml2_exception(ex, 'port')

    def update_binding_profile(self, context):
        """update binding profile"""
        original_profile = context.original['binding:profile']
        current_profile = context.current['binding:profile']
        if original_profile.get('vm_name'):
            context.current['binding:profile'].update({'vm_name': original_profile['vm_name']})
        if self._need_update_migrate_host(current_profile):
            current_profile['migrating-host'] = current_profile['migrating_to']
            if self.vhost_user:
                current_profile['vhost_user'] = 'ce1800v'

        if util.ACUtil.is_compute_port(context.current):
            allocation = self._allocate_vlan(context)
            if allocation:
                context.current['binding:profile'].update({'plugin-allocate-vlan': allocation})

        high_priority_physnet = ml2_port_util.get_high_priority_physnet(context)
        if high_priority_physnet:
            context.current['binding:profile'].update({'physical_network': high_priority_physnet})

        ml2_port_util.set_fw_enabled(context)

    @classmethod
    def update_floating_ip_address(cls, context):
        """update floating ip address"""
        LOG.info(_LI('Start to update associate fip: %s'), context.original)
        original_fixed_ips = context.original.get('fixed_ips')
        current_fixed_ips = context.current.get('fixed_ips')
        original_subnet_ips = ml2_util.get_port_subnet_and_ip(original_fixed_ips)
        current_subnet_ips = ml2_util.get_port_subnet_and_ip(current_fixed_ips)
        LOG.info(_LI('original subnet ips: %s , current subnet ips: %s'), original_subnet_ips, current_subnet_ips)
        subnet_ids = []
        for (subnet_id, subnet) in current_subnet_ips.items():
            if not subnet or not original_subnet_ips.get(subnet_id):
                continue
            if netaddr.valid_ipv4(subnet) and subnet != original_subnet_ips.get(subnet_id):
                subnet_ids.append(subnet_id)
        if not subnet_ids:
            return
        LOG.info(_LI('vm port: %s ip address has changed,the subnet ids is: %s, need to update fip'),
                 context.current['id'], subnet_ids)
        l3plugin = ncu.get_service_plugin()['L3_ROUTER_NAT']
        if not l3plugin:
            LOG.info(_LI('No l3 router nat service, return'))
            return
        filters = {'port_id': [context.current['id']]}
        admin_context = ncu.neutron_context.get_admin_context()
        fips = l3plugin.get_floatingips(admin_context, filters)
        LOG.info(_LI('Get floating ip info %s:'), fips)
        if not fips:
            LOG.info(_LI('No associate fip, return'))
            return
        fip_fixed_ip_address = fips[0].get('fixed_ip_address', None)
        for subnet_id in subnet_ids:
            if not ml2_util.validate_ip_address_in_subnet(admin_context, subnet_id, fip_fixed_ip_address):
                LOG.info(_LI('Fip ip address: %s not in subnet: %s'), fip_fixed_ip_address, subnet_id)
                continue
            current_ip = current_subnet_ips.get(subnet_id)
            LOG.debug(_LI("Fip %s ip_address need update to %s."), fips[0]['id'], current_ip)
            l3plugin.update_floatingip_info(admin_context, fips[0]['id'], {
                'floatingip': {'port_id': context.current['id'], 'fixed_ip_address': current_ip}})
            LOG.info(_LI("[AC]Huawei AC update floating IP successful."))

    def _hw_update_ac_record(self, context, vnic_type, host_id):
        """
        因为ac设备对共享ex-route的限制，更新port时需要对首子port进行保序
        """
        LOG.info('[AC] Update ac record')
        trunk_details = context.current.get('trunk_details', None)
        current_vif_details = context.current.get('binding:vif_details', None)
        original_vif_details = context.original.get('binding:vif_details', None)
        if current_vif_details:
            current_evs_plug = current_vif_details.get('vhostuser_evs_plug', None)
        else:
            current_evs_plug = ''
        if original_vif_details:
            original_evs_plug = original_vif_details.get('vhostuser_evs_plug', None)
        else:
            original_evs_plug = ''
        db_port_host_id = context.original.get('binding:host_id')
        network_id = context.current.get('network_id', None)
        router_external = vlan_transparent = False
        if network_id:
            network = ncu.get_core_plugin().get_network(context._plugin_context, network_id)
            if network:
                router_external = network.get('router:external', None)
                vlan_transparent = network.get('vlan_transparent', None)
        if self._filter_update_ac_record({"vnic_type": vnic_type,
                                          "trunk_details": trunk_details,
                                          "host_id": host_id,
                                          "db_port_host_id": db_port_host_id,
                                          "current_evs_plug": current_evs_plug,
                                          "original_evs_plug": original_evs_plug,
                                          "router_external": router_external,
                                          "vlan_transparent": vlan_transparent}):
            update_ac_record = threading.Thread(name="update_subports_thread", target=self.update_subport_thread,
                                                args=(context, host_id, trunk_details.get('trunk_id', None)))
            update_ac_record.start()

    def _hw_update_port_info(self, context, port_info):
        host_id = port_info.get('host-id', None)
        vif_type = port_info.get('vif-type', None)
        vnic_type = port_info.get('vnic-type', None)
        is_baremetal_port = ml2_port_util.is_baremetal_port(context.current)

        if host_id and host_id in self.logical_hostnames and not is_baremetal_port:
            port_info['host-id'] = ''
        if vif_type == ncu.VIF_TYPE_UNBOUND and vnic_type == 'vcenter':
            port_info['host-id'] = ''
        binding_profile = context.current.get('binding:profile', None)
        str_profile = ml2_port_util.get_ac_port_profile(context, port_info['uuid'], binding_profile)
        if str_profile:
            port_info['profile'] = str_profile
        port_info = ml2_port_util.process_subport_upgrade(context, context.current['id'], port_info)
        return port_info

    def _filter_update_ac_record(self, filter_attribute_dict):
        vnic_type = filter_attribute_dict.get("vnic_type")
        trunk_details = filter_attribute_dict.get("trunk_details")
        host_id = filter_attribute_dict.get("host_id")
        db_port_host_id = filter_attribute_dict.get("db_port_host_id")
        current_evs_plug = filter_attribute_dict.get("current_evs_plug")
        original_evs_plug = filter_attribute_dict.get("original_evs_plug")
        router_external = filter_attribute_dict.get("router_external")
        vlan_transparent = filter_attribute_dict.get("vlan_transparent")
        if router_external and vlan_transparent:
            return False
        flg_1 = ((vnic_type == 'direct' or self.vhost_user) and trunk_details and host_id != db_port_host_id)
        flg_2 = (current_evs_plug != original_evs_plug and trunk_details)
        flg_3 = (current_evs_plug == original_evs_plug and host_id != db_port_host_id and trunk_details)
        if flg_1 or flg_2 or flg_3:
            return True
        return False

    def update_port_precommit(self, context):
        """This function is to check whether the qos of port is deleting

        :param context: DB context for the network update
        :return: None
        """
        self.reliability_api.check_neutron_sync(context, ac_cnst.OPER_UPDATE, ac_cnst.NW_HW_PORTS,
                                                context.current['id'])
        dry_run_util.check_whether_dryrun_to_product(ac_cnst.NW_HW_PORTS, ac_cnst.OPER_UPDATE, context.original,
                                                     context.current)
        LOG.info(_LI("[AC]update port: %s."), context.current)
        if not ml2_port_util.is_edge_dc_vm_port(context):
            self.http_heart_beat.block_send_request_to_ac()

        if context.original.get('binding:host_id') and not context.current.get('binding:host_id'):
            for sub_port in context.current.get('trunk_details', {}).get('sub_ports', []):
                self.check_fm_for_port(sub_port.get('port_id'))
            self.check_fm_for_port(context.current['id'])
        if ncu.need_ac_status(ac_cnst.NW_HW_AC_STATUS_PORT):
            device_dict = {'id': context.current['id'], 'type': 'port', 'ac_proc_status': 'PENDING'}
            ac_status_dict = self.ac_proc_db_func.update_db_ac_proc_status(device_dict)
            LOG.info(_LI("[AC]Updating port, Update ac process status in Neutron DB successful:%s"), ac_status_dict)
        ac_osprofiler.record_chain_start("update port start,ID:" + context.current['id'])
        self._add_primary_interface_to_port(context)

        if self._not_handle_port_update_pre(context):
            return

        self._check_migrate_host_status(context)

        self._set_update_time(context, ac_cnst.NW_HW_PORTS)

        self.check_qos_dependent(context._plugin_context.session, context.current)

        ml2_port_util.update_port_context(context)

        self.update_dhcp_port_pre(context)

        if not context.current.get('fixed_ips') and context.original.get('fixed_ips'):
            self.check_fiexed_ip_update(context)
        if (context.original['binding:vnic_type'] == ac_cnst.VNIC_TYPE_BAREMETAL or context.current[
                'binding:vnic_type'] == ac_cnst.VNIC_TYPE_BAREMETAL) and ncu.get_ops_version() in [ac_cnst.FSP_21_0]:
            self._handle_baremetal_port_precommit(context)
            return
        if not self.filter_port_update_process:
            host_id = context.current.get('binding:host_id', None)
            vnic_type = context.current.get('binding:vnic_type', None)
            self._hw_update_ac_record(context, vnic_type, host_id)

    def _handle_baremetal_port_precommit(self, context):
        str_binding_profile = ml2_port_util.get_ac_port_profile(
            context, context.current['id'], context.current['binding:profile']) or '{}'
        str_binding_vnic_type = context.current['binding:vnic_type']
        str_binding_host_id = context.current['binding:host_id']
        LOG.info('[AC] Handle baremetal port precommit, host_id: %s, vnic_type: %s, profile: %s', str_binding_host_id,
                 str_binding_vnic_type, str_binding_profile)
        # baremetal场景下，父port的hostid从非空->空的时候，必须先更新所有的子port，父port走正常的postcommit更新
        if context.original['binding:host_id'] and not context.current['binding:host_id']:
            LOG.info('[AC] Handle baremetal port, host from %s to ""', context.original['binding:host_id'])
            self._process_baremetal_subports(context, binding_host_id='', binding_vnic_type=str_binding_vnic_type,
                                             binding_profile=str_binding_profile)
            self._update_baremetal_subports_in_db(context)
            return
        # baremetal场景下处理迁移，父子port均在此处处理
        if self._check_if_profile_changed_in_baremetal(context):
            LOG.info('[AC] Handle baremetal port in migrating, phase: start')
            self._process_baremetal_subports(context, binding_host_id='', binding_vnic_type=ac_cnst.NORMAL,
                                             binding_profile='{}')
            self._process_baremetal_parent_port(context, binding_host_id='', binding_vnic_type=ac_cnst.NORMAL,
                                                binding_profile='{}')
            LOG.info('[AC] Handle baremetal port in migrating, phase: middle')
            self._process_baremetal_parent_port(context, binding_host_id=str_binding_host_id,
                                                binding_vnic_type=ac_cnst.VNIC_TYPE_BAREMETAL,
                                                binding_profile=str_binding_profile)
            self._process_baremetal_subports(context, binding_host_id=str_binding_host_id,
                                             binding_vnic_type=ac_cnst.VNIC_TYPE_SUB_BAREMETAL,
                                             binding_profile=str_binding_profile, use_thread=True)
            LOG.info('[AC] Handle baremetal port in migrating, phase: end')
            self._update_baremetal_subports_in_db(context)

    @classmethod
    def _check_if_profile_changed_in_baremetal(cls, context, allow_empty=False):
        """
        检查baremetal场景下，binding:profile是否有变更，默认不处理local_link_information非空和空之间转换的场景
        """
        original_local_link_info = context.original['binding:profile'].get('local_link_information')
        current_local_link_info = context.current['binding:profile'].get('local_link_information')
        if not original_local_link_info or not current_local_link_info:
            return False or allow_empty
        original_local_link_info.sort(key=lambda x: x['port_id'])
        current_local_link_info.sort(key=lambda x: x['port_id'])
        LOG.debug('[AC] original lldp: %s, current lldp: %s', original_local_link_info, current_local_link_info)
        profile_changed = original_local_link_info != current_local_link_info
        if not profile_changed:
            original_local_group_info = context.original['binding:profile'].get('local_group_information')
            current_local_group_info = context.current['binding:profile'].get('local_group_information')
            if original_local_group_info and current_local_group_info and original_local_group_info.get(
                    'bond_mode') != current_local_group_info.get('bond_mode'):
                return True
        return profile_changed

    def _process_baremetal_parent_port(self, context, binding_host_id, binding_vnic_type, binding_profile):
        if not context.current.get('trunk_details'):
            return
        LOG.info('[AC] Processing update parent port start, '
                 'binding_host_id: %s, binding_vnic_type: %s, binding_profile: %s',
                 binding_host_id, binding_vnic_type, binding_profile)
        port_info = ACPortModel.ac_model_format(context.current, str(context.current['tenant_id']))
        port_info['host-id'] = binding_host_id
        port_info['profile'] = binding_profile
        port_info['vnic-type'] = binding_vnic_type
        dry_run_state = _get_dry_run_state(context.original, ac_cnst.NW_HW_PORTS)
        LOG.debug(_LI("[AC] Start to send rest request of port: %s"), port_info)
        self.__rest_request__(context, context.current['id'], port_info, ac_cnst.NW_HW_UPDATE_PORT,
                              dry_run_state=dry_run_state)
        LOG.info('[AC] Processing update parent port end')

    def _process_baremetal_subports(self, context, **kwargs):
        binding_host_id = kwargs.get('binding_host_id')
        binding_vnic_type = kwargs.get('binding_vnic_type')
        binding_profile = kwargs.get('binding_profile')
        use_thread = kwargs.get('use_thread', False)

        def process():
            if context.current.get('trunk_details') and context.current['trunk_details'].get('sub_ports'):
                update_subport_threadpool = eventlet.GreenPool(size=ac_cnst.MAX_SUBPORT_UPDATE_THREADS)
                LOG.info('[AC] Processing update subports start, '
                         'binding_host_id: %s, binding_vnic_type: %s, binding_profile: %s',
                         binding_host_id, binding_vnic_type, binding_profile)
                for sub_port in context.current['trunk_details']['sub_ports']:
                    subport_id = sub_port.get('port_id')
                    LOG.debug('[AC] Handle baremetal subport postcommit, subport_id: %s', subport_id)
                    update_subport_threadpool.spawn_n(self.process_update_subport, subport_id=subport_id,
                                                      host_id=binding_host_id,
                                                      trunk_id=None, migrating_host=None,
                                                      is_sub_baremetal_scenario=True,
                                                      binding_vnic_type=binding_vnic_type,
                                                      binding_profile=binding_profile)
                LOG.info('[AC] Processing update subports end')
                update_subport_threadpool.waitall()

        if not use_thread:
            process()
        else:
            threading.Thread(target=process).start()

    @classmethod
    def _update_baremetal_subports_in_db(cls, context):
        """
        当父port的host_id或者profile或者vnic_type改变的时候，更新对应的subports
        """
        LOG.info('[AC] Update baremetal subports in db')
        if context.current.get('trunk_details') and context.current['trunk_details'].get('sub_ports'):
            for sub_port in context.current['trunk_details']['sub_ports']:
                sub_port_id = sub_port.get('port_id')
                cls._update_baremetal_subport_in_db(context, sub_port_id)

    @classmethod
    def _update_baremetal_subport_in_db(cls, context, subport_id):
        LOG.info('[AC] Update baremetal subport in db, subport_id: %s', subport_id)
        with context._plugin_context.session.begin(subtransactions=True):
            port_binding = ncu.get_port_binding(context._plugin_context.session, subport_id)
            if port_binding:
                str_profile = jsonutils.dumps(context.current['binding:profile'])
                if str_profile:
                    str_profile = ml2_port_util.process_profile_for_db(str_profile)
                else:
                    str_profile = "{}"
                port_binding.profile = str_profile
                port_binding.vnic_type = context.current['binding:vnic_type']
                context._plugin_context.session.merge(port_binding)
                context._plugin_context.session.flush()

    def _check_migrate_host_status(self, context):
        migrating_to = context.current['binding:profile'].get('migrating_to')
        org_migrating_to = context.original['binding:profile'].get('migrating_to')
        if migrating_to == org_migrating_to:
            return
        if not util.ACUtil.is_compute_port(context.current):
            return
        if migrating_to and migrating_to != "null":
            self.client.query_host_status(migrating_to)

    def _not_handle_port_update_pre(self, context):
        if self.data_filter.not_in_white_or_in_black_list(
                context._plugin_context, context.current, ac_cnst.NW_HW_PORTS, context.current['id']):
            return True

        if self.data_filter.port_contain_anti_affinity(context.current):
            return True

        if context.current['status'] == ac_cnst.NEUTRON_STATUS_BUILD:
            LOG.info(_LI("[AC]Port status is BUILD, no need send to ac"))
            if context.original['status'] == ac_cnst.NEUTRON_STATUS_ERROR:
                ml2_port_util.set_port_status(context, context.current['id'], ac_cnst.NEUTRON_STATUS_ERROR)
            ac_osprofiler.record_chain_end_with_reason("port status is BUILD,return")
            return True
        return False

    @classmethod
    def check_fiexed_ip_update(cls, context):
        """This function is to check whether the fixed ip of port can update

        :param context: context info for the port update
        :return: None
        """
        host_id = context.original.get('binding:host_id') or context.current.get('binding:host_id')
        device_owner = context.original.get('device_owner') or context.current.get('device_owner')
        if not host_id or not device_owner:
            return

        if ml2_port_util.check_port(context.original) or ml2_port_util.check_port(context.current):
            raise ml2_exc.CanNotUpdateFixedIpException

    def update_dhcp_port_pre(self, context):
        """update dhcp port pre"""
        if util.ACUtil.is_dhcp_port(context.current):
            service_plugins = ncu.get_service_plugin()
            if 'GROUP_POLICY' in service_plugins and ml2_security_util.security_group_no_change(
                    context.original, context.current):
                return
            if self.ops_version in [ac_cnst.OPS_R, ac_cnst.OPS_T, ac_cnst.OPS_W]:
                if ml2_port_util.check_port_no_change(context.original, context.current):
                    LOG.info(_LI("[AC] dhcp port %s no change"), context.current['id'])
                    return
                try:
                    self.rest_service.get_ac_port_info(context.current['id'])
                except Exception as ex:
                    LOG.error(_LE("[AC] get ac dhcp port %s failed: %s"), context.current['id'], ex)
                    return
                rec_list = self.db_if.get_plugin_record_list(
                    self.db_if.get_session('write'),
                    ACPluginSchema(res_uuid=context.current['id'], neutron_id=-1, user_oper=ac_cnst.OPER_DELETE,
                                   state=ac_cnst.IN_PROCESS))
                if rec_list:
                    LOG.info(_LI("[AC] dhcp port %s is being deleted"), context.current['id'])
                    return
            if not self.check_subnet_ready(context):
                return
            port_info = ACPortModel.ac_model_format(context.current, str(context.current['tenant_id']))
            self.__rest_request__(context, context.current['id'], port_info, ac_cnst.NW_HW_UPDATE_PORT,
                                  dry_run_state=_get_dry_run_state(context.original, ac_cnst.NW_HW_PORTS))

    @staticmethod
    def delete_snats_for(context):
        dnat_plugin = ncu.get_service_plugin().get('dnat')
        if dnat_plugin:
            from networking_huawei.drivers.ac.db.dnat.dnat import DNAT
            session = context._plugin_context.session
            dnat_list = session.query(DNAT).filter(DNAT.fixed_port_id == context.current['id']).all()
            for dnat in dnat_list:
                try:
                    dnat_plugin.delete_dnat(context._plugin_context, dnat.id)
                except Exception:
                    LOG.info('[AC] dnat already been deleted, id: %s', dnat.id)

    def delete_pf_for_port(self, context):
        """delete pf for port"""
        pf_plugin = ncu.get_service_plugin().get('PORTFORWARDING')
        if not pf_plugin:
            return
        LOG.info("[AC]checking and deleting portforwarding related to the port: %s" % context.current['id'])
        from neutron.objects.port_forwarding import PortForwarding
        admin_ctx = ncu.neutron_context.get_admin_context()
        if ncu.get_ops_version() in {ac_cnst.FSP_8_0_0, ac_cnst.FSP_8_0_3, ac_cnst.FSP_21_0}:
            with admin_ctx.session.begin(subtransactions=True):
                pf_list = PortForwarding.get_objects(admin_ctx)
        else:
            with self.context_manager.reader.using(admin_ctx):
                pf_list = PortForwarding.get_objects(admin_ctx)
        for portforwarding in pf_list:
            if portforwarding.internal_port_id != context.current['id']:
                continue
            try:
                self.reliability_api.update_plugin_record(admin_ctx, portforwarding.id, {}, ac_cnst.NW_HW_DELETE_DNAT)
            except Exception as ex:
                LOG.error(_LE('[AC]Huawei AC delete portforwarding failed: %s'), ex)
                raise ex

    @neutron_version_util.restrict_version(ac_cnst.FSP_21_0)
    def update_parent_subports_in_precommit_delete_trunk(self, resource, event, trigger, **kwargs):
        """
        baremetal场景下，收到TRUNK的PRECOMMIT_DELETE消息时，需要先下线所有的subport
        """
        payload = kwargs.get('payload')
        context = payload.context
        parent_port_id = ml2_port_util._get_parent_port_id(payload)
        LOG.info('[AC] Before deleting trunk, device_id: %s, resource: %s, event: %s, trigger: %s', parent_port_id,
                 resource, event, trigger)
        try:
            self._update_parent_subports_in_precommit_delete_trunk(context, payload)
        except Exception as ex:
            LOG.error('[AC] Failed to update subports in huawei driver: %s', ex)
            raise
        LOG.info('[AC] Before deleting trunk, reset all subports successfully')

    def _update_parent_subports_in_precommit_delete_trunk(self, context, payload):
        parent_port_id = ml2_port_util._get_parent_port_id(payload)
        parent_port = ncu.get_core_plugin().get_port(context, parent_port_id)
        LOG.info('[AC] Get payload parent port: %s', parent_port)
        if not parent_port:
            return
        if not payload.original_trunk.sub_ports:
            return
        if parent_port.get('binding:vnic_type') != ac_cnst.VNIC_TYPE_BAREMETAL:
            return
        LOG.info('[AC] Get payload subports: %s', payload.original_trunk.sub_ports)
        binding_host_id = ''
        binding_vnic_type = ac_cnst.VNIC_TYPE_NORMAL
        binding_profile = '{}'
        for subport in payload.original_trunk.sub_ports:
            with context.session.begin(subtransactions=True):
                subport_id = subport.get('port_id')
                port_binding = ncu.get_port_binding(context.session, subport_id)
                if port_binding:
                    port_binding.host = binding_host_id
                    port_binding.profile = binding_profile
                    port_binding.vnic_type = binding_vnic_type
                    context.session.merge(port_binding)
        update_subport_threadpool = eventlet.GreenPool(size=ac_cnst.MAX_SUBPORT_UPDATE_THREADS)
        for subport in payload.original_trunk.sub_ports:
            subport_id = subport.get('port_id')
            update_subport_threadpool.spawn_n(self.process_update_subport, subport_id=subport_id,
                                              host_id=binding_host_id,
                                              trunk_id=None, migrating_host=None,
                                              is_sub_baremetal_scenario=True,
                                              binding_vnic_type=binding_vnic_type,
                                              binding_profile=binding_profile)
        update_subport_threadpool.waitall()

    @lock_db.wrap_db_lock(lock_db.DELETE_PORT_LOCK)
    def delete_port_precommit(self, context):
        """This function sends port delete message to AC

        :param context: DB context for the port delete
        :return: None
        """
        try:
            ac_osprofiler.record_chain_start("delete port start,ID:")
            LOG.info(_LI("[AC]Request AC to delete port: %s."), context.current)
            self.delete_snats_for(context)
            self.delete_pf_for_port(context)
            self.check_fm_for_port(context.current['id'])
            if self.data_filter.not_in_white_or_in_black_list(
                    context._plugin_context, context.current, ac_cnst.NW_HW_PORTS, context.current['id']):
                return

            if self.data_filter.port_contain_anti_affinity(context.current):
                return

            fixed_ips = context.current.get('fixed_ips', None)
            if ncu.IS_FSP and ACCommonUtil.is_port_contain_ipv4_address(fixed_ips):
                PublicService().delete_fip(context)

            if ncu.need_ac_status(ac_cnst.NW_HW_AC_STATUS_PORT):
                self.ac_proc_db_func.delete_db_ac_proc_status(context.current['id'])
                LOG.info(_LI("[AC] Deleting port, delete ac process status in Neutron DB successful, id: %s "),
                         context.current['id'])

            port_info = ACPortModel.ac_model_format(context.current, str(context.current['tenant_id']))

            self.delete_bm_mode_record(context, port_info)

            self.__rest_request__(context, context.current['id'], port_info, ac_cnst.NW_HW_DELETE_PORT,
                                  dry_run_state=_get_dry_run_state(context.current, ac_cnst.NW_HW_PORTS))

        except Exception as ex:
            LOG.error(_LE("[AC]Delete port: %s failed. exceptions, raise exception to plugin to rollback."),
                      context.current['id'])
            LOG.error(_LE("[AC] The exception is :%s."), ex)
            self.raise_ml2_exception(ex, 'port')

    def check_fm_for_port(self, port_id):
        """check and delete flow-mirrors related to the port"""
        fm_plugin = ncu.get_service_plugin().get('flow-mirror')
        if fm_plugin:
            from networking_huawei.drivers.ac.db.flow_mirror import flow_mirror_db
            LOG.info("[AC]checking the port : %s related to flow-mirror.", port_id)
            admin_ctx = ncu.neutron_context.get_admin_context()
            tass_db = flow_mirror_db.FlowMirrorDbMixin()
            filters = {'source_port': [port_id]}
            tap_flows = tass_db.get_tap_flows(admin_ctx, filters=filters)
            for tap_flow in tap_flows:
                try:
                    self.flow_mirror_plugin.delete_tap_flow(admin_ctx, tap_flow.get("id"))
                except Exception:
                    LOG.info('[AC] tap-flow already been deleted, id: %s', tap_flow.get("id"))
            LOG.info('[AC]flow-mirror related the port and delete flow-mirror success.')

    def delete_bm_mode_record(self, context, port_info):
        """delete bm mode record"""
        binding_profile = context.current.get('binding:profile', None)
        if binding_profile:
            local_link = binding_profile.get('local_link_information', None)
            if local_link and len(local_link) > 1:
                self.db_if.delete_bm_bond_mode_record(context._plugin_context.session, port_info['uuid'])

    def update_subport_thread(self, context, host_id, trunk_id):
        """update subnet port thread"""
        LOG.info("update_subport_thread start")
        sub_ports = [item['port_id'] for item in context.current['trunk_details']['sub_ports']]
        LOG.debug("[SubPortUpdate]sub ports:%s", sub_ports)
        bfd_leader_port_id = None

        admin_context = ncu.neutron_context.get_admin_context()
        query_result = admin_context.session.query(RouterExRoute, IPAllocation).join(
            IPAllocation, RouterExRoute.nexthop == IPAllocation.ip_address).filter(IPAllocation.port_id.in_(sub_ports))
        for (route, ips) in query_result:
            LOG.debug("[SubPortUpdate]find bfd_leader subport:route=%s,ip_address=%s", route, ips)
            if route.extra_opt.get('bfd_leader'):  # 标记成首Port
                if bfd_leader_port_id is None:
                    bfd_leader_port_id = ips.port_id
                elif bfd_leader_port_id != ips.port_id:
                    LOG.error("find multiple bfd_leader subport:bfd_leader_port_id=%s,route=%s,ip_address=%s",
                              bfd_leader_port_id, route, ips)
                    raise ExRouteMultiBfdLeader(ex_route=route)

        # 将parent port的migrating_to作为migrating-host给subport使用
        migrating_host = None
        current_profile = context.current['binding:profile']
        if self._need_update_migrate_host(current_profile):
            migrating_host = current_profile['migrating_to']
        original_host = ACPortModel.get_host_id_from_port(context.original)

        # host_A-->空:首子port最后更新
        # 空-->host_B:首子port先更新
        # host_A-->host_B:首子port先更新
        # host_A-->host_A,migrating_host_B-->host_B:首子port先更新
        # host_A-->host_A,migrating_host_B-->host_A,migrating_host_C:首子port先更新
        last_flg = original_host and not host_id
        LOG.debug("[SubPortUpdate]original_host=%s,migrating_host=%s,host_id=%s,last_flg=%s,bfd_leader_port_id=%s",
                  original_host,
                  migrating_host, host_id, last_flg, bfd_leader_port_id)
        if not last_flg and bfd_leader_port_id:  # 首子port先更新
            self.process_update_subport(subport_id=bfd_leader_port_id, host_id=host_id, trunk_id=trunk_id,
                                        migrating_host=migrating_host)
            LOG.debug(
                "[SubPortUpdate]deal bfd_leader subport first:port_id=%s,host_id=%s,trunk_id=%s,migrating_host=%s",
                bfd_leader_port_id, host_id, trunk_id, migrating_host)

        update_subport_pool = eventlet.GreenPool(size=ac_cnst.MAX_SUBPORT_UPDATE_THREADS)
        for elem in filter(lambda x: x != bfd_leader_port_id, sub_ports):  # 排除首子port
            update_subport_pool.spawn_n(self.process_update_subport, subport_id=elem, host_id=host_id,
                                        trunk_id=trunk_id, migrating_host=migrating_host)
            LOG.debug("[SubPortUpdate]deal other subport:port_id=%s,host_id=%s,trunk_id=%s,migrating_host=%s",
                      elem, host_id, trunk_id, migrating_host)

        if last_flg and bfd_leader_port_id:  # 首子port最后更新
            update_subport_pool.waitall()
            self.process_update_subport(subport_id=bfd_leader_port_id, host_id=host_id, trunk_id=trunk_id,
                                        migrating_host=migrating_host)
            LOG.debug("[SubPortUpdate]deal bfd_leader subport last:port_id=%s,host_id=%s,trunk_id=%s,migrating_host=%s",
                      bfd_leader_port_id, host_id, trunk_id, migrating_host)
        LOG.info("[SubPortUpdate]update_subport_thread end")

    def process_update_subport(self, **kwargs):
        """process update subport"""
        LOG.info("[AC] Process update subport start")
        subport_id = kwargs.get('subport_id')
        host_id = kwargs.get('host_id')
        trunk_id = kwargs.get('trunk_id')
        migrating_host = kwargs.get('migrating_host')
        is_sub_baremetal_scenario = kwargs.get('is_sub_baremetal_scenario', False)
        binding_vnic_type = kwargs.get('binding_vnic_type')
        binding_profile = kwargs.get('binding_profile', '{}')
        subport_context = ncu.neutron_context.get_admin_context()
        interface_query = ncu.get_core_plugin()
        subport_interface = interface_query.get_port(subport_context, subport_id)
        subport_info = ACPortModel.ac_model_format(subport_interface, str(subport_interface['tenant_id']))
        LOG.info(_LI("[AC] The subport info in db is: %s."), subport_info)
        subport_info['host-id'] = host_id
        if migrating_host:
            subport_info['profile'] = str(ast.literal_eval(jsonutils.dumps({'migrating-host': migrating_host})))
        session = self.db_if.get_session('write')
        old_port = ncu.get_port_binding(session, subport_id)
        dry_run_state = _get_dry_run_state(old_port, ac_cnst.NW_HW_PORTS)
        if self.vhost_user and not self.ops_version.startswith('FusionSphere'):
            if host_id:
                subport_info['device-owner'] = 'trunk:subport'
                subport_info['device-id'] = trunk_id
                device_owner = 'trunk:subport'
                subport_device_id = trunk_id
            else:
                subport_info['device-owner'] = ''
                subport_info['device-id'] = ''
                device_owner = ''
                subport_device_id = ''
            LOG.info(_LI("[AC] Start to send request of subport: %s"), subport_info)
            self.reliability_api.update_plugin_record(subport_context, subport_id, subport_info,
                                                      ac_cnst.NW_HW_UPDATE_PORT, dry_run_state=dry_run_state)
            self.db_if.update_subport_device_info(subport_id, subport_device_id, device_owner, session=session)
        elif is_sub_baremetal_scenario:
            subport_info['profile'] = binding_profile
            subport_info['vnic-type'] = binding_vnic_type
            LOG.info(_LI("[AC] Start to send rest request of subport: %s"), subport_info)
            self.__rest_request__(subport_context, subport_id, subport_info, ac_cnst.NW_HW_UPDATE_PORT,
                                  dry_run_state=dry_run_state)
            return
        else:
            LOG.info(_LI("[AC] Start to send rest request of subport: %s"), subport_info)
            self.__rest_request__(subport_context, subport_id, subport_info, ac_cnst.NW_HW_UPDATE_PORT,
                                  dry_run_state=dry_run_state)
        with session.begin(subtransactions=True):
            if old_port:
                old_port.host = host_id
                session.merge(old_port)
                session.flush()

    def __rest_request__(self, context, source_id, entry_info, operation, dry_run_state=DryRunStateEnum.DEFAULT):
        LOG.info(_LI("[AC]Start to request."))
        if entry_info.get('device-owner') == 'trunk:subport':
            self.reliability_api.update_plugin_record(context, source_id, entry_info, operation,
                                                      dry_run_state=dry_run_state)
        else:
            self.reliability_api.update_plugin_record(context._plugin_context, source_id, entry_info, operation,
                                                      dry_run_state=dry_run_state)

    def check_vlan_transparency(self, context):
        """Currently huawei driver support vlan transparency."""
        return True

    def _get_bind_port_level(self, network_id, host_id, context):
        physical_network = self.physical_network
        if context.current['binding:profile'].get("physical_network", None):
            physical_network = context.current['binding:profile'].get("physical_network", None)

        high_priority_physnet = ml2_port_util.get_high_priority_physnet(context)
        if high_priority_physnet:
            physical_network = high_priority_physnet
        bind_port_info = {'input': {'host-id': host_id, 'physical-network': physical_network, 'network-id': network_id,
                                    'port-id': context.current['id']}}
        ac_osprofiler.record_call_chain("get vlan from AC start")
        response = self.rest_service.send_bind_port_request(bind_port_info)

        if response:
            output = response.get('huawei-ac-neutron-bind-port-level:output', None)
            LOG.debug(_LI('[AC] Bind port level response get from AC: %s'), output)

            if output:
                next_segment = output.get('next-segment', {})

                network_type = next_segment.get('network-type', None)
                network_type_prefix = ac_cnst.NETWORK_TYPE_PREFFIX
                if not network_type or not network_type.startswith(network_type_prefix):
                    self._handle_bind_lever_error("network type in bind port level response is None.")

                segmentation_id = next_segment.get('segmentation-id', None)
                if not segmentation_id:
                    self._handle_bind_lever_error("segmentation id in bind port level response is None.")

                dynamic_segment = {api.PHYSICAL_NETWORK: physical_network, api.NETWORK_TYPE: network_type[40:],
                                   api.SEGMENTATION_ID: segmentation_id}
                ac_osprofiler.record_call_chain("get vlan from AC success:" + str(dynamic_segment))
                return dynamic_segment
        self._handle_bind_lever_error("Failed to get bind port level response from AC.")
        return dict()

    def _handle_bind_lever_error(self, error):
        LOG.error(_LE('[AC] %s'), error)
        ac_osprofiler.record_chain_exception("error")
        self.raise_ml2_exception(error, 'port')

    def _set_bind_port_level(self, network_id, host_id, dynamic_segment, port_id=None):
        network_type = ac_cnst.NETWORK_TYPE_PREFFIX
        network_type += dynamic_segment[api.NETWORK_TYPE]
        next_segment = {'network-type': network_type, 'segmentation-id': dynamic_segment[api.SEGMENTATION_ID]}
        bind_port_info = {'input': {'host-id': host_id, 'physical-network': dynamic_segment[api.PHYSICAL_NETWORK],
                                    'network-id': network_id, 'next-segment': next_segment, 'port-id': port_id}}
        self.rest_service.send_bind_port_request(bind_port_info)

    def _bind_vmware_port(self, context, segment):
        LOG.info(_LI('[AC]Begin to bind vmware port,context:%s,original context:%s,segment:%s.'), context.current,
                 context.original, segment)
        ac_osprofiler.record_call_chain("bind vmware port")
        network_id = context.current.get('network_id')
        port_id = context.current.get('id')
        host_id = context.current.get('binding:host_id')
        vcenter_id = self.vcenter_host_info.get(host_id)

        if vcenter_id:
            physical_network_list = self.vcenter_network_info.get(vcenter_id, [])
        else:
            cluster_id = self.vcenter_cluster_info.get(host_id, {})
            vcenter_id = self.vcenter_host_info.get(cluster_id, {})
            physical_network_list = self.vcenter_network_info.get(vcenter_id, [])
        LOG.info(_LI('[AC]%s physical network list:%s'), port_id, physical_network_list)

        for physical_network in physical_network_list:
            param_dict = {'host_id': host_id, 'network_id': network_id, 'physical_network': physical_network,
                          'port_id': port_id, 'segment': segment}
            try:
                if self._check_dynamic_segment_exists(context, param_dict):
                    return
                self._allocated_and_binding(context, param_dict)
            except Exception as ex:
                LOG.error(_LE('[AC]Failed to bind vmware port %s:%s'), port_id, ex)
                ac_osprofiler.record_call_chain("bind vmware port fail" + str(ex))
                return

    def _allocated_and_binding(self, context, param_dict):
        network_id = param_dict.get('network_id')
        physical_network = param_dict.get('physical_network')
        host_id = param_dict.get('host_id')
        port_id = param_dict.get('port_id')
        segment = {api.PHYSICAL_NETWORK: physical_network, api.NETWORK_TYPE: "vlan"}
        dynamic_segment = context.allocate_dynamic_segment(segment)
        if dynamic_segment:
            LOG.info(_LI('[AC]The dynamic segment %s allocate successfully,network id %s'), dynamic_segment, network_id)
            if host_id not in self.logical_hostnames:
                ac_osprofiler.record_call_chain("segment allocated,set to AC:" + str(dynamic_segment))
                self._set_bind_port_level(network_id, host_id, dynamic_segment, port_id)
            context.continue_binding(segment.get(api.ID), [dynamic_segment])

    def _check_dynamic_segment_exists(self, context, param_dict):
        network_id = param_dict.get('network_id')
        physical_network = param_dict.get('physical_network')
        host_id = param_dict.get('host_id')
        port_id = param_dict.get('port_id')
        if self.ops_version in OPS_VERSIONS:
            from neutron.db import segments_db
            dynamic_segment = segments_db.get_dynamic_segment(
                ncu.neutron_context.get_admin_context(), network_id, physical_network)
        else:
            dynamic_segment = ml2db.get_dynamic_segment(self.db_if.get_session(), network_id, physical_network)
        if dynamic_segment:
            LOG.info(_LI('[AC]The dynamic segment %s already exists,no need to allocate.'), dynamic_segment)
            if host_id not in self.logical_hostnames:
                ac_osprofiler.record_call_chain("segment already exists,set to AC")
                self._set_bind_port_level(network_id, host_id, dynamic_segment, port_id)
            context.continue_binding(param_dict.get('segment')[api.ID], [dynamic_segment])
            return True
        return False

    def _bind_trunk_parent_port(self, context, segment):
        LOG.info(_LI('[AC]Begin to bind trunk parent port,context:%s,segment:%s'), context.__dict__, segment)
        ac_osprofiler.record_call_chain("bind trunk parent sriov port to flat")
        physical_network = segment.get('physical_network')
        network_id = context.current.get('network_id')
        dynamic_segment = {api.PHYSICAL_NETWORK: physical_network, api.NETWORK_TYPE: 'flat', api.SEGMENTATION_ID: 0}
        self._hw_add_network_segment(network_id, dynamic_segment)
        LOG.info(_LI('[AC] dynamic segment: %s'), dynamic_segment)
        context.continue_binding(segment[api.ID], [dynamic_segment])

    def add_network_segment(self, context):
        """add network segment"""
        LOG.info(_LI('[AC] Begin to add network segment: %s'), context.current)
        ac_osprofiler.record_call_chain("start add network segment")
        for segment in context.segments_to_bind:
            self._process_network_vxlan(context, segment)

    def _hw_add_network_segment(self, network_id, dynamic_segment):
        if self.ops_version == ac_cnst.OPS_N:
            ml2db.add_network_segment(ncu.neutron_context.get_admin_context(), network_id, dynamic_segment,
                                      is_dynamic=True)
        elif self.ops_version in OPS_VERSIONS:
            from neutron.db import segments_db
            segments_db.add_network_segment(
                ncu.neutron_context.get_admin_context(), network_id, dynamic_segment, is_dynamic=True)
        else:
            ml2db.add_network_segment(self.db_if.get_session('write'), network_id, dynamic_segment, is_dynamic=True)

    def _process_vxlan_with_device_owner(self, context, segment):
        host_id = context.current.get('binding:host_id', None)
        network_id = context.current.get('network_id', None)
        device_owner = context.current.get('device_owner', None)

        if not ml2_util.filter_device_owner_process(device_owner):
            context.set_binding(segment["id"], 'service', {})
            return
        try:
            if ACPortModel.is_f5_port(context.current):
                logical_host_id = ACPortModel.convert_host_id(self.LB_host_mappings, host_id)
                LOG.info(_LI("port[%s]: convert %s to %s"), context.current["id"], host_id, logical_host_id)
                ac_osprofiler.record_call_chain("bind f5 port")
                host_id = logical_host_id

            dynamic_segment = self._get_dynamic_segment(context, network_id, host_id)

            self._hw_add_network_segment(network_id, dynamic_segment)
            if device_owner in {ac_cnst.DEVICE_OWNER_F5_V1, ac_cnst.DEVICE_OWNER_F5_V2} or device_owner.startswith(
                    DEVICE_OWNER_LB):
                context.set_binding(dynamic_segment.get(api.ID), 'service', {})
            else:
                context.continue_binding(segment[api.ID], [dynamic_segment])

            LOG.info(_LI('[AC] Huawei AC add network segment successfully.'))
        except Exception as ex:
            ac_osprofiler.record_chain_exception("fail to bind port:" + str(ex))
            LOG.error(_LE('[AC] Failed to add network segment in huawei driver: %s'), traceback.format_exc())
            raise ex

    def _get_dynamic_segment(self, context, network_id, host_id):
        profile = context.current.get('binding:profile', {})
        network = self.db_base_plugin.get_network(context._plugin_context, network_id)
        vxlan_physnet_prefix = network.get('vxlan_physnet_prefix')
        if vxlan_physnet_prefix and not ACPortModel.is_f5_port(context.current):
            allocation = self._allocate_vlan(context)
            physical_network = ml2_util.get_h3c_physical_network(context, vxlan_physnet_prefix)
            dynamic_segment = {api.PHYSICAL_NETWORK: physical_network, api.NETWORK_TYPE: 'vlan',
                               api.SEGMENTATION_ID: allocation}
            cvk_esxi_host = profile.get('cvk_host') or profile.get('esxi_host')
            if host_id and cvk_esxi_host:
                self._set_bind_port_level(network_id, cvk_esxi_host, dynamic_segment)
        elif context.current.get('device_owner', None) == 'trunk:subport':
            dynamic_segment = self.get_vxlan_subport_dynamic_segment(context)
            return dynamic_segment
        else:
            dynamic_segment = self._get_bind_port_level(network_id, host_id, context)
        return dynamic_segment

    def _process_network_vxlan(self, context, segment):
        """ process network vxlan"""
        network_id = context.current.get('network_id', None)
        device_owner = context.current.get('device_owner', None)

        if ml2_util.is_segment_ac_vxlan(segment):
            self.add_segment_vxlan(context, segment, device_owner)
        elif ml2_util.is_segment_ac_vlan(segment):
            self.add_segment_vlan(context, network_id, segment)
        else:
            LOG.info(_LI('[AC]No need to add network segment in huawei driver: network type is not vxlan.'))
            ac_osprofiler.record_call_chain("vlan network no need bind")

    def add_segment_vlan(self, context, network_id, segment):
        """add segment vlan"""
        router_external = vlan_transparent = False
        if network_id:
            network = ncu.get_core_plugin().get_network(context._plugin_context, network_id)
            if network:
                router_external = network.get('router:external', False)
                vlan_transparent = network.get('vlan_transparent', False)
        if router_external or not vlan_transparent:
            vnic_type = context.current.get('binding:vnic_type')
            trunk_details = context.current.get('trunk_details')
            if vnic_type == 'direct' and (trunk_details or self.sub_port_need_set_flat(context)):
                self._bind_trunk_parent_port(context, segment)

    def sub_port_need_set_flat(self, context):
        """ whether sub port need set flat """
        if not self.data_filter.port_contain_anti_affinity(context.current):
            return False
        profile = context.current.get('binding:profile', {})
        anti_affinity_port_id = profile.get('anti_affinity_port', None)
        anti_affinity_port = ncu.get_core_plugin().get_port(context._plugin_context, anti_affinity_port_id)
        if anti_affinity_port.get('trunk_details'):
            return True
        return False

    def add_segment_vxlan(self, context, segment, device_owner):
        """add segment vxlan"""
        if self._is_vmware_port(context):
            self._bind_vmware_port(context, segment)
        elif device_owner:
            self._process_vxlan_with_device_owner(context, segment)
        else:
            LOG.error(_LE('[AC] Failed to add network segment in huawei driver: device owner is None.'))

    def _if_fusioncompute_logical_host(self, context, network):
        if context.current.get('binding:host_id') in self.logical_hostnames and not self._is_vmware_port(
                context) and not network.get('vxlan_physnet_prefix'):
            return True
        return False

    def _filter_bind_port(self, context):
        if self.data_filter.not_in_white_or_in_black_list(
                context._plugin_context, context.current, ac_cnst.NW_HW_PORTS, context.current['id']):
            return True

        network_id = context.current.get('network_id')
        network = self.db_base_plugin.get_network(context._plugin_context, network_id)

        if ml2_util.is_compute_external(context):
            LOG.info(_LI('[AC] port has already contained a vlan,no need send bind request.'), context.current)
            return True

        if self._if_fusioncompute_logical_host(context, network):
            LOG.info(_LI('[AC] %s host id is FusionCompute logical host,there is no need to bind port.'),
                     context.current)
            ac_osprofiler.record_chain_end_with_reason("port no host,return")
            return True

        segments = ml2_util.hw_get_segment(context, network_id)
        LOG.info(_LI('[AC] network %s segments %s'), network_id, segments)
        if len(segments) > 1 and not ml2_util.if_vhost_compute(context):
            LOG.info(_LI('[AC]there is no need to bind port %s'), context.current)
            ac_osprofiler.record_chain_end_with_reason("port already bind segment is " + str(segments) + ",return")
            return True
        return False

    def bind_port(self, context):
        """This function sends bind port to VM message to AC

        :param context: DB context for the port binding
        :return: None
        """
        ac_osprofiler.record_chain_start("bind port start,ID:" + context.current['id'])
        if self._filter_bind_port(context):
            return
        if context.current['binding:profile'].get('sdn_binding_status') == 'failed':
            vif_details = {}
            for segment in context.segments_to_bind:
                context.set_binding(segment[api.ID], ncu.VIF_TYPE_BINDING_FAILED, vif_details)
            return
        if not (ml2_util.if_vhost_compute(context) or ml2_port_util.is_baremetal_port(context.current)):
            self.add_network_segment(context)
        else:
            # Don't bind direct port. Only bind normal port.
            if context._binding.vnic_type == ncu.VNIC_DIRECT or context._binding.vif_type == VIF_TYPE_VHOST_USER:
                ac_osprofiler.record_chain_end_with_reason("no need to bind VNIC_DIRECT or VIF_TYPE_VHOST_USER")
                raise Exception
            for segment in context.segments_to_bind:
                ml2_port_util.bind_port_once(context, segment)
        ac_osprofiler.record_chain_end()

    def check_qos_dependent(self, session, res_info):
        """Check whether the resources of business dependnet is being
        deleting.
        """
        if 'qos_policy_id' in res_info and res_info['qos_policy_id'] is not None:
            qos_policy_id = res_info['qos_policy_id']
            dependent_res = self.db_if.get_plugin_record_list(
                session, ACPluginSchema(neutron_id=-1, res_uuid=qos_policy_id))
            if dependent_res is None:
                return
            LOG.info(_LI("When deal %s, detect dependent qos %s"), res_info['id'], dependent_res)
            for qos_to_deal in dependent_res:
                if qos_to_deal['user_oper'] == ac_cnst.OPER_DELETE:
                    raise ml2_exc.MechanismDriverError()

    def _allocate_vlan(self, context):
        network_id = context.current.get('network_id')
        cluster_id = context.current.get('binding:host_id')
        if network_id and cluster_id:
            LOG.info(_LI('[AC] begin to allocate vlan for cluster(%s) and network(%s)'), cluster_id, network_id)
            context = context._plugin_context
            network = self.db_base_plugin.get_network(context, network_id)
            if 'vxlan_physnet_prefix' not in network or not self.context_manager:
                LOG.info(_LI('[AC] no vxlan physnet prefix in network'))
                return None
            with self.context_manager.writer.using(context):
                allocated = context.session.query(AllocateVlanSchema).filter(
                    AllocateVlanSchema.cluster_id == cluster_id, AllocateVlanSchema.network_id == network_id,
                ).one_or_none()
                if allocated:
                    LOG.info(_LI('[AC] vlan(%s) for cluster(%s) and network(%s) is already allocated'), allocated.vlan,
                             cluster_id, network_id)
                    return allocated.vlan
                allocated = context.session.query(AllocateVlanSchema).filter(
                    AllocateVlanSchema.cluster_id == cluster_id).all()
                vlan_set = ml2_util.parse_vlan_ranges(set(int(a.vlan) for a in allocated))
                if not vlan_set:
                    error_msg = 'no more vlan available for ' + cluster_id
                    raise BadRequest(resource='vlan', msg=error_msg)
                allocated = AllocateVlanSchema(cluster_id=cluster_id, network_id=network_id, vlan=vlan_set.pop())
                context.session.add(allocated)
                LOG.info(_LI('[AC] allocate vlan(%s) for cluster(%s) and network(%s)'), allocated.vlan, cluster_id,
                         network_id)
                return allocated.vlan
        return None

    def raise_ml2_exception(self, ex, res_type):
        """raise ml2 exception"""
        if self.ops_version in [ac_cnst.OPS_P]:
            raise BadRequest(resource=res_type, msg=ex)
        raise ml2_exc.MechanismDriverError(error=ex)

    def get_vxlan_subport_dynamic_segment(self, ctx):
        try:
            LOG.info(_LI('[AC] Begin to add subport dynamic segment'))
            trunk = \
                ncu.get_service_plugin().get("trunk").get_trunk(context=ctx._plugin_context,
                                                                trunk_id=ctx.current['device_id'])
            LOG.info(_LI('[AC] trunk: %s'), trunk)
            vlan_id = None
            for sp in trunk.get("sub_ports", []):
                if sp['port_id'] == ctx.current['id']:
                    vlan_id = sp.get("segmentation_id", None)
            if vlan_id:
                dynamic_segment = {
                    api.PHYSICAL_NETWORK: self.physical_network,
                    api.NETWORK_TYPE: 'vlan',
                    api.SEGMENTATION_ID: vlan_id
                }
                return dynamic_segment
            else:
                LOG.error(_LE('[AC] Failed to get subport dynamic_segment'))
                return dict()
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to get subport dynamic segment : %s'), ex)
            return dict()
