#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ac ipv6 ns qos policy driver"""

from oslo_log import log as logging
from neutron.db import ipv6_ns_qos_policy_db as nqp_db
try:
    from neutron.services.ipv6_ns_qos_policy.drivers import \
        Ipv6NsQosPolicyAbstractDriver
except ImportError:
    class Ipv6NsQosPolicyAbstractDriver(object):
        fake = None
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.model.ipv6_ns_qos_policy_model \
    import ACIpv6NsQoSpolicyModel
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei._i18n import _LI, _LE

LOG = logging.getLogger(__name__)


class HuaweiIpv6NsQosPolicyDriver(nqp_db.Ipv6NsQoSPolicyDbMixin,
                                  Ipv6NsQosPolicyAbstractDriver):
    """HuaweiIpv6NsQosPolicyDriver"""
    def __init__(self, *args, **kwargs):
        LOG.info(_LI("[AC] Init Huawei ipv6 ns qos policy driver."))
        self.ac_driver = ACReliabilityAPI(ac_const.NW_HW_IPV6_NS_QOS_POLICY)
        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for Huawei ipv6 ns qos policy driver."))

    def add_ipv6_ns_qos_policy(self, policy):
        """add ipv6 ns qos policy"""
        LOG.info(_LI("[AC] Begin to create ipv6 ns qos policy: %s."), policy)
        try:
            policy_info = ACIpv6NsQoSpolicyModel.ac_model_format(policy)
            self.ac_driver.update_plugin_record(
                [],
                policy_info['uuid'],
                policy_info,
                ac_const.NW_HW_CREATE_IPV6_NS_QOS_POLICY)
        except Exception as except_msg:
            LOG.error(_LE("[AC] Failed to create ipv6 ns qos policy in "
                          "huawei driver: %s."), except_msg)
            raise
        LOG.info(_LI("[AC] Huawei AC create ipv6 ns qos policy successfully."))

    def update_ipv6_ns_qos_policy(self, policy_id, new_policy_db):
        """update ipv6 ns qos policy"""
        LOG.info(_LI("[AC] Begin to update ipv6 ns qos policy: %s."),
                 new_policy_db)
        try:
            policy_info = ACIpv6NsQoSpolicyModel.ac_model_format(new_policy_db)
            self.ac_driver.update_plugin_record(
                [],
                policy_id,
                policy_info,
                ac_const.NW_HW_UPDATE_IPV6_NS_QOS_POLICY)
        except Exception as except_msg:
            LOG.error(_LE("[AC] Failed to update ipv6 ns qos policy in "
                          "huawei driver: %s."), except_msg)
            raise
        LOG.info(_LI("[AC] Huawei AC update ipv6 ns qos policy successfully."))

    def remove_ipv6_ns_qos_policy(self, policy_id):
        """remove ipv6 ns qos policy"""
        LOG.info(_LI("[AC] Begin to delete ipv6 ns qos policy: %s."), policy_id)
        try:
            self.ac_driver.update_plugin_record(
                [],
                policy_id,
                {},
                ac_const.NW_HW_DELETE_IPV6_NS_QOS_POLICY)
        except Exception as except_msg:
            LOG.error(_LE("[AC] Failed to delete ipv6 ns qos policy "
                          "in huawei driver: %s."), except_msg)
            raise
        LOG.info(_LI("[AC] Huawei AC delete ipv6 ns qos policy successfully."))

