#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Implementation of the Neutron QoS driver."""

import datetime
import threading

from neutron.db.qos import models as qos_db_model
from neutron.objects.qos import policy as policy_object
from oslo_log import log as logging

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.client.service import DryRunStateEnum
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common import osprofiler_warp as ac_osprofiler
from networking_huawei.drivers.ac.common.util import DataFilterUtil
from networking_huawei.drivers.ac.model.qos_model import ACQoSpolicyModel
from networking_huawei.drivers.ac.plugins.dry_run import dry_run_util
from networking_huawei.drivers.ac.sync.message_reliability_api import ACReliabilityAPI

try:
    from neutron.common import exceptions as n_exc
except ImportError:
    from neutron_lib.exceptions import qos as n_exc
try:
    from neutron.db import api as db_api
except ImportError:
    from neutron_lib.db import api as db_api
try:
    from neutron.services.qos.notification_drivers.qos_base import \
        QosServiceNotificationDriverBase
except ImportError:
    class QosServiceNotificationDriverBase(object):
        """Fake QoS service notification driver base class."""
        fake = None
try:
    from neutron.services.qos.drivers.base import DriverBase
except ImportError:
    try:
        from neutron_lib.services.qos.base import DriverBase
    except ImportError:
        class DriverBase(object):
            """Fake QoS driver base class."""
            fake = None

LOG = logging.getLogger(__name__)

DEAL_POLICY = threading.local()
DRIVER = None

DB_INTEGER_MAX_VALUE = 2 ** 31 - 1
INGRESS_DIRECTION = 'ingress'
EGRESS_DIRECTION = 'egress'
VALID_DIRECTIONS = (INGRESS_DIRECTION, EGRESS_DIRECTION)
VALID_DSCP_MARKS = [0, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34,
                    36, 38, 40, 46, 48, 56]
VALID_COS_MARKS = [0, 1, 2, 3, 4, 5, 6, 7]

SUPPORTED_RULES = {
    'cos_marking': {
        "cos_mark": {'type:values': VALID_COS_MARKS}
    },
    'bandwidth_limit': {
        "max_kbps": {
            'type:range': [0, DB_INTEGER_MAX_VALUE]},
        "max_burst_kbps": {
            'type:range': [0, DB_INTEGER_MAX_VALUE]},
        "direction": {
            'type:values': VALID_DIRECTIONS}
    },
    'dscp_marking': {
        "dscp_mark": {'type:values': VALID_DSCP_MARKS}
    }
}


class HuaweiQosServiceNotificationDriver(QosServiceNotificationDriverBase,
                                         DriverBase):
    """Huawei AC service notification driver for QoS."""

    def __init__(self, *args, **kwargs):
        LOG.info(_LI("[AC] Init Huawei QoS driver."))
        self.ops_version = ncu.get_ops_version()
        self.data_filter = DataFilterUtil()
        if not hasattr(QosServiceNotificationDriverBase, 'fake'):
            if self.ops_version in ac_const.OPS_VERSION_O_PQRTW_6_21:
                super(HuaweiQosServiceNotificationDriver, self).__init__(
                    name='openvswitch',
                    vif_types=['ovs', 'vhostuser'],
                    vnic_types=['normal'],
                    supported_rules=SUPPORTED_RULES,
                    requires_rpc_notifications=True)
            else:
                super(HuaweiQosServiceNotificationDriver, self).__init__()
        elif not hasattr(DriverBase, 'fake'):
            super(HuaweiQosServiceNotificationDriver, self). \
                __init__(*args, **kwargs)
        self.ac_driver = ACReliabilityAPI(ac_const.NW_HW_QOS)
        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for Huawei QoS driver."))

    @classmethod
    def get_description(cls):
        """Huawei AC QoS service notification driver description."""
        return "Notify Huawei AC to create/update/delete QoS policy."

    def _check_qos_policy_in_use(self, context, policy_id):
        if self.ops_version in [ac_const.OPS_Q, ac_const.OPS_R, ac_const.OPS_T, ac_const.OPS_W,
                                ac_const.FSP_6_5, ac_const.FSP_21_0]:
            self._check_qos_policy_in_use_ops_high(context, policy_id)
        elif self.ops_version == ac_const.OPS_P and \
                self._has_db_context_reader():
            self._check_qos_policy_in_use_ops_high(context, policy_id)
        else:
            self._check_qos_policy_in_use_ops_low(context, policy_id)

    @classmethod
    def _has_db_context_reader(cls):
        try:
            from neutron.objects.base import NeutronDbObject
            return hasattr(NeutronDbObject, "db_context_reader")
        except ImportError:
            return False

    @classmethod
    def _check_qos_policy_in_use_ops_high(cls, context, policy_id):
        """ops_version in [ac_const.OPS_Q, ac_const.OPS_R, ac_const.OPS_T,
        ac_const.FSP_6_5, ac_const.FSP_21_0]
        """
        from neutron.objects.qos import binding
        from neutron.objects.qos.policy import QosPolicy
        from neutron.objects import base as base_db
        binding_models = {'port': binding.QosPolicyPortBinding,
                          'network': binding.QosPolicyNetworkBinding}
        if hasattr(binding, 'QosPolicyFloatingIPBinding'):
            binding_models['fip'] = binding.QosPolicyFloatingIPBinding
        with QosPolicy.db_context_writer(context):
            for obj_type, obj_class in binding_models.items():
                pager = base_db.Pager(limit=1)
                binding_obj = obj_class.get_objects(context,
                                                    policy_id=policy_id,
                                                    _pager=pager)
                if binding_obj:
                    raise n_exc.QosPolicyInUse(
                        policy_id=policy_id,
                        object_type=obj_type,
                        object_id=binding_obj[0]['%s_id' % obj_type])

    def _check_qos_policy_in_use_ops_low(self, context, policy_id):
        """ops_version not in [ac_const.OPS_Q, ac_const.OPS_R,
        ac_const.OPS_T, ac_const.FSP_6_5, ac_const.FSP_21_0]
        """
        models = (
            ('network', qos_db_model.QosNetworkPolicyBinding),
            ('port', qos_db_model.QosPortPolicyBinding)
        )
        with db_api.autonested_transaction(context.session):
            for obj_type, model in models:
                if self.ops_version in [ac_const.OPS_N, ac_const.OPS_O,
                                        ac_const.OPS_P]:
                    from neutron.objects.db import api as obj_db_api
                    binding_obj = obj_db_api.get_object(
                        context, model, policy_id=policy_id)
                else:
                    binding_obj = db_api.get_object(
                        context, model, policy_id=policy_id)
                if binding_obj:
                    raise n_exc.QosPolicyInUse(
                        policy_id=policy_id,
                        object_type=obj_type,
                        object_id=binding_obj['%s_id' % obj_type])

    def create_policy(self, context, policy):
        """Create policy."""
        LOG.info(_LI("[AC] Begin to create QoS policy: %s."), policy)
        ac_osprofiler.record_chain_start("create QoS policy start,ID:" + policy['id'])

        if self.data_filter.not_in_white_or_in_black_list(context, policy, ac_const.NW_HW_QOS_POLICY, policy['id']):
            return
        policy_info = ACQoSpolicyModel.ac_model_format(policy)
        try:
            self.ac_driver.update_plugin_record(context, policy_info.get('uuid'), policy_info,
                                                ac_const.NW_HW_CREATE_QOS_POLICY)
        except Exception as except_msg:
            LOG.error(_LE("[AC] Failed to create QoS policy in huawei driver:%s."), except_msg)
            ac_osprofiler.record_chain_exception_end("create QoS policy fail")
            raise except_msg
        ac_osprofiler.record_chain_end_with_reason("create QoS policy success")
        LOG.info(_LI("[AC] Huawei AC create QoS policy successfully."))

    def update_policy_precommit(self, context, policy):
        """If update_policy_precommit registered in openstack,
        use it instead of update_policy.
        """
        LOG.debug("[AC] update QoS policy: %s in update_policy_precommit.",
                  policy)
        self.update_policy(context, policy)
        DEAL_POLICY.mapping = {policy.id: datetime.datetime.utcnow()}

    def update_policy(self, context, policy):
        """Update policy."""
        if hasattr(DEAL_POLICY, 'mapping') and policy.id in DEAL_POLICY.mapping:
            DEAL_POLICY.mapping = {}
            return None

        LOG.info("[AC] Begin to update QoS policy: %s.", policy)
        ac_osprofiler.record_chain_start("update QoS policy start,ID:" + policy['id'])

        if self.data_filter.not_in_white_or_in_black_list(context, policy, ac_const.NW_HW_QOS_POLICY, policy['id']):
            return None

        original_db_policy = get_policy_obj(ncu.neutron_context.get_admin_context(), policy.id)
        new_db_policy = get_policy_obj(context, policy.id)
        dry_run_util.check_whether_dryrun_to_product(ac_const.NW_HW_QOS_POLICY, ac_const.OPER_UPDATE,
                                                     original_db_policy, new_db_policy)
        LOG.info('[AC] Update qos policy, context:%s, original_policy:%s, new_policy:%s', context, original_db_policy,
                 new_db_policy)
        if not original_db_policy:
            dry_run_state = DryRunStateEnum.NO_EXIST
        else:
            if dry_run_util.is_dry_run_data(original_db_policy, ac_const.NW_HW_QOS_POLICY):
                dry_run_state = DryRunStateEnum.DRY_RUN_DATA
            else:
                dry_run_state = DryRunStateEnum.REAL_DATA
        policy_info = ACQoSpolicyModel.ac_model_format(new_db_policy) if new_db_policy else {}
        LOG.info("[AC] Sent to ac qos policy info: %s", policy_info)
        try:
            self.ac_driver.update_plugin_record(
                context, policy.id, policy_info, ac_const.NW_HW_UPDATE_QOS_POLICY, dry_run_state=dry_run_state)
        except Exception as except_msg:
            LOG.error("[AC] Failed to update QoS policy in huawei driver: %s.", except_msg)
            ac_osprofiler.record_chain_exception_end("update QoS policy fail")
            raise except_msg
        ac_osprofiler.record_chain_end_with_reason("update QoS policy success")
        LOG.info("[AC] Huawei AC update QoS policy successfully.")
        return None

    def delete_policy_precommit(self, context, policy):
        """If delete_policy_precommit registered in openstack,use it instead of delete_policy"""
        LOG.debug("[AC] delete QoS policy: %s in delete_policy_precommit.",
                  policy)
        self.delete_policy(context, policy)
        DEAL_POLICY.mapping = {policy.id: datetime.datetime.utcnow()}

    def delete_policy(self, context, policy):
        """Delete policy."""
        if hasattr(DEAL_POLICY, 'mapping') and policy.id in DEAL_POLICY.mapping:
            DEAL_POLICY.mapping = {}
            return

        LOG.info(_LI("[AC] Begin to delete QoS policy: %s."), policy)
        ac_osprofiler.record_chain_start("delete QoS policy start,ID:" + policy.id)

        if self.data_filter.not_in_white_or_in_black_list(context, policy, ac_const.NW_HW_QOS_POLICY, policy['id']):
            return

        self._check_qos_policy_in_use(context, policy.id)

        db_policy = get_policy_obj(context, policy.id)
        policy_info = ACQoSpolicyModel.ac_model_format(db_policy) if db_policy else {}
        LOG.info("[AC] policy info in db: %s", policy_info)
        try:
            self.ac_driver.update_plugin_record(context, policy.id, policy_info, ac_const.NW_HW_DELETE_QOS_POLICY)
        except Exception as except_msg:
            LOG.error(_LE("[AC] Failed to delete QoS policy in huawei driver:%s."), except_msg)
            ac_osprofiler.record_chain_exception_end("delete QoS policy fail")
            raise except_msg
        ac_osprofiler.record_chain_end_with_reason("delete QoS policy success")
        LOG.info(_LI("[AC] Huawei AC delete QoS policy successfully."))

    @staticmethod
    def create():
        """Create QoS service notification driver."""
        return HuaweiQosServiceNotificationDriver(
            name='openvswitch',
            vif_types=['ovs', 'vhostuser'],
            vnic_types=['normal'],
            supported_rules=SUPPORTED_RULES,
            requires_rpc_notifications=True)


def register():
    """Register QoS service notification driver."""
    global DRIVER
    if not DRIVER:
        DRIVER = HuaweiQosServiceNotificationDriver.create()
    LOG.info('HuaweiQosServiceNotificationDriver registered')


def get_policy_obj(context, policy_id):
    """获取数据库中的qos_policy

    :param context: context
    :param policy_id: policy_id
    :return: qos_policy
    """
    if hasattr(policy_object.QosPolicy, "get_by_id"):
        # OPS L version
        obj = policy_object.QosPolicy.get_by_id(context, policy_id)
    elif hasattr(policy_object.QosPolicy, "get_object"):
        # OPS M version
        obj = policy_object.QosPolicy.get_object(context, id=policy_id)
    elif hasattr(policy_object.QosPolicy, "get_objects"):
        policies = policy_object.QosPolicy.get_objects(context, id=policy_id)
        obj = policies[0] if policies else None
    else:
        obj = None

    if obj is None:
        raise n_exc.QosPolicyNotFound(policy_id=policy_id)
    return obj
