#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""snat plugin"""

from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.db.snat.snat import SnatDbMixin
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.extensions.snat import snat as extension
from networking_huawei.drivers.ac.model.snat_model import ACSNATModel
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI

LOG = logging.getLogger(__name__)


class SnatPlugin(extension.SnatBase, SnatDbMixin):
    """SnatPlugin"""
    supported_extension_aliases = [extension.SNAT]

    def __init__(self):
        super(SnatPlugin, self).__init__()
        self.reliability_api = ACReliabilityAPI(ac_constants.NW_HW_SNATS)

    def get_plugin_name(self):
        """get plugin name"""
        return extension.SNAT

    def get_plugin_type(self):
        """get plugin type"""
        return extension.SNAT

    def get_plugin_description(self):
        """get plugin description"""
        return extension.SNAT

    def create_snat(self, context, snat):
        """create snat"""
        LOG.info(_LI('[AC] Begin to create SNAT: %s'), snat)
        snat_db = super(SnatPlugin, self).create_snat_db(context, snat['snat'])
        snat_info = ACSNATModel.ac_model_format_v2(snat_db)
        try:
            self.reliability_api.update_plugin_record(
                context, snat_db['id'], snat_info,
                ac_constants.NW_HW_CREATE_SNAT)
        except Exception as ex:
            LOG.error(_LE('[AC] Huawei AC create SNAT failed: %s'), ex)
            super(SnatPlugin, self).delete_snat_db(context, snat_db['id'])
            raise ex
        LOG.info(_LI('[AC] End of creating SNAT: %s'), snat_db)
        return snat_db

    def update_snat(self, context, snat_id, snat):
        """update snat"""
        LOG.info(_LI('[AC] Begin to update SNAT(%s): %s'), snat_id, snat)
        org_snat_db = super(SnatPlugin, self).get_snat_db(context, snat_id)
        cur_snat_db = super(SnatPlugin, self).update_snat_db(
            context, snat_id, snat['snat'])
        snat_info = ACSNATModel.ac_model_format_v2(cur_snat_db)
        try:
            self.reliability_api.update_plugin_record(
                context, snat_id, snat_info,
                ac_constants.NW_HW_UPDATE_SNAT)
        except Exception as ex:
            LOG.error(_LE('[AC] Huawei AC create SNAT failed: %s'), ex)
            super(SnatPlugin, self).update_snat_db(context, snat_id, org_snat_db)
            raise ex
        LOG.info(_LI('[AC] End of updating SNAT: %s'), cur_snat_db)
        return cur_snat_db

    def delete_snat(self, context, snat_id):
        """delete snat"""
        LOG.info(_LI('[AC] Begin to delete SNAT: %s'), snat_id)
        self.reliability_api.update_plugin_record(
            context, snat_id, {}, ac_constants.NW_HW_DELETE_SNAT)
        super(SnatPlugin, self).delete_snat_db(context, snat_id)
        LOG.info(_LI('[AC] End of deleting SNAT: %s'), snat_id)

    def get_snat(self, context, snat_id, fields=None):
        """get snat"""
        return super(SnatPlugin, self).get_snat_db(context, snat_id, fields)

    def get_snats(self, context, filters=None, fields=None):
        """get snats"""
        return super(SnatPlugin, self).get_snats_db(context, filters, fields)
