#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""vpc connection plugin"""

from oslo_config import cfg
from oslo_log import log as logging

from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import osprofiler_warp as ac_osprofiler
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.db.vpc_connection import vpc_connection_db
from networking_huawei.drivers.ac.extensions.vpc_connection import vpc_connection as vpcconn
from networking_huawei.drivers.ac.model.vpc_connection_model import ACVpcConnectionModel
from networking_huawei.drivers.ac.sync.message_reliability_api import ACReliabilityAPI

LOG = logging.getLogger(__name__)


class HuaweiVpcConnectionPlugin(vpcconn.VpcConnectionBase,
                                vpc_connection_db.VpcConnectionDbMixin):
    """Implementation of the Huawei AC Vpc Connection Service Plugin."""
    supported_extension_aliases = ['ac-vpc-connection']

    def __init__(self):
        LOG.info(_LI("[AC] Init Huawei vpc connection plugin."))
        super(HuaweiVpcConnectionPlugin, self).__init__()
        self.ac_reliability = ACReliabilityAPI(
            ac_constants.NW_HW_VPC_CONNECTION)

        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for Huawei vpc connection plugin."))

    def get_plugin_type(self):
        """get plugin type"""
        return ac_constants.NW_HW_VPC_CONNECTION

    def get_plugin_description(self):
        """get plugin description"""
        return 'Huawei vpc connection service plugin'

    @classmethod
    def optimize_vpc_conn_info(cls, vpc_conn_info):
        """optimize_vpc_conn_info"""
        if vpc_conn_info['local-firewall-enable'] is None or \
                vpc_conn_info['local-firewall-enable'] == 'None':
            vpc_conn_info.pop('local-firewall-enable')
        if vpc_conn_info['peer-firewall-enable'] is None or \
                vpc_conn_info['peer-firewall-enable'] == 'None':
            vpc_conn_info.pop('peer-firewall-enable')
        if vpc_conn_info['priority'] is None or \
                vpc_conn_info['priority'] == 'None':
            vpc_conn_info.pop('priority')

    def create_vpc_connection(self, context, vpc_connection):
        """create vpc connection"""
        LOG.info(_LI("[AC] Begin to create vpc connection: %s "),
                 vpc_connection)
        vpc_connection_data = vpc_connection['vpc_connection']
        if cfg.CONF.huawei_ac_config.cmcc_env:
            vpc_connection_data['mode'] = 1
            if vpc_connection_data['local_firewall_enable'] is None or \
                    vpc_connection_data['local_firewall_enable'] == 'None':
                vpc_connection_data['local_firewall_enable'] = True
            if vpc_connection_data['peer_firewall_enable'] is None or \
                    vpc_connection_data['peer_firewall_enable'] == 'None':
                vpc_connection_data['peer_firewall_enable'] = True
        local_cidrs = vpc_connection_data['local_cidrs']
        peer_cidrs = vpc_connection_data['peer_cidrs']
        validate.validate_cidrs(local_cidrs, peer_cidrs)
        firewall_mode = vpc_connection_data['mode']
        local_firewall_enable = vpc_connection_data['local_firewall_enable']
        peer_firewall_enable = vpc_connection_data['peer_firewall_enable']
        fw_enabled = vpc_connection_data['fw_enabled']
        validate.validate_firewall_mode(firewall_mode, fw_enabled,
                                        local_firewall_enable,
                                        peer_firewall_enable)
        ac_osprofiler.record_chain_start("create vpc connection start")

        status = ac_constants.NEUTRON_STATUS_ACTIVE
        vpc_conn_dict = super(HuaweiVpcConnectionPlugin, self). \
            create_db_vpc_connection(context, vpc_connection, status)
        LOG.info(_LI("[AC] Create vpc connection record in Neutron DB "
                     "successful: %s "), vpc_conn_dict)
        vpc_conn_info = ACVpcConnectionModel.ac_model_format(vpc_conn_dict)
        self.optimize_vpc_conn_info(vpc_conn_info)

        ac_osprofiler.record_chain_start("create vpc connection start,ID:%s"
                                         % vpc_conn_dict['id'])
        try:
            self.ac_reliability.update_plugin_record(
                context, vpc_conn_info.get('uuid'), vpc_conn_info,
                ac_constants.NW_HW_CREATE_VPC_CONNECTION)
        except Exception as ex:
            LOG.error(_LE("[AC]Huawei AC create vpc connection failed.Roll back"
                          ",Delete vpc connection in Neutron DB.catch: %s"), ex)
            super(HuaweiVpcConnectionPlugin, self). \
                delete_db_vpc_connection(context, vpc_conn_info.get('uuid'))
            ac_osprofiler.record_chain_exception_end("create vpc connection "
                                                     "fail")
            raise ex

        LOG.info(_LI("[AC] Huawei AC create vpc connection successful."))
        ac_osprofiler.record_chain_end_with_reason("create vpc connection "
                                                   "success")
        return vpc_conn_dict

    def delete_vpc_connection(self, context, vpc_connection_id):
        """delete vpc connection"""
        LOG.info(_LI("[AC] Begin to delete vpc connection : %s"), vpc_connection_id)
        ac_osprofiler.record_chain_start("delete vpc connection start,ID:" +
                                         vpc_connection_id)

        try:
            LOG.debug("Start to send the request of delete vpc connection")
            self.ac_reliability. \
                update_plugin_record(context,
                                     vpc_connection_id,
                                     {},
                                     ac_constants.NW_HW_DELETE_VPC_CONNECTION)
        except Exception as value:
            LOG.error(_LE("[AC] AC delete vpc connection "
                          "failed for %s."), value)
            ac_osprofiler.record_chain_exception_end("delete vpc connection "
                                                     "fail")
            raise value

        LOG.info(_LI("[AC] AC delete vpc connection successful."))

        try:
            super(HuaweiVpcConnectionPlugin, self). \
                delete_db_vpc_connection(context, vpc_connection_id)
        except Exception as value:
            LOG.error(_LE("[AC] Failed to delete vpc connetion "
                          "in Neutron DB for %s."), value)
            raise value
        ac_osprofiler.record_chain_end_with_reason("delete vpc connection "
                                                   "success")
        LOG.info(_LI("[AC] Delete vpc connection record in "
                     "Neutron DB successful."))

    def prepare_vpc_conn_info(self, vpc_conn):
        """validate and modify date before send to controller

        :param vpc_conn: input data
        :return: vpc connection date send to controller
        """
        new_vpc_conn_info = ACVpcConnectionModel. \
            ac_model_format(vpc_conn)
        firewall_mode = new_vpc_conn_info.get('mode')
        local_firewall_enable = new_vpc_conn_info.get('local-firewall-enable')
        peer_firewall_enable = new_vpc_conn_info.get('peer-firewall-enable')

        validate.validate_firewall_mode(firewall_mode, None,
                                        local_firewall_enable,
                                        peer_firewall_enable)

        self.optimize_vpc_conn_info(new_vpc_conn_info)
        return new_vpc_conn_info

    def update_vpc_connection(self, context, vpc_connection_id, vpc_connection):
        """update vpc connection"""
        LOG.info(_LI("[AC] Begin to update vpc connection:%s as:%s"),
                 vpc_connection_id, vpc_connection)

        ac_osprofiler.record_chain_start("update vpc connection start,ID:%s"
                                         % vpc_connection_id)

        original_vpc_conn = super(HuaweiVpcConnectionPlugin, self). \
            get_db_vpc_connection(context, vpc_connection_id)
        LOG.debug("[AC] Get original vpc connection in Neutron DB, %s",
                  original_vpc_conn)
        update_vpc_conn = super(HuaweiVpcConnectionPlugin, self). \
            update_db_vpc_connection(context, vpc_connection_id, vpc_connection,
                                     original_vpc_conn)
        LOG.info(_LI("[AC] Updated vpc connection record in Neutron DB "
                     "successful: %s"), update_vpc_conn)

        try:
            new_vpc_conn_info = self.prepare_vpc_conn_info(update_vpc_conn)
            LOG.debug("Start to send the request of update vpc connection")
            self.ac_reliability. \
                update_plugin_record(context,
                                     new_vpc_conn_info.get('uuid'),
                                     new_vpc_conn_info,
                                     ac_constants.NW_HW_UPDATE_VPC_CONNECTION)
        except Exception as value:
            LOG.error(_LE("[AC] AC update vpc connection failed for %s "
                          "Roll back to the previous vpc connetcion "
                          "info."), value)
            ac_osprofiler.record_chain_exception_end("update vpc connection "
                                                     "fail")
            super(HuaweiVpcConnectionPlugin, self). \
                update_db_vpc_connection(context, vpc_connection_id,
                                         {'vpc_connection': original_vpc_conn})
            raise value

        LOG.info(_LI("[AC] Huawei AC update vpc connection successful."))
        LOG.info("update return: %s", update_vpc_conn)
        ac_osprofiler.record_chain_end_with_reason("update vpc connection "
                                                   "success")
        return update_vpc_conn

    def get_vpc_connection(self, context, vpc_connection_id, fields=None):
        """get vpc connection"""
        LOG.info(_LI("[AC] Begin to get vpc connection : %s"), vpc_connection_id)
        rest_info = super(HuaweiVpcConnectionPlugin, self). \
            get_db_vpc_connection(context, vpc_connection_id, fields=fields)
        LOG.info(_LI("[AC] End to get vpc connection."))
        return rest_info

    def get_vpc_connections(self, context, filters=None, fields=None):
        """get vpc connections"""
        LOG.info(_LI("[AC] Begin to get vpc connections."))

        return super(HuaweiVpcConnectionPlugin, self). \
            get_db_vpc_connections(context, filters=filters, fields=fields)
