#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""vpc connection custom plugin"""

from oslo_log import log as logging
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.extensions.vpc_connection_custom import \
    vpc_connection_custom as vpcconn
from networking_huawei.drivers.ac.model.vpc_connection_model \
    import ACVpcConnectionModel
from networking_huawei.drivers.ac.plugins.vpc_connection \
    .vpc_connection_plugin import \
    HuaweiVpcConnectionPlugin
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei.drivers.ac.db.vpc_connection \
    import vpc_connection_custom_db
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler

LOG = logging.getLogger(__name__)


class HuaweiVpcConnectionCustomPlugin(vpcconn.VpcConnectionCustomBase,
                                      vpc_connection_custom_db.
                                      VpcConnectionCustomDbMixin):
    """Implementation of the Huawei AC Vpc Connection Service Plugin."""
    supported_extension_aliases = ['vpc-connection-custom']

    def __init__(self):
        LOG.info(_LI("[AC] Init Huawei vpc connection plugin."))
        super(HuaweiVpcConnectionCustomPlugin, self).__init__()
        self.ac_reliability = ACReliabilityAPI(
            ac_constants.NW_HW_VPC_CONNECTION)

        LOG.info(_LI("[AC] Initialization finished successfully "
                     "for Huawei vpc connection plugin."))

    def get_plugin_type(self):
        """get plugin type"""
        return 'vpc_connection'

    def get_plugin_description(self):
        """get plugin description"""
        return 'Huawei vpc connection service plugin'

    def create_vpc_connection(self, context, vpc_connection):
        """create vpc connection"""
        LOG.info(_LI("[AC] Begin to create vpc connection: %s "),
                 vpc_connection)
        local_cidrs = vpc_connection['vpc_connection']['local_cidrs']
        peer_cidrs = vpc_connection['vpc_connection']['peer_cidrs']
        validate.validate_cidrs(local_cidrs, peer_cidrs)
        ac_osprofiler.record_chain_start("create vpc connection start")

        status = ac_constants.NEUTRON_STATUS_ACTIVE
        if vpc_connection['vpc_connection'].get('priority'):
            vpc_connection['vpc_connection']['priority'] = \
                int(vpc_connection['vpc_connection']['priority'])
        if vpc_connection['vpc_connection'].get('mode'):
            vpc_connection['vpc_connection']['mode'] = \
                int(vpc_connection['vpc_connection']['mode'])
        vpc_conn_dict = super(HuaweiVpcConnectionCustomPlugin, self). \
            create_db_vpc_connection(context, vpc_connection, status)
        LOG.info(_LI("[AC] Create vpc connection record in Neutron DB "
                     "successful: %s "), vpc_conn_dict)
        vpc_conn_info = ACVpcConnectionModel.ac_model_format(vpc_conn_dict)
        if vpc_conn_info['priority'] is None or \
                vpc_conn_info['priority'] == 'None':
            vpc_conn_info.pop('priority')

        ac_osprofiler.record_chain_start("create vpc connection start,ID:"
                                         + vpc_conn_dict['id'])
        try:
            LOG.debug("Start to send the request of create vpc connection")
            self.ac_reliability.update_plugin_record(
                context, vpc_conn_info['uuid'], vpc_conn_info,
                ac_constants.NW_HW_CREATE_VPC_CONNECTION)
        except Exception as ex:
            LOG.error(_LE("[AC] Huawei AC create vpc connection failed. "
                          "Roll back: Delete vpc connection in Neutron DB. "
                          "catch: %s"), ex)
            super(HuaweiVpcConnectionCustomPlugin, self). \
                delete_db_vpc_connection(context, vpc_conn_info['uuid'])
            ac_osprofiler.record_chain_exception_end("create vpc connection "
                                                     "fail")
            raise

        LOG.info(_LI("[AC] Huawei AC create vpc connection successful."))
        ac_osprofiler.record_chain_end_with_reason("create vpc connection "
                                                   "success")
        return self._make_vpc_conn_custom_dict(vpc_conn_dict)

    def delete_vpc_connection(self, context, vpc_connection_id):
        """delete vpc connection"""
        LOG.info(_LI("[AC] Begin to delete vpc connection : %s"),
                 vpc_connection_id)
        ac_osprofiler.record_chain_start("delete vpc connection start,ID:" +
                                         vpc_connection_id)

        try:
            LOG.debug("Start to send the request of delete vpc connection")
            self.ac_reliability. \
                update_plugin_record(context, vpc_connection_id, {},
                                     ac_constants.NW_HW_DELETE_VPC_CONNECTION)
        except Exception as value:
            LOG.error(_LE("[AC] AC delete vpc connection "
                          "failed for %s."), value)
            ac_osprofiler.record_chain_exception_end("delete vpc connection "
                                                     "fail")
            raise

        LOG.info(_LI("[AC] AC delete vpc connection successful."))

        try:
            super(HuaweiVpcConnectionCustomPlugin, self). \
                delete_db_vpc_connection(context, vpc_connection_id)
        except Exception as value:
            LOG.error(_LE("[AC] Failed to delete vpc connetion "
                          "in Neutron DB for %s."), value)
            ac_osprofiler.record_chain_end_with_reason("[AC] Failed to delete "
                                                       "vpc connetion")
            raise
        ac_osprofiler.record_chain_end_with_reason("delete vpc connection "
                                                   "success")
        LOG.info(_LI("[AC] Delete vpc connection record in "
                     "Neutron DB successful."))

    def update_vpc_connection(self, context, vpc_connection_id, vpc_connection):
        """update vpc connection"""
        new_vpc_conn_info, original_vpc_conn, update_vpc_conn = \
            self._update_vpc_connection_before(
                context, vpc_connection, vpc_connection_id)
        try:
            LOG.debug("Start to send the request of update vpc connection")
            self.ac_reliability.update_plugin_record(
                context, new_vpc_conn_info['uuid'], new_vpc_conn_info,
                ac_constants.NW_HW_UPDATE_VPC_CONNECTION)
        except Exception as value:
            LOG.error(_LE("[AC] AC update vpc connection failed for %s. "
                          "Roll back to the previous vpc connetcion "
                          "info."), value)
            ac_osprofiler.record_chain_exception_end("update vpc connection "
                                                     "fail")
            if original_vpc_conn['priority'] == 'None':
                original_vpc_conn.pop('priority')
            else:
                original_vpc_conn['priority'] = int(original_vpc_conn
                                                    ['priority'])
            original_vpc_conn['mode'] = int(original_vpc_conn['mode'])
            super(HuaweiVpcConnectionCustomPlugin, self). \
                update_db_vpc_connection(context, vpc_connection_id,
                                         {'vpc_connection': original_vpc_conn})
            raise

        LOG.info(_LI("[AC] Huawei AC update vpc connection successful."))
        LOG.info("update return: %s", update_vpc_conn)
        ac_osprofiler.record_chain_end_with_reason("update vpc connection "
                                                   "success")
        return self._make_vpc_conn_custom_dict(update_vpc_conn)

    def _update_vpc_connection_before(self, context, vpc_connection,
                                      vpc_connection_id):
        """update_vpc_connection before"""
        LOG.info(_LI("[AC] Begin to update vpc connection:%s as:%s"),
                 vpc_connection_id, vpc_connection)
        ac_osprofiler.record_chain_start("update vpc connection start,ID:"
                                         + vpc_connection_id)
        if vpc_connection['vpc_connection'].get('priority'):
            vpc_connection['vpc_connection']['priority'] = \
                int(vpc_connection['vpc_connection']['priority'])
        if vpc_connection['vpc_connection'].get('mode'):
            vpc_connection['vpc_connection']['mode'] = \
                int(vpc_connection['vpc_connection']['mode'])
        original_vpc_conn = super(HuaweiVpcConnectionCustomPlugin, self). \
            get_db_vpc_connection(context, vpc_connection_id)
        LOG.debug("[AC] Get original vpc connection in Neutron DB, %s",
                  original_vpc_conn)
        update_vpc_conn = super(HuaweiVpcConnectionCustomPlugin, self). \
            update_db_vpc_connection(context, vpc_connection_id, vpc_connection,
                                     original_vpc_conn)
        LOG.info(_LI("[AC] Updated vpc connection record in Neutron DB "
                     "successful: %s"), update_vpc_conn)
        new_vpc_conn_info = ACVpcConnectionModel.ac_model_format(
            update_vpc_conn)
        HuaweiVpcConnectionPlugin.optimize_vpc_conn_info(new_vpc_conn_info)
        return new_vpc_conn_info, original_vpc_conn, update_vpc_conn

    def get_vpc_connection(self, context, vpc_connection_id, fields=None):
        """get vpc connection"""
        LOG.info(_LI("[AC] Begin to get vpc connection : %s"),
                 vpc_connection_id)
        rest_info = super(HuaweiVpcConnectionCustomPlugin, self). \
            get_db_vpc_connection(context, vpc_connection_id, fields=fields)
        LOG.info(_LI("[AC] End to get vpc connection."))
        return rest_info

    def get_vpc_connections(self, context, filters=None, fields=None,
                            sorts=None, limit=None, marker_obj=None,
                            page_reverse=False):
        """get vpc connections"""
        LOG.info(_LI("[AC] Begin to get vpc connections."))

        return super(HuaweiVpcConnectionCustomPlugin, self). \
            get_db_vpc_connections(context, filters=filters, fields=fields,
                                   sorts=sorts, limit=limit,
                                   marker_obj=marker_obj,
                                   page_reverse=page_reverse)
