#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""vpnaas plugin"""

import copy
import netaddr

try:
    try:
        from FSSecurity import crypt
    except ImportError:
        from FSComponentUtil import crypt
except Exception:
    crypt = None
try:
    # import vpn models for openstack L version
    from neutron_vpnaas.db.vpn import vpn_models as vpn_models
except ImportError:
    try:
        # import vpn models for openstack K version
        from neutron_vpnaas.db.vpn import vpn_db as vpn_models
    except ImportError:
        # import vpn models from fusionsphere 6.0
        from neutron.db.vpn import vpn_db as vpn_models
try:
    # import vpn db for openstack
    from neutron_vpnaas.db.vpn import vpn_db
    from neutron_vpnaas.db.vpn import vpn_validator
except ImportError:
    # import vpn db for fusionsphere 6.0
    from neutron.db.vpn import vpn_db
try:
    # import vpn extensions for openstack
    from neutron_vpnaas.extensions import vpnaas
except ImportError:
    # import vpn extensions for fusionsphere 6.0
    from neutron.extensions import vpnaas
try:
    # import vpn plugin for openstack
    from neutron_vpnaas.services.vpn.plugin import VPNPlugin
except ImportError:
    # import vpn plugin for fusionsphere 6.0
    from neutron.services.vpn.plugin import VPNPlugin

import networking_huawei.drivers.ac.common.constants as ac_const
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.model.ipsecvpn_model \
    import ACVPNServiceModel, ACIKEpolicyModel, \
    ACIPSecpolicyModel, ACIPSecSiteConnModel, ACEndPointGroupModel
from networking_huawei.drivers.ac.sync.message_reliability_api \
    import ACReliabilityAPI
from networking_huawei._i18n import _LI
from networking_huawei._i18n import _LE
from networking_huawei.drivers.ac.common import osprofiler_warp as \
    ac_osprofiler
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu

LOG = ncu.ac_log.getLogger(__name__)

VPN_MODELS = [vpn_models.IKEPolicy,
              vpn_models.IPsecPolicy,
              vpn_models.VPNService]


class HuaweiIPsecVPNValidator(vpn_validator.VpnReferenceValidator):
    """Huawei AC validation routines for VPN resources."""

    def _check_router(self, context, router_id):
        LOG.info(_LI("[AC]Check router begin. "
                     "Context:%s, router:%s"), context, router_id)
        return

    def _validate_peer_address(self, ip_version, router):
        if hasattr(router, 'gw_port') and router.gw_port:
            super(HuaweiIPsecVPNValidator, self)._validate_peer_address(
                ip_version, router)


class HuaweiIPsecVPNPlugin(VPNPlugin):
    """Implementation of the Huawei AC VPN Service Plugin."""

    def __init__(self):
        LOG.info(_LI("[AC]Init Huawei vpnaas plugin."))
        super(HuaweiIPsecVPNPlugin, self).__init__()
        self.vpn_driver = HuaweiIPsecVPNDriver()

        try:
            vpn_db.subscribe()
        except Exception as ex:
            LOG.error(_LE('[AC] Failed to load db subscribe in '
                          'huawei VPN plugin: %s'), ex)
        LOG.info(_LI("[AC]Initialization finished successfully "
                     "for Huawei vpnaas plugin."))

    def get_plugin_description(self):
        """get plugin description"""
        return 'Huawei VPN service plugin'

    def _get_validator(self):
        return HuaweiIPsecVPNValidator()

    @classmethod
    def __validate_resource_in_use__(cls, context, model, res_uuid):
        with context.session.begin(subtransactions=True):
            if issubclass(model, vpn_models.IKEPolicy) and \
                    context.session.query(vpn_models.IPsecSiteConnection). \
                            filter_by(ikepolicy_id=res_uuid).first():
                LOG.error(_LE("[AC]IKE policy is in use."))
                raise vpnaas.IKEPolicyInUse(ikepolicy_id=res_uuid)
            if issubclass(model, vpn_models.IPsecPolicy) and \
                    context.session.query(vpn_models.IPsecSiteConnection). \
                            filter_by(ipsecpolicy_id=res_uuid).first():
                LOG.error(_LE("[AC]IPSec policy is in use."))
                raise vpnaas.IPsecPolicyInUse(ipsecpolicy_id=res_uuid)
            if issubclass(model, vpn_models.VPNService) and \
                    context.session.query(vpn_models.IPsecSiteConnection). \
                            filter_by(vpnservice_id=res_uuid).first():
                LOG.error(_LE("[AC]VPN service is in use."))
                raise vpnaas.VPNServiceInUse(vpnservice_id=res_uuid)

    def __qry_and_del_ipsec_site_conn__(self, context, model, res_uuid):
        if model not in VPN_MODELS:
            return
        with context.session.begin(subtransactions=True):
            connections = context.session.query(vpn_models.IPsecSiteConnection)
            if issubclass(model, vpn_models.IKEPolicy):
                connections = connections.filter_by(ikepolicy_id=res_uuid)
            if issubclass(model, vpn_models.IPsecPolicy):
                connections = connections.filter_by(ipsecpolicy_id=res_uuid)
            if issubclass(model, vpn_models.VPNService):
                connections = connections.filter_by(vpnservice_id=res_uuid)
            if hasattr(vpn_models, 'VPNEndpointGroup') and \
                    issubclass(model, vpn_models.VPNEndpointGroup):
                connections = connections.filter_by(local_ep_group_id=res_uuid)
                if not connections:
                    connections = connections.filter_by(
                        peer_ep_group_id=res_uuid)
            for connect in connections:
                super(HuaweiIPsecVPNPlugin, self). \
                    delete_ipsec_site_connection(context, connect['id'])

    @classmethod
    def _get_gateway_ips(cls, router):
        """call by create_vpnservice()"""
        v4_ip = v6_ip = None
        if hasattr(router, 'gw_port') and router.gw_port:
            for fixed_ip in router.gw_port.get('fixed_ips'):
                addr = fixed_ip.get('ip_address')
                vers = netaddr.IPAddress(addr).version
                if vers == 4:
                    v4_ip = v4_ip or addr
                else:
                    v6_ip = v6_ip or addr
        return v4_ip, v6_ip

    def __rollback_ipsecpolicy__(self, context, ipsecpolicy_id, ipsecpolicy):
        ipsecp = ipsecpolicy.get('ipsecpolicy')
        validator = self._get_validator()
        with context.session.begin(subtransactions=True):
            validator.validate_ipsec_policy(context, ipsecp)
            if context.session.query(vpn_models.IPsecSiteConnection). \
                    filter_by(ipsecpolicy_id=ipsecpolicy_id).first():
                LOG.info(_LI("[AC]IPSec site connection has been created"
                             " in VPN DB, "
                             "roll back to the original IPSec policy."))
            ipsecp_db = self._get_resource(
                context, vpn_models.IPsecPolicy, ipsecpolicy_id)
            if ipsecp:
                if ipsecp.get('lifetime'):
                    if ipsecp.get('lifetime').get('units'):
                        ipsecp['lifetime_units'] = ipsecp.get('lifetime').get(
                            'units')
                    if ipsecp.get('lifetime').get('value'):
                        ipsecp['lifetime_value'] = ipsecp.get('lifetime').get(
                            'value')
                ipsecp_db.update(ipsecp)
        return self._make_ipsecpolicy_dict(ipsecp_db)

    def __rollback_ikepolicy__(self, context, ikepolicy_id, ikepolicy):
        ike = ikepolicy['ikepolicy']
        with context.session.begin(subtransactions=True):
            if context.session.query(vpn_models.IPsecSiteConnection).filter_by(
                    ikepolicy_id=ikepolicy_id).first():
                LOG.info(_LI("[AC]IPSec site connection has been created"
                             " in VPN DB, "
                             "roll back to the original IKE policy."))
            ike_db = self._get_resource(
                context, vpn_models.IKEPolicy, ikepolicy_id)
            if ike:
                if ike.get('lifetime'):
                    if ike.get('lifetime').get('units'):
                        ike['lifetime_units'] = ike.get('lifetime').get('units')
                    if ike.get('lifetime').get('value'):
                        ike['lifetime_value'] = ike.get('lifetime').get('value')
                ike_db.update(ike)
        return self._make_ikepolicy_dict(ike_db)

    def create_ikepolicy(self, context, ikepolicy):
        """create ike policy"""
        LOG.info(_LI("[AC]Begin to create IKE policy"))

        try:
            created_ikepolicy = super(HuaweiIPsecVPNPlugin, self). \
                create_ikepolicy(context, ikepolicy)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to create IKE policy "
                          "in Neutron DB for %s."), value)
            raise
        LOG.info(_LI("[AC]Created IKE policy record in Neutron DB successful"))

        ac_osprofiler.record_chain_start("create ike policy start,ID:"
                                         + created_ikepolicy['id'])
        try:
            self.vpn_driver.create_ikepolicy(context, created_ikepolicy)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC create IKE policy failed for %s."
                          "Roll back to delete IKE policy in Neutron DB."),
                      value)
            ac_osprofiler.record_chain_exception_end("create ike policy rule "
                                                     "fail")
            self.__qry_and_del_ipsec_site_conn__(context,
                                                 vpn_models.IKEPolicy,
                                                 created_ikepolicy['id'])
            super(HuaweiIPsecVPNPlugin, self). \
                delete_ikepolicy(context, created_ikepolicy['id'])
            raise

        LOG.info(_LI("[AC]Huawei AC create IKE policy successful."))
        ac_osprofiler.record_chain_end_with_reason("create ike policy success")
        return created_ikepolicy

    def delete_ikepolicy(self, context, ikepolicy_id):
        """delete ike policy"""
        LOG.info(_LI("[AC]Begin to delete IKE policy : %s"), ikepolicy_id)
        ac_osprofiler.record_chain_start("delete ike policy start,ID:" +
                                         ikepolicy_id)

        self.__validate_resource_in_use__(context, vpn_models.IKEPolicy,
                                          ikepolicy_id)

        try:
            self.vpn_driver.delete_ikepolicy(context, ikepolicy_id)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC delete IKE policy failed for %s."),
                      value)
            ac_osprofiler.record_chain_exception_end("delete ike policy fail")
            raise
        LOG.info(_LI("[AC]Huawei AC delete IKE policy successful."))

        try:
            super(HuaweiIPsecVPNPlugin, self). \
                delete_ikepolicy(context, ikepolicy_id)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to delete IKE policy "
                          "in Neutron DB for %s."), value)
            raise
        ac_osprofiler.record_chain_end_with_reason("delete ike policy success")
        LOG.info(_LI("[AC]Delete IKE policy record in Neutron DB successful."))

    def update_ikepolicy(self, context, ikepolicy_id, ikepolicy):
        """update ike policy"""
        LOG.info(_LI("[AC]Begin to update IKE policy : %s"), ikepolicy_id)
        ac_osprofiler.record_chain_start("update ike policy start,ID:" +
                                         ikepolicy_id)
        original_ikepolicy = self.get_ikepolicy(context, ikepolicy_id)

        try:
            update_ikepolicy = super(HuaweiIPsecVPNPlugin, self). \
                update_ikepolicy(context, ikepolicy_id, ikepolicy)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to update IKE policy "
                          "in Neutron DB for %s."), value)
            ac_osprofiler.record_chain_exception_end("update ike policy fail")
            raise
        LOG.info(_LI("[AC]Updated IKE policy record in Neutron DB successful"))

        try:
            self.vpn_driver.update_ikepolicy(context, update_ikepolicy)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC update IKE policy failed for %s."),
                      value)
            self.__rollback_ikepolicy__(context, ikepolicy_id,
                                        {'ikepolicy': original_ikepolicy})
            raise

        LOG.info(_LI("[AC]Huawei AC update IKE policy successful."))
        ac_osprofiler.record_chain_end_with_reason("update ike policy success")
        return update_ikepolicy

    def create_ipsecpolicy(self, context, ipsecpolicy):
        """create ipsec policy"""
        LOG.info(_LI("[AC]Begin to create IPSec policy: %s "), ipsecpolicy)

        try:
            ipsecpolicy_dict = super(HuaweiIPsecVPNPlugin, self). \
                create_ipsecpolicy(context, ipsecpolicy)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to update IPSec policy "
                          "in Neutron DB for %s."), value)
            raise
        LOG.info(_LI("[AC]Create IPSec policy record in Neutron DB "
                     "successful: %s"), ipsecpolicy_dict)
        ac_osprofiler.record_chain_start("create ipsec policy start,ID:"
                                         + ipsecpolicy_dict['id'])
        try:
            self.vpn_driver.create_ipsecpolicy(context, ipsecpolicy_dict)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC create IPSec policy failed for %s."
                          "Roll back to delete IPSec policy in DB."), value)
            self.__qry_and_del_ipsec_site_conn__(context,
                                                 vpn_models.IPsecPolicy,
                                                 ipsecpolicy_dict['id'])
            super(HuaweiIPsecVPNPlugin, self). \
                delete_ipsecpolicy(context, ipsecpolicy_dict['id'])
            ac_osprofiler.record_chain_exception_end("create ipsec policy rule"
                                                     " fail")
            raise

        LOG.info(_LI("[AC]Huawei AC create IPSec policy successful."))
        ac_osprofiler.record_chain_end_with_reason("create ipsec policy "
                                                   "success")
        return ipsecpolicy_dict

    def delete_ipsecpolicy(self, context, ipsecpolicy_id):
        """delete ipsec policy"""
        LOG.info(_LI("[AC]Begin to delete IPSec policy: %s"), ipsecpolicy_id)
        self.__validate_resource_in_use__(context, vpn_models.IPsecPolicy,
                                          ipsecpolicy_id)
        ac_osprofiler.record_chain_start("delete ipsec policy start,ID:" +
                                         ipsecpolicy_id)

        try:
            self.vpn_driver.delete_ipsecpolicy(context, ipsecpolicy_id)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC delete IPSec policy failed for %s."),
                      value)
            ac_osprofiler.record_chain_exception_end("delete ipsec policy "
                                                     "fail")
            raise
        LOG.info(_LI("[AC]Huawei AC delete IPSec policy successful."))

        try:
            super(HuaweiIPsecVPNPlugin, self). \
                delete_ipsecpolicy(context, ipsecpolicy_id)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to delete IPSec policy "
                          "in Neutron DB for %s."), value)
            raise
        ac_osprofiler.record_chain_end_with_reason("delete ipsec policy "
                                                   "success")
        LOG.info(_LI("[AC]Delete IPSec policy record "
                     "in Neutron DB successful."))

    def update_ipsecpolicy(self, context, ipsecpolicy_id, ipsecpolicy):
        """update ipsec policy"""
        LOG.info(_LI("[AC]Begin to update IPSec policy: %s"), ipsecpolicy)
        original_ipsecpolicy = self.get_ipsecpolicy(context, ipsecpolicy_id)
        LOG.info(_LI("[AC]Original IPSec policy: %s"), original_ipsecpolicy)
        ac_osprofiler.record_chain_start("update ipsec policy start,ID:"
                                         + ipsecpolicy_id)
        try:
            ipsecpolicy_dict = super(HuaweiIPsecVPNPlugin, self). \
                update_ipsecpolicy(context, ipsecpolicy_id, ipsecpolicy)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to update IPSec policy "
                          "in Neutron DB for %s."), value)
            ac_osprofiler.record_chain_exception_end("update ipsec policy "
                                                     "fail")
            raise
        LOG.info(_LI("[AC]Update IPSec policy DB successful: %s"),
                 ipsecpolicy_dict)

        try:
            self.vpn_driver.update_ipsecpolicy(context, ipsecpolicy_dict)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC update IPSec policy failed for %s."
                          "Roll back to the original IPSec policy."), value)
            self.__rollback_ipsecpolicy__(
                context, ipsecpolicy_id, {'ipsecpolicy': original_ipsecpolicy})
            raise

        ac_osprofiler.record_chain_end_with_reason("update ipsec policy "
                                                   "success")
        LOG.info(_LI("[AC]Huawei AC update IPSec policy successful."))
        return ipsecpolicy_dict

    def create_vpnservice(self, context, vpnservice):
        """create vpn service"""
        LOG.info(_LI("[AC]Begin to create VPN service : %s "), vpnservice)

        try:
            created_vpn = super(HuaweiIPsecVPNPlugin, self). \
                create_vpnservice(context, vpnservice)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to create VPN service in Neutron DB, "
                          "for : %s"), value)
            raise
        LOG.info(_LI("[AC]Created VPN service record in Neutron DB "
                     "successful: %s"), created_vpn)
        ac_osprofiler.record_chain_start("create vpn start,ID:" +
                                         created_vpn['id'])
        try:
            if hasattr(VPNPlugin, "set_external_tunnel_ips"):
                # OPS L\M version
                vpn_record = self._get_vpnservice(context, created_vpn['id'])
                v4_ip = v6_ip = None
                if vpnservice['vpnservice'].get('external_v4_ip'):
                    v4_ip = vpnservice['vpnservice']['external_v4_ip']
                if vpnservice['vpnservice'].get('external_v6_ip'):
                    v6_ip = vpnservice['vpnservice']['external_v6_ip']
                if (not v4_ip) and (not v6_ip):
                    v4_ip, v6_ip = self._get_gateway_ips(vpn_record.router)
                created_vpn = self.set_external_tunnel_ips(
                    context,
                    created_vpn['id'],
                    v4_ip=v4_ip,
                    v6_ip=v6_ip)
            self.vpn_driver.create_vpnservice(context, created_vpn)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC create VPN service failed for %s."
                          "Roll back to delete VPN service in Neutron DB."),
                      value)
            ac_osprofiler.record_chain_exception_end("create vpn rule fail")
            self.__qry_and_del_ipsec_site_conn__(context,
                                                 vpn_models.VPNService,
                                                 created_vpn['id'])
            super(HuaweiIPsecVPNPlugin, self). \
                delete_vpnservice(context, created_vpn['id'])
            raise
        ac_osprofiler.record_chain_end_with_reason("create vpn success")
        LOG.info(_LI("[AC]Huawei AC create VPN service successful."))
        return created_vpn

    def delete_vpnservice(self, context, vpnservice_id):
        """delete vpn service"""
        LOG.info(_LI("[AC]Begin to delete VPN service: %s"), vpnservice_id)
        ac_osprofiler.record_chain_start("delete vpn start,ID:" +
                                         vpnservice_id)

        self.__validate_resource_in_use__(context, vpn_models.VPNService,
                                          vpnservice_id)

        try:
            self.vpn_driver.delete_vpnservice(context, vpnservice_id)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC delete VPN service failed for %s."),
                      value)
            ac_osprofiler.record_chain_exception_end("delete vpn fail")
            raise
        LOG.info(_LI("[AC]Huawei AC delete VPN service successful."))

        try:
            super(HuaweiIPsecVPNPlugin, self). \
                delete_vpnservice(context, vpnservice_id)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to delete VPN service "
                          "in Neutron DB for %s."), value)
            raise
        ac_osprofiler.record_chain_end_with_reason("delete vpn success")
        LOG.info(_LI("[AC]Delete VPN service record "
                     "in Neutron DB successful."))

    def update_vpnservice(self, context, vpnservice_id, vpnservice):
        """update vpn service"""
        LOG.info(_LI("[AC]Begin to update VPN service : %s "), vpnservice)
        ac_osprofiler.record_chain_start("update vpn start,ID:" +
                                         vpnservice_id)
        original_vpnservice = self.get_vpnservice(context, vpnservice_id)
        LOG.info(_LI("[AC]Backup old VPN service : %s"), original_vpnservice)

        try:
            update_vpnservice = super(HuaweiIPsecVPNPlugin, self). \
                update_vpnservice(context, vpnservice_id, vpnservice)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to update VPN service "
                          "in Neutron DB for %s."), value)
            ac_osprofiler.record_chain_exception_end("update vpn fail")
            raise
        LOG.info(_LI("[AC]Updated VPN service record in Neutron DB "
                     "successful: %s"), update_vpnservice)

        try:
            self.vpn_driver. \
                update_vpnservice(context, update_vpnservice)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC update VPN service failed for %s."),
                      value)
            original_vpnservice['status'] = ac_const.NEUTRON_STATUS_ERROR
            super(HuaweiIPsecVPNPlugin, self). \
                update_vpnservice(context, vpnservice_id,
                                  {'vpnservice': original_vpnservice})
            raise
        ac_osprofiler.record_chain_end_with_reason("update vpn success")
        LOG.info(_LI("[AC]Huawei AC update VPN service successful."))
        return update_vpnservice

    def create_endpoint_group(self, context, endpoint_group):
        """create endpoint"""
        LOG.info(_LI('[AC] Begin to create endpoint group: %s'),
                 endpoint_group)

        try:
            ep_group_dict = super(HuaweiIPsecVPNPlugin, self). \
                create_endpoint_group(context, endpoint_group)
        except Exception as value:
            LOG.error(_LE('[AC] Failed to create endpoint group in '
                          'Neutron DB for %s.'), value)
            raise
        LOG.info(_LI('[AC] Create endpoint group DB successfully: %s'),
                 ep_group_dict)
        ac_osprofiler.record_chain_start("create endpoint group start,ID:"
                                         + ep_group_dict['id'])
        try:
            self.vpn_driver.create_endpoint_group(context, ep_group_dict)
        except Exception as value:
            LOG.error(_LE('[AC] Failed to create endpoint group: %s'), value)
            ac_osprofiler.record_chain_exception_end("create endpoint group "
                                                     "rule fail")

            self.__qry_and_del_ipsec_site_conn__(context,
                                                 vpn_models.VPNEndpointGroup,
                                                 ep_group_dict['id'])

            super(HuaweiIPsecVPNPlugin, self). \
                delete_endpoint_group(context, ep_group_dict['id'])
            raise

        LOG.info(_LI('[AC] Huawei AC create endpoint group successfully.'))
        ac_osprofiler.record_chain_end_with_reason("create endpoint group "
                                                   "success")
        return ep_group_dict

    def delete_endpoint_group(self, context, endpoint_group_id):
        """delete endpoint"""
        LOG.info(_LI('[AC] Begin to delete endpoint group: %s'),
                 endpoint_group_id)
        ac_osprofiler.record_chain_start("delete endpoint group start,ID:"
                                         + endpoint_group_id)

        self.check_endpoint_group_not_in_use(context, endpoint_group_id)

        try:
            self.vpn_driver.delete_endpoint_group(context, endpoint_group_id)
        except Exception as value:
            LOG.error(_LE('[AC] Failed to delete endpoint group: %s'), value)
            ac_osprofiler.record_chain_exception_end("delete endpoint group "
                                                     "fail")
            raise
        LOG.info(_LI("[AC] Huawei AC delete endpoint group successfully."))

        try:
            super(HuaweiIPsecVPNPlugin, self). \
                delete_endpoint_group(context, endpoint_group_id)
        except Exception as value:
            LOG.error(_LE("[AC] Failed to delete endpoint group in "
                          "Neutron DB for %s."), value)
            raise
        ac_osprofiler.record_chain_end_with_reason("delete endpoint group "
                                                   "success")
        LOG.info(_LI("[AC] Delete endpoint group DB successfully."))

    def update_endpoint_group(self, context, endpoint_group_id,
                              endpoint_group):
        """update endpoint"""
        LOG.info(_LI("[AC] Begin to update endpoint group: %s"),
                 endpoint_group)
        ac_osprofiler.record_chain_start("update endpoint group start,ID:"
                                         + endpoint_group_id)

        original_ep_group = self.get_endpoint_group(context, endpoint_group_id)
        LOG.info(_LI("[AC] Original endpoint group: %s"), original_ep_group)

        try:
            ep_group_dict = super(HuaweiIPsecVPNPlugin, self). \
                update_endpoint_group(context, endpoint_group_id,
                                      endpoint_group)
        except Exception as value:
            LOG.error(_LE("[AC] Failed to update endpoint group in "
                          "Neutron DB for %s."), value)
            ac_osprofiler.record_chain_exception_end("update endpoint group "
                                                     "fail")
            raise
        LOG.info(_LI("[AC] Update endpoint group DB successfully: %s"),
                 ep_group_dict)

        try:
            self.vpn_driver.update_endpoint_group(context, ep_group_dict)
        except Exception as value:
            LOG.error(_LE("[AC] Failed to update endpoint group: %s"), value)
            super(HuaweiIPsecVPNPlugin, self). \
                update_endpoint_group(context, endpoint_group_id,
                                      {'endpoint_group': original_ep_group})
            raise

        LOG.info(_LI("[AC]Huawei AC update endpoint group successfully."))
        ac_osprofiler.record_chain_end_with_reason("update endpoint group "
                                                   "success")
        return ep_group_dict

    def create_ipsec_site_connection(self, context, ipsec_site_connection):
        """create ipsec site connection"""
        ipsec_conn_log = \
            copy.deepcopy(ipsec_site_connection)['ipsec_site_connection']
        ipsec_conn_log = validate.validate_log_record(
            ipsec_conn_log, ac_const.NW_HW_IPSEC_SITE_CONNECTION)
        LOG.info(_LI("[AC]Begin to create IPSec site connection: %s"),
                 ipsec_conn_log)
        if ncu.IS_FSP:
            psk = ipsec_site_connection['ipsec_site_connection']['psk']
            ipsec_site_connection['ipsec_site_connection']['psk'] = \
                crypt.encrypt(psk)

        try:
            ipsec_conn_dict = super(HuaweiIPsecVPNPlugin, self). \
                create_ipsec_site_connection(context, ipsec_site_connection)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to create IPSec site connection in "
                          "Neutron DB for %s."), value)
            raise
        ipsec_conn_log = copy.deepcopy(ipsec_conn_dict)
        ipsec_conn_log = validate.validate_log_record(
            ipsec_conn_log, ac_const.NW_HW_IPSEC_SITE_CONNECTION)
        LOG.info(_LI("[AC]Create IPSec site connection DB successful: %s"),
                 ipsec_conn_log)
        ac_osprofiler.record_chain_start("create ipsec site connection start, "
                                         "ID:"
                                         + ipsec_conn_dict['id'])

        try:
            if ncu.IS_FSP:
                ipsec_conn_dict['psk'] = psk
            self.vpn_driver. \
                create_ipsec_site_connection(context, ipsec_conn_dict)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC create IPSec site connection failed "
                          "for %s. Roll back to delete IPSec site connection "
                          "in DB."), value)
            ac_osprofiler.record_chain_exception_end("create "
                                                     "ipsec site connection "
                                                     "fail")
            super(HuaweiIPsecVPNPlugin, self). \
                delete_ipsec_site_connection(context, ipsec_conn_dict['id'])
            raise

        LOG.info(_LI("[AC]Huawei AC create IPSec site connection successful."))
        # change psk of the return value to encoded value
        if ncu.IS_FSP:
            ipsec_conn_dict['psk'] = \
                ipsec_site_connection['ipsec_site_connection']['psk']
        ac_osprofiler.record_chain_end_with_reason("create "
                                                   "ipsec site connection "
                                                   "success")
        return ipsec_conn_dict

    def delete_ipsec_site_connection(self, context, ipsec_conn_id):
        """delete ipsec site connection"""
        LOG.info(_LI("[AC]Begin to delete IPSec site connection: %s"),
                 ipsec_conn_id)
        ac_osprofiler.record_chain_start("delete ipsec site connection start, "
                                         "ID:"
                                         + ipsec_conn_id)

        try:
            self.vpn_driver. \
                delete_ipsec_site_connection(context, ipsec_conn_id)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC delete IPSec site connection failed "
                          "for %s."), value)
            raise
        LOG.info(_LI("[AC]Huawei AC delete IPSec site connection successful."))

        try:
            super(HuaweiIPsecVPNPlugin, self). \
                delete_ipsec_site_connection(context, ipsec_conn_id)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to delete IPSec site connection "
                          "in Neutron DB for %s."), value)
            ac_osprofiler.record_chain_exception_end("delete ipsec site "
                                                     "connection fail")
            raise
        ac_osprofiler.record_chain_end_with_reason("delete ipsec site "
                                                   "connection success")
        LOG.info(_LI("[AC]Delete IPSec site connection record "
                     "in Neutron DB successful."))

    def update_ipsec_site_connection(self, context, ipsec_conn_id,
                                     ipsec_site_connection):
        """update ipsec site connection"""
        ipsec_conn_log = \
            copy.deepcopy(ipsec_site_connection)['ipsec_site_connection']
        ipsec_conn_log = validate.validate_log_record(
            ipsec_conn_log, ac_const.NW_HW_IPSEC_SITE_CONNECTION)
        LOG.info(_LI("[AC]Begin to update IPSec site connection: %s"),
                 ipsec_conn_log)

        original_ipsec_conn = \
            self.get_ipsec_site_connection(context, ipsec_conn_id)
        ipsec_conn_log = copy.deepcopy(original_ipsec_conn)
        ipsec_conn_log = validate.validate_log_record(
            ipsec_conn_log, ac_const.NW_HW_IPSEC_SITE_CONNECTION)
        LOG.debug(_LI("[AC]Original IPSec site connection: %s"),
                  ipsec_conn_log)
        ipsec_conn_dict = self._update_ipsec_conn_db(
            context, ipsec_conn_id, ipsec_site_connection)
        ipsec_conn_log = copy.deepcopy(ipsec_conn_dict)
        ipsec_conn_log = validate.validate_log_record(
            ipsec_conn_log, ac_const.NW_HW_IPSEC_SITE_CONNECTION)
        LOG.info(_LI("[AC]Update IPSec site connection DB successful: %s"),
                 ipsec_conn_log)

        self._update_ipsec_site(context, ipsec_conn_dict, ipsec_conn_id,
                                original_ipsec_conn)

        LOG.info(_LI("[AC]AC update IPSec site connection successful."))
        # change psk of the return value to encoded value
        if ncu.IS_FSP:
            ipsec_conn_dict['psk'] = ipsec_conn_dict['psk']
        ac_osprofiler.record_chain_end_with_reason("update ipsec site "
                                                   "connection success")
        return ipsec_conn_dict

    def _update_ipsec_site(self, context, ipsec_conn_dict, ipsec_conn_id,
                           original_ipsec_conn):
        """ update_ipsec_site_connection: update ipsec site"""
        try:
            if ncu.IS_FSP:
                decode_psk = crypt.decrypt(ipsec_conn_dict['psk'])
                ipsec_conn_dict['psk'] = decode_psk
        except TypeError as ex:
            LOG.error(_LE('[AC] decrypt psk failed: %s'), ex)
        except Exception as ex:
            LOG.error(_LE('[AC] decrypt psk failed: %s'), ex)
            original_ipsec_conn['status'] = ac_const.NEUTRON_STATUS_ERROR
            original_ipsec_site_conn = \
                {'ipsec_site_connection': original_ipsec_conn}
            super(HuaweiIPsecVPNPlugin, self). \
                update_ipsec_site_connection(context, ipsec_conn_id,
                                             original_ipsec_site_conn)
            ac_osprofiler.record_chain_exception_end("update ipsec site "
                                                     "connection decrypt fail")
            raise
        try:
            self.vpn_driver. \
                update_ipsec_site_connection(context, ipsec_conn_dict)
        except Exception as value:
            LOG.error(_LE("[AC]Huawei AC update IPSec site connection failed "
                          "for %s. Roll back to the original IPSec site "
                          "connection."), value)
            ac_osprofiler.record_chain_exception_end("update ipsec site "
                                                     "connection fail")
            original_ipsec_conn['status'] = ac_const.NEUTRON_STATUS_ERROR
            original_ipsec_site_conn = \
                {'ipsec_site_connection': original_ipsec_conn}
            super(HuaweiIPsecVPNPlugin, self). \
                update_ipsec_site_connection(context, ipsec_conn_id,
                                             original_ipsec_site_conn)
            raise

    def _update_ipsec_conn_db(self, context, ipsec_conn_id,
                              ipsec_site_connection):
        """update_ipsec_site_connection:update ipsec site connection"""
        ac_osprofiler.record_chain_start(
            "update ipsec site connection start,ID:" + ipsec_conn_id)
        try:
            if ncu.IS_FSP and 'psk' in \
                    ipsec_site_connection['ipsec_site_connection']:
                # update ipsec conn site, check that if the psk is updated
                # if updated, encode the psk
                new_update_psk = \
                    ipsec_site_connection['ipsec_site_connection']['psk']
                ipsec_site_connection['ipsec_site_connection']['psk'] = \
                    crypt.encrypt(new_update_psk)

            ipsec_conn_dict = super(HuaweiIPsecVPNPlugin, self). \
                update_ipsec_site_connection(context, ipsec_conn_id,
                                             ipsec_site_connection)
        except Exception as value:
            LOG.error(_LE("[AC]Failed to update IPSec site connection "
                          "in Neutron DB for %s."), value)
            raise
        return ipsec_conn_dict


class HuaweiIPsecVPNDriver(object):
    """Huawei AC VPN Service Driver class for IPsec."""

    def __init__(self):
        LOG.info(_LI("[AC]Init Huawei vpnaas driver."))
        self.vpn_reliability = ACReliabilityAPI(ac_const.NW_HW_VPNAAS)
        LOG.info(_LI("[AC]Initialization finished successfully "
                     "for Huawei vpnaas driver."))

    def create_ikepolicy(self, context, ikepolicy_dict):
        """create ike policy"""
        LOG.debug("[AC]IPSec VPN driver create IKE policy, context : %s",
                  context)
        tenant_name = ACKeyStoneIf.get_tenant_name(ikepolicy_dict,
                                                   context)
        ikepolicy_info = ACIKEpolicyModel. \
            ac_model_format(ikepolicy_dict, tenant_name)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ikepolicy_info.get('uuid'),
                                 ikepolicy_info,
                                 ac_const.NW_HW_CREATE_IKEPOLICY)

    def delete_ikepolicy(self, context, ikepolicy_id):
        """delete ike policy"""
        LOG.debug("[AC]IPSec VPN driver delete IKE policy, context : %s",
                  context)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ikepolicy_id,
                                 {},
                                 ac_const.NW_HW_DELETE_IKEPOLICY)

    def update_ikepolicy(self, context, ikepolicy_dict):
        """update ike policy"""
        LOG.debug("[AC]IPSec VPN driver update IKE policy, context : %s",
                  context)
        tenant_name = ACKeyStoneIf.get_tenant_name(ikepolicy_dict,
                                                   context)
        ikepolicy_info = ACIKEpolicyModel. \
            ac_model_format(ikepolicy_dict, tenant_name)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ikepolicy_info.get('uuid'),
                                 ikepolicy_info,
                                 ac_const.NW_HW_UPDATE_IKEPOLICY)

    def create_ipsecpolicy(self, context, ipsecpolicy_dict):
        """create ipsec policy"""
        LOG.debug("[AC]IPSec VPN driver create IPSec policy, context: %s",
                  context)
        tenant_name = ACKeyStoneIf.get_tenant_name(ipsecpolicy_dict,
                                                   context)
        ipsecpolicy_info = ACIPSecpolicyModel. \
            ac_model_format(ipsecpolicy_dict, tenant_name)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ipsecpolicy_info.get('uuid'),
                                 ipsecpolicy_info,
                                 ac_const.NW_HW_CREATE_IPSECPOLICY)

    def delete_ipsecpolicy(self, context, ipsecpolicy_id):
        """delete ipsec policy"""
        LOG.debug("[AC]IPSec VPN driver delete IPSec policy, context: %s",
                  context)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ipsecpolicy_id,
                                 {},
                                 ac_const.NW_HW_DELETE_IPSECPOLICY)

    def update_ipsecpolicy(self, context, ipsecpolicy_dict):
        """update ipsec policy"""
        LOG.debug("[AC]IPSec VPN driver update IPSec policy, context: %s",
                  context)
        tenant_name = ACKeyStoneIf.get_tenant_name(ipsecpolicy_dict,
                                                   context)
        ipsecpolicy_info = ACIPSecpolicyModel. \
            ac_model_format(ipsecpolicy_dict, tenant_name)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ipsecpolicy_info.get('uuid'),
                                 ipsecpolicy_info,
                                 ac_const.NW_HW_UPDATE_IPSECPOLICY)

    def create_vpnservice(self, context, vpnservice_dict):
        """create vpn service"""
        LOG.debug("[AC]IPSec VPN driver create vpn service, context : %s",
                  context)
        tenant_name = ACKeyStoneIf.get_tenant_name(vpnservice_dict,
                                                   context)
        vpnservice_info = ACVPNServiceModel. \
            ac_model_format(vpnservice_dict, tenant_name)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 vpnservice_info.get('uuid'),
                                 vpnservice_info,
                                 ac_const.NW_HW_CREATE_VPNSERVICE)

    def delete_vpnservice(self, context, vpnservice_id):
        """delete vpn service"""
        LOG.debug("[AC]IPSec VPN driver delete vpn service, context : %s",
                  context)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 vpnservice_id,
                                 {},
                                 ac_const.NW_HW_DELETE_VPNSERVICE)

    def update_vpnservice(self, context, vpnservice_dict):
        """update vpn service"""
        LOG.debug("[AC]IPSec VPN driver update vpn service, context : %s",
                  context)
        tenant_name = ACKeyStoneIf.get_tenant_name(vpnservice_dict,
                                                   context)
        vpnservice_info = ACVPNServiceModel. \
            ac_model_format(vpnservice_dict, tenant_name)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 vpnservice_info.get('uuid'),
                                 vpnservice_info,
                                 ac_const.NW_HW_UPDATE_VPNSERVICE)

    def create_endpoint_group(self, context, ep_group_dict):
        """create endpoint group"""
        LOG.debug("[AC] IPSec VPN driver create endpoint group, context: %s",
                  context)

        ep_group_info = ACEndPointGroupModel. \
            ac_model_format(ep_group_dict, context.tenant_name)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ep_group_info.get('uuid'),
                                 ep_group_info,
                                 ac_const.NW_HW_CREATE_EP_GROUP)

    def delete_endpoint_group(self, context, ep_group_id):
        """delete endpoint group"""
        LOG.debug("[AC] IPSec VPN driver delete endpoint group, context: %s",
                  context)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ep_group_id,
                                 {},
                                 ac_const.NW_HW_DELETE_EP_GROUP)

    def update_endpoint_group(self, context, ep_group_dict):
        """update endpoint group"""
        LOG.debug("[AC] IPSec VPN driver update endpoint group, context: %s",
                  context)

        ep_group_info = ACEndPointGroupModel. \
            ac_model_format(ep_group_dict, context.tenant_name)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ep_group_info.get('uuid'),
                                 ep_group_info,
                                 ac_const.NW_HW_UPDATE_EP_GROUP)

    def create_ipsec_site_connection(self, context, ipsec_conn_dict):
        """create ipsec site connection"""
        LOG.debug("[AC]IPSec VPN driver create IPSec site connection, "
                  "context: %s", context)
        tenant_name = ACKeyStoneIf.get_tenant_name(ipsec_conn_dict,
                                                   context)
        ipsec_conn_info = ACIPSecSiteConnModel. \
            ac_model_format(ipsec_conn_dict, tenant_name)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ipsec_conn_info.get('uuid'),
                                 ipsec_conn_info,
                                 ac_const.NW_HW_CREATE_IPSEC_SITE_CONN)

    def delete_ipsec_site_connection(self, context, ipsec_conn_id):
        """delete ipsec site connection"""
        LOG.debug("[AC]IPSec VPN driver delete IPSec site connection, "
                  "context : %s", context)

        self.vpn_reliability. \
            update_plugin_record(context,
                                 ipsec_conn_id,
                                 {},
                                 ac_const.NW_HW_DELETE_IPSEC_SITE_CONN)

    def update_ipsec_site_connection(self, context, ipsec_conn_dict):
        """update ipsec site connection"""
        LOG.debug("[AC]IPSec VPN driver update IPSec site connection, "
                  "context : %s", context)
        tenant_name = ACKeyStoneIf.get_tenant_name(ipsec_conn_dict,
                                                   context)
        ipsec_conn_info = ACIPSecSiteConnModel.ac_model_format(
            ipsec_conn_dict, tenant_name)

        self.vpn_reliability.update_plugin_record(
            context, ipsec_conn_info.get('uuid'), ipsec_conn_info,
            ac_const.NW_HW_UPDATE_IPSEC_SITE_CONN)
