#!/usr/bin/env python
# -*- coding:utf-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""Single data inconsistency check and recover"""

import ast
import copy
import traceback
import eventlet
from oslo_config import cfg
from oslo_serialization import jsonutils

try:
    from neutron.common.exceptions import NotFound
except ImportError:
    from neutron_lib.exceptions import NotFound

from networking_huawei._i18n import _LI, _LE
from networking_huawei.common import constants as constants
from networking_huawei.common import exceptions as ml2_exc
from networking_huawei.common.exceptions import GetAcDataError
from networking_huawei.drivers.ac.client.service import ACRestUtils, \
    RequestServiceParas, ACReSTService
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.common import validate
from networking_huawei.drivers.ac.client import restclient
from networking_huawei.drivers.ac.client.formatter import ACNeutronDataFormatter
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.external.ext_if import ACKeyStoneIf
from networking_huawei.drivers.ac.sync import util

requests = eventlet.import_patched('requests.__init__')
LOG = ncu.ac_log.getLogger(__name__)

REST_OPER_DICT = {
    constants.REST_POST: ac_const.OPER_CREATE,
    constants.REST_UPDATE: ac_const.OPER_UPDATE,
    constants.REST_DELETE: ac_const.OPER_DELETE
}


class CheckAndRecover(object):
    """Single data inconsistency check and recover class"""
    def __init__(self):
        self.rest_client = restclient.ACReSTClient()
        self.rest_service = ACReSTService()
        self.neutron_formatter = ACNeutronDataFormatter()
        self.db_if = ACdbInterface()
        self.timeout = \
            float(cfg.CONF.huawei_ac_config.request_timeout - 5)
        self.ac_url = ''
        self.base_url = ''
        self.get_base_url()

    def check_inconsistency(self, res_info):
        """
        Check the difference for single data between AC and the
        cloud platform. get ac data by restconf interface
        """
        LOG.info(_LI("Check inconsistency start :%s"), res_info)
        res_type = res_info.get('attribute')
        res_id = res_info.get('res_id')
        method = constants.REST_GET

        if res_type == ac_const.NW_HW_EXROUTE:
            ac_data = ast.literal_eval(res_id)
            res_id = copy.deepcopy(ac_data)
            method = constants.REST_POST
            result, ac_data = self.query_exroute_from_ac(res_id, res_type, method)
            if result == ac_const.NW_HW_ERROR:
                raise GetAcDataError(reason=str(ac_data))
        else:
            result, ac_data = self.send_data_to_ac(res_id, res_type,
                                                   method, res_data=None)
            if result == ac_const.NW_HW_ERROR:
                raise GetAcDataError(reason=str(ac_data))
        LOG.info(_LI("Get ac data :%s"), ac_data)

        neutron_data = {}
        admin_context = ncu.neutron_context.get_admin_context()
        try:
            neutron_data = self.get_neutron_info(admin_context, res_id, res_type)
        except NotFound:
            LOG.error(_LI("Get neutron data error, data is not exists"))
        except Exception as ex:
            LOG.error(_LI("Get neutron data raise exception: %s"),
                      str(ex.message))
            raise
        if neutron_data:
            attr_time = self.get_attribute_time(res_id, res_type)
            if attr_time:
                neutron_data['updated-at'] = attr_time.update_time. \
                    strftime(ac_const.ISO8601_TIME_FORMAT)
                neutron_data['created-at'] = attr_time.create_time. \
                    strftime(ac_const.ISO8601_TIME_FORMAT)

        return self.compare_ac_and_neutron_data(res_id, res_type,
                                                ac_data, neutron_data)

    def compare_ac_and_neutron_data(self, res_id, res_type,
                                    ac_data, neutron_data):
        """
        Compare the data on the Agile Controller-DCN with that
        on the cloud platform. If the data exists,
        compare the data based on the update time.
        """
        res_cfg = 'huawei-ac-neutron' + ':' + str(res_type)
        result_data = []
        result_body = {'id': res_id,
                       'tenant_name': '',
                       'res_name': '',
                       'res_type': res_type,
                       'create_time': '',
                       'update_time': '',
                       'status': '',
                       'detail': ''}
        if res_type == ac_const.NW_HW_EXROUTE:
            result_body = self.compare_data_for_exroute(res_id, result_body, ac_data, neutron_data)
            return [result_body]
        if ac_data and not neutron_data:
            result_body = self.get_check_result_body(result_body, ac_data)
            result_body['status'] = 'Missing in neutron'
        elif not ac_data and not neutron_data:
            result_body['status'] = 'Data is not exist'
        else:
            result_body = self.get_check_result_body(result_body, neutron_data)
            validate.validate_log_record(neutron_data, res_type)
            result_body['detail'] = neutron_data
            if not ac_data and neutron_data:
                result_body['status'] = 'Missing in controller'
            elif ac_data and neutron_data:
                result_body['status'] = 'No different'
                neutron_update_time = neutron_data.get('updated-at')
                ac_info = ac_data.get(res_cfg, [])
                ac_update_time = ac_info[0].get('updated-at') \
                    if (ac_info and isinstance(ac_info, list)) else None
                if neutron_update_time != ac_update_time:
                    result_body['status'] = 'Data is not same'
            else:
                result_body['status'] = 'No different'

        result_data.append(result_body)
        return result_data

    def compare_data_for_exroute(self, res_id, result_body, ac_data, neutron_data):
        result_body['id'] = \
            "{'router_id':'%s', 'nexthop':'%s', 'destination':'%s', 'type':'%s'}" \
            % (res_id.get('router_id'), res_id.get('nexthop'),
               res_id.get('destination'), res_id.get('type'))
        if ac_data and not neutron_data:
            result_body['status'] = 'Missing in neutron'
        elif not ac_data and not neutron_data:
            result_body['status'] = 'Data is not exist'
        elif not ac_data and neutron_data:
            result_body['status'] = 'Missing in controller'
            result_body['detail'] = neutron_data
        else:
            result_body['status'] = 'No different'
            result_body['detail'] = neutron_data
        return result_body

    @classmethod
    def get_check_result_body(cls, result_body, res_data):
        """Get result data body for response to ac"""
        result_body['res_name'] = res_data.get('name')
        result_body['tenant_name'] = res_data.get('tenant-name')
        result_body['create_time'] = res_data.get('created-at')
        result_body['update_time'] = res_data.get('updated-at')
        return result_body

    def send_data_to_ac(self, res_id, res_type, method, res_data):
        """Send data to ac"""
        res_url = util.ACUtil.form_resource_url(
            ac_const.NW_HW_URL, method, res_type, res_id)
        if res_type == ac_const.NW_HW_EXROUTE:
            if method == constants.REST_POST:
                res_url = ac_const.ADD_EXROUTES_URL
            elif method == constants.REST_DELETE:
                res_url = ac_const.REMOVE_EXROUTES_URL
                method = constants.REST_POST
        try:
            ret = self.rest_client. \
                send(method, '%s%s' % (self.base_url, res_url),
                     res_id, res_data, 0, self.timeout)

            res_code = int(ret.status_code)
            res_json = jsonutils.loads(ret.content) if ret.content else None

            res = {'res_type': res_type}
            rest_info = jsonutils.loads(res_data). \
                get(res.get('res_type'), [{}])[0] if res_data else None

            result_code = self.update_db_state(method, res, res_code, rest_info)
            if result_code == ac_const.DATA_NOT_FOUND:
                res_json = {}
            return result_code, res_json

        except (requests.Timeout):
            LOG.error(_LE("[AC] AC request timeout error, "
                          "method: %(method)s, url: %(url)s, id: %(res_id)s"),
                      {'method': method, 'url': res_url, 'res_id': res_id})
            raise

        except Exception as ex:
            LOG.error(_LE("[AC] AC request exception, traceback: %s"),
                      traceback.format_exc())
            raise ml2_exc.MechanismDriverError(method=method, url=res_url,
                                               error=ex.message)

    def get_base_url(self):
        """Combine the body of the basic URL."""
        host_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip. \
            replace(' ', '').split(',')
        if len(host_list) == 1:
            self.base_url = '%s%s%s%s' % (
                ac_const.HTTPS_HEADER, host_list[0], ":",
                str(ac_const.rest_server_port))
        else:
            self.base_url = '%s%s%s%s' % (
                ac_const.HTTPS_HEADER, ac_const.DEFAULT_AC_IP, ":",
                str(ac_const.rest_server_port))

    def get_neutron_info(self, context, res_id, res_type):
        """Get neutron db info by type and id"""
        tenant_name = ''
        if res_type == ac_const.NW_HW_EXROUTE:
            router_id = res_id.get('router_id')
            attribute = {'nexthop': res_id.get('nexthop'),
                         'destination': res_id.get('destination'),
                         'type': res_id.get('type')}
            res_data = self.neutron_formatter.set_exroute(
                context, tenant_name, router_id=router_id, attribute=attribute)
        else:
            formatter_func = copy.deepcopy(ac_const.FORMATTER_FUNC)
            res_data = getattr(self.neutron_formatter, formatter_func.get(res_type))(
                context, tenant_name, res_id)
        if 'tenant-name' in res_data and not res_data.get('tenant-name'):
            tenant_id = res_data.get('tenant-id')
            tenant_name = ACKeyStoneIf. \
                get_tenant_name_by_id_from_keystone(tenant_id)
            if tenant_name and tenant_name != tenant_id:
                tenant_name = tenant_name + "(" + tenant_id + ")"
            if tenant_name:
                res_data['tenant-name'] = tenant_name
            LOG.debug(_LI("get_tenant_name from keystone %s"), tenant_name)
        return res_data

    def check_is_recover_gw_port(self, res_type, operation, neutron_data):
        """Check whether the router is associated with external network."""
        if res_type == ac_const.NW_HW_ROUTERS \
                and operation == constants.REST_POST \
                and neutron_data.get('gateway-port-id', None):
            if self.check_gw_port_inconsistency(neutron_data):
                return True
        return False

    def check_gw_port_inconsistency(self, neutron_data):
        """Step 1: check whether the port exists on the AC."""
        gw_port_id = neutron_data.get('gateway-port-id')
        res_info = {'attribute': ac_const.NW_HW_PORTS, 'res_id': gw_port_id}
        check_result = self.check_inconsistency(res_info)
        if check_result \
                and check_result[0].get('status') == 'Missing in controller':
            return True
        return False

    def recover_router_with_gw_port(self, context, neutron_data):
        """If the router has gw port and the gw port not exist in AC,
        first step: create router without gw port and create gw port
        """
        LOG.info("Begin to create router without gw port and create gw port")
        gw_port_id = neutron_data.get('gateway-port-id')
        result_body = {}

        # Step 1: create router without gateway port
        tmp_data = copy.deepcopy(neutron_data)
        tmp_data.pop('gateway-port-id', None)
        tmp_data.pop('external-gateway-info', None)
        LOG.info("Get tmp data1: %s", tmp_data)
        rest_paras = RequestServiceParas(
            context.session, '', ac_const.NW_HW_ROUTERS, constants.REST_POST,
            '', gw_port_id, tmp_data)
        res_data = ACRestUtils.generate_rest_data(rest_paras)
        LOG.info("Get tmp data2: %s", res_data)
        result, res_json = self.send_data_to_ac(gw_port_id,
                                                ac_const.NW_HW_ROUTERS,
                                                constants.REST_POST,
                                                res_data)
        if result == ac_const.NW_HW_SUCCESS:
            self.create_recover_record(constants.REST_POST,
                                       ac_const.NW_HW_ROUTERS,
                                       tmp_data, context.session)
        else:
            result_body['status'] = 'error'
            result_body['exec_result'] = 'error'
            result_body['error_message'] = str(res_json)
            return result_body

        # Step 2: create gateway port
        tmp_gw_data = self.get_neutron_info(context, gw_port_id,
                                            ac_const.NW_HW_PORTS)
        gw_port_data = self.set_res_data_time(context.session,
                                              tmp_gw_data, gw_port_id,
                                              ac_const.NW_HW_PORTS)
        rest_paras = RequestServiceParas(
            context.session, '', ac_const.NW_HW_PORTS, constants.REST_POST,
            '', gw_port_id, gw_port_data)
        res_data = ACRestUtils.generate_rest_data(rest_paras)
        result, res_json = self.send_data_to_ac(gw_port_id,
                                                ac_const.NW_HW_PORTS,
                                                constants.REST_POST,
                                                res_data)
        if result != ac_const.NW_HW_SUCCESS:
            result_body['status'] = 'error'
            result_body['exec_result'] = 'error'
            result_body['error_message'] = str(res_json)
            return result_body

        return result_body

    def recover_inconsistency(self, res_info):
        """recover inconsistency"""
        LOG.info(_LI("Recover inconsistency start :%s"), res_info)
        res_type = res_info.get('attribute')
        res_id = res_info.get('res_id') \
            if res_type != ac_const.NW_HW_EXROUTE \
            else ast.literal_eval(res_info.get('res_id'))
        operation = res_info.get('oper')
        admin_context = ncu.neutron_context.get_admin_context()
        neutron_data = {}
        result_data = []
        if operation != constants.REST_DELETE:
            neutron_data = self.get_neutron_info(admin_context, res_id, res_type)
            neutron_data = self.set_res_data_time(admin_context.session, neutron_data, res_id, res_type)
            if not neutron_data and res_type == ac_const.NW_HW_EXROUTE:
                LOG.info(_LI("Recover inconsistency exroute failed, data is null"))
                return result_data
        try:
            if self.check_is_recover_gw_port(res_type, operation, neutron_data):
                self.recover_router_with_gw_port(admin_context, neutron_data)
                operation = constants.REST_UPDATE
            resource_id = res_id
            if res_type == ac_const.NW_HW_EXROUTE:
                resource_id = res_id.get('router_id')
                if operation == constants.REST_DELETE:
                    neutron_data = self._format_exroute(res_id)
            rest_paras = RequestServiceParas(
                admin_context.session, '', res_type, operation, '', resource_id, neutron_data)
            res_data = ACRestUtils.generate_rest_data(rest_paras)
            result, res_json = self.send_data_to_ac(resource_id, res_type, operation, res_data)
            result_body = self.init_result_body(res_id, res_type, operation)
            if res_type == ac_const.NW_HW_EXROUTE:
                result_body['id'] = "{'router_id':'%s', 'nexthop':'%s', 'destination':'%s', 'type':'%s'}" \
                                    % (resource_id, res_id.get('nexthop'), res_id.get('destination'),
                                       res_id.get('type'))
            if operation != constants.REST_DELETE:
                self.update_result_body(result_body, neutron_data)
            if result == ac_const.NW_HW_SUCCESS:
                self.create_recover_record(operation, res_type, neutron_data, None)
                result_body['status'] = 'success'
                result_body['exec_result'] = 'success'
            else:
                result_body['status'] = 'error'
                result_body['exec_result'] = 'error'
                result_body['error_message'] = str(res_json)
            LOG.info(_LI("Recover inconsistency get result :%s"), result_body)
            result_data.append(result_body)
            return result_data

        except Exception as ex:
            LOG.error(_LE("Recover inconsistency data failed, controller returns status code %s."), str(ex))
            raise

    def _format_exroute(self, exroutes_dict):
        exroutes_info = {
            'router-id': exroutes_dict.get("router_id")
        }
        if 'nexthop' in exroutes_dict:
            exroutes_info['nexthop'] = exroutes_dict['nexthop']
        if 'destination' in exroutes_dict:
            exroutes_info['destination'] = exroutes_dict['destination']
        if exroutes_dict.get('type'):
            exroutes_info['type'] = int(exroutes_dict['type'])
        return exroutes_info

    def init_result_body(self, res_id, res_type, operation):
        result_body = {'id': res_id,
                       'tenant_name': '',
                       'res_name': '',
                       'res_type': res_type,
                       'create_time': '',
                       'update_time': '',
                       'status': '',
                       'oper': operation,
                       'sync_type': '',
                       'exec_result': '',
                       'error_message': ''}
        return result_body

    def update_result_body(self, result_body, neutron_data):
        if neutron_data.get('tenant-name'):
            result_body['tenant_name'] = neutron_data.get('tenant-name')
        elif neutron_data.get('tenant-id'):
            result_body['tenant_name'] = neutron_data.get('tenant-id')
        result_body['res_name'] = neutron_data.get('name')
        result_body['create_time'] = neutron_data.get('created-at')
        result_body['update_time'] = neutron_data.get('updated-at')

    def update_db_state(self, method, res, res_code, rest_info):
        """update db state"""
        if requests.codes.ok <= res_code < requests.codes.multiple_choices:
            LOG.info(_LI('Send request successfully and AC process ok, '
                         'response: %s'), res_code)
            self.update_db_state_for_ok(method, res, rest_info)
            return ac_const.NW_HW_SUCCESS
        elif method == 'GET' and res_code == requests.codes.not_found:
            return ac_const.DATA_NOT_FOUND
        else:
            LOG.error(_LE("Send data failed, controller returns status "
                          "code %s."), res_code)
            self.update_db_state_for_error(method, res, rest_info)
            return ac_const.NW_HW_ERROR

    def update_db_state_for_ok(self, method, res, rest_info):
        if rest_info and not self.rest_service.is_need_short_feedback(res, rest_info, method):
            LOG.debug("[AC] Long feedback need to update status "
                      " to down or active %s", rest_info)
            if util.ACUtil.check_resource_have_state(res['res_type']):
                status = self._get_res_status(res, rest_info)
                session = self.db_if.get_session()
                self.db_if.update_neutron_db_state(
                    session,
                    rest_info['uuid'],
                    res['res_type'],
                    status)

    def update_db_state_for_error(self, method, res, rest_info):
        if rest_info and not self.rest_service. \
                is_need_short_feedback(res, rest_info, method):
            LOG.debug("[AC] Long feedback need to update status "
                      "to error %s", rest_info)
            if util.ACUtil.check_resource_have_state(res['res_type']):
                status = ac_const.NEUTRON_STATUS_ERROR

                session = self.db_if.get_session()
                self.db_if.update_neutron_db_state(
                    session,
                    rest_info['uuid'],
                    res['res_type'],
                    status)

    def create_recover_record(self, method, res_type, res_info, session):
        """
        After the single data recover is complete,
        write the db with short feedback request for waiting status report.
        """
        res = {'res_type': res_type}
        if self.rest_service.is_need_short_feedback(res, res_info, method):
            operation = REST_OPER_DICT.get(method)
            uuid = res_info['uuid']
            self.db_if.create_plugin_record(session, res_info, (uuid, operation, res_type), ac_const.COMPLETE)

    def get_attribute_time(self, res_id, res_type):
        """ get update time from neutron db """
        if res_type == ac_const.NW_HW_EXROUTE:
            return None
        if res_type != ac_const.NW_HW_PORTS:
            res_info = self.db_if.get_res_attribute(res_id=res_id,
                                                    res_type=res_type)
        else:
            res_info = self.db_if.get_port_attribute(res_id=res_id)
        LOG.info('Get attribute time %s for res id: %s', res_info, res_id)
        return res_info

    def set_res_data_time(self, session, res_data, res_id, res_type):
        """Set the res data create and update time"""
        if res_type == ac_const.NW_HW_EXROUTE:
            return res_data
        rec = self.get_attribute_time(res_id, res_type)
        if rec:
            res_data['updated-at'] = rec.update_time. \
                strftime(ac_const.ISO8601_TIME_FORMAT)
            res_data['created-at'] = rec.create_time. \
                strftime(ac_const.ISO8601_TIME_FORMAT)
        else:
            network_id = None
            if res_type == ac_const.NW_HW_PORTS and 'network-id' in res_data:
                network_id = res_data['network-id']
            updated_at = ACCommonUtil.get_standard_current_time(). \
                strftime(ac_const.ISO8601_TIME_FORMAT)
            self.db_if.update_attribute_record(session, res_id, res_type,
                                               updated_at=updated_at,
                                               network_id=network_id)
            res_data['updated-at'] = updated_at
            res_data['created-at'] = updated_at
        return res_data

    def query_exroute_from_ac(self, res_id, res_type, method):
        res_data = self.build_query_exroute_req(res_id)
        res_url = ac_const.NW_HW_EXROUTE_QUERY_URL
        LOG.info("query ac url:%s,req:%s", res_url, res_data)
        try:
            ret = self.rest_client. \
                send(method, '%s%s' % (self.base_url, res_url),
                     res_id, res_data, 0, self.timeout)

            res_code = int(ret.status_code)
            if requests.codes.ok <= res_code < requests.codes.multiple_choices:
                responce_content = jsonutils.loads(ret.content) if ret.content else None
                res_json = responce_content.get('huawei-ac-neutron-exroutes:output') if responce_content else None
                result_code = ac_const.NW_HW_SUCCESS
            else:
                LOG.error(_LE("Send data failed, controller returns status code %s."), res_code)
                res_json = jsonutils.loads(ret.content) if ret.content else None
                result_code = ac_const.NW_HW_ERROR
            return result_code, res_json

        except (requests.Timeout):
            LOG.error(_LE("[AC] AC request timeout error, "
                          "method: %(method)s, url: %(url)s, id: %(res_id)s"),
                      {'method': method, 'url': res_url, 'res_id': res_id})
            raise

        except Exception as ex:
            LOG.error(_LE("[AC] AC request exception, traceback: %s"),
                      traceback.format_exc())
            raise ml2_exc.MechanismDriverError(method=method, url=res_url,
                                               error=ex.message)

    def build_query_exroute_req(self, res_id):
        req = {
            'huawei-ac-neutron-exroutes:input': {
                'destination': res_id.get('destination'),
                'nexthop': res_id.get('nexthop'),
                'router-id': res_id.get('router_id'),
                'type': res_id.get('type')
            }
        }
        return jsonutils.dumps(req)

    @staticmethod
    def _get_res_status(res, rest_info):
        """get res status"""
        def _check_rest_info(rest_info):
            return 'admin-state-up' in rest_info and not rest_info['admin-state-up'] \
                and rest_info.get('device-owner') != 'neutron:LOADBALANCERV2'

        if res['res_type'] == ac_const.NW_HW_PORTS and _check_rest_info(rest_info):
            status = ac_const.NEUTRON_STATUS_DOWN
        else:
            status = ac_const.NEUTRON_STATUS_ACTIVE
        return status
