#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""controll task"""

import copy
from random import randint
from time import sleep
import datetime

from networking_huawei.drivers.ac.db.schema import ACNeutronStateSchema, \
    ACPluginSchema
from oslo_config import cfg
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging

try:
    from oslo_service import loopingcall
except ImportError:
    from neutron.openstack.common import loopingcall
from networking_huawei._i18n import _LI, _LE
from networking_huawei.drivers.ac.db import dbif
from networking_huawei.drivers.ac.db import schema as dbschema
from networking_huawei.drivers.ac.common import constants as ac_const
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.sync import util
from networking_huawei.drivers.ac.sync import neutron_sync
from networking_huawei.drivers.ac.db.dbif import ACdbInterface

LOG = logging.getLogger(__name__)


class ACCommonControlTask(object):
    """AC Common Control Task"""

    def __init__(self):
        self.timer = loopingcall.FixedIntervalLoopingCall(
            self.perform_control_tasks)
        self._interval = ac_const.CONTROL_TASK_STEP_INTERVAL
        self._step_counter = 1
        self.control_tasks = []
        self._max_counter_limit = 1

    def start(self):
        """start control task"""
        LOG.info(_LI("Starting control task thread for an interval of %d "
                     "seconds."), self._interval)
        self.timer.start(self._interval,
                         initial_delay=ac_const.CONTROL_TASK_STEP_INTERVAL)

    def register_looping_control_tasks(self, func, interval):
        """Register the maintenance control operations with the timer value.

        :param func: Function to call when the maintenance is required
        :param interval: Interval between the call
        """
        LOG.debug('Registering looping function: %s for interval: %d',
                  func.__name__, interval)
        self.control_tasks.append(
            {"func": func, "interval": interval, "loop": True})
        if self._max_counter_limit % interval != 0:
            self._max_counter_limit *= interval
        LOG.debug('Max counter limit: %s', self._max_counter_limit)

    def register_non_looping_control_tasks(self, func, interval):
        """Register the non-looping maintenance control operations with the
        timer value.

        :param func: Function to call when the maintenance is required
        :param interval: Interval required before the call
        """
        LOG.debug('Registering non looping function: %s for interval: %d',
                  func.__name__, interval)
        self.control_tasks.append(
            {"func": func, "interval": interval, "loop": False})
        if self._max_counter_limit % interval != 0:
            self._max_counter_limit *= interval
        LOG.debug('Max counter limit: %s', self._max_counter_limit)

    def perform_control_tasks(self):
        """perform control tasks"""
        try:
            session = ACdbInterface().get_session()
            for control_task in self.control_tasks:
                self._handle_control_task(session, control_task)
        except Exception:
            LOG.exception(_LE('Exception while performing control task'))
        finally:
            if self._max_counter_limit == self._step_counter:
                self._step_counter = 1
            else:
                self._step_counter += 1

    def _handle_control_task(self, session, control_task):
        if (self._step_counter % control_task["interval"]) == 0:
            self.execute_control_task(
                session, control_task=control_task["func"])
            if not control_task["loop"]:
                self.control_tasks.remove(control_task)

    @classmethod
    def execute_control_task(cls, session, control_task):
        """execute control task"""
        task_details = control_task.__name__
        if control_task.__doc__:
            LOG.debug('%s:%s', task_details, control_task.__doc__)

        try:
            LOG.debug("Starting control maintenance operation: %s.",
                      task_details)
            control_task(session=session)
        except Exception:
            LOG.exception(_LE("Exception during maintenance operation %s."),
                          task_details)


class ACMaintenance(object):
    """AC Maintenance"""

    def __init__(self):
        self._dbif = dbif.ACdbInterface()
        self.timeout = float(cfg.CONF.huawei_ac_config.request_timeout)
        self.timeout_retry = int(cfg.CONF.huawei_ac_config.timeout_retry)
        self.neutron_name = ACCommonUtil.get_neutron_server_name()
        self._util = util.ACUtil()
        host_list = cfg.CONF.huawei_ac_agent_config.rpc_server_ip.replace(
            ' ', '').split(',')
        if len(host_list) == 1:
            self._url = '%s%s%s%s' % (
                ac_const.HTTPS_HEADER, host_list[0], ":",
                str(ac_const.rest_server_port))
        else:
            self._url = '%s%s%s%s' % (
                ac_const.HTTPS_HEADER, ac_const.DEFAULT_AC_IP, ":",
                str(ac_const.rest_server_port))

    def handle_long_pending_in_process_records(self, session):
        """Handle and long pending process on in-process state after neutron
            server restart.

        :param session: Current session info.
        """
        time_delta_s = self.timeout
        incremental_time = time_delta_s
        index = self.timeout_retry
        while index > 0:
            incremental_time = util.ACUtil.calculate_incremental_timeout(
                incremental_time)
            time_delta_s += incremental_time
            index -= 1

        # Adding additional 10 seconds for safe purpose
        time_delta_s += 10

        LOG.debug('Time delta for processing current maintenance task is %f',
                  time_delta_s)

        local_seq_num = ac_const.PLUGIN_INVALID_SEQ_NUM
        while True:
            plugin_data = self._dbif.get_next_plugin_record(
                session, ACPluginSchema(seq_num=local_seq_num,
                                        state=ac_const.IN_PROCESS),
                time_delta=time_delta_s)
            if not plugin_data:
                break
            LOG.debug('Updating the records long pending in process '
                      'state(seq_num = %d).', plugin_data.seq_num)
            self._dbif.update_plugin_record(
                session, ACPluginSchema(seq_num=plugin_data.seq_num,
                                        state=ac_const.WAIT))

    # Every 2 minutes update the access time of the neutron server to keep the
    # neutron server updated and thus informing others that this one is alive.
    def handle_server_status_update(self, session):
        """Periodically update the access time to inform others about the alive
         status.

        :param session: Current session info.
        """
        self._dbif.update_access_time_server_record(session=session)

    # Periodically at every 10 minutes check the other neutron servers in the
    # cluster mode are alive or not. If not available, consider it as dead and
    # assign its jobs to the leader neutron server. If the leader itself has
    # dead, then assign the jobs to the new leader. And if the dead neutron
    # server was on neutron sync, then perform a neutron sync in the server
    # which has identified the dead neutron server.
    def handle_server_alive_check(self, session):
        """Periodically check other neutron server alive status.

        :param session: Current session info.
        """
        LOG.debug('Server alive check.')
        server_list = self._dbif.get_dead_neutron_server_list(session)
        if not server_list:
            LOG.debug('Not found any dead neutron servers.')
            return

        dead_leader = {}
        need_neutron_sync = False

        for server in server_list:
            LOG.debug('Found %s dead.', server.neutron_name)
            if server.is_leader:
                LOG.error('Found the leader neutron server(%d) dead.',
                          server.neutron_id)
                dead_leader = {'neutron_id': server.neutron_id,
                               'access_time': server.access_time,
                               'neutron_name': server.neutron_name}
                LOG.error('dead leader: %s', dead_leader)
            else:
                LOG.error(
                    _LE('Neutron server %s is detected as down, please '
                        'check immediately.'), server.neutron_name)
            if server.state in [ac_const.NEUTRON_SYNC, ac_const.CONSISTENCY_CHECK]:
                LOG.info('Server is in the neutron sync state.')
                need_neutron_sync = True
                self._util.clear_dependent_suspended_records(session,
                                                             res_type=None)
                self._util.clear_neutron_sync_records(
                    session, server.neutron_id)

            self._dbif.update_server_record(
                ACNeutronStateSchema(
                    neutron_id=server.neutron_id, is_leader=False,
                    state=ac_const.ERROR, sync_res=''),
                session=session)

        need_neutron_sync = self._handle_dead_leader(
            dead_leader, need_neutron_sync, session)

        alive_server_list = self._dbif.get_alive_server_record_list(session)
        leader_server = util.ACUtil.get_leader_in_server_list(
            alive_server_list)

        if dead_leader:
            LOG.critical(_LE('Neutron server leader %s is detected as down, '
                             'please check immediately. New leader is %s.'),
                         dead_leader['neutron_name'],
                         leader_server.neutron_name)

        self._assign_job(session, server_list, leader_server)

        if need_neutron_sync:
            LOG.info(_LI('Dead neutron server was on neutron sync, '
                         're-triggering again.'))
            # Start neutron sync after 5 seconds
            neutron_sync.ACNeutronSync(ac_const.NEUTRON_SYNC_PARAM_DEFAULT)

    def _handle_dead_leader(self, dead_leader, need_neutron_sync, session):
        if dead_leader:
            LOG.info('dead leader: %s', dead_leader)
            timedelta_s = datetime.timedelta(minutes=10)
            access_time = dead_leader['access_time']
            sync_time_str = cfg.CONF.huawei_ac_config.neutron_sync_time
            current_time = datetime.datetime.utcnow()
            try:
                sync_time = datetime.datetime.strptime(
                    sync_time_str, '%H:%M:%S')

                expect_sync_time = datetime.datetime(year=current_time.year,
                                                     month=current_time.month,
                                                     day=current_time.day,
                                                     hour=sync_time.hour,
                                                     minute=sync_time.minute,
                                                     second=sync_time.second)
            except ValueError:
                sync_time = self._get_sync_time(sync_time_str)
                weekday = sync_time_str.split(' ', 1)[0]
                dow = ['mon', 'monday', 'tue', 'tuesday', 'wed', 'wednesday',
                       'thu', 'thursday', 'fri', 'friday', 'sat', 'saturday',
                       'sun', 'sunday']
                weekday_num = dow.index(weekday.lower()) // 2
                daily_sync_time = datetime.datetime(year=current_time.year,
                                                    month=current_time.month,
                                                    day=current_time.day,
                                                    hour=sync_time.hour,
                                                    minute=sync_time.minute,
                                                    second=sync_time.second)
                expect_sync_time = daily_sync_time - datetime.timedelta(
                    (daily_sync_time.weekday() - weekday_num) % 7)

            LOG.debug('expect_sync_time: %s', expect_sync_time)
            LOG.debug('access_time: %s', access_time)
            time_diff = expect_sync_time - access_time
            LOG.debug('time_diff: %s', time_diff)

            if access_time < expect_sync_time and time_diff < timedelta_s:
                LOG.info(_LI('Found the dead leader does not perform '
                             'neutron sync at %s. Trigger neutron sync.'),
                         sync_time)
                need_neutron_sync = True
            self.handle_leader_selection(session)
        return need_neutron_sync

    @classmethod
    def _get_sync_time(cls, sync_time_str):
        try:
            sync_time = datetime.datetime.strptime(
                sync_time_str, '%a %H:%M:%S')
        except ValueError:
            sync_time = datetime.datetime.strptime(
                sync_time_str, '%A %H:%M:%S')
        return sync_time

    def _assign_job(self, session, server_list, leader_server):
        for server in server_list:
            rec_list = self._dbif.get_plugin_record_list(
                session, ACPluginSchema(neutron_id=server.neutron_id))
            for record in rec_list:
                LOG.debug('Reassign the job %d to new server.', record.seq_num)
                self._dbif.update_plugin_record(
                    session,
                    ACPluginSchema(neutron_id=leader_server.neutron_id,
                                   state=ac_const.WAIT),
                    old_value=record)

    # If user has changed the neutron server name, then the neutron state DB
    # record can be leaked forever. So this control task will check the neutron
    # server record which has not updated the access time and crossed more than
    # one hour and delete the records by considering it as permanently
    # unavailable for service.
    def handle_long_pending_error_servers(self, session):
        """Periodically check any long pending error state neutron servers.

        :param session: Current session info.
        """
        server_list = self._dbif.get_server_record_list(
            session, state=ac_const.ERROR)
        now = datetime.datetime.utcnow()
        timedelta_s = datetime.timedelta(
            minutes=ac_const.SERVER_ERR_REC_INTERVAL)
        for server in server_list:
            if server.access_time < (now - timedelta_s):
                LOG.error('Deleting long pending server record, neutron '
                          'id: %d', server.neutron_id)
                self._dbif.delete_server_record(session, server.neutron_id)

    @classmethod
    def get_leader(cls, server_list):
        """get leader"""
        leader_sync_type = [ac_const.NEUTRON_SYNC_INTERVAL_AND_RESTART,
                            ac_const.NEUTRON_SYNC_INTERVAL]
        new_leader = None
        actual_leader = None
        for server in server_list:
            if server.is_leader:
                actual_leader = server
            if not new_leader:
                new_leader = server
            if new_leader.sync_type not in leader_sync_type and \
                    server.sync_type in leader_sync_type:
                new_leader = server
        return actual_leader, new_leader

    # Perform the leader selection operation and find the leader, this is a
    # onetime activity during startup, after one time execution this will be
    # removed from the task list.
    def handle_leader_selection(self, session):
        """Perform leader selection.

        :param session: Current session info.
        """
        server_list = self._dbif.get_alive_server_record_list(session)
        if not server_list:
            LOG.debug('No alive neutron servers in leader selection.')
            return

        actual_leader, new_leader = self.get_leader(server_list)

        if actual_leader:
            if actual_leader.neutron_id == new_leader.neutron_id:
                LOG.debug('No change in the selected leader. Leader is %s, '
                          'neutron id:%d', actual_leader.neutron_name,
                          actual_leader.neutron_id)
                return
            else:
                LOG.debug('Found new leader. Resetting the leadership of '
                          'existing leader %s, neutron id:%d',
                          actual_leader.neutron_name, actual_leader.neutron_id)
                self._dbif.update_server_record(
                    ACNeutronStateSchema(neutron_id=actual_leader.neutron_id,
                                         is_leader=False))

        LOG.debug('Assigning new leader as %s, neutron id:%d.', new_leader.
                  neutron_name, new_leader.neutron_id)
        self._dbif.update_server_record(
            ACNeutronStateSchema(neutron_id=new_leader.neutron_id,
                                 is_leader=True))

        sleep(randint(1, 10))
        server_list = self._dbif.get_alive_server_record_list(
            session)
        already_found_leader = False
        for server in server_list:
            if server.is_leader:
                LOG.debug('Leader neutron server %s:%d.',
                          server.neutron_name, server.neutron_id)
                if already_found_leader:
                    LOG.debug('Multiple leader detected, withdraw claim.')
                    self._dbif.update_server_record(
                        ACNeutronStateSchema(neutron_id=server.neutron_id,
                                             is_leader=False))
                already_found_leader = True

    def handle_neutron_sync_at_reboot_after_leader_sel(self, session):
        """Handle neutron sync start after restart of server.

        :param session: Current session info.
        """

        server = self._dbif.get_server_record(
            session, neutron_name=self.neutron_name)
        sync_type = cfg.CONF.huawei_ac_config.neutron_sync_type
        sync_param = copy.deepcopy(ac_const.NEUTRON_SYNC_PARAM_DEFAULT)
        if sync_type in [ac_const.NEUTRON_SYNC_INTERVAL_AND_RESTART,
                         ac_const.NEUTRON_SYNC_INTERVAL]:
            LOG.debug("Neutron sync triggered to happen on configured "
                      "neutron_sync_time in neutron server %s",
                      self.neutron_name)
            sync_param.update({"looping": True,
                               "sync_op": ac_const.SYNC_OP_SYNC_DATA})
            neutron_sync.ACNeutronSync(sync_param)
        if server.state in [ac_const.NEUTRON_SYNC, ac_const.CONSISTENCY_CHECK]:
            LOG.info(_LI("Neutron sync triggered on neutron server %s as it "
                         "was on neutron-sync before restart."),
                     self.neutron_name)
            self._util.clear_dependent_suspended_records(session,
                                                         res_type=None)
            self._util.clear_neutron_sync_records(session, server.neutron_id)
            self._dbif.update_server_record(
                ACNeutronStateSchema(neutron_id=server.neutron_id,
                                     state=ac_const.NORMAL,
                                     sync_res=''))
            neutron_sync.ACNeutronSync(sync_param)
        else:
            sync_type = cfg.CONF.huawei_ac_config.neutron_sync_type
            if sync_type in [ac_const.NEUTRON_SYNC_INTERVAL_AND_RESTART,
                             ac_const.NEUTRON_SYNC_RESTART]:
                LOG.info(_LI("Neutron sync triggered on restart of system on "
                             "neutron server %s"), self.neutron_name)
                neutron_sync.ACNeutronSync(sync_param)

    def handle_long_pending_validation_records(self, session):
        """Handle long pending validation records.

        :param session: Current session info.
        """

        validation_records = self._dbif.read_dependency(session)
        plugin_seq_list = self.get_plugin_seq_num_list(session)

        for val_rec in validation_records:
            if val_rec.dep_seq_num not in plugin_seq_list:
                LOG.info(_LI('Cleaning the record in validation for seq_num'
                             ': %d, dep_seq_num: %d'), val_rec.res_seq_num,
                         val_rec.dep_seq_num)
                self._dbif.delete_dependency(rec=val_rec)

    # Get a list of plugin records seq_number
    def get_plugin_seq_num_list(self, session):
        """Query plugin records from db based on the conditions.

        :param session:     Current session info where the operation is
                              performed. Used to access db.
        :return: List of plugin records or None
        """
        if not session:
            session = self._dbif.get_session()

        plugin_rec = session.query(dbschema.ACPluginSchema.seq_num).all()
        if plugin_rec is None:
            return []
        seq_list = [res[0] for res in plugin_rec]
        return seq_list

    def handle_segments_records(self, session):
        """clean invalid segments.
        """
        self._dbif.handle_segments_records(session)
