#!/usr/bin/env python
# -*- coding:utf-8 -*-
# Copyright 2016 Huawei Technologies Co. Ltd. All rights reserved.
"""ac message reliability driver"""

import re
import datetime
import os
import sys
import six
from oslo_config import cfg

from networking_huawei._i18n import _LI
from networking_huawei.drivers.ac.common.neutron_compatible_util import \
    ac_log as logging
from networking_huawei.drivers.ac.common import constants as ac_constants
from networking_huawei.drivers.ac.db.dbif import ACdbInterface
from networking_huawei.drivers.ac.db.schema import ACNeutronStateSchema
from networking_huawei.drivers.ac.db.utcnow import utcnow
from networking_huawei.drivers.ac.sync.normal_sync import ACNormalSync
from networking_huawei.drivers.ac.sync.cloud_alarm_sync import CloudAlarmSync
from networking_huawei.drivers.ac.sync.failed_resource_sync \
    import ACFailedResourceSync
from networking_huawei.drivers.ac.sync.websocket_msg_sync \
    import WebsocketMsgSync
from networking_huawei.drivers.ac.sync.verify_cert import ACCertSync
from networking_huawei.drivers.ac.sync import control_task
from networking_huawei.drivers.ac.common.util import ACCommonUtil
from networking_huawei.drivers.ac.sync import http_heart_beat
from networking_huawei.drivers.ac.common import neutron_compatible_util as ncu
from networking_huawei.drivers.ac.common.validate import ParamNotConfigured, \
    ParamValueInvalid, validate_rpc_server_ip, validate_ip_address

LOG = logging.getLogger(__name__)

CERT_EXPIRED_WARNING_TIME_CFG = \
    'huawei_driver_config.huawei_ac_config.cert_expiration_warning_time'
CERT_EXPIRED_WARNING_TIME = '30'


class Singleton(object):
    """ To make only one time the Reliability driver is initialised. """
    _instance = None
    _initialized_once = False

    def __new__(cls, *args, **kwargs):
        """ __new__ method """
        if not isinstance(cls._instance, cls):
            cls._instance = object.__new__(cls, *args, **kwargs)
        return cls._instance


def validate_auth_username():
    """validate auth username"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'ac_auth_username'):
        raise ParamNotConfigured(para='ac_auth_username')
    if not cfg.CONF.huawei_ac_config.ac_auth_username:
        raise ParamNotConfigured(para='ac_auth_username')
    if len(cfg.CONF.huawei_ac_config.ac_auth_username) >= \
            ac_constants.USER_NAME_MAX_LEN:
        raise ParamValueInvalid(
            para='ac_auth_username',
            value=cfg.CONF.huawei_ac_config.ac_auth_username)


def validate_auth_password():
    """validate auth password """
    if not hasattr(cfg.CONF.huawei_ac_config, 'ac_auth_password'):
        raise ParamNotConfigured(para='ac_auth_password')
    if not cfg.CONF.huawei_ac_config.ac_auth_password:
        raise ParamNotConfigured(para='ac_auth_password')
    if len(cfg.CONF.huawei_ac_config.ac_auth_password) >= \
            ac_constants.USER_PASS_MAX_LEN:
        raise ParamValueInvalid(
            para='ac_auth_password',
            value=cfg.CONF.huawei_ac_config.ac_auth_password)


def validate_websocket_key_password():
    """validate websocket key password"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'websocket_key_password'):
        raise ParamNotConfigured(para='websocket_key_password')
    if not cfg.CONF.huawei_ac_config.websocket_key_password:
        raise ParamNotConfigured(para='websocket_key_password')
    if len(cfg.CONF.huawei_ac_config.websocket_key_password) >= \
            ac_constants.USER_PASS_MAX_LEN:
        raise ParamValueInvalid(
            para='websocket_key_password',
            value=cfg.CONF.huawei_ac_config.websocket_key_password)


def validate_cert_expiration_warning_time():
    """validate cert expiration warning time"""
    if not hasattr(cfg.CONF.huawei_ac_config,
                   'cert_expiration_warning_time'):
        raise ParamNotConfigured(para='cert_expiration_warning_time')
    if cfg.CONF.huawei_ac_config.cert_expiration_warning_time is None:
        raise ParamNotConfigured(para='cert_expiration_warning_time')
    if int(cfg.CONF.huawei_ac_config.cert_expiration_warning_time) <= 0 \
            or int(cfg.CONF.huawei_ac_config.cert_expiration_warning_time) \
            > sys.maxsize:
        raise ParamValueInvalid(
            para='cert_expiration_warning_time',
            value=cfg.CONF.huawei_ac_config.cert_expiration_warning_time)


def validate_cloud_name():
    """validate cloud name"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'cloud_name'):
        raise ParamNotConfigured(para='cloud_name')
    if not cfg.CONF.huawei_ac_config.cloud_name:
        raise ParamNotConfigured(para='cloud_name')
    if len(cfg.CONF.huawei_ac_config.cloud_name) >= \
            ac_constants.CLOUD_STR_MAX_LEN:
        raise ParamValueInvalid(
            para='cloud_name',
            value=cfg.CONF.huawei_ac_config.cloud_name)


def validate_ops_version():
    """validate ops version"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'OPS_version'):
        raise ParamNotConfigured(para='OPS_version')
    if not cfg.CONF.huawei_ac_config.OPS_version:
        raise ParamNotConfigured(para='OPS_version')
    if cfg.CONF.huawei_ac_config.OPS_version not in [
            ac_constants.OPS_K,
            ac_constants.OPS_L,
            ac_constants.OPS_M,
            ac_constants.OPS_N,
            ac_constants.OPS_O,
            ac_constants.OPS_P,
            ac_constants.OPS_Q,
            ac_constants.OPS_R,
            ac_constants.OPS_T,
            ac_constants.OPS_W,
            ac_constants.OPS_EZ_M,
            ac_constants.FSP_6_1,
            ac_constants.FSP_6_3_0,
            ac_constants.FSP_6_3_1,
            ac_constants.FSP_6_5,
            ac_constants.FSP_6_5_1,
            ac_constants.FSP_6_5_private,
            ac_constants.FSP_6_5_NFVI,
            ac_constants.FSP_8_0_0,
            ac_constants.FSP_8_0_3,
            ac_constants.FSP_21_0
    ]:
        raise ParamValueInvalid(
            para='OPS_version',
            value=cfg.CONF.huawei_ac_config.OPS_version)


def validate_physical_net():
    """validate physical net"""
    if not cfg.CONF.huawei_ac_config.physical_network:
        raise ParamNotConfigured(para='physical_network')


def validate_keystone_tenant():
    """validate keystone tenant"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'keystone_tenant'):
        raise ParamNotConfigured(para='keystone_tenant')
    if not cfg.CONF.huawei_ac_config.keystone_tenant:
        raise ParamNotConfigured(para='keystone_tenant')
    if len(cfg.CONF.huawei_ac_config.keystone_tenant) >= \
            ac_constants.USER_NAME_MAX_LEN:
        raise ParamValueInvalid(
            para='keystone_tenant',
            value=cfg.CONF.huawei_ac_config.keystone_tenant)


def validate_keystone_user():
    """validate keystone user"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'keystone_user'):
        raise ParamNotConfigured(para='keystone_user')
    if not cfg.CONF.huawei_ac_config.keystone_user:
        raise ParamNotConfigured(para='keystone_user')
    if len(cfg.CONF.huawei_ac_config.keystone_user) >= \
            ac_constants.USER_NAME_MAX_LEN:
        raise ParamValueInvalid(
            para='keystone_user',
            value=cfg.CONF.huawei_ac_config.keystone_user)


def validate_keystone_passwd():
    """ validate keystone passwd """
    if not hasattr(cfg.CONF.huawei_ac_config, 'keystone_passwd'):
        raise ParamNotConfigured(para='keystone_passwd')
    if not cfg.CONF.huawei_ac_config.keystone_passwd:
        raise ParamNotConfigured(para='keystone_passwd')
    if len(cfg.CONF.huawei_ac_config.keystone_passwd) >= \
            ac_constants.USER_PASS_MAX_LEN:
        raise ParamValueInvalid(
            para='keystone_passwd',
            value=cfg.CONF.huawei_ac_config.keystone_passwd)


def validate_auth_url():
    """validate auth url"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'auth_url'):
        raise ParamNotConfigured(para='auth_url')
    if not cfg.CONF.huawei_ac_config.auth_url:
        raise ParamNotConfigured(para='auth_url')


def validate_request_timeout():
    """validate request timeout"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'request_timeout'):
        raise ParamNotConfigured(para='request_timeout')
    if not cfg.CONF.huawei_ac_config.request_timeout:
        raise ParamNotConfigured(para='request_timeout')
    if (int(cfg.CONF.huawei_ac_config.request_timeout) < 30 or
            int(cfg.CONF.huawei_ac_config.request_timeout) > 600):
        raise ParamValueInvalid(
            para='request_timeout',
            value=cfg.CONF.huawei_ac_config.request_timeout)


def validate_timeout_retry():
    """validate timeout retry"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'timeout_retry'):
        raise ParamNotConfigured(para='timeout_retry')
    if cfg.CONF.huawei_ac_config.timeout_retry is None:
        raise ParamNotConfigured(para='timeout_retry')
    if (int(cfg.CONF.huawei_ac_config.timeout_retry) < 0 or
            int(cfg.CONF.huawei_ac_config.timeout_retry) > 10):
        raise ParamValueInvalid(
            para='timeout_retry',
            value=cfg.CONF.huawei_ac_config.timeout_retry)


def validate_token_retry():
    """validate token retry"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'token_retry'):
        raise ParamNotConfigured(para='token_retry')
    if not cfg.CONF.huawei_ac_config.token_retry:
        raise ParamNotConfigured(para='token_retry')
    if (int(cfg.CONF.huawei_ac_config.token_retry) < 1 or
            int(cfg.CONF.huawei_ac_config.token_retry) > 10):
        raise ParamValueInvalid(
            para='token_retry',
            value=cfg.CONF.huawei_ac_config.token_retry)


def validate_allocate_vlan_ranges():
    """validate allocate vlan ranges"""
    allocate_vlan_ranges = cfg.CONF.huawei_ac_config.allocate_vlan_ranges
    for vlan_range in allocate_vlan_ranges:
        if not re.match(r'\d+:\d+', vlan_range):
            raise ParamValueInvalid(
                para='allocate_vlan_ranges', value=allocate_vlan_ranges)
        for value in vlan_range.split(':'):
            if int(value) not in six.moves.range(2, 4095):
                raise ParamValueInvalid(para='allocate_vlan_ranges',
                                        value=allocate_vlan_ranges)


def validate_neutron_sync_type():
    """validate neutron sync type"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'neutron_sync_type'):
        raise ParamNotConfigured(para='neutron_sync_type')
    if not cfg.CONF.huawei_ac_config.neutron_sync_type:
        raise ParamNotConfigured(para='neutron_sync_type')
    if len(cfg.CONF.huawei_ac_config.neutron_sync_type) > \
            ac_constants.NEUTRON_SYNC_TYPE_STR_MAX_LEN:
        raise ParamValueInvalid(
            para='neutron_sync_type',
            value=cfg.CONF.huawei_ac_config.neutron_sync_type)
    if cfg.CONF.huawei_ac_config.neutron_sync_type \
            not in [ac_constants.NEUTRON_SYNC_NONE,
                    ac_constants.NEUTRON_SYNC_INTERVAL]:
        raise ParamValueInvalid(
            para='neutron_sync_type',
            value=cfg.CONF.huawei_ac_config.neutron_sync_type)


def validate_bare_metal_bond_mode():
    """validate bare metal bond mode"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'bare_metal_bond_mode'):
        raise ParamNotConfigured(para='bare_metal_bond_mode')
    if cfg.CONF.huawei_ac_config.bare_metal_bond_mode not in [
            ac_constants.BOND_MODE_0,
            ac_constants.BOND_MODE_1,
            ac_constants.BOND_MODE_4
    ]:
        raise ParamValueInvalid(
            para='bare_metal_bond_mode',
            value=cfg.CONF.huawei_ac_config.bare_metal_bond_mode)


def validate_sg_sync_times():
    """validate sg sync times"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'security_group_sync_times'):
        raise ParamNotConfigured(para='security_group_sync_times')
    if not cfg.CONF.huawei_ac_config.security_group_sync_times:
        raise ParamNotConfigured(para='security_group_sync_times')
    if (int(cfg.CONF.huawei_ac_config.security_group_sync_times) < 1 or
            int(cfg.CONF.huawei_ac_config.security_group_sync_times) > 10):
        raise ParamValueInvalid(
            para='security_group_sync_times',
            value=cfg.CONF.huawei_ac_config.security_group_sync_times)

    if cfg.CONF.huawei_ac_config.external_access_mode \
            not in [ac_constants.GLOBAL_SNAT, ac_constants.FLEX_SNAT]:
        raise ParamValueInvalid(
            para='external_access_mode',
            value=cfg.CONF.huawei_ac_config.external_access_mode)


def validate_network_list_matching():
    """validate network list matching"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'network_list_matching'):
        raise ParamNotConfigured(para='network_list_matching')
    if not cfg.CONF.huawei_ac_config.network_list_matching:
        raise ParamNotConfigured(para='network_list_matching')
    if cfg.CONF.huawei_ac_config.network_list_matching not in \
            ['prefix', 'suffix']:
        raise ParamValueInvalid(
            para='network_list_matching',
            value=cfg.CONF.huawei_ac_config.network_list_matching)


def validate_detect_period():
    """validate detect period"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'ac_detect_period'):
        raise ParamNotConfigured(para='ac_detect_period')
    if not cfg.CONF.huawei_ac_config.ac_detect_period:
        raise ParamNotConfigured(para='ac_detect_period')
    if (int(cfg.CONF.huawei_ac_config.ac_detect_period) < 60 or
            int(cfg.CONF.huawei_ac_config.ac_detect_period) > 300):
        raise ParamValueInvalid(
            para='ac_detect_period',
            value=cfg.CONF.huawei_ac_config.ac_detect_period)


def validate_neutron_sync_time():
    """validate neutron sync time"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'neutron_sync_time'):
        raise ParamNotConfigured(para='neutron_sync_time')
    if not cfg.CONF.huawei_ac_config.neutron_sync_time:
        raise ParamNotConfigured(para='neutron_sync_time')
    try:
        date_d = datetime.datetime.strptime(
            cfg.CONF.huawei_ac_config.neutron_sync_time, '%H:%M:%S')
    except ValueError:
        try:
            date_d = datetime.datetime.strptime(
                cfg.CONF.huawei_ac_config.neutron_sync_time,
                '%a %H:%M:%S')
        except ValueError:
            try:
                date_d = datetime.datetime.strptime(
                    cfg.CONF.huawei_ac_config.neutron_sync_time,
                    '%A %H:%M:%S')
            except ValueError:
                raise ParamValueInvalid(
                    para='neutron_sync_time',
                    value=cfg.CONF.huawei_ac_config.neutron_sync_time)
    if date_d.second >= 60:
        raise ParamValueInvalid(
            para='neutron_sync_time',
            value=cfg.CONF.huawei_ac_config.neutron_sync_time)


def validate_error_retry_count():
    """validate error retry count"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'error_retry_count'):
        raise ParamNotConfigured(para='error_retry_count')
    if not cfg.CONF.huawei_ac_config.error_retry_count and \
            cfg.CONF.huawei_ac_config.error_retry_count != 0:
        raise ParamNotConfigured(para='error_retry_count')
    if (int(cfg.CONF.huawei_ac_config.error_retry_count) < 0 or
            int(cfg.CONF.huawei_ac_config.error_retry_count) > 3):
        raise ParamValueInvalid(
            para='error_retry_count',
            value=cfg.CONF.huawei_ac_config.error_retry_count)


def validate_error_retry_interval():
    """validate error retry interval"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'error_retry_interval'):
        raise ParamNotConfigured(para='error_retry_interval')
    if not cfg.CONF.huawei_ac_config.error_retry_interval:
        raise ParamNotConfigured(para='error_retry_interval')
    if (int(cfg.CONF.huawei_ac_config.error_retry_interval) < 1 or
            int(cfg.CONF.huawei_ac_config.error_retry_interval) > 180):
        raise ParamValueInvalid(
            para='error_retry_interval',
            value=cfg.CONF.huawei_ac_config.error_retry_interval)


def validate_response_time():
    """validate response time"""
    if not hasattr(cfg.CONF.huawei_ac_config, 'ac_response_time'):
        raise ParamNotConfigured(para='ac_response_time')
    if not cfg.CONF.huawei_ac_config.ac_response_time:
        raise ParamNotConfigured(para='ac_response_time')
    if (int(cfg.CONF.huawei_ac_config.ac_response_time) < 60 or
            int(cfg.CONF.huawei_ac_config.ac_response_time) > 1200):
        raise ParamValueInvalid(
            para='ac_response_time',
            value=cfg.CONF.huawei_ac_config.ac_response_time)


class ACMessageReliabilityDriver(Singleton):
    """ ac message reliability driver """

    def __init__(self):
        """init method"""
        # If already initialized once in one process, no need again
        if Singleton._initialized_once is True:
            return
        # Initialize the consistency normal sync frame work and thread.
        LOG.info(_LI("Huawei AC plugin consistency normal sync initializing."))
        Singleton._initialized_once = True
        ACMessageReliabilityDriver._config_validation()
        if ncu.SUPPORT_VERIFY_CERT:
            ACMessageReliabilityDriver.reload_cert()
            ACMessageReliabilityDriver.check_cert_warning_time_cfg()
        self.neutron_name = ACCommonUtil.get_neutron_server_name()
        self.ac_db_interface = ACdbInterface()
        self._init_server_record()
        ACNormalSync()
        http_heart_beat_obj = http_heart_beat.HttpHeatBeat2AC()
        http_heart_beat_obj.init_http_heart_beat()
        ACCertSync()
        ACFailedResourceSync()
        WebsocketMsgSync()
        CloudAlarmSync()
        self.maintenance = control_task.ACMaintenance()
        self.control_thread = self._start_control_thread()
        LOG.info(_LI("Huawei AC plugin consistency normal sync initialization "
                     "success."))

    def _start_control_thread(self):
        """start control thread"""
        control = control_task.ACCommonControlTask()
        control.register_non_looping_control_tasks(
            self.maintenance.handle_leader_selection,
            ac_constants.LEADER_SELECTION_INTERVAL)
        control.register_looping_control_tasks(
            self.maintenance.handle_server_status_update,
            ac_constants.ACCESS_TIME_UPDATE)
        control.register_looping_control_tasks(
            self.maintenance.handle_long_pending_in_process_records,
            ac_constants.LONG_PENDING_IN_PROCESS)
        control.register_looping_control_tasks(
            self.maintenance.handle_server_alive_check,
            ac_constants.SERVER_ALIVE_CHECK_INTERVAL)
        control.register_looping_control_tasks(
            self.maintenance.handle_segments_records,
            ac_constants.CLEAN_INVALID_SEGMENTS_INTERVAL)
        control.register_non_looping_control_tasks(
            self.maintenance.handle_neutron_sync_at_reboot_after_leader_sel,
            ac_constants.NEUTRON_SYNC_AT_REBOOT)
        control.start()
        return control

    @staticmethod
    def _config_validation():
        """ config validation"""
        validate_auth_username()
        validate_auth_password()
        validate_websocket_key_password()
        if ncu.SUPPORT_VERIFY_CERT:
            validate_cert_expiration_warning_time()
        validate_cloud_name()
        validate_ops_version()
        validate_physical_net()
        validate_keystone_tenant()
        validate_keystone_user()
        validate_keystone_passwd()
        validate_auth_url()
        validate_request_timeout()
        validate_timeout_retry()
        validate_token_retry()
        validate_response_time()
        validate_error_retry_interval()
        validate_error_retry_count()
        validate_neutron_sync_time()
        validate_detect_period()
        validate_network_list_matching()
        validate_sg_sync_times()
        validate_bare_metal_bond_mode()
        validate_rpc_server_ip()
        validate_neutron_sync_type()
        validate_allocate_vlan_ranges()

    def _init_server_record(self):
        """ Create or update the server record and create dummy-record
        """

        server_state = None
        session = self.ac_db_interface.get_session('write')
        server = self.ac_db_interface.get_server_record(
            session, neutron_name=self.neutron_name)

        if server:
            if server.state not in [ac_constants.NEUTRON_SYNC,
                                    ac_constants.CONSISTENCY_CHECK]:
                server_state = ac_constants.NORMAL

            self.ac_db_interface.update_server_record(ACNeutronStateSchema(
                neutron_id=server.neutron_id, create_time=utcnow(),
                sync_type=cfg.CONF.huawei_ac_config.neutron_sync_type,
                state=server_state))
        else:
            self.ac_db_interface.create_server_record()

    @staticmethod
    def check_cert_warning_time_cfg():
        """ check cert warning time cfg """
        if not ncu.get_neutron_ext_param(CERT_EXPIRED_WARNING_TIME_CFG):
            cert_warning_time_cfg = \
                {CERT_EXPIRED_WARNING_TIME_CFG: CERT_EXPIRED_WARNING_TIME}
            ncu.add_neutron_ext_param(cert_warning_time_cfg)

    @staticmethod
    def reload_cert():
        """reload cert"""
        dir_path = os.path.dirname(os.path.realpath(__file__))
        config_cert = {'https_ca_cert': '../client/ssl_cacert.pem',
                       'websocket_ca_cert':
                           '../ac_agent/rpc/websocket/trust.cer',
                       'websocket_public_key':
                           '../ac_agent/rpc/websocket/client.cer',
                       'websocket_private_key':
                           '../ac_agent/rpc/websocket/client_key.pem'
                      }
        for key, value in six.iteritems(config_cert):
            if not hasattr(cfg.CONF.huawei_ac_config, key) \
                    or not cfg.CONF.huawei_ac_config[key]:
                LOG.info(
                    "[AC]user has no config for %s, use default" % key)
            else:
                cert_file = os.path.join(dir_path, value)
                conf_item = cfg.CONF.huawei_ac_config[key]
                flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
                with os.fdopen(os.open(cert_file, flags, 0o600), 'w') as cert:
                    cert.write(conf_item)
        LOG.info("[AC]reload cert finished")
